package org.postgresforest;

import java.sql.*;
import java.util.*;
import java.util.concurrent.*;

import net.jcip.annotations.*;

import org.postgresforest.apibase.*;
import org.postgresforest.exception.*;
import org.postgresforest.util.*;

@NotThreadSafe public class Driver implements java.sql.Driver {
    
    /**
     * Class.forName時にDriverManagerに登録するためのstatic初期化ブロック
     */
    static {
        try {
            Class.forName("org.postgresql.Driver");
            java.sql.DriverManager.registerDriver(new org.postgresforest.Driver());
        } catch (SQLException e) {
            e.printStackTrace();
        } catch (ClassNotFoundException e) {
            e.printStackTrace();
        }
    }
    
    public boolean acceptsURL(String arg0) {
        try {
            ForestUrl.parseUrl(arg0);
            return true;
        } catch (ForestException e) {
            return false;
        }
    }
    
    public Connection connect(String arg0, Properties arg1) throws SQLException {
        final Pair2<ForestUrl, Properties> pair;
        try {
             pair = ForestUrl.parseUrl(arg0);
        } catch (NonForestUrlException e) {
            // ForestのURLと認識できない場合、nullを返す（JDBCの仕様参照）
            // これ以外の例外が発生した場合は、そのままユーザ側に返す
            return null;
        }
        final ForestUrl forestUrl = pair.getFirst();
        
        // EntrypointCommonResourceのコンストラクタに渡すoptionを作る。
        // 接続文字列中に含まれていたオプション列（ForestUrl.parseUrlで取得済み）と
        // Driver.connectの引数として与えられたオプション列とを合成する。
        // その上で、このプロパティ中に、user/passwordが含まれていることを確認する
        final Properties newProp = pair.getSecond();
        for (final Object key : Collections.list(arg1.propertyNames())) {
            final Object val = arg1.getProperty(key.toString());
            if (val != null) {
                newProp.setProperty(key.toString(), val.toString());
            }
        }
        // ForestのJDBCエントリポイント（ForestConnection）と、このエントリポイントを
        // コントロールするオブジェクトを生成し、相互に関連付ける。
        final EntrypointCommonResource epResource = new EntrypointCommonResource(forestUrl, newProp);
        try {
            epResource.init();
            final List<PgUrl> pgUrlList = epResource.getUdbPgUrlList();
            final Callable<Connection> task0 = new DriverTask.Connect(pgUrlList.get(0), (Properties) newProp.clone());
            final Callable<Connection> task1 = new DriverTask.Connect(pgUrlList.get(1), (Properties) newProp.clone());
            final Callable<Connection> dummy = new DriverTask.Connect(null, null);
            // DriverクラスからConnectionを生成する際は、縮退している系のタスクを
            // ダミータスクとして用意しておく。このダミータスクは、何も行わない。
            // 通常の他のAPIであれば、executeXxxApiの内部で縮退状態などをチェックし、
            // 縮退と判断されればその系のリソースを解放してしまうためチェック関数の中で
            // 使用する対象のリソース解放を行っているため、そのまま実行して問題ないが
            // コネクションの生成に関しては全く新しいリソースから新規に生成するため、
            // API呼び出しの前の段階で縮退のチェックを行い、API実行を行わざるを得ない
            final List<Connection> conList = epResource.executeAllApiWithPreCheck(task0, task1, dummy);
            final ForestConnection con = new ForestConnection(epResource, conList);
            epResource.setTargetConnection(con);
            return con;
        } catch (SQLException e) {
            // EntrypointCommonResourceクラスの作成に成功した場合、必ずリリースが必要。
            // ここで例外が発生せず解放しなかった場合は、ForestConnectionクラスに解放を任せる。
            epResource.releaseEntryPointCommonResource();
            throw e;
        } catch (RuntimeException e) {
            epResource.releaseEntryPointCommonResource();
            throw e;
        }
    }
    
    public int getMajorVersion() {
        return 0;
    }
    
    public int getMinorVersion() {
        return 0;
    }
    
    public boolean jdbcCompliant() {
        return false;
    }
    
    public DriverPropertyInfo[] getPropertyInfo(String url, Properties info)
            throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
}
