/*
                                                                                                                                                                 
Copyright (C) 2006 NTT DATA Corporation
                                                                                                                                                                 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.
                                                                                                                                                                 
This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
                                                                                                                                                                 
*/

package com.clustercontrol.performance.util;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Iterator;

import javax.ejb.CreateException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.naming.NamingException;

import com.clustercontrol.bean.FacilityConstant;
import com.clustercontrol.bean.FacilityInfo;
import com.clustercontrol.bean.FacilityTreeItem;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorScopeRelationData;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorScopeRelationLocal;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorScopeRelationUtil;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorScopeSnapLocal;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorScopeSnapUtil;

/**
 * DBに対してのファシリティツリー情報の読み書きを実行するクラス
 * （情報の参照先、保存先がDBであり、LDAPではない）
 *
 */
public class FacilityTreeDB {

	/**
	 * スコープツリー情報をDBに書き込みます。
	 * 
	 * @param collectorID 収集ID
	 * @param fItem ファシリティのツリー情報
	 */
	public void insertFacilityTree(String collectorID, FacilityTreeItem fItem) {
		insertFacilityTree(collectorID, fItem, null);
	}
	
	/**
	 * スコープツリー情報をDBに書き込みます。
	 * 
	 * @param collectorID 収集ID
	 * @param fItem ファシリティのツリー情報
	 * @param writtenSet 既にDBに書き込み済みのファシリティIDのセット。(初回呼出しの際はnullを与える)
	 */
	private void insertFacilityTree(String collectorId, FacilityTreeItem fItem, HashSet<String> writtenSet) {
//		ScopeRelationDAO srdao =  new ScopeRelationDAO();
//		ScopeSnapDAO sndao = new ScopeSnapDAO();
		// 重複してDB書き込みをしないために、書き込み済みのものを管理
		if(writtenSet == null){
			writtenSet = new HashSet<String>(); 
		}
		
		FacilityInfo fInfo = fItem.getData(); //ScopeSnapDAOに渡します。
		FacilityTreeItem[] fArray = fItem.getChildren(); //子ファシリティを取得
		
		CollectorScopeRelationData  srData = new CollectorScopeRelationData();
		srData.setCollectorId(collectorId);
		
		if(fItem.getParent() != null ){
			//親があればrelationのDTOにFacilityIDを代入
			srData.setParentFacilityId(fItem.getParent().getData().getFacilityId());
		}else{
			//無ければ空文字を代入
			srData.setParentFacilityId("");
		}
		//自分のFacilityIDを代入
		srData.setFacilityId(fInfo.getFacilityId());
		
		try {
			// まだ書き込まれていないファシリティの場合はDBに書き込む
			if(writtenSet.contains(fInfo.getFacilityId()) == false){
//				sndao.insertScopeSnap(collectorID, fInfo);
				
				int typeInt = fItem.getData().getType();
				String typeStr = null;
				String platform = null;
				
				if(typeInt == FacilityConstant.TYPE_NODE){
					typeStr = "node";
					platform = Repository.getPlatformId(fItem.getData().getFacilityId());
				} else if (typeInt == FacilityConstant.TYPE_SCOPE){
					typeStr = "scope";
					platform = "SCOPE";  // プラットフォームは定義できない
				} else {
					typeStr = "comp";
					platform = "COMP";  // プラットフォームは定義できない
				}
				
				CollectorScopeSnapUtil.getLocalHome().create(
						collectorId, 
						fItem.getData().getFacilityId(), 
						platform, 
						fItem.getData().getFacilityName(), 
						typeStr);
				writtenSet.add(fInfo.getFacilityId());
			}
			// ファシリティ間の関係をDBに書き込む
//			srdao.insertScopeRelation(srData);
			CollectorScopeRelationUtil.getLocalHome().create(
					collectorId, 
					srData.getFacilityId(), 
					srData.getParentFacilityId());
		} catch (CreateException e) {
			throw new EJBException(e);
		} catch (NamingException e){
			throw new EJBException(e);
		}
		
		for(int i= 0; i < fArray.length;i++){
			insertFacilityTree(collectorId, fArray[i], writtenSet);
		}
	}
	
	/**
	 * ファシリティツリーを取得します。
	 * @return facilityTree ファシリティツリー
	 * 
	 * @ejb.interface-method
	 *  view-type="both"
	 * 
	 * @ejb.persistence 
	 */
	public FacilityTreeItem getFacilityTree(String collectorId) {
	
		try {
			/*
			 * テンポラリー領域をhtとして、ツリー構造を構成します。 親子関係を持つ状態をhtに格納していき、rootFalicityから
			 * 連結された状態を返り値とします。
			 */
			// テンポラリーのハッシュテーブル
			Hashtable<String, FacilityTreeItem> ht = new Hashtable<String, FacilityTreeItem>();
			
			HashMap<String, FacilityInfo> sncol = new HashMap<String, FacilityInfo>();
			Collection<CollectorScopeRelationLocal> srcol = null;

			// DAOと結果セット
			FacilityTreeItem currentItem = null;
			FacilityTreeItem parentItem = null;

			CollectorScopeRelationLocal srData = null;
			String current = null;
			String parent = null;

			Collection<CollectorScopeSnapLocal> col = CollectorScopeSnapUtil
					.getLocalHome().findByCollectorId(collectorId);
			Iterator<CollectorScopeSnapLocal> itr = col.iterator();

			while (itr.hasNext()) {
				CollectorScopeSnapLocal bean = itr.next();

				FacilityInfo finfo = new FacilityInfo();
				finfo.setFacilityId(bean.getFacilityId());
				finfo.setFacilityName(bean.getFacilityName());
				
				// FIXME
				// DBでのスコープ情報には無効なノードは含まれないため、常に有効にする。
				finfo.setValid(true);

				String typeStr = bean.getFacilityType();
				int typeInt = 0;
				if (typeStr.equals("node")) {
					typeInt = FacilityConstant.TYPE_NODE;
				} else if (typeStr.equals("scope")) {
					typeInt = FacilityConstant.TYPE_SCOPE;
				} else if (typeStr.equals("comp")) {
					typeInt = FacilityConstant.TYPE_COMPOSITE;
				}
				finfo.setType(typeInt);

				sncol.put(bean.getFacilityId(), finfo);
			}

			srcol = CollectorScopeRelationUtil.getLocalHome()
					.findByCollectorId(collectorId);

			String topFacility = null; // 最上位のファシリティ(コンポジット)

			if (sncol.isEmpty() || srcol.isEmpty()) {
			} else {
				Iterator<CollectorScopeRelationLocal> it = srcol.iterator();

				while (it.hasNext()) {
					srData = it.next();

					// 自分と親のファシリティIDを取得
					current = srData.getFacilityId();
					parent = srData.getParentFacilityId();

					/*
					 * ツリーを構成するために以下の操作を行う 1. 親、自分のインスタンスを生成 2.自分setParent,
					 * 親addChildrenで繋ぐ
					 */

					// 自分の型がコンポジットでない場合は、親が存在するため、それを作成する
					if (((FacilityInfo) sncol.get(current)).getType() != FacilityConstant.TYPE_COMPOSITE) {
						if (ht.containsKey(parent)) {
							// 親が存在したらロードしておく
							parentItem = (FacilityTreeItem) ht.get(parent);
						} else {
							// 親が存在しないときには、取り合えず親を作成(その親はnullとしておく)
							parentItem = new FacilityTreeItem(
									(FacilityTreeItem) null,
									(FacilityInfo) sncol.get(parent));
							ht.put(parent, parentItem);
						}
					} else {
						// 自分の型がコンポジットの場合は、親は存在しない
						parentItem = null;
						topFacility = ((FacilityInfo) sncol.get(current))
								.getFacilityId();
					}

					// 自分のインスタンスを生成し親と繋ぐ
					if (ht.containsKey(current)) {
						// 既にインスタンスが生成されている場合は、setParent、addChildrenで繋ぐ
						currentItem = (FacilityTreeItem) ht.get(current);

						// 既に親が存在するノードの場合は、そのインスタンスに新たに親を関連づけると、
						// 既に登録されていた親との関係が切れてしまうため、新規にインスタンスを作成し、
						// そのインスタンスを登録する。
						if (currentItem.getData().getType() == FacilityConstant.TYPE_NODE
								&& currentItem.getParent() != null) {
							FacilityInfo info = new FacilityInfo();
							info.setFacilityId(currentItem.getData()
									.getFacilityId());
							info.setFacilityName(currentItem.getData()
									.getFacilityName());
							info.setType(currentItem.getData().getType());
							// 親を設定してインスタンスを生成する(同時に親スコープの子として自分が登録される)
							currentItem = new FacilityTreeItem(parentItem, info);
						} else {
							// まだ親が登録されていないファシリティ(スコープもしくはノード)の場合は、親を設定する。
							currentItem.setParent(parentItem);

							// 親のスコープの子として自分を登録する
							parentItem.addChildren(currentItem);
						}
					} else {
						// 自分が存在しなければ生成する
						currentItem = new FacilityTreeItem(parentItem,
								(FacilityInfo) sncol.get(current));

						ht.put(current, currentItem);
					}
				}
			}
			// 取得したファシリティツリーをコンポジットアイテムに繋ぐ
			FacilityTreeItem top = (FacilityTreeItem) ht.get(topFacility);

			return top;
		} catch (NamingException e) {
		} catch (FinderException e) {
		}
		return null;
	}
}
