/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 */

package com.clustercontrol.performance.dao;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.sql.Types;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import javax.ejb.EJBException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.commons.util.ConnectionManager;
import com.clustercontrol.performance.bean.CollectedDataInfo;

/**
 * 収集した性能値をDBに保存、読み出しするDAOクラス
 * 
 * 
 * @version 2.0.0
 * @since 1.0
 */
public class CalculatedDataDAO {
	
//	ログ出力
	protected static Log m_log = LogFactory.getLog(CalculatedDataDAO.class);
	
	/**
	 * コンストラクター
	 */
	public CalculatedDataDAO(){
	}
	
	/**
	 * 収集された性能値を収集ID、SNMPのOID、ファシリティID、期間（開始の時刻、終了の時刻）で検索します。
	 *
	 * 
	 * @param collectorID 収集ID
	 * @param collectMethod 収集方法
	 * @param itemCode　収集項目コード
	 * @param deviceName  デバイス名
	 * @param facilityID ファシリティID
	 * @param startDate (検索の）開始時刻
	 * @param stopDate　（検索の）終了時刻
	 * @return　収集された性能値のコレクション
	 * @throws EJBException
	 */
	public List<CollectedDataInfo> select(
			String collectorID, 
			String collectMethod,
			String itemCode,
			String deviceName,
			String facilityID,
			Date startDate,
			Date stopDate )
	throws EJBException{
		
		m_log.debug("select() start :" + collectorID + " " + itemCode + " " + deviceName + " " + facilityID);
		ArrayList<CollectedDataInfo> ret = new ArrayList<CollectedDataInfo>();  // DBから取得したデータを格納するリスト
		
		Connection conn = null;
		PreparedStatement stmt =null;
		ResultSet res = null; 
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			
			//		SQL文の作成
			String sql = "SELECT * FROM CC_CALCULATED_DATA WHERE " + 
			"COLLECTORID=? AND " +
			"COLLECT_METHOD=? AND " +
			"ITEM_CODE=? AND " + 
			"DEVICE_NAME=? AND " + 
			"FACILITYID=? AND " + 
			"DATE_TIME BETWEEN ? AND ?" +
			"ORDER BY DATE_TIME";
			
			// SQL文発行準備
			stmt = conn.prepareStatement(sql);
			
			Timestamp start = new Timestamp(startDate.getTime());
			Timestamp stop  = new Timestamp(stopDate.getTime());
			
			stmt.setString(1, collectorID);
			stmt.setString(2, collectMethod);
			stmt.setString(3, itemCode);
			stmt.setString(4, deviceName);
			stmt.setString(5, facilityID);
			stmt.setTimestamp(6, start);
			stmt.setTimestamp(7, stop);
			
			res = stmt.executeQuery();
			
			while(res.next()) {
				Date d = new Date(res.getTimestamp("DATE_TIME").getTime());
				CollectedDataInfo data;
				if(res.getObject("VALUE") != null){
					data = new CollectedDataInfo(facilityID, itemCode, deviceName, collectMethod, d, res.getDouble("VALUE"));
				} else {
					data = new CollectedDataInfo(facilityID, itemCode, deviceName, collectMethod, d, Double.NaN);
				}
				
//				CollectedDataInfo data = new CollectedDataInfo(facilityID, itemCode, deviceName, collectMethod);
//				data.setDate(new Date(res.getTimestamp("DATE_TIME").getTime()));
//				
//				if(res.getObject("VALUE") != null){	
//					data.setValue(res.getDouble("VALUE"));
//				} else {
//					data.setValue(Double.NaN);  // DBにnullが入っている場合は性能値が算出不能だった場合
//				}
				ret.add(data);
			}
		} catch (SQLException e) {
			m_log.error("select() error :" + facilityID + " EJBException ",e);
			throw new EJBException(e.getMessage());
		}  finally{
			
			//コネクション、結果セット、プリペアドステートメントのクロース
			try {
				if(res != null){
					res.close();
				}
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("select() error :" + facilityID + " EJBException ");
				throw new EJBException(e1.getMessage());
			}
		}	
		m_log.debug("select() end   :" + collectorID + " " + itemCode + " " + deviceName + " " + facilityID);
		
		return ret;
	}
	
	/**
	 * 実績収集結果を収集ID、ファシリティID、検索期間（開始日時および終了日時）の組み合わせで検索し、収集性能値DTOクラスの配列として返す。
	 * 
	 * @param collectorID
	 *            収集ID
	 * @param facilityID
	 *            ファシリティID
	 * @param startDate
	 *            検索対象となる開始日時
	 * @param stopDate
	 *            検索対象となる終了日時
	 * @return 該当する収集性能値DTOクラスの配列（順序は収集日時順とは限らない）
	 * @throws EJBException
	 * 
	 * @author NTT DATA takahatat
	 */
	public ArrayList<CollectedDataInfo> select(String collectorID, String facilityID, Date startDate, Date stopDate)
			throws EJBException {

		m_log.debug("select() start :" + collectorID + " " + facilityID);
		ArrayList<CollectedDataInfo> ret = new ArrayList<CollectedDataInfo>();
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();

			// SQL文の作成
			String sql = "SELECT date_time, item_code, device_name, collect_method, value FROM cc_calculated_data WHERE collectorid = ? AND facilityid = ? AND date_time >= ? AND date_time < ?";

			// SQL文発行準備
			stmt = conn.prepareStatement(sql);

			Timestamp startTimestamp = new Timestamp(startDate.getTime());
			Timestamp stopTimestamp = new Timestamp(stopDate.getTime());

			stmt.setString(1, collectorID);
			stmt.setString(2, facilityID);
			stmt.setTimestamp(3, startTimestamp);
			stmt.setTimestamp(4, stopTimestamp);

			res = stmt.executeQuery();

			// 配列への性能値の格納
			while (res.next()) {
				if (res.getObject("VALUE") != null) {
					ret.add(new CollectedDataInfo(facilityID, res.getString("ITEM_CODE"), res.getString("DEVICE_NAME"), res.getString("COLLECT_METHOD"), res.getTimestamp("DATE_TIME"), res.getDouble("VALUE")));
				}
			}
		} catch (SQLException e) {
			m_log.error("select() error :" + collectorID + " " + facilityID + " EJBException ", e);
			throw new EJBException(e.getMessage());
		} finally {

			// コネクション、結果セット、プリペアドステートメントのクロース
			try {
				if (res != null) {
					res.close();
				}
				if (stmt != null) {
					stmt.close();
				}
				if (conn != null) {
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("select() error :" + collectorID + " " + facilityID + " EJBException ");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("select() end   :" + collectorID + " " + facilityID);

		return ret;
	}
	
	/**
	 * 収集した性能値を収集IDで削除します。
	 * 
	 * 収集設定に紐付けられた性能値を一括で削除します。
	 * 
	 * @param collectorID　収集ID
	 * @throws SQLException
	 */
	public void delete(String collectorID)
	throws EJBException{
		
		m_log.debug("delete() start :" + collectorID );

		Connection conn = null;
		PreparedStatement stmt = null;
		
		try{
			conn = ConnectionManager.getConnectionManager().getConnection();
			//conn.setAutoCommit(false);
			//SQL文の定義
			String sql = "DELETE FROM CC_CALCULATED_DATA WHERE COLLECTORID = ? ";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1,collectorID);
			
			stmt.executeUpdate();
		} catch (SQLException e) {
			m_log.error("delete() error :" + collectorID + " EJBException ");
			throw new EJBException(e.getMessage());
		}  finally{
			
			//コネクション、結果セット、プリペアドステートメントのクロース
			try {
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("delete() error :" + collectorID + " EJBException ");
				throw new EJBException(e1.getMessage());
			}
			m_log.debug("delete() end   :" + collectorID );
		}	
	}
	
	/**
	 * 収集した性能値をインサートします。
	 * 
	 * @param collectorID　収集ID
	 * @param data 計算済み性能値情報
	 * @throws SQLException
	 */
	public void insert(String collectorID, CollectedDataInfo data)
	throws EJBException{
		
		m_log.debug("insert() start : " + collectorID +
				" " + data.getCollectMethod() +
				" " + data.getItemCode() +
				" " + data.getDeviceName() +
				" " + data.getFacilityId());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		Timestamp time = new Timestamp(data.getDate().getTime());
		
		try{
			conn = ConnectionManager.getConnectionManager().getConnection();
			//SQL文の定義
			String sql = "INSERT INTO CC_CALCULATED_DATA " +
			"(COLLECTORID, " +
			"COLLECT_METHOD, " +
			"ITEM_CODE, " +
			"DEVICE_NAME, " +
			"DATE_TIME, " +
			"FACILITYID, " +
			"VALUE) " +
			"values (?,?,?,?,?,?,?);";
			
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1, collectorID);
			stmt.setString(2, data.getCollectMethod());
			stmt.setString(3, data.getItemCode());
			stmt.setString(4, data.getDeviceName());
			stmt.setTimestamp(5, time);
			stmt.setString(6, data.getFacilityId());
			if(Double.isNaN(data.getValue())){
				stmt.setNull(7, Types.DOUBLE);  // 性能値が算出不能だった場合はnullを書き込む
			} else {
				stmt.setDouble(7, data.getValue());
			}
			
			int row = stmt.executeUpdate();
			
			if (row != 1) {
				String msg = "result row is not 1";
				throw new SQLException(msg);
			}
		} catch (SQLException e) {
			m_log.error("insert() error :" + collectorID + " EJBException ", e);
			throw new EJBException(e.getMessage());
		}  finally{
			
			//コネクション、結果セット、プリペアドステートメントのクロース
			try {
				if(stmt != null){
					stmt.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("insert() error :" + collectorID + " EJBException ", e1);
				throw new EJBException(e1.getMessage());
			}
			m_log.debug("insert() end   : " + collectorID +
					" " + data.getCollectMethod() +
					" " + data.getItemCode() +
					" " + data.getDeviceName() +
					" " + data.getFacilityId());
		}	
	}
	/**
	 * 保存期間を過ぎた実績値を削除します。
	 * 
	 * @param collectorID 収集ID
	 * @param startDate 開始時刻
	 * @throws EJBException
	 */
	public void deleteOverPresave(String collectorID , Date startDate)
		throws EJBException{
			
			m_log.debug("deleteOrverPresave() start :" + collectorID + " " + startDate.toString());

			Connection conn = null;
			PreparedStatement stmt = null;
			Timestamp time = new Timestamp(startDate.getTime());
			
			try{
				conn = ConnectionManager.getConnectionManager().getConnection();
				//conn.setAutoCommit(false);
				//SQL文の定義
				String sql = "DELETE FROM CC_CALCULATED_DATA WHERE COLLECTORID = ? AND DATE_TIME < ?";
				//SQL文のセット
				stmt = conn.prepareStatement(sql);
				
				stmt.setString(1,collectorID );
				stmt.setTimestamp(2,time);
				
				stmt.executeUpdate();
			} catch (SQLException e) {
				m_log.debug("deleteOrverPresave() error :" + collectorID + " " + startDate.toString() + "  EJBException ");

				throw new EJBException(e.getMessage());
			}  finally{
				
				//コネクション、結果セット、プリペアドステートメントのクロース
				try {
					if(stmt != null){
						stmt.close();
					}
					if(conn != null){
						conn.close();
					}
				} catch (SQLException e1) {
					m_log.debug("deleteOrverPresave() error :" + collectorID + " " + startDate.toString() + "  EJBException ");
					throw new EJBException(e1.getMessage());
				}
				m_log.debug("deleteOrverPresave() end :" + collectorID + " " + startDate.toString() + "  EJBException ");
			}	
	}

/**
 * 収集回数を再カウントします。
 * 
 * @param collectorID
 * @return
 * @throws EJBException
 */
	public int recountFromDB(String collectorID)throws EJBException{
		
		m_log.debug("recountFromDB() start :" + collectorID );
		
		Connection conn = null;
		PreparedStatement stmt =null;
		ResultSet res = null; 
		int ret = 0;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			
			//		SQL文の作成
			String sql = "SELECT COUNT(*) AS COLLECT_COUNT FROM CC_calculated_data " +
			"WHERE collectorid =? group by item_code , facilityid limit 1";
			
			// SQL文発行準備
			stmt = conn.prepareStatement(sql);
			
			
			stmt.setString(1,collectorID);
			
			res = stmt.executeQuery();
			
			if(res.next()){
				ret = res.getInt("COLLECT_COUNT");
			}
		} catch (SQLException e) {
			m_log.error("recountFromDB() error :" + collectorID + " EJBException ",e);
			throw new EJBException(e.getMessage());
		}  finally{
			
			//コネクション、結果セット、プリペアドステートメントのクロース
			try {
				if(res != null){
					res.close();
				}
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("recountFromDB() error :" + collectorID + " EJBException ");
				throw new EJBException(e1.getMessage());
			}
		}	
		m_log.debug("recountFromDB() end   :" + collectorID + " ");
		
		return ret;
	}
}
