/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 */

package com.clustercontrol.performance.action;

import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.events.SelectionAdapter;

import com.clustercontrol.bean.FacilityTreeItem;
import com.clustercontrol.performance.bean.GraphDisplayRange;
import com.clustercontrol.performance.bean.GraphProperty;
import com.clustercontrol.performance.composite.RecordGraphComposite;
import com.clustercontrol.performance.util.CollectorItemCodeFactory;
import com.clustercontrol.performance.util.Messages;
import com.clustercontrol.performance.bean.CollectedDataInfo;
import com.clustercontrol.performance.bean.CollectedDataSet;
import com.clustercontrol.performance.bean.CollectorItemInfo;
import com.clustercontrol.performance.ejb.bmp.RecordCollectorData;

/**
 * 実績グラフを描画するためのアクションクラス
 * 
 * @version 1.0
 * @since 1.0
 *  
 */
public class DrawRecordGraphAction extends SelectionAdapter {
    RecordGraphComposite m_composite;

    // 既にマネージャから取得済みのデータを保持する

    private int m_graphType;

    private RecordCollectorData m_collectorData;

    private List<CollectorItemInfo> m_itemInfoList;

//    private FacilityTreeItem m_facilityTree;
    
    // 表示対象収集項目
    private CollectorItemInfo m_currentItem;

    // 表示対象ファシリティ
    private FacilityTreeItem m_targetFacility;
    
    private long m_graphStartTime;
    private long m_graphEndTime;

    // 既に描画されている範囲をリストで保持する
    private ArrayList<GraphDisplayRange> m_gotTimeArea;

    /**
     * 新しい DrawRecordGraphAction オブジェクトを割り当てます。
     *
     */
    public DrawRecordGraphAction() {
        this.m_gotTimeArea = new ArrayList<GraphDisplayRange>();
    }

    /**
     * 新しい DrawRecordGraphAction オブジェクトを割り当てます。
     * 
     * @param collectorData 実績収集の収集定義
     * @param itemInfoList 収集項目のリスト
     */
    public DrawRecordGraphAction(RecordCollectorData collectorData, List<CollectorItemInfo> itemInfoList) {
        this.m_collectorData = collectorData;
        this.m_itemInfoList = itemInfoList;
    }

    /**
     * 呼び出し元となるグラフコンポジットを設定します。
     * 
     * @param composite 呼び出し元となるコンポジット
     */
    public void setComposite(RecordGraphComposite composite) {
        this.m_composite = composite;
    }

    /**
     * グラフ描画対象の収集定義情報を設定します。（表示する収集を変更時に設定）
     * 
     * @param collectorData 実績収集の収集定義
     * @param itemInfoList　収集項目のリスト
     * @param facilityTree ファシリティツリー
     */
    public void setData(RecordCollectorData collectorData, List<CollectorItemInfo> itemInfoList) {
        this.m_collectorData = collectorData;
        this.m_itemInfoList = itemInfoList;
//        this.m_facilityTree = facilityTree;
    }

    /**
     * グラフを描画します。
     * 
     * @param facilityTree ファシリティツリー
     * @param itemCode 収集項目コード
     * @param graphType グラフ種別
     */
    public void drawGraph(
    		FacilityTreeItem facilityTree, 
    		CollectorItemInfo itemInfo,
            int graphType, 
            Date startDate, 
            Date endDate) {
    	// 表示設定がまったく変更になっていない場合は再描画しない
    	if(m_targetFacility == facilityTree && 
    			m_currentItem == itemInfo &&
    			m_graphStartTime == startDate.getTime() &&
    			m_graphEndTime == endDate.getTime()){
    		return;
    	}
    	
        // 現在表示されているグラフを消去
    	m_composite.clearData();
    	
        // グラフを生成します。
        if (m_collectorData == null || itemInfo == null) {
            return;
        }

        String itemCode = itemInfo.getItemCode(); // 収集項目コード
        String deviceName = itemInfo.getDeviceName();

        // 収集項目名をタイトルに設定
        String title = CollectorItemCodeFactory.getFullItemName(itemCode, deviceName);

//        if (CollectorItemCodeFactory.isDeviceSupport(itemCode)) {
//            title = title + "(" + itemInfo.getDeviceName() + ")";
//        }

        if (graphType == GraphProperty.TYPE1) {
            title = title + " "
                    + Messages.getString("SCOPE_REPRESENTING_VALUE"); // "スコープ代表値"
        } else if (graphType == GraphProperty.TYPE2) {
            title = title + " "
                    + Messages
                            .getString("DETAIL_SCOPE_REPRESENTING_VALUE"); // "スコープ代表値詳細"
        } else if (graphType == GraphProperty.TYPE3) {
            title = title + " "
                    + Messages
                            .getString("SUB_SCOPE_REP_VAL"); // "サブスコープ代表値"
        }

//        Date stopDate; // 収集終了時刻
//        if (m_collectorData.getStatus() == CollectorRunStatus.RUNNING) { // 収集中の場合
//            long stopTime = m_collectorData.getStartDate().getTime()
//                    + m_collectorData.getRealCollectPeriod();
//            stopDate = new Date(stopTime);
//        } else {
//        	stopDate = m_collectorData.getStopDate();
//        }

        this.m_composite.createGraph(title, // タイトル
        		Messages.getString("TIME"), // 横軸ラベル
        		CollectorItemCodeFactory.getMeasure(itemCode), // 縦軸ラベル
        		graphType, // グラフタイプ
        		CollectorItemCodeFactory.isRangeFixed(itemCode), // 軸固定フラグ
        		m_collectorData.getIntervalSec(), // インターバル
        		startDate, // グラフ表示開始時刻
        		endDate // グラフ表示終了時刻
//        		m_collectorData.getStartDate(), // 収集開始時刻
//        		m_collectorData.getLastCollectDate() //最終収集時刻
        );


        
        
        
        if (m_collectorData == null || itemInfo == null) {
            return;
        }
        
//        GraphDisplayRange dr = getDisplayRange(
//        		new GraphDisplayRange(startDate.getTime(), endDate.getTime()));
//
//        long startTime;
//        long endTime;
//        
//        if (dr == null) {
//            // 新規に描画する範囲はない
//        	return;
//        } else {
//        	startTime = dr.getStartTime();
//        	endTime = dr.getEndTime();
//        }
//
//        // グラフに表示される時刻の前後2プロット分を余分に取得する。
//        startDate = new Date(startTime
//        		- this.m_collectorData.getIntervalSec() * 2 * 1000); // 秒単位
//        endDate = new Date(endTime
//        		+ this.m_collectorData.getIntervalSec() * 2 * 1000); // 秒単位

        switch (graphType) {
        case GraphProperty.TYPE1:
        	drawGraph1(itemInfo, facilityTree, startDate, endDate);
        	break;
        case GraphProperty.TYPE2:
        	drawGraph2(itemInfo, facilityTree, startDate, endDate);
        	break;
        case GraphProperty.TYPE3:
        	drawGraph3(itemInfo, facilityTree, startDate, endDate);
        	break;
        default:
        	break;
        }

//        // 描画した範囲を描画済み範囲リストに追加する
//        addGotTimeArea(dr);
        
    	// グラフ表示した内容で更新
    	this.m_targetFacility = facilityTree;
        this.m_currentItem = itemInfo;
        this.m_graphType = graphType;
        this.m_graphStartTime = startDate.getTime();
        this.m_graphEndTime = endDate.getTime();
    }

    /**
     * グラフ種別を設定します。
     * 
     * @param graphType
     */
//    public void setGraphType(int graphType) {
//        this.m_graphType = graphType;
//    }

    /**
     * グラス種別を取得します。
     * 
     * @return グラフ種別
     */
    public int getGraphType() {
        return this.m_graphType;
    }

//    /**
//     * グラフ描画対象の収集項目IDを設定します。
//     * 
//     * @param itemID 収集項目ID
//     */
//    public void setTargetItemID(int itemID) {
//        Iterator itr = this.m_itemInfoList.iterator();
//
//        while (itr.hasNext()) {
//            CollectorItemInfo itemInfo = (CollectorItemInfo) itr.next();
//            if (itemInfo.getCollectorItemID() == itemID) {
//                this.m_currentItem = itemInfo;
//            }
//        }
//    }

    /**
     * グラフ描画対象の収集項目IDを取得します。
     * 
     * @return 収集項目ID
     */
    public CollectorItemInfo getTargetItem() {
        return this.m_currentItem;
    }

//    /**
//     * グラフ描画対象のファシリティ情報を設定します。
//     * 
//     * @param facilityTree ファシリティツリー
//     */
//    public void setTargetFacility(FacilityTreeItem facilityTree) {
//        this.m_facilityTree = facilityTree;
//
//        // グラフを生成します。
//        createGraph();
//    }

    /**
     * グラフ描画対象のファシリティ情報を取得します。
     * 
     * @return ファシリティツリー
     */
    public FacilityTreeItem getTargetFacility() {
        return this.m_targetFacility;
    }

    /**
     * グラフを生成します。
     */
    public void createGraph() {

    }

//    /**
//     * Widgetが選択されたときのアクションを定義します。
//     */
//    public synchronized void widgetSelected(
//            org.eclipse.swt.events.SelectionEvent e) {
//        double lowerBound = (this.m_composite.getHorizontalBar().getSelection()) * 1000D;
//        double upperBound = (this.m_composite.getHorizontalBar().getThumb() + this.m_composite
//                .getHorizontalBar().getSelection()) * 1000D;
//
//        this.drawGraph((long) lowerBound, (long) upperBound);
//
//        this.m_composite.setRange(lowerBound, upperBound);
//    }

    /**
     * 
     * 指定の収集項目の代表値グラフを表示します
     * 
     * @param itemInfo 収集項目情報
     * @param treeItem ファシリティツリー情報
     * @param startDate 描画対象開始時刻
     * @param endDate　描画対象終了時刻
     */
    public synchronized void drawGraph1(CollectorItemInfo itemInfo,
            FacilityTreeItem treeItem, Date startDate, Date endDate) {
    	
        // マネージャからデータを取得
        String facilityId = treeItem.getData().getFacilityId();

        RecordController controller = RecordController.getInstance();
        // マネージャとの接続に失敗した場合はエラーダイアログを表示
        if (controller == null) {
            MessageDialog.openError(this.m_composite.getShell(), 
            		Messages.getString("MESSAGE_0020"),  // "確認"
            		Messages.getString("CONNECTION_ERROR_MESSAGE"));
            return;
        }

        CollectedDataSet dataSet = controller.getRecordCollectedData(
        		facilityId, itemInfo, startDate, endDate);
        // 性能データの取得に失敗した場合はエラーダイアログを表示
        if (dataSet == null) {
            MessageDialog.openError(this.m_composite.getShell(), 
            		Messages.getString("MESSAGE_0020"),  // "確認"
            		Messages.getString("GET_COLLECTEDDATA_ERROR_MESSAGE"));
            return;
        }

        List<CollectedDataInfo> dataList = dataSet
                .getCollectedDataList(facilityId, itemInfo);

        if (dataList.size() != 0) { // 収集中のグラフ表示ではマネージャから値を取得できていない場合もあるため
            // グラフに値を設定
            setItemData(itemInfo.getItemCode(), itemInfo
                    .getDeviceName(), dataList);
        }
    }
    
    /**
     * 指定の収集項目の詳細情報を内訳グラフとして表示します。（指定の収集項目のグラフを描画するわけではない）
     * 
     * @param itemInfo 収集項目情報
     * @param treeItem ファシリティツリー情報
     * @param startDate 描画対象開始時刻
     * @param endDate　描画対象終了時刻
     */
    public synchronized void drawGraph2(CollectorItemInfo itemInfo,
            FacilityTreeItem treeItem, Date startDate, Date endDate) {
        // 内訳グラフ描画に必要な収集項目のリストを取得
        List<CollectorItemInfo> itemInfoList = selectItemInfo(itemInfo, this.m_itemInfoList);

        // 内訳グラフの表示に必要な項目が収集されていない場合はエラー
        if (itemInfoList == null || itemInfoList.size() == 0) {
            // エラー処理
            MessageDialog.openError(this.m_composite.getShell(), 
            		Messages.getString("MESSAGE_0020"), 
            		Messages.getString("DRAW_RECORD_GRAPH_ERROR_MESSAGE02"));
            return;
        }

        // マネージャからデータを取得
        String facilityId = treeItem.getData().getFacilityId();
 
        RecordController controller = RecordController.getInstance();
        // マネージャとの接続に失敗した場合はエラーダイアログを表示
        if (controller == null) {
            MessageDialog.openError(this.m_composite.getShell(), 
            		Messages.getString("MESSAGE_0020"), 
            		Messages.getString("CONNECTION_ERROR_MESSAGE"));
            return;
        }

        CollectedDataSet dataSet = controller.getRecordCollectedData(
        		facilityId, itemInfoList, startDate, endDate);
        // 性能データの取得に失敗した場合はエラーダイアログを表示
        if (dataSet == null) {
            MessageDialog.openError(this.m_composite.getShell(), 
            		Messages.getString("MESSAGE_0020"), 
            		Messages.getString("GET_COLLECTEDDATA_ERROR_MESSAGE"));
            return;
        }

        ArrayList<List<CollectedDataInfo>> addedDataList = new ArrayList<List<CollectedDataInfo>>();
        // 面グラフにするために値を加工
        Iterator<CollectorItemInfo> itr = itemInfoList.iterator();
        List<CollectedDataInfo> collectedDataListTotal = null; // 性能値の合計を保持するリスト
        while (itr.hasNext()) {
            CollectorItemInfo buffer = itr.next();
            List<CollectedDataInfo> collectedDataList = dataSet.getCollectedDataList(
            		treeItem.getData().getFacilityId(), buffer);

            if(addedDataList.size() == 0){
            	collectedDataListTotal = collectedDataList;
            } else {
            	// 前の項目までの合計値
            	collectedDataListTotal = getSumData(collectedDataListTotal, collectedDataList);
            }

           	addedDataList.add(collectedDataListTotal);
        }

        // 時刻の最大値と最小値を設定
        List<CollectedDataInfo> bufferList = addedDataList.get(0); // どれも同じ時刻であるため最初の要素で求める

        if (bufferList.size() != 0) { // 収集中のグラフ表示ではマネージャから値を取得できていない場合もあるため

        	// グラフに値を設定
        	int itemSize = itemInfoList.size();
        	for (int i = 0; i < itemSize; i++) {
        		CollectorItemInfo buffer = itemInfoList.get(i);
        		List<CollectedDataInfo> collectedDataList = addedDataList.get(i);
        		setItemData(buffer.getItemCode(), buffer.getDeviceName(), collectedDataList);
        	}
        }
    }

    /**
     * 指定の収集項目のサブスコープ代表値グラフを表示します
     * 
     * @param itemInfo 収集項目情報
     * @param treeItem ファシリティツリー情報
     * @param startDate 描画対象開始時刻
     * @param endDate　描画対象終了時刻
     */
    public synchronized void drawGraph3(CollectorItemInfo itemInfo,
            FacilityTreeItem treeItem, Date startDate, Date endDate) {
        // 子スコープを設定する
        FacilityTreeItem[] children = treeItem.getChildren();
        ArrayList<String> targetFacilitys = new ArrayList<String>();
        ArrayList<String> targetFacilityNames = new ArrayList<String>();
        for (int i = 0; i < children.length; i++) {
            targetFacilitys.add(children[i].getData().getFacilityId());
            targetFacilityNames.add(children[i].getData().getFacilityName());
        }

        // マネージャからデータを取得

        RecordController controller = RecordController.getInstance();
        // マネージャとの接続に失敗した場合はエラーダイアログを表示
        if (controller == null) {
            MessageDialog.openError(this.m_composite.getShell(), 
            		Messages.getString("MESSAGE_0020"), 
            		Messages.getString("CONNECTION_ERROR_MESSAGE"));
            return;
        }

        CollectedDataSet dataSet = controller.getRecordCollectedData(
        		targetFacilitys, 
        		itemInfo, 
        		startDate, 
        		endDate);
        // 性能データの取得に失敗した場合はエラーダイアログを表示
        if (dataSet == null) {
            MessageDialog.openError(this.m_composite.getShell(), 
            		Messages.getString("MESSAGE_0020"), 
            		Messages.getString("GET_COLLECTEDDATA_ERROR_MESSAGE"));
            return;
        }

		// グラフに値を設定
		Iterator<String> itr = targetFacilitys.iterator();
		int i = 0;
		while (itr.hasNext()) {
			String targetFacility = itr.next();
			List<CollectedDataInfo> collectedDataList = 
					dataSet.getCollectedDataList(targetFacility, itemInfo);
			setFacilityData(i, collectedDataList, targetFacilityNames);
			i++;
		}
    }

    /**
	 * 凡例を収集項目名として、グラフに値を設定します。（スコープ代表値グラフ、内訳グラフの場合）
	 * 
	 * @param composite
	 *            グラフを関連づけるコンポジット
	 * @param itemCode
	 *            収集項目コード
	 * @param data
	 *            性能値
	 */
    private void setItemData(String itemCode, String deviceName, List<CollectedDataInfo> data) {
        this.m_composite.setGraphData(CollectorItemCodeFactory.getFullItemName(
                itemCode, deviceName), data);
    }

    /**
     * 凡例をファシリティ名として、グラフに値を設定します。（サブスコープ代表値グラフの場合）
     * 
     * @param composite グラフを関連づけるコンポジット
     * @param itemCode 収集項目コード
     * @param data 性能値
     */
    private void setFacilityData(int index, List<CollectedDataInfo> data, ArrayList<String> facilityNames) {
        this.m_composite.setSubscopeGraphData(index, data, facilityNames);
    }

    /**
     * 収集時の収集項目定義の中から内訳表示を行うのに必要な収集項目を選びます。 必要な収集項目に満たない場合は、nullを返します。
     * 
     * @param itemList 収集項目のリスト
     */
    private List<CollectorItemInfo> selectItemInfo(CollectorItemInfo target, List<CollectorItemInfo> itemList) {
        ArrayList<CollectorItemInfo> returnList = new ArrayList<CollectorItemInfo>();

        // 内訳表示に必要な収集項目コードを取得
        List<String> itemCodeList = CollectorItemCodeFactory.getSubItemCode(target
                .getItemCode());

        // デバイス名を取得
        String deviceName = target.getDeviceName();

        // 内訳表示に必要な収集項目コードの数だけループ
        Iterator<String> itrItemCodeList = itemCodeList.iterator();
        while (itrItemCodeList.hasNext()) {
            String itemCode = itrItemCodeList.next();

            CollectorItemInfo buffer = null;

            Iterator<CollectorItemInfo> itrItemList = itemList.iterator();
            while (itrItemList.hasNext()) {
                CollectorItemInfo itemInfo = itrItemList.next();
                // itemList内に収集項目コードとデバイス名が同じ要素が重複して存在しないことが前提
                if (itemInfo.getItemCode().equals(itemCode)
                        && itemInfo.getDeviceName().equals(deviceName)) {
                    buffer = itemInfo;
                }
            }

            if (buffer == null) {
                // 収集時に指定された収集項目では、内訳表示を行うには不十分である。
                return null;
            } else {
                returnList.add(buffer);
            }
        }
        return returnList;
    }

    /**
     * 2つの収集済み性能値リストの和をとります。
     */
    private List<CollectedDataInfo> getSumData(List<CollectedDataInfo> dataSet1, List<CollectedDataInfo> dataSet2) {
        ArrayList<CollectedDataInfo> dataList = new ArrayList<CollectedDataInfo>();

        Iterator<CollectedDataInfo> itr1 = dataSet1.iterator();
        Iterator<CollectedDataInfo> itr2 = dataSet2.iterator();

        while (itr1.hasNext() && itr2.hasNext()) {
        	CollectedDataInfo data1 = itr1.next();
            CollectedDataInfo data2 = itr2.next();
            
            // 時刻がずれている場合は、別時刻の収集値であるため加算しない
            if(data1.getDate().getTime() == data2.getDate().getTime()){
            	CollectedDataInfo returnData = new CollectedDataInfo(
            			data1.getFacilityId(), 
            			data1.getItemCode(),
            			data1.getDeviceName(),
            			data1.getCollectMethod(),
            			data1.getDate(), 
            			data1.getValue() + data2.getValue());
            	dataList.add(returnData);
            }
        }

        return dataList;
    }

    /**
     * 描画済み範囲のリストに範囲を追加する。
     * 
     * @param dr 描画済みリストに追加する範囲
     */
    private void addGotTimeArea(GraphDisplayRange dr) {
        // 初めての描画の場合
        if (m_gotTimeArea.size() == 0) {
            m_gotTimeArea.add(dr);

            return;
        }

        Iterator<GraphDisplayRange> itr = m_gotTimeArea.iterator();

        while (itr.hasNext()) {
            GraphDisplayRange drBuffer = (GraphDisplayRange) itr.next();

            if (dr.getEndTime() < drBuffer.getStartTime()) {
                // 既に描画されている範囲と重なる範囲がないため、リストに追加する
                int index = m_gotTimeArea.indexOf(drBuffer);
                m_gotTimeArea.add(index, dr);
                return;
            } else if (drBuffer.getStartTime() <= dr.getEndTime()
                    && dr.getEndTime() <= drBuffer.getEndTime()) {
                // 新規追加対象の範囲の終了時刻がリストの中にある既に描画済み範囲のひとつに収まる場合
                // その描画済み範囲と統合(範囲を広げる)する

                // 追加の表示範囲の分だけ表示済み範囲を広げる
                drBuffer.setStartTime(dr.getStartTime());

                return;
            } else if (drBuffer.getStartTime() <= dr.getStartTime()
                    && dr.getStartTime() <= drBuffer.getEndTime()) {
                // 新規追加対象の範囲の開始時刻がリストの中にある既に描画済み範囲のひとつに収まる場合
                // その描画済み範囲と統合(範囲を広げる)する

                // 追加の表示範囲の分だけ表示済み範囲を広げる
                drBuffer.setEndTime(dr.getEndTime());

                // 次の要素の範囲を調べ終了と開始の時刻がクロスするようなら統合する。
                if (itr.hasNext()) {
                    GraphDisplayRange nextDrBuffer = (GraphDisplayRange) itr
                            .next();
                    if (nextDrBuffer.getStartTime() <= drBuffer.getEndTime()) {
                        // 終了時刻の大きいほうを新たな終了時刻として統合
                        drBuffer.setEndTime(Math.max(nextDrBuffer.getEndTime(),
                                nextDrBuffer.getEndTime()));

                        // 統合された側は削除する
                        m_gotTimeArea.remove(nextDrBuffer);
                    }
                }
                return;
            }
        }
        m_gotTimeArea.add(dr);
    }

    /**
     * 描画希望範囲をあたえると、描画済みリストと照合しまだ描画されていない範囲を返す。 全ての範囲が既に描画済みの場合はnullを返す。
     * 
     * @param dr
     *            描画希望範囲
     * @return まだ描画されていない範囲
     */
    private GraphDisplayRange getDisplayRange(GraphDisplayRange dr) {
        // リストにまだなにも登録されていない場合
        if (m_gotTimeArea.size() == 0) {
            return dr;
        }

        GraphDisplayRange returnDr = new GraphDisplayRange();
        returnDr.setStartTime(dr.getStartTime());
        returnDr.setEndTime(dr.getEndTime());

        Iterator<GraphDisplayRange> itr = m_gotTimeArea.iterator();

        while (itr.hasNext()) {
            GraphDisplayRange drBuffer = (GraphDisplayRange) itr.next();

            // 開始時刻の判定
            if (drBuffer.getStartTime() <= dr.getStartTime()
                    && dr.getStartTime() <= drBuffer.getEndTime()) {
                returnDr.setStartTime(drBuffer.getEndTime());
            }

            // 終了時刻の判定
            if (drBuffer.getStartTime() <= dr.getEndTime()
                    && dr.getEndTime() <= drBuffer.getEndTime()) {
                returnDr.setEndTime(drBuffer.getStartTime());
            }
        }

        if (returnDr.getStartTime() >= returnDr.getEndTime()) {
            returnDr = null;
        }

        return returnDr;
    }
}