/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.jobmanagement.composite;

import java.text.DateFormat;
import java.util.ArrayList;
import java.util.Date;

import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Table;

import com.clustercontrol.jobmanagement.action.GetRunSchedule;
import com.clustercontrol.jobmanagement.action.GetRunScheduleTableDefine;
import com.clustercontrol.jobmanagement.composite.action.VerticalBarSelectionListener;
import com.clustercontrol.util.Messages;

/**
 * ジョブ[実行予定]ビューの右側用のコンポジットクラスです。
 * 
 * @version 2.0.0
 * @since 2.0.0
 */
public class RunScheduleComposite extends Composite {
	/** テーブルビューア */
    protected RunScheduleTableViewer m_viewer = null;
    /** テーブル */
    protected Table m_table = null;
    /** 件数用ラベル */
    protected Label m_labelCount = null;
    /** 基準日 */
    protected Date m_base = null;

    /**
     * コンストラクタ
     * 
     * @param parent 親のコンポジット
     * @param style スタイル
     * @param base 基準日
     * 
     * @see org.eclipse.swt.SWT
     * @see org.eclipse.swt.widgets.Composite#Composite(Composite parent, int style)
     * @see #initialize()
     */
    public RunScheduleComposite(Composite parent, int style, Date base) {
        super(parent, style);
        m_base = base;
        initialize();
    }

    /**
     * コンポジットを構築します。
     */
    private void initialize() {
        GridLayout layout = new GridLayout(1, true);
        this.setLayout(layout);
        layout.marginHeight = 0;
        layout.marginWidth = 0;

        m_table = new Table(this, SWT.H_SCROLL | SWT.V_SCROLL
                | SWT.FULL_SELECTION | SWT.SINGLE | SWT.BORDER);
        m_table.setHeaderVisible(true);
        m_table.setLinesVisible(true);

        GridData gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.grabExcessVerticalSpace = true;
        m_table.setLayoutData(gridData);

        m_labelCount = new Label(this, SWT.RIGHT);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        m_labelCount.setLayoutData(gridData);
        
        m_viewer = new RunScheduleTableViewer(m_table);
        m_viewer.createTableColumn(GetRunScheduleTableDefine.get(m_base), 0, 1);
        
        m_viewer.getTable().getVerticalBar().addSelectionListener(
        		new VerticalBarSelectionListener(this));
    }

    /**
     * テーブルビューアーを更新します。<BR>
     * 基準日からの実行予定マーク情報を取得し、テーブルビューアーにセットします。
     * <p>
	 * <ol>
	 * <li>基準日からの実行予定マーク情報を取得します。</li>
	 * <li>テーブルビューアーに実行予定マーク情報をセットします。</li>
	 * </ol>
	 * 
     * @see com.clustercontrol.jobmanagement.action.GetRunSchedule#get(Date)
     */
    public void update() {
        ArrayList info = null;

        GetRunSchedule getSchedule = new GetRunSchedule();
        info = getSchedule.get(m_base);
        if (info == null) {
        	info = new ArrayList();
        }
        
        m_viewer.setInput(info);

        Object[] args = { DateFormat.getDateInstance().format(m_base) };
        m_labelCount.setText(Messages.getString("specified.period", args));
    }

    /**
     * このコンポジットが利用するテーブルビューアを返します。
     * 
     * @return テーブルビューア
     */
    public TableViewer getTableViewer() {
        return m_viewer;
    }

    /**
     * このコンポジットが利用するテーブルを返します。
     * 
     * @return テーブル
     */
    public Table getTable() {
        return m_viewer.getTable();
    }
	
	/**
	 * 基準日を返します。
	 * 
	 * @return 基準日
	 */
	public Date getBaseDate() {
		return m_base;
	}
	
	/**
	 * 基準日を設定します。
	 * 
	 * @param base 基準日
	 */
	public void setCalendarId(Date base) {
		m_base = base;
	}
}