/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.notify.dialog;

import java.util.ArrayList;

import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;

import com.clustercontrol.bean.NotifyTypeConstant;
import com.clustercontrol.bean.PriorityColorConstant;
import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.bean.StatusExpirationConstant;
import com.clustercontrol.bean.StatusValidPeriodConstant;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.notify.action.AddNotify;
import com.clustercontrol.notify.action.GetNotify;
import com.clustercontrol.notify.action.ModifyNotify;
import com.clustercontrol.notify.bean.NotifyInfo;
import com.clustercontrol.notify.bean.NotifyInfoDetail;
import com.clustercontrol.notify.bean.NotifyStatusInfo;
import com.clustercontrol.util.Messages;

/**
 * 通知（ステータス）作成・変更ダイアログクラス<BR>
 * 
 * @version 3.0.0
 * @since 3.0.0
 */
public class NotifyStatusCreateDialog extends NotifyBasicCreateDialog {
	
	/** カラム数（重要度）。 */
	private static final int WIDTH_PRIORITY 	= 1;
	
	/** カラム数（チェックボックス）。 */
	private static final int WIDTH_CHECK 		= 1;
	
    // ----- instance フィールド ----- //
	
	/** 通知タイプ 
	 * @see com.clustercontrol.bean.NotifyTypeConstant
	 */
	private final Integer m_notifyType = NotifyTypeConstant.TYPE_STATUS;
	
	/** 入力値を保持するオブジェクト。 */
	private NotifyInfo inputData = null;
	
	/** 入力値の正当性を保持するオブジェクト。 */
	private ValidateResult validateResult = null;
	
	/** ステータス 通知（重要度：通知） チェックボックス。 */ 
	private Button m_checkStatusNormalInfo = null;
	/** ステータス 通知（重要度：警告） チェックボックス。 */
	private Button m_checkStatusNormalWarning = null;
	/** ステータス 通知（重要度：危険） チェックボックス。 */
	private Button m_checkStatusNormalCritical = null;
	/** ステータス 通知（重要度：不明） チェックボックス。 */
	private Button m_checkStatusNormalUnknown = null;
	
	/** ステータス 抑制（重要度：通知） チェックボックス。 */ 
	private Button m_checkStatusInhibitionInfo = null;
	/** ステータス 抑制（重要度：警告） チェックボックス。 */
	private Button m_checkStatusInhibitionWarning = null;
	/** ステータス 抑制（重要度：危険） チェックボックス。 */
	private Button m_checkStatusInhibitionCritical = null;
	/** ステータス 抑制（重要度：不明） チェックボックス。 */
	private Button m_checkStatusInhibitionUnknown = null;
	
	/**  ステータス通知 ステータス情報の存続期間 コンボボックス。 */
	private Combo m_comboStatusValidPeriod = null;
	
	/** ステータス通知 存続期間経過後の処理（情報を削除する） ラジオボタン。  */
	private Button m_radioStatusDelete = null;
	
	/** ステータス通知 存続期間経過後の処理（更新されていない旨のメッセージに置換える） ラジオボタン。 */
	private Button m_radioStatusUpdate = null;
	
	/**  ステータス通知 更新時の重要度 コンボボックス。 */
	private Combo m_comboStatusPriority = null;
	
	
	// ----- コンストラクタ ----- //
	
	/**
	 * 作成用ダイアログのインスタンスを返します。
	 * 
	 * @param parent 親のシェルオブジェクト
	 */
	public NotifyStatusCreateDialog(Shell parent) {
		super(parent);
	}
	
	/**
	 * 変更用ダイアログのインスタンスを返します。
	 * 
	 * @param parent 親のシェルオブジェクト
	 * @param notifyId 変更する通知情報の通知ID
	 * @param referenceFlg 参照フラグ（参照のみで変更不可の場合、<code> true </code>）
	 */
	public NotifyStatusCreateDialog(Shell parent, String notifyId, boolean referenceFlg) {
		super(parent, notifyId, referenceFlg);
	}
	
    // ----- instance メソッド ----- //
	
	/**
	 * ダイアログエリアを生成します。
	 * 
	 * @param parent 親のコンポジット
	 * 
	 * @see com.clustercontrol.notify.dialog.NotifyBasicCreateDialog#customizeDialog(Composite)
	 * @see com.clustercontrol.notify.action.GetNotify#getNotify(String)
	 * @see #setInputData(NotifyInfo)
	 */
    protected void customizeDialog(Composite parent) {
    	
    	super.customizeDialog(parent);
    	
        // 通知IDが指定されている場合、その情報を初期表示する。
        NotifyInfo info = null; 
        if(this.notifyId != null){
        	info = new GetNotify().getNotify(this.notifyId);        	
        }
        else{
        	info = new NotifyInfo();
        }
        this.setInputData(info);
        
    }
    
	/**
	 * 親のクラスから呼ばれ、各通知用のダイアログエリアを生成します。
	 * 
	 * @param parent 親のコンポジット
	 * 
	 * @see com.clustercontrol.notify.dialog.NotifyBasicCreateDialog#customizeDialog(Composite)
	 */
    protected void customizeSettingDialog(Composite parent) {
    	Shell shell = this.getShell();

        // タイトル
        shell.setText(Messages.getString("dialog.notify.status.create.modify"));

        // 変数として利用されるラベル
        Label label = null;
        // 変数として利用されるグリッドデータ
        GridData gridData = null;

        // レイアウト
        GridLayout layout = new GridLayout(1, true);
        layout.marginWidth = 10;
        layout.marginHeight = 10;
        layout.numColumns = 15;
        parent.setLayout(layout);

        /*
         * ステータス通知
         */
		label = new Label(parent, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("notifies.status") + " : ");
        
        /*
         * 大グループ
         */
        // ステータス通知グループ
        Group groupStatus1 = new Group(parent, SWT.NONE);
        layout = new GridLayout(1, true);
        layout.marginWidth = 5;
        layout.marginHeight = 5;
        layout.numColumns = 15;
        groupStatus1.setLayout(layout);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        groupStatus1.setLayoutData(gridData);

        /*
         * 重要度 ごとの設定
         */
        // ラベル（重要度）
        label = new Label(groupStatus1, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_PRIORITY;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("priority"));
        
        // ラベル（通知する）
        label = new Label(groupStatus1, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_CHECK;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("notify.event.treatment.normal"));
        
        // ラベル（抑制する）
        label = new Label(groupStatus1, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_CHECK;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("notify.event.treatment.inhibition"));
        
        // 空白
        label = new Label(groupStatus1, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 12;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        
        
        // ステータス 重要度：通知
        label = this.getLabelPriority(groupStatus1, Messages.getString("info"),PriorityColorConstant.COLOR_INFO);
        this.m_checkStatusNormalInfo = this.getCheckStatusNormal(groupStatus1);
        this.m_checkStatusInhibitionInfo = this.getCheckStatusInhibition(groupStatus1);
        
        this.m_checkStatusNormalInfo.addSelectionListener(new SelectionAdapter() {
        	public void widgetSelected(SelectionEvent e) {
        		m_checkStatusInhibitionInfo.setEnabled(m_checkStatusNormalInfo.getSelection());
        		setEnabledForStatusDuration();
        	}
        });
        
        // 空白
        label = new Label(groupStatus1, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 12;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        
        
        // ステータス 重要度：警告
        label = this.getLabelPriority(groupStatus1, Messages.getString("warning"),PriorityColorConstant.COLOR_WARNING);
        this.m_checkStatusNormalWarning = this.getCheckStatusNormal(groupStatus1);
        this.m_checkStatusInhibitionWarning = this.getCheckStatusInhibition(groupStatus1);
        
        this.m_checkStatusNormalWarning.addSelectionListener(new SelectionAdapter() {
        	public void widgetSelected(SelectionEvent e) {
        		m_checkStatusInhibitionWarning.setEnabled(m_checkStatusNormalWarning.getSelection());
        		setEnabledForStatusDuration();
        	}
        });
        
        // 空白
        label = new Label(groupStatus1, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 12;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);

        
        // ステータス 重要度：危険
        label = this.getLabelPriority(groupStatus1, Messages.getString("critical"),PriorityColorConstant.COLOR_CRITICAL);
        this.m_checkStatusNormalCritical = this.getCheckStatusNormal(groupStatus1);
        this.m_checkStatusInhibitionCritical = this.getCheckStatusInhibition(groupStatus1);
        
        this.m_checkStatusNormalCritical.addSelectionListener(new SelectionAdapter() {
        	public void widgetSelected(SelectionEvent e) {
        		m_checkStatusInhibitionCritical.setEnabled(m_checkStatusNormalCritical.getSelection());
        		setEnabledForStatusDuration();
        	}
        });
        
        // 空白
        label = new Label(groupStatus1, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 12;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        
        
        // ステータス 重要度：不明
        label = this.getLabelPriority(groupStatus1, Messages.getString("unknown"),PriorityColorConstant.COLOR_UNKNOWN);
        this.m_checkStatusNormalUnknown = this.getCheckStatusNormal(groupStatus1);
        this.m_checkStatusInhibitionUnknown = this.getCheckStatusInhibition(groupStatus1);
        
        this.m_checkStatusNormalUnknown.addSelectionListener(new SelectionAdapter() {
        	public void widgetSelected(SelectionEvent e) {
        		m_checkStatusInhibitionUnknown.setEnabled(m_checkStatusNormalUnknown.getSelection());
        		setEnabledForStatusDuration();
        	}
        });
        
        // 空白
        label = new Label(groupStatus1, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 12;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        
        // 空行
        label = new Label(groupStatus1, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        
        
        /*
         * ステータス情報の存続期間（分）
         */
        // ラベル
        label = new Label(groupStatus1, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 4;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("notify.status.valid.period") + " : ");
        // コンボボックス
        this.m_comboStatusValidPeriod = new Combo(groupStatus1, SWT.DROP_DOWN | SWT.READ_ONLY);
        gridData = new GridData();
        gridData.horizontalSpan = 2;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.m_comboStatusValidPeriod.setLayoutData(gridData);
        this.m_comboStatusValidPeriod.add(StatusValidPeriodConstant.STRING_UNLIMITED);
        this.m_comboStatusValidPeriod.add(StatusValidPeriodConstant.STRING_MIN_10);
        this.m_comboStatusValidPeriod.add(StatusValidPeriodConstant.STRING_MIN_20);
        this.m_comboStatusValidPeriod.add(StatusValidPeriodConstant.STRING_MIN_30);
        this.m_comboStatusValidPeriod.add(StatusValidPeriodConstant.STRING_HOUR_1);
        this.m_comboStatusValidPeriod.add(StatusValidPeriodConstant.STRING_HOUR_3);
        this.m_comboStatusValidPeriod.add(StatusValidPeriodConstant.STRING_HOUR_6);
        this.m_comboStatusValidPeriod.add(StatusValidPeriodConstant.STRING_HOUR_12);
        this.m_comboStatusValidPeriod.add(StatusValidPeriodConstant.STRING_DAY_1);
        this.m_comboStatusValidPeriod.setText(StatusValidPeriodConstant.STRING_MIN_10);
        
        // 空白
        label = new Label(groupStatus1, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 9;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        
        // ラベル
        label = new Label(groupStatus1, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("notify.status.invalid.period.treatment") + " : ");
        
        /*
         * 中グループ
         */
        // ステータス情報有効期間経過後の扱い通知グループ
        Group groupStatus2 = new Group(groupStatus1, SWT.NONE);
        layout = new GridLayout(1, true);
        layout.marginWidth = 5;
        layout.marginHeight = 5;
        layout.numColumns = 15;
        groupStatus2.setLayout(layout);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        groupStatus2.setLayoutData(gridData);
        
        // 情報を削除する
        this.m_radioStatusDelete = new Button(groupStatus2, SWT.RADIO);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.m_radioStatusDelete.setLayoutData(gridData);
        this.m_radioStatusDelete.setText(Messages.getString("notify.status.invalid.period.delete"));

        // 更新されていない旨のメッセージに置換える。
        this.m_radioStatusUpdate = new Button(groupStatus2, SWT.RADIO);
        gridData = new GridData();
        gridData.horizontalSpan = 6;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.m_radioStatusUpdate.setLayoutData(gridData);
        this.m_radioStatusUpdate.setText(Messages.getString("notify.status.invalid.period.updatet"));
        this.m_radioStatusUpdate.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(SelectionEvent e) {
            	m_comboStatusPriority.setEnabled(m_radioStatusUpdate.getSelection());
            }
        });
        
        /*
         * ステータス情報 重要度
         */
        // ラベル
        label = new Label(groupStatus2, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 2;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("priority") + " : ");
        // コンボボックス
        this.m_comboStatusPriority = new Combo(groupStatus2, SWT.DROP_DOWN | SWT.READ_ONLY);
        gridData = new GridData();
        gridData.horizontalSpan = 2;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.m_comboStatusPriority.setLayoutData(gridData);
        this.m_comboStatusPriority.add(PriorityConstant.STRING_CRITICAL);
        this.m_comboStatusPriority.add(PriorityConstant.STRING_WARNING);
        this.m_comboStatusPriority.add(PriorityConstant.STRING_INFO);
        this.m_comboStatusPriority.add(PriorityConstant.STRING_UNKNOWN);
        this.m_comboStatusPriority.setText(PriorityConstant.STRING_WARNING);
        // 空白
        label = new Label(groupStatus2, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 5;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);   

    }

	/**
	 * 入力値を保持した通知情報を返します。
	 * 
	 * @return 通知情報
	 */
    public NotifyInfo getInputData() {
        return this.inputData;
    }

	/**
	 * 引数で指定された通知情報の値を、各項目に設定します。
	 * 
	 * @param notify 設定値として用いる通知情報
	 */
    protected void setInputData(NotifyInfo notify) {

        this.inputData = notify;
        
        // 通知基本情報
        super.m_notifyBasic.setInputData(notify);
        
        // 抑制情報
        super.m_notifyInhibition.setInputData(notify);
      
        // ステータス情報
        if (notify.getNotifyType() != null && notify.getNotifyType().intValue() == this.m_notifyType.intValue() && notify.getNotifyInfoDetail() != null) {
        	ArrayList<NotifyInfoDetail> list = notify.getNotifyInfoDetail();
        	if (list != null) {
        		for (int index=0; index<list.size(); index++){
        			NotifyStatusInfo status = (NotifyStatusInfo)list.get(index);
        			if(status != null){
        				// 通知の場合
                    	if(PriorityConstant.TYPE_INFO ==  status.getPriority()){
                            this.setInputDataForStatus(status, this.m_checkStatusNormalInfo, 	this.m_checkStatusInhibitionInfo);
                            //通知のみステータス通知の共通部分の設定を行う
                            if (status.getStatusValidPeriod() != null) {
                            	this.m_comboStatusValidPeriod.setText(StatusValidPeriodConstant.typeToString(status.getStatusValidPeriod().intValue()));
                            }
                            if (status.getStatusInvalidFlg() != null && status.getStatusInvalidFlg().intValue() == StatusExpirationConstant.TYPE_DELETE) {
                            	this.m_radioStatusDelete.setSelection(true);
                            } else {
                            	this.m_radioStatusUpdate.setSelection(true);
                            }
                            this.m_comboStatusPriority.setText(PriorityConstant.typeToString(status.getStatusUpdatePriority().intValue()));
                    	}
                    	// 警告の場合
                    	else if(PriorityConstant.TYPE_WARNING ==  status.getPriority()){
                            this.setInputDataForStatus(status, this.m_checkStatusNormalWarning, 	this.m_checkStatusInhibitionWarning);
                    	}
                    	// 危険の場合
                    	else if(PriorityConstant.TYPE_CRITICAL ==  status.getPriority()){
                            this.setInputDataForStatus(status, this.m_checkStatusNormalCritical, 	this.m_checkStatusInhibitionCritical);
                    	}
                    	// 不明の場合
                    	else if(PriorityConstant.TYPE_UNKNOWN ==  status.getPriority()){
                            this.setInputDataForStatus(status, this.m_checkStatusNormalUnknown, 	this.m_checkStatusInhibitionUnknown);
                    	}
        			}
        		}
        		
        	}
        }
        // 新規追加の場合
        else {
            this.m_radioStatusUpdate.setSelection(true);
        }

        // 有効／無効
        if (notify.getValidFlg() != null && notify.getValidFlg() == ValidConstant.TYPE_INVALID) {
            this.m_confirmValid.setSelection(false);
        }
        else {
        	this.m_confirmValid.setSelection(true);
        }
        
        // 入力制御
        if(this.referenceFlg){
        	this.setEnabled(false);
        }
        else{
        	this.setEnabled(true);
        }
        
    }
    
	/**
	 * 引数で指定された通知ステータス情報の値を、各項目に設定します。
	 * 
	 * @param info 設定値として用いる通知ステータス情報
	 * @param checkStatusNormal 通知チェックボックス
	 * @param checkStatusInhibition 抑制チェックボックス
	 */
    private void setInputDataForStatus(NotifyStatusInfo info, 
    		Button checkStatusNormal, 
    		Button checkStatusInhibition
    		) {
    	// ステータス通知
        if (info.getValidFlg() != null && info.getValidFlg().intValue() == ValidConstant.TYPE_VALID) {
        	checkStatusNormal.setSelection(true);
        } 
        // 抑制
        if (info.getInhibitionFlg() != null && info.getInhibitionFlg().intValue() == ValidConstant.TYPE_VALID) {
        	checkStatusInhibition.setSelection(true);
        }
    }

	/**
	 * 入力値を設定した通知情報を返します。<BR>
	 * 入力値チェックを行い、不正な場合は<code>null</code>を返します。
	 * 
	 * @return 通知情報
	 * 
	 * @see #createInputDataForStatus(ArrayList, int, Button, Button)
	 * @see #setValidateResult(String, String)
	 */
    protected NotifyInfo createInputData() {
        NotifyInfo info = new NotifyInfo();

        // 通知タイプの設定
    	info.setNotifyType(m_notifyType);
        
        // 通知基本情報
        validateResult = super.m_notifyBasic.createInputData(info);
        if(validateResult != null){
        	return null;
        }
        
        // 抑制情報
        validateResult = super.m_notifyInhibition.createInputData(info);
        if(validateResult != null){
        	return null;
        }
        
        // ステータス情報
        // 各重要度で１つも通知が選択されていない場合はエラー
    	if(!this.m_checkStatusNormalInfo.getSelection() && 
    			!this.m_checkStatusNormalWarning.getSelection() && 
    			!this.m_checkStatusNormalCritical.getSelection() && 
    			!this.m_checkStatusNormalUnknown.getSelection()){
        	this.setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.notify.13"));
                return null;
    	}
        
        ArrayList<NotifyInfoDetail> list = new ArrayList<NotifyInfoDetail>();
        
        list = this.createInputDataForStatus(list, PriorityConstant.TYPE_INFO,		this.m_checkStatusNormalInfo,		this.m_checkStatusInhibitionInfo);
        list = this.createInputDataForStatus(list, PriorityConstant.TYPE_WARNING,	this.m_checkStatusNormalWarning,	this.m_checkStatusInhibitionWarning);
        list = this.createInputDataForStatus(list, PriorityConstant.TYPE_CRITICAL,	this.m_checkStatusNormalCritical,	this.m_checkStatusInhibitionCritical);
        list = this.createInputDataForStatus(list, PriorityConstant.TYPE_UNKNOWN,	this.m_checkStatusNormalUnknown, 	this.m_checkStatusInhibitionUnknown);
        
        if(list == null){
        	return null;
        }
        info.setNotifyInfoDetail(list);
        
        // 有効/無効
        if (this.m_confirmValid.getSelection()) {
        	info.setValidFlg(ValidConstant.TYPE_VALID);
        } else {
        	info.setValidFlg(ValidConstant.TYPE_INVALID);
        }

        return info;
    }

	/**
	 * 入力値を設定した通知イベント情報を返します。<BR>
	 * 入力値チェックを行い、不正な場合は<code>null</code>を返します。
	 * 
	 * @return 通知ステータス情報
	 * 
	 * @see #setValidateResult(String, String)
	 */
    protected ArrayList<NotifyInfoDetail> createInputDataForStatus(
    		ArrayList<NotifyInfoDetail> list,
    		int priority,
    		Button checkStatusNormal, 
    		Button checkStatusInhibition
    		) {
    
    	if(list == null)
    		return null;
    	
    	NotifyStatusInfo status = new NotifyStatusInfo();
    	
    	// 重要度
    	status.setPriority(priority);
    	
    	// ステータス通知
        if (checkStatusNormal.getSelection()) {
        	status.setValidFlg(new Integer(ValidConstant.TYPE_VALID));
        } else {
        	status.setValidFlg(new Integer(ValidConstant.TYPE_INVALID));
        }
        
        // 抑制
        if (checkStatusInhibition.getSelection()) {
            // 抑制がチェックされているが、抑制条件が指定されていない場合
        	if(checkStatusNormal.getSelection() && this.m_notifyInhibition.m_radioNoInhibition.getSelection()){
                this.setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.notify.16"));
                return null;
        	}
    		status.setInhibitionFlg(new Integer(ValidConstant.TYPE_VALID));
        } else {
        	status.setInhibitionFlg(new Integer(ValidConstant.TYPE_INVALID));
        }
        
        // ステータス通知の共通内容（重要度に関係なく全て同じものを設定する）
        // ステータス情報の存続期間
        if (this.m_comboStatusValidPeriod.getText() != null
        		&& !"".equals(this.m_comboStatusValidPeriod.getText())){
        	status.setStatusValidPeriod(StatusValidPeriodConstant.stringToType(this.m_comboStatusValidPeriod.getText()));
        } else {
        	this.setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.notify.12"));
        	return null;
        }
        
        // 存続期間経過後の処理
        if (this.m_radioStatusDelete.getSelection()) {
        	status.setStatusInvalidFlg(StatusExpirationConstant.TYPE_DELETE);
        } else {
            status.setStatusInvalidFlg(StatusExpirationConstant.TYPE_UPDATE);
        }
        
        // 更新されていない場合に通知する際の重要度
        if (this.m_comboStatusPriority.getText() != null
                && !"".equals((this.m_comboStatusPriority.getText()).trim())) {
        	status.setStatusUpdatePriority(new Integer(PriorityConstant.stringToType(this.m_comboStatusPriority.getText())));
        }
        
        list.add(status);
        return list;
    }
    
	/**
	 * 入力値チェックをします。
	 * 
	 * @return 検証結果
	 */
    protected ValidateResult validate() {
        // 入力値生成
        this.inputData = this.createInputData();

        if (this.inputData != null) {
            return super.validate();
        } else {
            return validateResult;
        }
    }
    
    /**
     * 入力値をマネージャに登録します。
     * 
     * @return true：正常、false：異常
     * 
     * @see com.clustercontrol.dialog.CommonDialog#action()
     */
    protected boolean action() {
    	boolean result = false;
    	
    	NotifyInfo info = this.getInputData();
    	if(info != null){
            if(this.notifyId == null){
            	// 作成の場合
            	result = new AddNotify().add(info);
            }
            else{
            	// 変更の場合
            	result = new ModifyNotify().modify(info);
            }
    	}
    	
    	return result;
    }

	/**
	 * ＯＫボタンのテキストを返します。
	 * 
	 * @return ＯＫボタンのテキスト
	 */
    protected String getOkButtonText() {
        return Messages.getString("ok");
    }

	/**
	 * キャンセルボタンのテキストを返します。
	 * 
	 * @return キャンセルボタンのテキスト
	 */
    protected String getCancelButtonText() {
        return Messages.getString("cancel");
    }
    
	/**
	 * コンポジットの選択可/不可を設定します。
	 * 
	 * @param enable 選択可の場合、<code> true </code>
	 */
    protected void setEnabled(boolean enable) {

    	super.m_notifyBasic.setEnabled(enable);
    	super.m_notifyInhibition.setEnabled(enable);
    	this.setEnabledForStatuses(enable);
    	this.setEnabledForStatusDuration();
    	
    }    
    
    /**
	 * ステータスの通知関連コンポジットの選択可/不可を設定します。
	 * 
	 * @param enable 選択可の場合、<code> true </code>
	 * 
	 * @see #setEnabledForStatuses(boolean, Button, Button)
	 */
    private void setEnabledForStatuses(boolean enable) {

    	// 通知チェックボックス
    	this.m_checkStatusNormalInfo.setEnabled(enable);
    	this.m_checkStatusNormalWarning.setEnabled(enable);
    	this.m_checkStatusNormalCritical.setEnabled(enable);
    	this.m_checkStatusNormalUnknown.setEnabled(enable);
    	
    	//　抑制チェックボックス
    	if(enable) {
	    	this.m_checkStatusInhibitionInfo.setEnabled(this.m_checkStatusNormalInfo.getSelection());
	    	this.m_checkStatusInhibitionWarning.setEnabled(this.m_checkStatusNormalWarning.getSelection());
	    	this.m_checkStatusInhibitionCritical.setEnabled(this.m_checkStatusNormalCritical.getSelection());
	    	this.m_checkStatusInhibitionUnknown.setEnabled(this.m_checkStatusNormalUnknown.getSelection());
    	}

    }
    
    /**
     * ステータスの通知情報の選択可/不可を設定します。
     * 
     */
    private void setEnabledForStatusDuration() {
    	
    	boolean enable = false;
    	
    	// 全重要度の「通知」のチェックボックスのチェック内容と表示/非表示を確認する。
    	if ((this.m_checkStatusNormalInfo.getSelection() && this.m_checkStatusNormalInfo.getEnabled()) || 
    			(this.m_checkStatusNormalWarning.getSelection() && this.m_checkStatusNormalWarning.getEnabled()) || 
    			(this.m_checkStatusNormalCritical.getSelection() && this.m_checkStatusNormalCritical.getEnabled()) || 
    			(this.m_checkStatusNormalUnknown.getSelection() && this.m_checkStatusNormalUnknown.getEnabled())) {
    	
    		enable = true;
    	}
    	else {
    		enable = false;
    	}
    		
    	this.m_comboStatusValidPeriod.setEnabled(enable);
    	this.m_radioStatusDelete.setEnabled(enable);
    	this.m_radioStatusUpdate.setEnabled(enable);
        if (enable) {
            this.m_comboStatusPriority.setEnabled(this.m_radioStatusUpdate.getSelection());
        } else {
            this.m_comboStatusPriority.setEnabled(enable);
        }

    }
     
	/**
	 * 無効な入力値の情報を設定します。
	 * 
	 * @param id ID
	 * @param message メッセージ
	 */
    protected void setValidateResult(String id, String message) {

        this.validateResult = new ValidateResult();
        this.validateResult.setValid(false);
        this.validateResult.setID(id);
        this.validateResult.setMessage(message);
    }
    
	/**
	 * ボタンを生成します。<BR>
	 * 参照フラグが<code> true </code>の場合は閉じるボタンを生成し、<code> false </code>の場合は、デフォルトのボタンを生成します。
	 * 
	 * @param parent ボタンバーコンポジット
	 * 
	 * @see #createButtonsForButtonBar(Composite)
	 */
    protected void createButtonsForButtonBar(Composite parent) {
    	
        if(!this.referenceFlg){
        	super.createButtonsForButtonBar(parent);
        }
        else{
        	// 閉じるボタン
            this.createButton(parent, IDialogConstants.CANCEL_ID, IDialogConstants.CLOSE_LABEL, false);
        }
    }
    
	/**
	 * 重要度のラベルを返します。
	 * 
	 * @param parent 親のコンポジット
	 * @param text ラベルに表示するテキスト
	 * @param background ラベルの背景色
	 * @return 生成されたラベル
	 */
    protected Label getLabelPriority(Composite parent,
    		String text,
    		Color background
    		) {
    	
    	// ラベル（重要度）
    	Label label = new Label(parent, SWT.NONE);
    	GridData gridData = new GridData();
    	gridData.horizontalSpan = WIDTH_PRIORITY;
    	gridData.horizontalAlignment = GridData.FILL;
    	gridData.grabExcessHorizontalSpace = true;
    	label.setLayoutData(gridData);
    	label.setText(text + " : ");
    	label.setBackground(background);
        
        return label;
    }
    
	/**
	 * ステータスの通知のチェックボックスを返します。
	 * 
	 * @param parent 親のコンポジット
	 * @return 生成されたチェックボックス
	 */
    protected Button getCheckStatusNormal(Composite parent) {
    	
        // チェックボックス（通知する）
    	Button button = new Button(parent, SWT.CHECK);
        GridData gridData = new GridData();
        gridData.horizontalSpan = WIDTH_CHECK;
        gridData.horizontalAlignment = GridData.CENTER;
        gridData.grabExcessHorizontalSpace = true;
        button.setLayoutData(gridData);
        
        return button;
    }
    
    /**
	 * ステータスの抑制のチェックボックスを返します。
	 * 
	 * @param parent 親のコンポジット
	 * @return 生成されたチェックボックス
	 */
    protected Button getCheckStatusInhibition(Composite parent) {
    	
        // チェックボックス（抑制する）
        Button button = new Button(parent, SWT.CHECK);
        GridData gridData = new GridData();
        gridData.horizontalSpan = WIDTH_CHECK;
        gridData.horizontalAlignment = GridData.CENTER;
        gridData.grabExcessHorizontalSpace = true;
        button.setLayoutData(gridData);
        
        return button;
    }
    
}