/*
                                                                                                
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
 
*/

package com.clustercontrol.snmppoller.jmx;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;

import javax.naming.CompositeName;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.Name;
import javax.naming.NamingException;

import org.jboss.naming.NonSerializableFactory;
import org.jboss.system.ServiceMBeanSupport;

import com.clustercontrol.snmppoller.SnmpSharedTable;

/**
 * @jmx.mbean
 *     name="user:service=SnmpPollerService,name=SnmpPollerService"
 *     description="SnmpPollerService MBean"
 *     extends="org.jboss.system.ServiceMBean"
 * 
 * @jboss.service servicefile="snmppoller"
 */
public class SnmpPollerService extends ServiceMBeanSupport implements SnmpPollerServiceMBean{
	private String jndiName = "SnmpPollerService";  // JNDIに登録する際の名前
	private String groupName = "SNMP"; // Quartzに登録する際の名前
	private String quartzJndiName = null;  // Quartzをlookupする際のJNDI名

	private int defaultPort = -1;
	private int defaultVersion = 1;
	private String defaultCommunity = "public";
	private int defaultRetries = -1;
	private int defaultTimeout = 1000;
	
	// 複数OIDに対してポーリングする際、レスポンスのOIDのインデックスが一致しているかのチェックフラグ
	private boolean indexCheckFlg = false; 
	
	private String checkInterval = "34 */1 * * * ? *"; // チェックジョブの起動条件
	private List oidList;
		
    private SnmpSharedTable snmpSharedTable;
	
	public SnmpPollerService(){
		oidList = new ArrayList();
	}
	
    /**
     * @jmx.managed-attribute
     * @return
     */
    public String getJndiName() {
        return jndiName;
    }
    
    /**
     * @jmx.managed-attribute
     * 
     * @param jndiName
     * @throws NamingException 
     * @throws NamingException
     */
    public void setJndiName(String jndiName) throws NamingException {
        log.info("setJndiName() SnmpService : " + jndiName);
    	
        String oldName = this.jndiName;
        this.jndiName = jndiName;
        
        if (super.getState() == STARTED) {
        	try {
        		unbind(oldName);
        	} catch (NamingException e){
        		NamingException ne = new NamingException("Failed to unbind");
        		ne.setRootCause(e);
        		log.error(ne);
        	}

        	try{
        		rebind();
        	} catch (NamingException e){
        		NamingException ne = new NamingException("Failed to rebind");
        		ne.setRootCause(e);
        		log.error(ne);
        	}
        }
    }
    
    /**
     * グループ名を返します。
     * @return グループ名
     * 
     * @jmx.managed-attribute
     */
	public String getGroupName() {
		return groupName;
	}

	/**
	 * グループ名を設定します。
	 * @param groupName グループ名
	 * 
     * @jmx.managed-attribute
	 */
	public void setGroupName(String groupName) {
		this.groupName = groupName;
	}
    
	/**
	 * QuartzのサービスのJNDI名を返します。
	 * @return QuartzサービスのJNDI名
	 *
     * @jmx.managed-attribute
	 */
	public String getQuartzJndiName() {
		return quartzJndiName;
	}

	/**
	 * QuartzのサービスのJNDI名を設定します。
	 * @param jndiName QuartzサービスのJNDI名
	 * 
     * @jmx.managed-attribute
   	 */
	public void setQuartzJndiName(String jndiName) {
		quartzJndiName = jndiName;
	}
	
    /**
     * @jmx.managed-attribute
     */
    public String getName() {
        return "SnmpPollerService(" + jndiName + ")";
    }
    
    /**
     * @jmx.managed-attribute
     * 
     * @return
     */
	public int getDefaultPort() {
		return defaultPort;
	}

	/**
     * @jmx.managed-attribute
	 * 
	 * @param defaultPort
	 */
	public void setDefaultPort(int defaultPort) {
		this.defaultPort = defaultPort;
	}

    /**
     * @jmx.managed-attribute
     */
	public int getDefaultVersion() {
		return defaultVersion;
	}

    /**
     * @jmx.managed-attribute
     * @param defaultVersion
     */
	public void setDefaultVersion(int defaultVersion) {
		this.defaultVersion = defaultVersion;
	}

    /**
     * @jmx.managed-attribute
     */
	public String getDefaultCommunity() {
		return defaultCommunity;
	}

    /**
     * @jmx.managed-attribute
     * @param defaultCommunity
     */
	public void setDefaultCommunity(String defaultCommunity) {
		this.defaultCommunity = defaultCommunity;
	}
	
	/**
     * @jmx.managed-attribute
	 * 
	 * @return
	 */
	public int getDefaultRetries() {
		return defaultRetries;
	}

	/**
     * @jmx.managed-attribute
	 * 
	 * @param defaultRetries
	 */
	public void setDefaultRetries(int defaultRetries) {
		this.defaultRetries = defaultRetries;
	}

	/**
     * @jmx.managed-attribute
	 * 
	 * @return
	 */
	public int getDefaultTimeout() {
		return defaultTimeout;
	}

	/**
     * @jmx.managed-attribute
	 * 
	 * @param defaultTimeout
	 */
	public void setDefaultTimeout(int defaultTimeout) {
		this.defaultTimeout = defaultTimeout;
	}
	
    /**
     * @jmx.managed-attribute
     */
    public String getOids() {
    	String oidSetString = null;
    	
    	Iterator itr = oidList.iterator();
    	while(itr.hasNext()){
    		String oid = (String)itr.next();
    		oidSetString = oidSetString + "\n" + oid;
    	}
    	
    	return oidSetString;
    }

    /**
     * @jmx.managed-attribute
     * @param oids
     */
    public void setOids(String oids) {
    	ByteArrayInputStream inputStream = new ByteArrayInputStream(oids.getBytes());
    	Properties properties = new Properties();
    	try {
    		properties.load(inputStream);
    	} catch (IOException e) {
    		IOException ioe = new IOException("Failed to set OID list.");
    		log.error(ioe);
    	}
    	Enumeration enu = properties.elements();
    	while(enu.hasMoreElements()){
    		oidList.add((String)enu.nextElement());
    	}
    }
    
    /**
     * @jmx.managed-attribute
     */
    public String getCheckInterval() {
		return checkInterval;
	}

    /**
     * @jmx.managed-attribute
     * @param checkInterval
     */
	public void setCheckInterval(String checkInterval) {
		this.checkInterval = checkInterval;
	}
    
    /**
     * @jmx.managed-attribute
     */
    public boolean getIndexCheckFlg() {
		return indexCheckFlg;
	}

    /**
     * @jmx.managed-attribute
     * @param indexCheckFlg
     */
	public void setIndexCheckFlg(boolean indexCheckFlg) {
		this.indexCheckFlg = indexCheckFlg;
	}
    
    public void createService() {
        log.info("Create SnmpPollerService(" + jndiName + ")");
 
        snmpSharedTable = new SnmpSharedTable(jndiName, groupName);
        snmpSharedTable.setQuartzJndiName(quartzJndiName);
        snmpSharedTable.setCheckInterval(checkInterval);
        snmpSharedTable.setDefaultPort(defaultPort);
        snmpSharedTable.setDefaultVersion(defaultVersion);
        snmpSharedTable.setDefaultCommunity(defaultCommunity);
        snmpSharedTable.setDefaultRetries(defaultRetries);
        snmpSharedTable.setDdefaultTimeout(defaultTimeout);
        snmpSharedTable.setOidList(oidList);
        snmpSharedTable.setIndexCheckFlg(indexCheckFlg);

        log.info("Created SnmpPollerService(" + jndiName + ")");
    }
    
    public void startService() throws NamingException {
        log.info("Start SnmpPollerService(" + jndiName + ")");
    	// 周期チェックジョブを登録する
    	snmpSharedTable.scheduleCheckJob();
        
    	rebind();
        log.info("Started SnmpPollerService(" + jndiName + ")");
    }
    
    public void stopService() throws NamingException {
        log.info("Stop SnmpPollerService(" + jndiName + ")");
    	unbind(jndiName);
    	
        // 周期チェックジョブを削除する
        snmpSharedTable.deleteCheckJob();
        
        // テーブルを全て削除する
        snmpSharedTable.deleteTableAll();
        
        log.info("Stoped SnmpPollerService(" + jndiName + ")");
    }
    
    public void destroyService() throws Exception {
        log.info("Destroy SnmpPollerService(" + jndiName + ")");
        
        snmpSharedTable = null;

        log.info("Destroyed SnmpPollerService(" + jndiName + ")");
    }
    
    private static Context createContext(Context rootCtx, Name name) throws NamingException {
    	Context subctx = rootCtx;
    	
    	for (int n = 0; n < name.size(); n++) {
    		String atom = name.get(n);
    		
    		try {
    			Object obj = subctx.lookup(atom);
    			subctx = (Context) obj;
    		} catch (NamingException e) {
    			// 存在しない場合は、サブコンテキストを生成
    			subctx = subctx.createSubcontext(atom);
    		}
    	}
    	
    	return subctx;
    }
    
    private void rebind() throws NamingException {
    	InitialContext rootCtx = new InitialContext();
    	
    	Name fullName = rootCtx.getNameParser("").parse(jndiName);
    	Name parentName = fullName;
    	if(fullName.size() > 1){
    		parentName = fullName.getPrefix(fullName.size()-1);
    	} else {
    		parentName = new CompositeName();
    	}
    	
    	Context parentCtx = createContext(rootCtx, parentName);
    	Name atomName = fullName.getSuffix(fullName.size()-1);
    	String atomStirng = atomName.get(0);
    	
    	NonSerializableFactory.rebind(parentCtx, atomStirng, snmpSharedTable);
    }
    
    private void unbind(String jndiName) throws NamingException {
    	InitialContext rootCtx = null;
    	
    	try {
    		rootCtx = new InitialContext();
    		
    		Name fullName = rootCtx.getNameParser("").parse(jndiName);
    		Name atomName = fullName.getSuffix(fullName.size() - 1);
    		String atom = atomName.get(0);
    		
    		rootCtx.unbind(jndiName);
    		NonSerializableFactory.unbind(atom);
    	} finally {
    		if(rootCtx != null) { 
    			rootCtx.close(); 
    		}
    	}
    }
}
