/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 */

package com.clustercontrol.snmppoller;

import java.net.InetAddress;
import java.net.SocketException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.opennms.protocols.snmp.SnmpEndOfMibView;
import org.opennms.protocols.snmp.SnmpHandler;
import org.opennms.protocols.snmp.SnmpNoSuchInstance;
import org.opennms.protocols.snmp.SnmpNoSuchObject;
import org.opennms.protocols.snmp.SnmpNull;
import org.opennms.protocols.snmp.SnmpObjectId;
import org.opennms.protocols.snmp.SnmpParameters;
import org.opennms.protocols.snmp.SnmpPduPacket;
import org.opennms.protocols.snmp.SnmpPduRequest;
import org.opennms.protocols.snmp.SnmpPeer;
import org.opennms.protocols.snmp.SnmpSession;
import org.opennms.protocols.snmp.SnmpSyntax;
import org.opennms.protocols.snmp.SnmpTimeTicks;
import org.opennms.protocols.snmp.SnmpUInt32;
import org.opennms.protocols.snmp.SnmpV2Error;
import org.opennms.protocols.snmp.SnmpVarBind;

import com.clustercontrol.sharedtable.bean.DataTable;

/**
 * SNMPのポーリングを実行するクラス
 * 
 * @version 2.0.0
 * @since 2.0.0
 */
public class SnmpPoller implements SnmpHandler {
	protected static Log m_log = LogFactory.getLog(SnmpPoller.class);

	/** IPアドレス */
	private String m_ipAddress;

	// 収集のターゲットとなるOID（全てのポーリングが終了するまで値を保持）
	private String[] m_oidText;

	// 収集を開始するOID（ポーリングの度に更新）
	private ArrayList<SnmpObjectId> m_startOids;
	
	// 収集を終了するOID（ポーリングの度に更新）
	private ArrayList<SnmpObjectId> m_stopOids;
	
	// 取得したデータを格納するテーブル
	DataTable m_dataTable;

	// メッセージ 
	private String m_message = null;

	// 待機フラグ
	private boolean m_waitFlg;
	
	// 収集したOIDの最後のインデックスをチェックするか否かのフラグ	
	private boolean m_indexCheckFlg = false;
	
	// 全体的なポーリングをリトライするか否かのフラグ
	private boolean m_retryFlg = false;
	
	// 全体的なポーリングのリトライ回数
	private int m_retryCount= 1;
	
	// エラーコード
	private int m_errorCode = 0;

	// エラーコード
	private final int Normal = 0; // 正常に動作いている場合
	private final int TimeOutError = 1; // 問い合わせがタイムアウトした場合
	private final int InternalError = 2; // タイムアウト以外のエラー

	/**
	 * メインルーチン　
	 * IPアドレスと　Hashtableを受け取り、
	 * ポーリングした結果をHashtableに代入する
	 */
	public void polling(
			InetAddress ipAddress, 
			int port, 
			int version,
			String community, 
			int retries, 
			int timeout, 
			List oidList,
			boolean indexCheckFlg,
			DataTable table) {
		// デバッグ出力
		if (m_log.isDebugEnabled()) {
			m_log.debug("polling() start :" + ipAddress.toString());
			m_log.debug("Port      : " + port);
			m_log.debug("Version   : " + version);
			m_log.debug("Community : " + community);
			m_log.debug("Retries   : " + retries);
			m_log.debug("Timeout   : " + timeout);
			m_log.debug("IndexCheckFlg : " + indexCheckFlg);
		}

		m_errorCode = Normal;
		m_dataTable = table;

		m_ipAddress = ipAddress.toString();

		m_indexCheckFlg = indexCheckFlg;
		
		SnmpPeer peer = new SnmpPeer(ipAddress);
		if (port != -1)
			peer.setPort(port);

		if (timeout != -1)
			peer.setTimeout(timeout);

		if (retries != -1)
			peer.setRetries(retries);

		// peer　を初期化
		SnmpParameters parms = peer.getParameters();
		parms.setVersion(version);
		if (community != null)
			parms.setReadCommunity(community);

		//　sessionを生成
		SnmpSession session = null;
		try {
			session = new SnmpSession(peer);
		} catch (SocketException e) {
			m_log.error("polling() warning  :" + ipAddress.toString()
					+ " SocketException creating the SNMP session");
			m_errorCode = InternalError;
			return;
		}
		session.setDefaultHandler(this);

		try {
			m_oidText = new String[oidList.size()];
			m_startOids = new ArrayList<SnmpObjectId>(oidList.size());
			m_stopOids = new ArrayList<SnmpObjectId>(oidList.size());
			int i = 0;

			SnmpPduRequest pdu = new SnmpPduRequest(SnmpPduPacket.GETNEXT);
			pdu.setRequestId(SnmpPduPacket.nextSequence());
			
			Iterator itr = oidList.iterator();
			while (itr.hasNext()) {
				m_oidText[i] = (String) itr.next();
				
				// OIDの最後が0で終わる場合は、.0を削除する
				// GETNEXTで取得するため
				if(m_oidText[i].endsWith(".0")){
					m_oidText[i] = m_oidText[i].substring(0, m_oidText[i].lastIndexOf(".0"));
				}
				
				// 開始OIDを設定
				SnmpObjectId startOid = new SnmpObjectId(m_oidText[i]);

				// 終了するOIDを設定
				SnmpObjectId stopOid = new SnmpObjectId(m_oidText[i]);
				int[] ids = stopOid.getIdentifiers();
				++ids[ids.length - 1];
				stopOid.setIdentifiers(ids);

				SnmpObjectId oId = new SnmpObjectId(m_oidText[i]);
				pdu.addVarBind(new SnmpVarBind(oId));
				m_startOids.add(startOid);
				m_stopOids.add(stopOid);
				
				i++;
			}
			
			for (int count = 0; count < m_retryCount+1; count++) {
				synchronized (session) {
					m_waitFlg = true;
					session.send(pdu);
					m_log.debug("polling() : " + m_ipAddress + " send SnmpPduRequest");

					if (m_waitFlg) {
						session.wait();
					}
				}
				
				if(m_retryFlg == false){
					//　リトライしない
					break;
				}
			}
		} catch (InterruptedException e) {
			m_log.error("polling() warning :" + ipAddress.toString()
					+ " polling failed at InterruptedException");
			m_errorCode = InternalError;
			return;
		} finally {
			try {

				session.close();

			} catch (Exception e) {
				/**
				 * FIXME
				 * Joe-SNMPの不具合の回避コード
				 * Joe-SNMPで不具合が修正されたら削除すること
				 */
				m_log
						.warn("polling():" + m_ipAddress
								+ " Session close failed");
			}
		}

		// 例外処理(未実装)
		if (this.m_errorCode == TimeOutError) {

		} else if (this.m_errorCode == InternalError) {

		}

		// デバッグ出力
		if (m_log.isDebugEnabled()) {
			m_log.debug("polling() end :" + ipAddress.toString());
		}
	}

	public void snmpReceivedPdu(SnmpSession session, int cmd, SnmpPduPacket pdu) {
		//		m_log.debug("snmpReceivedPdu() start");

		try {
			long time = System.currentTimeMillis(); // 取得時刻

			SnmpPduRequest req = null;
			if (pdu instanceof SnmpPduRequest) {
				req = (SnmpPduRequest) pdu;
			} else {
				m_log.error("polling() error :" + session.toString()
						+ " Received non-request pdu");
				synchronized (session) {
					m_errorCode = InternalError;
					session.notify();
				}
				return;
			}

			if (pdu.getCommand() != SnmpPduPacket.RESPONSE) {
				m_log
						.error("polling() error :" + session.getPeer().getPeer().toString()
								+ "  Received non-response command "
								+ pdu.getCommand());
				synchronized (session) {
					m_errorCode = InternalError;
					session.notify();
				}
				return;
			}

			if (req.getErrorStatus() != 0) {
				SnmpVarBind[] bind = req.toVarBindArray();
				for(int i=0; i<bind.length; i++){
					m_log.error("polling() error :" + session.getPeer().getPeer().toString()
							+ "  Error Status " + req.getErrorStatus()
							+ "  " + bind[i].getName());
				}
				synchronized (session) {
					m_errorCode = InternalError;
					session.notify();
				}
				return;
			}

			// 次の pduを作成
			SnmpPduRequest nxt = new SnmpPduRequest(SnmpPduPacket.GETNEXT);
			nxt.setRequestId(SnmpPduPacket.nextSequence());

			// インデックスチェックフラグが設定されている場合にOIDの最後のインデックスが
			// 一致しているか否かをチェックするのに使用
			String lastIndexStr = null;

			// 次回のポーリングでpduにセットするOIDの開始と終了のOIDを保持する
			ArrayList<SnmpObjectId> nextStartOids = new ArrayList<SnmpObjectId>(m_startOids.size());
			ArrayList<SnmpObjectId> nextStopOids = new ArrayList<SnmpObjectId>(m_stopOids.size());
			
			// pduの戻りとして帰ってきた SnmpVarBindを順次処理する
			int length = pdu.getLength();
			for (int i = 0; i < length; i++) {
				SnmpVarBind var = pdu.getVarBindAt(i);
				
				// ポーリングの終了判定
				// 条件：
				// 戻りの型が EndOfMibView の場合
				// 停止条件のOIDにマッチした場合は終了
				if (var.getValue().typeId() == SnmpEndOfMibView.ASNTYPE
						|| ((m_stopOids.get(i) != null && m_stopOids.get(i).compare(var.getName()) < 0))) {
			
					// 該当OIDの収集を終了します
					if (m_log.isDebugEnabled()) {
						m_log.debug(m_ipAddress + " stop polling. " + i + ", "
								+ m_startOids.get(i) + ", " + m_stopOids.get(i)
								+ " - " + var.getName());
					}
					continue;
				}

				// OIDを設定
				String oidString = var.getName().toString();

				// デバッグ出力
				if (m_log.isDebugEnabled()) {
					m_log.debug("get " + m_ipAddress + " " + oidString + " "
							+ time + " " + var.getValue().getClass().getName()
							+ " " + var.getValue().toString());
				}

				if (!m_startOids.get(i).isRootOf(var.getName())) {
					// 帰ってきたMIB値のOIDが開始OIDのツリーの配下でない場合は、
					// 該当OIDの収集を終了します
					if (m_log.isDebugEnabled()) {
						m_log.debug(m_ipAddress + " stop polling.. " + i + ", "
								+ m_startOids.get(i) + ", " + m_stopOids.get(i)
								+ " - " + var.getName());
					}
					continue;
				} else {
					boolean putFlg = false;  // テーブルに値をputしたか否かを示すフラグ
					if (var.getValue() instanceof SnmpNoSuchInstance) {
						// 何もしない
					} else if (var.getValue() instanceof SnmpNoSuchObject) {
						// 何もしない
					} else if (var.getValue() instanceof SnmpV2Error) {
						// 何もしない
					} else if (var.getValue() instanceof SnmpNull) {
						// 何もしない
					} else if (var.getValue() instanceof SnmpTimeTicks) {
						// デバッグ
						if (m_log.isDebugEnabled()) {
							m_log.debug("set " + m_ipAddress + "," + oidString
									+ "," + time + ","
									+ var.getValue().toString());
						}

						long value = ((SnmpUInt32) var.getValue()).getValue();
						m_dataTable.putValue(oidString, time, String.valueOf(value));
						putFlg = true;
					} else {
						// デバッグ
						if (m_log.isDebugEnabled()) {
							m_log.debug("set " + m_ipAddress + "," + oidString
									+ "," + time + ","
									+ var.getValue().toString());
						}

						m_dataTable.putValue(oidString, time, var.getValue().toString());
						putFlg = true;
					}
					
					// インデックスチェックフラグが設定されている場合は、OIDの最後のインデックスが
					// 一致しているか否かをチェックする
					if(putFlg == true && m_indexCheckFlg == true){
						if(lastIndexStr == null){
							lastIndexStr = oidString.substring(oidString.lastIndexOf("."));
						} else {
							String indexStr = oidString.substring(oidString.lastIndexOf("."));

							// OIDの最後のインデックスが一致しない場合
							if (!lastIndexStr.equals(indexStr)) {
								m_log.debug("Index not match. " + m_ipAddress
										+ ", " + lastIndexStr + ", " + oidString);
								
								// 格納されたMIB値を削除します。
								for(int j=0; j<m_oidText.length; j++){
									removeTableValue(m_oidText[j]);
								}
								
								// 収集を終了します
								synchronized (session) {
									// リトライするようにフラグを設定する
									m_retryFlg = true;
									session.notify();
								}
								return;
							}
							lastIndexStr = indexStr;
						}
					}
					
					nxt.addVarBind(new SnmpVarBind(var.getName()));
					// 次のポーリングで収集するものだけを追加
					nextStartOids.add(m_startOids.get(i));
					nextStopOids.add(m_stopOids.get(i));
				}
			}
			
			if (nxt.getLength() != 0) {
				//				m_log.debug("session.send() start");
				m_startOids = nextStartOids;
				m_stopOids = nextStopOids;
				session.send(nxt, this);
			} else {
				// 収集を終了します
				synchronized (session) {
					session.notify();
				}
				return;
			}
		} catch (Exception e) { // 何か例外が生じた場合は収集を停止する
			m_log.error("InternalError ", e);

			// 収集を終了します
			synchronized (session) {
				m_errorCode = InternalError;
				m_message = e.getMessage();
				session.notify();
			}
		}
	}

	/* (非 Javadoc)
	 * @see org.opennms.protocols.snmp.SnmpHandler#snmpInternalError(org.opennms.protocols.snmp.SnmpSession, int, org.opennms.protocols.snmp.SnmpSyntax)
	 */
	public void snmpInternalError(SnmpSession session, int err, SnmpSyntax pdu) {
		String message = "InternalError. The error code is " + err + ". IP:"
				+ m_ipAddress + " OID:" + m_oidText[0];
		m_log.error("snmpInternalError():" + session.toString() + " " + message);

		synchronized (session) {
			m_errorCode = InternalError;
			m_message = message;
			
			// 既に収集されたものも含めて全て削除する
			for (int j = 0; j < m_oidText.length; j++) {
				// 収集中のOID配下のOIDの値をテーブルから削除
				removeTableValue(m_oidText[j]);
			}
			
			m_waitFlg = false;
			session.notify();
		}
	}

	/* (非 Javadoc)
	 * @see org.opennms.protocols.snmp.SnmpHandler#snmpTimeoutError(org.opennms.protocols.snmp.SnmpSession, org.opennms.protocols.snmp.SnmpSyntax)
	 */
	public void snmpTimeoutError(SnmpSession session, SnmpSyntax pdu) {
//		m_log.warn("snmpTimeoutError():"
//				+ session.getPeer().getPeer().toString() + " "
//				+ ((SnmpPduRequest) pdu).toVarBindArray()[0].getName()
//				+ " polling failed at TimeoutError");
		
		SnmpVarBind[] bind = ((SnmpPduRequest) pdu).toVarBindArray();
		String ipAddress = session.getPeer().getPeer().toString();
		String message = "Polling failed at TimeoutError." + " IP:" + ipAddress
				+ " OID:" + bind[0].getName();
		
		// より詳細なメッセージを作成
		String addMessage = "";
		if(m_log.isDebugEnabled()){
			for (int i = 1; i < bind.length; i++) {
				addMessage = addMessage + "\n\t" + ipAddress + " " + bind[i].getName();
			}
		}
		
		m_log.warn(message + addMessage);
				
		synchronized (session) {
			m_errorCode = TimeOutError;
			m_message = message;
			
			// 既に収集されたものも含めて全て削除する
			for (int j = 0; j < m_oidText.length; j++) {
				// 収集中のOID配下のOIDの値をテーブルから削除
				removeTableValue(m_oidText[j]);
			}
			
			session.notify();
		}
	}

	/**
	 * 収集指定のOID配下のMIB値を全て削除する
	 * 
	 * 例)
	 * .1.3.6.1.2.1.2.2.1   を対象に収集している場合で
	 * .1.3.6.1.2.1.2.2.1.3 を収集の際にこのメソッドを呼び出した場合
	 * .1.3.6.1.2.1.2.2.1   配下のMIB値は全て削除されるため
	 * 既にテーブルに格納済みの
	 * .1.3.6.1.2.1.2.2.1.1 のMIB値および .1.3.6.1.2.1.2.2.1.2
	 * のMIB値を削除する
	 */
	private void removeTableValue(String oidText) {
		// テーブルに格納されているOIDのセットを取得
		Set<String> oidSet = m_dataTable.keySet();

		String[] fullOids = (String[]) oidSet
				.toArray(new String[oidSet.size()]);

		for (int i = 0; i < fullOids.length; i++) {

			String fullOid = fullOids[i];

			// もし先頭の部分が一致するならそのツリーの子孫と判定
			if (fullOid.startsWith(oidText)) {
				// テーブルに格納されている値を削除
				m_dataTable.removeValue(fullOid);
			}
		}
	}
}
