/*
                                                                                                
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
 
*/

package com.clustercontrol.sharedtable.bean;

import java.util.HashMap;
import java.util.Set;

/**
 * キーとオブジェクトのマッピングを保持するクラス
 * 
 * @version 2.0.0
 * @since 2.0.0
 */
public class DataTable implements Cloneable {
	HashMap<String, ValueObject> m_hm;   // キーと値オブジェクトのマッピングを保持するマップ
	long m_lastReference = 0;	// 最終参照日時	
	long m_lastReferenceInterval;  // 最終更新時刻とその前の参照時刻との差分の時間
	
	/**
	 * 新しい空のテーブルを作成します。
	 */
	public DataTable(){
		m_hm = new HashMap<String, ValueObject>();
		m_lastReference = System.currentTimeMillis();
	}

	private DataTable(HashMap<String, ValueObject> hp, long lastReference){
		m_hm = hp;
	}
	
	/**
	 * 指定された値と指定されたキーをこのテーブルに関連付けます。
	 * 
	 * @param key 指定される値が関連付けられるキー
	 * @param date 指定される値に関連付けられる時刻
	 * @param value 指定されるキーに関連付けられる値
	 */
	synchronized public void putValue(String key, long date, Object value) {
		m_hm.put(key, new ValueObject(key, date, value));
	}

	/**
	 * 指定されたテーブルから全ての値をテーブルにコピーします。
	 * 
	 * @param table 格納する値のセットが格納されているテーブル
	 */
	synchronized public void putValueAll(DataTable table) {
		m_hm.putAll(table.m_hm);
	}

	/**
	 * テーブルで保持していた値を全て削除し、
	 * 指定されたテーブルで全ての値を書き換えます。
	 * 
	 * @param table 格納する値のセットが格納されているテーブル
	 */
	synchronized public void setValueAll(DataTable table) {
		m_hm.clear();
		m_hm.putAll(table.m_hm);
	}
	
	/**
	 * 指定されたキーにマップされている値を返します。
	 * 
	 * @param key 関連付けられた値が返されるキー
	 * @return 指定されたキーにマッピングしている値オブジェクト。
	 *          このキーに対する値オブジェクがテーブルにない場合は null
	 */
	synchronized public ValueObject getValue(String key){
		ValueObject vo = (ValueObject)m_hm.get(key);
	
		// 最終更新時刻からの経過時間を算出
		long now = System.currentTimeMillis();
		m_lastReferenceInterval = now - m_lastReference;
		
		m_lastReference = now;  // 現時刻を設定
		
		return vo;
	}
	
	/**
	 * 指定されたキーにマップされている値を削除します。
	 * 
	 * @param key 関連付けられた値のキー
	 * @return 指定されたキーにマッピングしている値オブジェクト。
	 *          このキーに対する値オブジェクがテーブルにない場合は null
	 */
	synchronized public ValueObject removeValue(String key){
		return m_hm.remove(key);
	}

	/**
	 * 現在のテーブルを複製した新たなテーブルを返します。
	 * 
	 * @return 現在のテーブルを複製した新たなテーブル
	 */
	synchronized public DataTable getValueAll(){
		DataTable st = (DataTable)this.clone();
		
		// 最終更新時刻からの経過時間を算出
		long now = System.currentTimeMillis();
		m_lastReferenceInterval = now - m_lastReference;
		
		m_lastReference = System.currentTimeMillis();  // 現時刻を設定
		return st;
	}
	
	// clone メソッドをオーバーライド
	protected Object clone(){
		DataTable st = new DataTable((HashMap)m_hm.clone(), this.m_lastReference);
		
		return st;
	}

	/**
	 * 最終参照時刻を返します。
	 * @return 最終参照時刻
	 */
	public long getLastReference() {
		return m_lastReference;
	}
	
	/**
	 * 最後の参照とその前の参照までの時間を返す。
	 * @return 最後の参照とその前の参照までの時間(ミリ秒)
	 */
	public long getLastReferenceInterval(){
		return m_lastReferenceInterval;
	}
	
	/**
	 * キーのセットを返します。
	 * @return キーのセット
	 */
	public Set keySet(){
		return this.m_hm.keySet();
	}
}
