/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.snmp.util;

import java.net.InetAddress;
import java.net.SocketException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.opennms.protocols.snmp.SnmpHandler;
import org.opennms.protocols.snmp.SnmpObjectId;
import org.opennms.protocols.snmp.SnmpParameters;
import org.opennms.protocols.snmp.SnmpPduPacket;
import org.opennms.protocols.snmp.SnmpPduRequest;
import org.opennms.protocols.snmp.SnmpPeer;
import org.opennms.protocols.snmp.SnmpSession;
import org.opennms.protocols.snmp.SnmpSyntax;
import org.opennms.protocols.snmp.SnmpTimeTicks;
import org.opennms.protocols.snmp.SnmpUInt32;
import org.opennms.protocols.snmp.SnmpV2Error;
import org.opennms.protocols.snmp.SnmpVarBind;

import com.clustercontrol.util.Messages;

/**
 * SNMP実行クラス
 *
 * @version 2.0.0
 * @since 2.0.0
 */
public class RequestSnmp implements SnmpHandler {
	
	protected static Log m_log = LogFactory.getLog( RequestSnmp.class );
	
	/**
	 * The version of the SNMP protocol used to communicate
	 */
	private int		m_version 	= 1;
	
	/**
	 * The community string used to "authenticate" the request.
	 */
	private String		m_community  	= "public";
	
	/**
	 * The number of retries to use.
	 */
	int		m_retries 	= -1;
	
	/**
	 * The time period to wait before considering the last transmission
	 * a failure. This should be in milliseconds.
	 */	
	int		m_timeout	= 1000;
	
	/**
	 * The port where request are sent & received from.
	 */	
	int		m_port		= -1;
	
	/** IPアドレス */
	private String m_ipAddress;
	
	/** 収集を開始したOID */
	private SnmpObjectId m_oid;
	
	/** 収集を開始したOID */
	private String m_oidText;
	
	/** 取得値 */
	private String m_value;
	
	/** 取得日時 */
	private long m_date;
	
	/** メッセージ */
    private String m_message = null;

    /** 待機フラグ */
    boolean m_waitFlg;
    
	// エラーコード
	private int m_errorCode = 0;
	
	// エラーコード
	private final int Normal = 0;         // 正常に動作いている場合
	private final int TimeOutError = 1;   // 問い合わせがタイムアウトした場合
	private final int InternalError = 2;  // タイムアウト以外のエラー

	/**
	 * メインルーチン　
	 * IPアドレスと　Hashtableを受け取り、
	 * ポーリングした結果をHashtableに代入する
	 */
	public boolean polling (
			InetAddress ipAddress,
			String community,
			Integer portNumber,
			String oidText,
			Integer version
			)
	{	
		m_log.debug("polling() start :" + ipAddress.toString());
		
		m_errorCode = Normal;
		m_ipAddress = ipAddress.toString();
		m_community = community;
		m_port = portNumber;
		m_oidText = oidText;
		m_version = version;

		m_waitFlg = true;
		
//		if(version == SnmpVersionConstant.TYPE_V1){
//			m_version = SnmpSMI.SNMPV1;
//		} else if (version == SnmpVersionConstant.TYPE_V2){
//			m_version = SnmpSMI.SNMPV2;
//		}
		
		SnmpPeer peer = new SnmpPeer(ipAddress);
		
		if(m_port != -1)
			peer.setPort(m_port);
		
		if(m_timeout != -1)
			peer.setTimeout(m_timeout);
		
		if(m_retries != -1)
			peer.setRetries(m_retries);
		
		//
		// Initialize the peer
		//
		SnmpParameters parms = peer.getParameters();
		
		parms.setVersion(m_version);
		parms.setReadCommunity(m_community);
		
		//
		// Now create the session, set the initial request
		// and walk the tree!
		//
		SnmpSession session = null;
		try
		{
			session = new SnmpSession(peer);
		}
		catch(SocketException e)
		{
			m_log.debug("polling():" + m_ipAddress + " SocketException creating the SNMP session");
			
			m_message = Messages.getString("message.snmp.7") + " (" + e.getMessage() + ")";
			return false;
		}
		
		session.setDefaultHandler(this);

		try
		{
			// build the first request
			SnmpPduRequest pdu = new SnmpPduRequest(SnmpPduRequest.GET);
			
			m_oid = new SnmpObjectId(m_oidText);
			pdu.addVarBind(new SnmpVarBind(m_oid));
			
			synchronized(session)
			{
				session.send(pdu);
				if(m_waitFlg){
					session.wait();
				}
			}
		}
		catch(InterruptedException e) 
		{ 
			m_log.debug("polling():" + m_ipAddress + " polling failed at InterruptedException" );
			
			m_message = Messages.getString("message.snmp.7") + " (" + e.getMessage() + ")";
			return false;
		}
		finally
		{
			try{
				
			session.close();
			
			}
			catch(Exception e) 
			{ 
				/**
				 * FIXME
				 * Joe-SNMPの不具合の回避コード
				 * Joe-SNMPで不具合が修正されたら削除すること
				 */
				m_log.warn("polling():" + m_ipAddress + " Session close failed" );
			}
		}
		
		if(this.m_errorCode != Normal){
			return false;
		}
		return true;
		
	} 
	
	/* (非 Javadoc)
	 * @see org.opennms.protocols.snmp.SnmpHandler#snmpReceivedPdu(org.opennms.protocols.snmp.SnmpSession, int, org.opennms.protocols.snmp.SnmpPduPacket)
	 */
	public void snmpReceivedPdu(SnmpSession session, int cmd, SnmpPduPacket pdu) {
		try{
			// 取得時刻
			long time = System.currentTimeMillis();
			
			m_log.debug("snmpReceivedPdu():" +  session.toString());
			
			SnmpPduRequest req = null;
			if(pdu instanceof SnmpPduRequest)
			{
				req = (SnmpPduRequest)pdu;
			}
			
			if(pdu.getCommand() != SnmpPduPacket.RESPONSE)
			{
				m_log.debug("snmpReceivedPdu():" +  session.toString() + "  Received non-response command " + pdu.getCommand());
				synchronized(session)
				{
					m_errorCode = InternalError;
					m_message = Messages.getString("message.snmp.7") + " Received non-response command " + pdu.getCommand();
					session.notify();
				}
				return;
			}
			
			// 0:noError 	エラーなし
			// 1:TooBig 	PDUサイズが制限を越えた
			// 2:noSuchName 要求されたオブジェクトが存在しない
			// 3:badValue 	SetRequestに矛盾する値が含まれている
			// 4:readOnly 	未定義
			// 5:GenErr 	定義されていないエラー
			if(req.getErrorStatus() != 0)
			{
				m_log.error("snmpReceivedPdu():" +  session.toString() + "  Error Status " + req.getErrorStatus());
				
				synchronized(session)
				{
					m_errorCode = InternalError;
					m_message = Messages.getString("message.snmp.7") + " Error Status:" + req.getErrorStatus();
					session.notify();
				}
				return;
			}
			
			SnmpVarBind[] vars = pdu.toVarBindArray();
			
			for(int x = 0; x < vars.length; x++)
			{
				// デバッグ出力
				m_log.debug(
						vars[x].getName().toString() + 
						" " + time +
						" " + vars[x].getValue().getClass().getName() + 
						" " + vars[x].getValue().toString()
				);
				
				if(m_oid.isRootOf(vars[x].getName())) {
					if(vars[x].getValue() instanceof SnmpV2Error){
						// エラー
						m_errorCode = InternalError;
						m_message = Messages.getString("message.snmp.7") + " SnmpV2Error. Value:" + vars[x].getValue().toString();
					}
					else if(vars[x].getValue() instanceof SnmpTimeTicks){
						m_value = ((SnmpUInt32)vars[x].getValue()).toString();
						m_date = time;
					}
					else{
						m_log.debug("snmpReceivedPdu():" + vars[x].getName().toString() + " ," + time + " ," + vars[x].getValue().toString());

						m_value = vars[x].getValue().toString();
						m_date = time;
					}
				}
			}
			
			// 収集を終了します
			synchronized(session)
			{
				session.notify();
			}
			return;
			
		} catch (Exception e) { // 何か例外が生じた場合は収集を停止する

			// 収集を終了します
			synchronized(session)
			{
				m_errorCode = InternalError;
				m_message = Messages.getString("message.snmp.7") + " (" + e.getMessage() + ")";
				session.notify();
			}
		}
		
	}

	/* (非 Javadoc)
	 * @see org.opennms.protocols.snmp.SnmpHandler#snmpInternalError(org.opennms.protocols.snmp.SnmpSession, int, org.opennms.protocols.snmp.SnmpSyntax)
	 */
	public void snmpInternalError(SnmpSession session, int err, SnmpSyntax pdu) {
		m_log.warn("snmpInternalError():" + session.toString()
				+ " snmpInternalError. The error code is " + err);
		
		synchronized(session)
		{
			m_errorCode = InternalError;
			m_waitFlg = false;
			m_message = Messages.getString("message.snmp.7") + " snmpInternalError. The error code is " + err;
			session.notify();
		}
	}

	/* (非 Javadoc)
	 * @see org.opennms.protocols.snmp.SnmpHandler#snmpTimeoutError(org.opennms.protocols.snmp.SnmpSession, org.opennms.protocols.snmp.SnmpSyntax)
	 */
	public void snmpTimeoutError(SnmpSession session, SnmpSyntax pdu) {
		m_log.warn("snmpTimeoutError():" + session.getPeer().getPeer().toString()
				+ " " + ((SnmpPduRequest)pdu).toVarBindArray()[0].getName()
				+ " polling failed at TimeoutError" );
		
		synchronized(session)
		{
			m_errorCode = TimeOutError;
			m_message = Messages.getString("message.snmp.7") + " snmpTimeoutError." + session.getPeer().getPeer().toString()
									+ " " + ((SnmpPduRequest)pdu).toVarBindArray()[0].getName();
			session.notify();
		}
	}
	
	/**
	 * @return メッセージを戻します。
	 */
	public String getMessage() {
		return m_message;
	}
	
	/**
	 * @return 取得値を戻します。
	 */
	public String getValue() {
		return m_value;
	}
	
	/**
	 * @return 取得日時を戻します。
	 */
	public long getDate() {
		return m_date;
	}
	
	
	// テストコード
	public static void main(String[] args) {
		try {
			RequestSnmp test = new RequestSnmp();
			test.polling(InetAddress.getByName("10.68.154.157"),"public",161,".1.3.6.1.2.1.25.2.3.1.5.7",1 );
		} catch (Exception e) {
		}
	}
}
