/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 */

package com.clustercontrol.performance.composite;

import java.util.List;

import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;

import com.clustercontrol.bean.FacilityPath;
import com.clustercontrol.bean.FacilityTreeItem;
import com.clustercontrol.performance.bean.GraphProperty;
import com.clustercontrol.performance.util.Messages;
import com.clustercontrol.performanceMGR.ejb.bmp.RecordCollectorData;

/**
 * 複数の実績グラフのコンポジットをまとめて表示するコンポジットクラス
 * 
 * @version 1.0
 * @since 1.0
 *  
 */
public class RecordGraphHolderComposite extends Composite {

    private Label idLabel;

    private Label scopeLabel;

    private Composite baseCcomposite = null;

    private RecordGraphComposite[] composite = null;

    private String collectorID; // 表示対象の収集ID

    private String currentPath; // グラフ表示しているスコープパス

    // グラフの設定
    private GraphProperty[] graphPropertys;

    // グラフ表示用データ
    private RecordCollectorData collectorData;

    private List itemInfoList;

    private FacilityTreeItem facilityTree;

    /**
     * コンストラクター
     * 
     * @param parent
     * @param style
     */
    public RecordGraphHolderComposite(Composite parent, int style,
            int maxGraphSize) {
        super(parent, style);

        composite = new RecordGraphComposite[maxGraphSize]; // グラフ表示領域を生成

        initialize();
    }

    /**
     * GUIを初期化
     *  
     */
    private void initialize() {
        GridData gridData1 = new GridData();
        idLabel = new Label(this, SWT.NONE);
        idLabel.setLayoutData(gridData1);

        GridData gridData2 = new GridData();
        scopeLabel = new Label(this, SWT.NONE);
        scopeLabel.setLayoutData(gridData2);

        baseCcomposite = new Composite(this, SWT.NONE);
        baseCcomposite.setLayout(new FillLayout(SWT.VERTICAL));

        GridData gridData3 = new GridData();
        gridData3.grabExcessVerticalSpace = true;
        gridData3.horizontalAlignment = GridData.FILL;
        gridData3.verticalAlignment = GridData.FILL;
        baseCcomposite.setLayoutData(gridData3);

        GridLayout gridLayout = new GridLayout(1, true);
        gridLayout.marginHeight = 0;
        gridLayout.marginWidth = 0;	
        
        this.setLayout(gridLayout);
    }

    /**
     * グラフ描画用のデータを設定します。
     */
    public void setMasterData(RecordCollectorData collectorData,
            List itemInfoList, FacilityTreeItem facilityTree) {
        this.collectorData = collectorData;
        this.itemInfoList = itemInfoList;
        this.facilityTree = facilityTree;
        this.collectorID = collectorData.getCollectorID();
        // グラフ表示領域を消去
        if (graphPropertys != null) {
        	for (int i = 0; i < this.graphPropertys.length; i++) {
                disposeGraph(i);
            }
        }
    }

    /**
     * 指定の設定でグラフを再表示します。
     */
    public void redrawGraph(GraphProperty[] graphProperty) {
        this.graphPropertys = graphProperty;

        // ラベルを設定
        FacilityPath fp = new FacilityPath(Messages.getString("SEPARATOR"));
        currentPath = fp.getPath(graphProperty[0].getTargetFacility());
        setLabel();
        
        for (int i = 0; i < this.graphPropertys.length; i++) {
            // グラフ描画
            // グラフ種別が非表示の場合もしくは、
            // グラフ種別がサブスコープ代表値であるが、選択のスコープに子スコープがない場合
            if (graphProperty[i].getGraphType() == GraphProperty.INVISIBLE
                    || graphProperty[i].getGraphType() == GraphProperty.TYPE3
                    && graphProperty[i].getTargetFacility().getChildren().length == 0) {
            	// 結局、ここでは何もしない。
            } else {
                // ここが本来の仕事をするメインルート。
                // compositeに対する操作であることに注意。
                getGraph(i).drawGraph(graphProperty[i]);
            }          
        }
    }

    /**
     * 指定のグラフ領域を取得する（領域が存在しない場合は作成して返す）
     * 
     * @param index
     */
    private RecordGraphComposite getGraph(int graphIndex) {
        if (composite[graphIndex] == null) {
        	// グラフを表示する各々のためのcomposite.
            composite[graphIndex] = new RecordGraphComposite(baseCcomposite);
            // グラフ表示のためのデータを設定
            if (collectorData != null) {
                composite[graphIndex].setMasterData(collectorData, itemInfoList,
                        facilityTree);
            }
        }
        return composite[graphIndex];
    }

    /**
     * 指定のグラフ領域を削除する
     * 
     * @param index
     */
    private void disposeGraph(int graphIndex) {
        if (composite[graphIndex] != null) {
            composite[graphIndex].dispose();
            composite[graphIndex] = null;
        }
    }

    /**
     * ラベルを設定します。
     */
    private void setLabel() {
        idLabel.setText(Messages.getString("COLLECTION_ID") + " : " + collectorID);
        scopeLabel.setText(Messages.getString("SCOPE") +  " : "  + currentPath);
    }

    /**
     * グラフの表示設定を返します。
     * 
     * @return GraphProperty
     */
    public GraphProperty[] getGraphPropertys() {
        return this.graphPropertys;
    }

    /**
     * 収集IDを取得します。
     * 
     * @return collectorID
     */
    public String getCollectorID() {
        return collectorID;
    }
}