/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.monitor.factory;

import java.sql.Timestamp;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;

import javax.ejb.CreateException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.jms.JMSException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.EventConfirmConstant;
import com.clustercontrol.bean.ExclusionConstant;
import com.clustercontrol.bean.YesNoConstant;
import com.clustercontrol.commons.util.SendQueue;
import com.clustercontrol.monitor.bean.QueueConstant;
import com.clustercontrol.monitor.ejb.entity.EventLogLocal;
import com.clustercontrol.monitor.ejb.entity.EventLogUtil;
import com.clustercontrol.monitor.ejb.entity.EventMultiInfoLocal;
import com.clustercontrol.monitor.ejb.entity.EventMultiInfoPK;
import com.clustercontrol.monitor.ejb.entity.EventMultiInfoUtil;
import com.clustercontrol.monitor.message.LogOutputInfo;
import com.clustercontrol.monitor.message.LogOutputJobRunInfo;


/**
 * イベント情報を更新するクラスです。
 *
 * @version 2.0.0
 * @since 1.0.0
 */
public class OutputEventLog {
	
	/** ログ出力のインスタンス */
	protected static Log m_log = LogFactory.getLog( OutputEventLog.class );
	
	private LogOutputInfo m_logOutput = null;
	private Timestamp m_outputDate = null;

	/** イベントログ情報生成フラグ。 */
    private boolean m_insertFlg;
	
    /** イベントログ多重化情報生成フラグ。 */
    private boolean m_insertMultiFlg;
    
    /** イベントログ情報更新フラグ。 */
    private boolean m_updateFlg;
	
    /** イベントログ多重化情報更新フラグ。 */
    private boolean m_updateMultiFlg;
    
    /**
	 * イベント情報を出力します。<BR>
	 * 抑制なしの場合は、イベント情報を作成します。
	 * 抑制ありでイベントがすでに存在する場合は、最初のイベント情報の重複カウンタをカウントアップします。
	 * ログ出力情報の抑制時の状態が確認済の場合は、確認済みでイベントを挿入します。
	 * 
	 * @param logOutput ログ出力情報
	 * @param outputDate 受信日時
	 * @return 成功した場合は、 <code>true</code> 
	 */
	public boolean outputEventLog(LogOutputInfo logOutput, Timestamp outputDate) {
		
		this.m_logOutput = logOutput;
		this.m_outputDate = outputDate;
		m_insertFlg = false;
		m_insertMultiFlg = false;
		m_updateFlg = false;
		m_updateMultiFlg = false;
		
		boolean result = false;
		
		if(logOutput.isEventLogFlg())
		{
			Collection coll = null;
			Collection collMulti = null;
			EventLogLocal eventLog = null;
			EventMultiInfoLocal eventMultiInfo = null;
			
			// 多重化ID
			String multiId = logOutput.getMultiId();
			// イベントログ多重化情報用出力日時
			Timestamp outputDateMulti = null;
			
			// 抑制する場合
			if(logOutput.getExcludeFlg() == ExclusionConstant.TYPE_FREQUENCY || 
					logOutput.getExcludeFlg() == ExclusionConstant.TYPE_PERIOD || 
					logOutput.getExcludeFlg() == ExclusionConstant.TYPE_PRIORITY){
				
				try
				{
					// 情報発生日時から見て最新の抑制されていないイベントログ情報の検索
					coll = EventLogUtil.getLocalHome().findByGenerationDateOrder(
							logOutput.getMonitorId(), 
							logOutput.getPluginId(), 
							logOutput.getFacilityId(), 
							new Timestamp(logOutput.getGenerationDate().getTime()),
							new Integer(YesNoConstant.TYPE_NO));
					
					Iterator itr = coll.iterator();
					// すでにイベントログ情報が存在する場合
					if(itr.hasNext())
					{
						eventLog = (EventLogLocal)itr.next();
						
						// 多重化IDの指定がない場合 
						if(multiId == null){
							
							// 抑制対象の場合、イベントログ情報を更新
					        if(isInhibited(logOutput, eventLog)){
					        	m_updateFlg = true;
					        }
					        else{
					        	m_insertFlg = true;
					        }
						}
						// 多重化IDの指定がある場合
						else{
							
							try {
								// 多重化エンティティ情報の検索
								eventMultiInfo = EventMultiInfoUtil.getLocalHome().findByPrimaryKey(
										new EventMultiInfoPK(
												eventLog.getMonitorId(), 
												eventLog.getPluginId(),
												eventLog.getFacilityId(),
												eventLog.getOutputDate(),
												logOutput.getMultiId())	
								);
								
								// すでにイベントログ多重化情報が存在する場合
								if(eventMultiInfo != null){

									// 他の多重化IDからすでに同一の抑制メッセージが送信されている場合
									if(eventLog.getDuplicationCount().intValue() > eventMultiInfo.getDuplicationCount().intValue()){
										
										// イベントログ多重化ID情報の重複カウンタのみ更新
										m_updateMultiFlg = true;
									}
									// 他の多重化IDから同一の抑制メッセージが送信されていない場合
									else{
										// 抑制対象の場合、イベントログ情報とイベントログ多重化情報を更新
								        if(isInhibited(logOutput, eventLog)){
								        	m_updateFlg = true;
								        	m_updateMultiFlg = true;
								        }
								        else{
								        	m_insertFlg = true;
								        	m_insertMultiFlg = true;
								        }
									}
								}
								else{
									// 他の多重化IDからすでに同一の抑制メッセージが送信済みの為、
									// イベントログ多重化ID情報のみ生成
									m_insertMultiFlg = true;
									outputDateMulti = eventLog.getOutputDate();
								}
							} catch (FinderException e) {
								// 他の多重化IDからすでに同一の抑制メッセージが送信済みの為、
								// イベントログ多重化ID情報のみ生成
								m_insertMultiFlg = true;
								outputDateMulti = eventLog.getOutputDate();
							}
						}
					}
					// イベントログ情報が存在しない場合
					else
					{
						// イベントログ情報の生成
						m_insertFlg = true;
						// 多重化IDの指定がある場合、イベントログ多重化情報の生成
						if(multiId != null){
							m_insertMultiFlg = true;
						}
					}
				}
				catch (NamingException e)
				{
					m_log.error("outputEventLog():" + e.getMessage());
				}
				catch(FinderException e)
				{
				}
			}
			// 抑制しない場合
			else{
				
				// 多重化IDの指定がない場合 
				if(multiId == null){
					// イベントログ情報の生成
					m_insertFlg = true;
				}
				// 多重化IDの指定がある場合
				else{
					// 情報発生日時とオリジナルメッセージが同一のイベントログ情報の検索
					try {
						coll = EventLogUtil.getLocalHome().findByMessageOrg(
								logOutput.getMonitorId(), 
								logOutput.getPluginId(), 
								logOutput.getFacilityId(), 
								new Timestamp(logOutput.getGenerationDate().getTime()),
								logOutput.getMessageOrg());
						
						Iterator itr = coll.iterator();
						
						// イベントログ情報に同一のデータが存在する場合
						if(itr.hasNext()){
							while (itr.hasNext()) {
								eventLog = (EventLogLocal) itr.next();
								
								try {
									// イベントログ多重化情報の検索
									eventMultiInfo = EventMultiInfoUtil.getLocalHome().findByPrimaryKey(
										new EventMultiInfoPK(
												eventLog.getMonitorId(), 
												eventLog.getPluginId(),
												eventLog.getFacilityId(),
												eventLog.getOutputDate(),
												logOutput.getMultiId())	
									);
								} catch (FinderException e) {
									// 同じ多重化IDのメッセージが、イベントログ多重化情報に存在しない場合
									// 本メッセージは、すでに他の多重化IDを持つアプリケーションから送信されているので重複したメッセージとなる。
									
									// イベントログ多重化情報の生成用に、イベントログ情報の出力日時を保持
									outputDateMulti = eventLog.getOutputDate();
									break;
								}
							}
							
							if(outputDateMulti == null){
								// イベントログ情報の生成とイベントログ多重化情報の生成
								m_insertFlg = true;
								m_insertMultiFlg = true;
							}
							else{
								// イベントログ多重化情報の生成
								m_insertMultiFlg = true;
							}
						}
						// イベントログ情報に同一のデータが存在しない場合
						else{
							// イベントログ情報の生成とイベントログ多重化情報の生成
							m_insertFlg = true;
							m_insertMultiFlg = true;
						}
					} catch (NamingException e) {
						m_log.error("outputEventLog():" + e.getMessage());
					} catch (FinderException e) {
					}
				}
			}
	            			
			try
			{
				// イベントログ情報の生成
				if(m_insertFlg)
				{
					insertEventLog(logOutput, YesNoConstant.TYPE_NO);
					
					// ジョブ実行
					if(logOutput.getJobRun() != null && logOutput.getJobRun().getJobRun() == YesNoConstant.TYPE_YES){
						SendQueue queue = null;
						try {
							//Queueに送信
							queue = new SendQueue(QueueConstant.QUEUE_NAME_CALLJOB);
							queue.put(logOutput);
						}
						catch (Exception e) {
							m_log.debug("outputEventLog() : ジョブ実行送信エラー : " + e.getMessage());
						}
						finally{
							if(queue != null){
								try {
									queue.terminate();
								} catch (JMSException e) {
								}	
							}
						}
					}
				}
				// エンティティ情報の更新（抑制対象）
				if(m_updateFlg)
				{
					// 元のエンティティ情報を更新
					updateEventLog(eventLog);
					
					if(logOutput.getExcludeConfirmFlg() == EventConfirmConstant.TYPE_CONFIRMED ||
							logOutput.getExcludeConfirmFlg() == EventConfirmConstant.TYPE_UNCONFIRMED){
						
						// 抑制されたエンティティ情報の生成
						insertEventLog(logOutput, YesNoConstant.TYPE_YES);
						
						// ジョブ実行
						LogOutputJobRunInfo jobRunInfo = logOutput.getJobRun();
						if(jobRunInfo != null && jobRunInfo.getJobRun() == YesNoConstant.TYPE_YES){
							// 通知抑制と連動しない場合のみ、ジョブ実行
							if(jobRunInfo.getJobInhibitionFlg() == YesNoConstant.TYPE_NO){
								SendQueue queue = null;
								try {
									//Queueに送信
									queue = new SendQueue(QueueConstant.QUEUE_NAME_CALLJOB);
									queue.put(logOutput);
								}
								catch (Exception e) {
									m_log.debug("outputEventLog() : ジョブ実行送信エラー : " + e.getMessage());
								}
								finally{
									try {
										queue.terminate();
									} catch (JMSException e) {
									}	
								}
							}
						}
					}
				}
				
				// 多重化IDの指定がある場合
				if(multiId != null){
					
					// イベントログ多重化情報の生成
					if(m_insertMultiFlg){
						
						if(outputDateMulti == null){
							outputDateMulti = m_outputDate;
						}
						insertEventMultiInfo(logOutput, outputDateMulti);
					}
					// イベントログ多重化情報の更新
					if(m_updateMultiFlg){
						// 重複カウンタ更新
						updateEventMultiInfo(eventMultiInfo);
					}
				}
				result = true;
			}
			catch(NamingException e)
			{
				m_log.error("outputEventLog():" + e.getMessage());
			}
			catch(CreateException e)
			{
				m_log.error("outputEventLog():" + e.getMessage());
			}
			catch(EJBException e)
			{
				// NOT NULL のカラムにnullを送信した場合など
				m_log.error("outputEventLog():" + e.getMessage());
			}
		}
		return result;
	}
	
	/**
	 * イベント情報を作成します。
	 * 
	 * @param logOutput ログ出力情報
	 * @param inhibitedFlg 抑制フラグ
	 * @throws NamingException
	 * @throws CreateException
	 * 
	 * @see com.clustercontrol.bean.YesNoConstant
	 */
	public void  insertEventLog(LogOutputInfo logOutput , int inhibitedFlg) throws NamingException, CreateException{
	
		try
		{
			int confirmFlg = -1;
			if(inhibitedFlg == YesNoConstant.TYPE_YES){
				// 抑制されたデータの場合
				confirmFlg = logOutput.getExcludeConfirmFlg();
			}
			else{
				confirmFlg = logOutput.getConfirmFlg();
			}
			
			EventLogUtil.getLocalHome().create(
					logOutput.getMonitorId(),
					logOutput.getPluginId(),
					logOutput.getFacilityId(),
					logOutput.getScopeText(),
					logOutput.getApplication(),
					logOutput.getMessageId(),
					logOutput.getMessage(),
					logOutput.getMessageOrg(),
					new Integer(logOutput.getPriority()),
					new Integer(confirmFlg),
					null,
					new Integer(0),
					new Timestamp(logOutput.getGenerationDate().getTime()),
					m_outputDate,
					new Integer(inhibitedFlg)
					);
		}
		catch(NamingException e)
		{
			throw e;
		}
		catch(CreateException e)
		{
			// 同じキーを持つオブジェクトがすでに存在する場合、出力日時を再度取得しイベントログ挿入をリトライする
			if(e instanceof javax.ejb.DuplicateKeyException){
				Date now = new Date();
				m_outputDate = new Timestamp(now.getTime());

				this.insertEventLog(logOutput, inhibitedFlg);
			}
			else{
				throw e;
			}
		}
	}
	
	/**
	 * イベント多重化情報を作成します。
	 * 
	 * @param logOutput ログ出力情報
	 * @param outputDate 受信日時
	 * @throws NamingException
	 * @throws CreateException
	 */
	public void insertEventMultiInfo(LogOutputInfo logOutput, Timestamp outputDate) throws NamingException, CreateException{
		
		try {
			EventMultiInfoUtil.getLocalHome().create(
					logOutput.getMonitorId(),
					logOutput.getPluginId(),
					logOutput.getFacilityId(),
					outputDate,
					logOutput.getMultiId(),
					Integer.valueOf(0)
					);
			
		} catch (NamingException e) {
			throw e;
		} catch (CreateException e) {
			throw e;
		}
	}
	
	/**
	 * イベント情報を更新します。<BR>
	 * 引数で指定されたイベント情報の重複カウンタをカウントアップします。
	 * 
	 * @param eventLog イベント情報のローカルコンポーネントインターフェース
	 * @throws EJBException
	 */
	public void updateEventLog(EventLogLocal eventLog) throws EJBException{
	
		if(eventLog != null)
		{
			try
			{
				eventLog.setDuplicationCount(new Integer((eventLog.getDuplicationCount().intValue()+1)));
			}
			catch(EJBException e)
			{
				// NOT NULL のカラムにnullを送信した場合など
				throw e;
			}
		}
	}
	
	/**
	 * イベント多重化情報を更新します。<BR>
	 * 引数で指定されたイベント多重化情報の重複カウンタをカウントアップします。
	 * 
	 * @param EventMultiInfo イベント多重化情報のローカルコンポーネントインターフェース
	 * @throws EJBException
	 */
	public void updateEventMultiInfo(EventMultiInfoLocal EventMultiInfo) throws EJBException{
	
		if(EventMultiInfo != null)
		{
			try
			{
				EventMultiInfo.setDuplicationCount(new Integer((EventMultiInfo.getDuplicationCount().intValue()+1)));
			}
			catch(EJBException e)
			{
				throw e;
			}
		}
	}
	
	/**
	 * 引数で指定されたログ出力情報が抑制対象の場合、 <code>true</code>を返します。
	 * 
	 * @param logOutput ログ出力情報
	 * @param eventLog 抑制対象か確認するイベント情報のローカルコンポーネントインターフェース
	 * @return 抑制対象の場合、 <code>true</code> 
	 */
	public boolean isInhibited(LogOutputInfo logOutput, EventLogLocal eventLog){
		
		// 抑制回数の範囲内か
        if(logOutput.getExcludeFlg() == ExclusionConstant.TYPE_FREQUENCY
        		&& eventLog.getConfirmDate() == null) {
        	if(eventLog.getDuplicationCount().intValue() < logOutput.getExcludeNumber()){
				return true;
        	}
        }
        // 抑制期間の範囲内か
        else if(logOutput.getExcludeFlg() == ExclusionConstant.TYPE_PERIOD
        		&& eventLog.getConfirmDate() == null) {
        	
        	Calendar cal = Calendar.getInstance();
			cal.setTime(eventLog.getGenerationDate());
	        cal.add(Calendar.MINUTE, logOutput.getExcludePeriod());
        	
			if(cal.getTime().compareTo(logOutput.getGenerationDate()) > 0){
				return true;
	        }
        }
        //抑制する重要度か
        else if(logOutput.getExcludeFlg() == ExclusionConstant.TYPE_PRIORITY
        		&& eventLog.getConfirmDate() == null) {
        	if(eventLog.getPriority().intValue() == logOutput.getPriority()){
				return true;
        	}
        }
        return false;
	}
	
	
	/**
	 * 受信日時を返します。
	 * 
	 * @return 受信日時
	 */
	public Timestamp getOutputDate() {
		return m_outputDate;
	}
	
	/**
	 * 受信日時を設定します。
	 * 
	 * @param outputDate 受信日時
	 */
	public void setOutputDate(Timestamp outputDate) {
		m_outputDate = outputDate;
	}

	/**
     * イベント情報を作成した場合は、 <code>true</code> を返します。
     * 
     * @return イベント情報を作成した場合は、 <code>true</code>
     */
    public boolean isInsertFlg() {
        return m_insertFlg;
    }
}
