/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.jobmanagement.dao;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;

import javax.ejb.CreateException;
import javax.ejb.DuplicateKeyException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.commons.util.ConnectionManager;
import com.clustercontrol.jobmanagement.ejb.entity.JobFileMasterBean;
import com.clustercontrol.jobmanagement.ejb.entity.JobFileMasterPK;

/**
 * ジョブファイル転送マスタのDAOインターフェースを実装するクラスです。
 * 
 * @version 2.1.0
 * @since 2.1.0
 * 
 * @see com.clustercontrol.jobmanagement.ejb.entity.JobFileMasterBean
 * @see com.clustercontrol.jobmanagement.dao.JobFileMasterDAO
 */
public class JobFileMasterDAOImpl implements JobFileMasterDAO {
	
	/** ログ出力のインスタンス。 */
	protected static Log m_log = LogFactory.getLog(JobFileMasterDAOImpl.class);

	public void init() {
	}
	
	/** 
	 * SELECT文を発行します。
	 * 引数で指定されたプライマリキーで検索し、取得したデータベースの内容をEntity Beanに反映します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobFileMasterDAO#load(com.clustercontrol.jobmanagement.ejb.entity.JobFileMasterPK, com.clustercontrol.jobmanagement.ejb.entity.JobFileMasterBean)
	 */
	public void load(JobFileMasterPK pk, JobFileMasterBean ejb) throws EJBException {
		m_log.debug("load() start : " + pk.toString());

		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;

		try {
			conn = ConnectionManager.getConnectionManager().getConnection();

			//SQL文の定義
			String sql = "SELECT * FROM cc_job_file_mst WHERE job_id = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);

			stmt.setString(1, pk.getJob_id());

			res = stmt.executeQuery();

			if (res.next()) {
				//取得した値をBeanにセット
				ejb.setCheck_flg(res.getInt("check_flg"));
				ejb.setCompression_flg(res.getInt("compression_flg"));
				ejb.setDest_directory(res.getString("dest_directory"));
				ejb.setDest_facility_id(res.getString("dest_facility_id"));
				ejb.setDest_work_dir(res.getString("dest_work_dir"));
				ejb.setEffective_user(res.getString("effective_user"));
				ejb.setJob_id(res.getString("job_id"));
				ejb.setProcess_mode(res.getInt("process_mode"));
				ejb.setSrc_facility_id(res.getString("src_facility_id"));
				ejb.setSrc_file(res.getString("src_file"));
				ejb.setSrc_work_dir(res.getString("src_work_dir"));
			} else {
				String msg = "JobFileMaster data is not found.";
				m_log.error("load() error : " + pk.toString() + " SQLException " + msg);
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			m_log.error("load() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());
		} finally {
			//コネクション、結果セット、プリペアドステートメントのクローズ
			try {
				if (stmt != null) {
					stmt.close();
				}
				if (res != null) {
					res.close();
				}
				if (conn != null) {
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("load() error : " + pk.toString() + " SQLException " + e1.getMessage());
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("load() end : " + pk.toString());
	}

	/**
	 * UPDATE文を発行します。
	 * 引数で指定されたEntity Beanの内容でデータベースを更新します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobFileMasterDAO#store(com.clustercontrol.jobmanagement.ejb.entity.JobFileMasterBean)
	 */
	public void store(JobFileMasterBean ejb) throws EJBException {
		m_log.debug("store() start : " + ejb.getJob_id());

		Connection conn = null;
		PreparedStatement stmt = null;

		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			//SQL文の定義
			StringBuilder sql = new StringBuilder();
			sql.append("UPDATE cc_job_file_mst SET ");
			sql.append("process_mode = ?, ");
			sql.append("src_facility_id = ?, ");
			sql.append("dest_facility_id = ?, ");
			sql.append("src_file = ?, ");
			sql.append("src_work_dir = ?, ");
			sql.append("dest_directory = ?, ");
			sql.append("dest_work_dir = ?, ");
			sql.append("compression_flg = ?, ");
			sql.append("check_flg = ?, ");
			sql.append("effective_user = ? ");
			sql.append("WHERE job_id = ?");
			  
			//SQL文のセット
			stmt = conn.prepareStatement(sql.toString());
			
			stmt.setInt(1, ejb.getProcess_mode());
			stmt.setString(2, ejb.getSrc_facility_id());
			stmt.setString(3, ejb.getDest_facility_id());
			stmt.setString(4, ejb.getSrc_file());
			stmt.setString(5, ejb.getSrc_work_dir());
			stmt.setString(6, ejb.getDest_directory());
			stmt.setString(7, ejb.getDest_work_dir());
			stmt.setInt(8, ejb.getCompression_flg());
			stmt.setInt(9, ejb.getCheck_flg());
			stmt.setString(10, ejb.getEffective_user());
			stmt.setString(11, ejb.getJob_id());
			
			int row = stmt.executeUpdate();

			if (row != 1) {
				String msg = "result row is not 1";
				m_log.error("store() error : " + ejb.getJob_id() + " SQLException" + msg);
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			m_log.error("store() error : " + ejb.getJob_id() + " SQLException");
			throw new EJBException(e.getMessage());
		} finally {
			// コネクション、結果セット、プリペアドステートメントのクローズ
			try {
				if (stmt != null) {
					stmt.close();
				}
				if (conn != null) {
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("store() error : " + ejb.getJob_id() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("store() end : " + ejb.getJob_id());
	}

	/**
	 * DELETE文を発行します。
	 * 引数で指定されたプライマリキーでデータベースから削除します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobFileMasterDAO#remove(com.clustercontrol.jobmanagement.ejb.entity.JobFileMasterPK)
	 */
	public void remove(JobFileMasterPK pk) throws RemoveException, EJBException {
		m_log.debug("remove() start : " + pk.toString());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			
			//SQL文の定義
			String sql = "DELETE FROM cc_job_file_mst WHERE job_id = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1, pk.getJob_id());
			
			int row = stmt.executeUpdate();
			
			if (row != 1 ) {
				String msg = "result row is not 1";
				m_log.error("remove() error : " + pk.toString() + " SQLException" + msg);
				throw new EJBException(msg);	
			}
		} catch (SQLException e) {
			m_log.error("remove() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());
			
		} finally {
			try {
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("remove() error : " + pk.toString() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("remove() end : " + pk.toString());
	}

	/**
	 * INSERT文を発行します。
	 * 引数で指定されたEntity Beanの内容をデータベースに挿入し、プライマリキーを返します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobFileMasterDAO#create(com.clustercontrol.jobmanagement.ejb.entity.JobFileMasterBean)
	 */
	public JobFileMasterPK create(JobFileMasterBean ejb) throws CreateException, EJBException {
		m_log.debug("create() start : " + ejb.getJob_id());

		Connection conn = null;
		PreparedStatement stmt = null;
		JobFileMasterPK pk;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			//SQL文の定義
			StringBuilder sql = new StringBuilder();
			sql.append("INSERT INTO cc_job_file_mst (");
			sql.append("job_id, ");
			sql.append("process_mode, ");
			sql.append("src_facility_id, ");
			sql.append("dest_facility_id, ");
			sql.append("src_file, ");
			sql.append("src_work_dir, ");
			sql.append("dest_directory, ");
			sql.append("dest_work_dir, ");
			sql.append("compression_flg, ");
			sql.append("check_flg, ");
			sql.append("effective_user) ");
			sql.append("VALUES (?,?,?,?,?,?,?,?,?,?,?)");
			
			//SQL文のセット
			stmt = conn.prepareStatement(sql.toString());

			stmt.setString(1, ejb.getJob_id());
			stmt.setInt(2, ejb.getProcess_mode());
			stmt.setString(3, ejb.getSrc_facility_id());
			stmt.setString(4, ejb.getDest_facility_id());
			stmt.setString(5, ejb.getSrc_file());
			stmt.setString(6, ejb.getSrc_work_dir());
			stmt.setString(7, ejb.getDest_directory());
			stmt.setString(8, ejb.getDest_work_dir());
			stmt.setInt(9, ejb.getCompression_flg());
			stmt.setInt(10, ejb.getCheck_flg());
			stmt.setString(11, ejb.getEffective_user());
			
			int row =	stmt.executeUpdate();
			pk = new JobFileMasterPK(ejb.getJob_id());
			
			if (row != 1) {
				String msg = "result row is not 1";
				m_log.error("create() error : " + ejb.getJob_id() + " SQLException");
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			if(e.getSQLState().equals("23505")){
				m_log.error("create() error : " + ejb.getJob_id() + " SQLException" + "DuplicateKeyException ");
				throw new DuplicateKeyException(e.getMessage());
			}
			throw new CreateException(e.getMessage());
		}finally{
			try {
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("create() error : " + ejb.getJob_id() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("create() end : " + ejb.getJob_id());
		return pk;
	}

	/**
	 * 全件取得するSELECT文を発行します。
	 * 取得したデータのプライマリキーをコレクションに格納し返します。
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobFileMasterBean#ejbFindAll()
	 * @see com.clustercontrol.jobmanagement.dao.JobFileMasterDAO#findAll()
	 */
	public Collection findAll() throws FinderException {
		m_log.debug("findAll() start : ");
		
		ArrayList<JobFileMasterPK> ret = new ArrayList<JobFileMasterPK>();
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();	
			//SQL文の定義
			String sql = "SELECT * FROM cc_job_file_mst";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			res = stmt.executeQuery();
			
			while(res.next()) {
				JobFileMasterPK pk= new JobFileMasterPK(
						res.getString("job_id")
						);
				ret.add(pk);
			}
			
		} catch (SQLException e) {
			m_log.error("findAll() error :  SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("findAll() error :  SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("findAll() end : ");
		return ret;
	}

	/**
	 * 1件取得するSELECT文を発行します。
	 * 引数で指定されたプライマリキーで、データベースを検索します。
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobFileMasterBean#ejbFindByPrimaryKey(com.clustercontrol.jobmanagement.ejb.entity.JobFileMasterPK)
	 * @see com.clustercontrol.jobmanagement.dao.JobFileMasterDAO#findByPrimaryKey(com.clustercontrol.jobmanagement.ejb.entity.JobFileMasterPK)
	 */
	public JobFileMasterPK findByPrimaryKey(JobFileMasterPK pk) throws FinderException {
		m_log.debug("findByPrimaryKey() start : " + pk.toString());

		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			
			//SQL文の定義
			String sql = "SELECT * FROM cc_job_file_mst WHERE job_id = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);

			stmt.setString(1, pk.getJob_id());
			
			res = stmt.executeQuery();
			
			if (res.next()) {
				return pk;
			} else {
				String msg = "id " + pk.toString() + " not found.";
				throw new FinderException(msg);
			}
		} catch (SQLException e) {
			m_log.error("find() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("find() error : " + pk.toString() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
			m_log.debug("findByPrimaryKey() end : " + pk.toString());
		}
	}
}
