/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.calendar.dialog;

import java.util.ArrayList;
import java.util.Date;

import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.TableTree;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;

import com.clustercontrol.bean.Property;
import com.clustercontrol.bean.PropertyConstant;
import com.clustercontrol.calendar.action.AddCalendarProperty;
import com.clustercontrol.calendar.action.GetCalendarProperty;
import com.clustercontrol.calendar.action.ModifyCalendarProperty;
import com.clustercontrol.calendar.factory.CalendarProperty;
import com.clustercontrol.composite.PropertySheet;
import com.clustercontrol.dialog.CommonDialog;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.util.Messages;
import com.clustercontrol.util.PropertyUtil;

/**
 * カレンダの作成・変更ダイアログクラスです。
 * 
 * @version 2.2.0
 * @since 2.0.0
 */
public class CalendarDialog extends CommonDialog {
    /** ID */
    private String id = "";

    /** 変更用ダイアログ判別フラグ */
    private int mode;

    /** カレンダプロパティシート */
    private PropertySheet propertySheet = null;
    /**
     * 指定した形式のダイアログのインスタンスを返します。
     * 
     * @param parent
     *            親のシェルオブジェクト
     * @param id
     *            カレンダID
     * @param mode
     *            表示モード
     */
    public CalendarDialog(Shell parent, String uid, int mode) {
        super(parent);

        this.id = uid;
        this.mode = mode;
    }
    
    /**
     * ダイアログの初期サイズを返します。
     * 
     * @return 初期サイズ
     */
    protected Point getInitialSize() {
        return new Point(500, 600);
    }

    /**
     * ダイアログエリアを生成します。
     * 
     * @param parent
     *            親のインスタンス
     */
    protected void customizeDialog(Composite parent) {
        Shell shell = this.getShell();

        // タイトル
        shell.setText(Messages
                .getString("dialog.calendar.calendar.create.modify"));

        // レイアウト
        GridLayout layout = new GridLayout(1, true);
        layout.marginWidth = 10;
        layout.marginHeight = 10;
        parent.setLayout(layout);

        /*
         * 属性プロパティシート
         */

        // ラベル
        Label label = new Label(parent, SWT.LEFT);
        GridData gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.horizontalSpan = 1;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("attribute") + " : ");

        // プロパティシート
        TableTree table = new TableTree(parent, SWT.H_SCROLL | SWT.V_SCROLL
                | SWT.FULL_SELECTION | SWT.MULTI | SWT.BORDER);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.grabExcessVerticalSpace = true;
        gridData.horizontalSpan = 1;
        table.setLayoutData(gridData);

        this.propertySheet = new PropertySheet(table);

        // プロパティ取得及び設定
        Property property = null;
        if (this.mode == PropertyConstant.MODE_MODIFY) {
            property = new GetCalendarProperty().getProperty(this.id,
                    PropertyConstant.MODE_MODIFY);
        } 
        else if (this.mode == PropertyConstant.MODE_ADD) {
            property = new GetCalendarProperty().getProperty(this.id,
                    PropertyConstant.MODE_ADD);
        }
        else if (this.mode == PropertyConstant.MODE_SHOW) {
            property = new GetCalendarProperty().getProperty(this.id,
                    PropertyConstant.MODE_SHOW);
        }
        this.propertySheet.setInput(property);
        this.propertySheet.expandAll();

        // ラインを引く
        Label line = new Label(parent, SWT.SEPARATOR | SWT.HORIZONTAL);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.horizontalSpan = 1;
        line.setLayoutData(gridData);

        // 画面中央に
        Display display = shell.getDisplay();
        shell.setLocation((display.getBounds().width - shell.getSize().x) / 2,
                (display.getBounds().height - shell.getSize().y) / 2);
    }

    /**
     * 入力値チェックをします。
     * 
     * @return 検証結果
     */
    protected ValidateResult validate() {
    	ValidateResult result = null;

		//カレンダIDの入力チェック
        ArrayList values = PropertyUtil.getPropertyValue(getInputData(), CalendarProperty.ID);
		String value = (String)values.get(0);
		if(value.compareTo("") == 0){
            result = new ValidateResult();
            result.setID(Messages.getString("message.hinemos.1"));
            result.setMessage(Messages.getString("message.calendar.22"));
            result.setValid(false);
            return result;
		}
		
		//名前の入力チェック
        values = PropertyUtil.getPropertyValue(getInputData(), CalendarProperty.NAME);
		value = (String)values.get(0);
		if(value.compareTo("") == 0){
            result = new ValidateResult();
            result.setID(Messages.getString("message.hinemos.1"));
            result.setMessage(Messages.getString("message.calendar.23"));
            result.setValid(false);
            return result;
		}
		
		//有効期間(開始)の入力チェック
        values = PropertyUtil.getPropertyValue(getInputData(), CalendarProperty.VALID_TIME_FROM);
		if(!(values.get(0) instanceof Date)){
            result = new ValidateResult();
            result.setID(Messages.getString("message.hinemos.1"));
            String[] args = { "(" + Messages.getString("start") + ")" };
            result.setMessage(Messages.getString("message.calendar.24", args));
            result.setValid(false);
            return result;
		}
		
		//有効期間(終了)の入力チェック
        values = PropertyUtil.getPropertyValue(getInputData(), CalendarProperty.VALID_TIME_TO);
		if(!(values.get(0) instanceof Date)){
            result = new ValidateResult();
            result.setID(Messages.getString("message.hinemos.1"));
            String[] args = { "(" + Messages.getString("end") + ")" };
            result.setMessage(Messages.getString("message.calendar.24", args));
            result.setValid(false);
            return result;
		}
		
        return result;
    }
    
    /**
     * 入力値をマネージャに登録します。
     * 
     * @return true：正常、false：異常
     * 
     * @see com.clustercontrol.dialog.CommonDialog#action()
     */
    protected boolean action() {
    	boolean result = false;
    	
    	Property property = this.getInputData();
    	if(property != null){
            if (this.mode == PropertyConstant.MODE_MODIFY) {
            	// 変更の場合
            	result = new ModifyCalendarProperty().edit(property);
            } 
            else if (this.mode == PropertyConstant.MODE_ADD) {
            	// 作成の場合
            	result = new AddCalendarProperty().add(property);
            }
            else {
            	// 表示の場合
            	result = true;
            }
    	}
    	
    	return result;
    }

    /**
     * 入力値を保持したデータモデルを生成します。
     * 
     * @return データモデル
     */
    public Property getInputData() {
        return (Property) this.propertySheet.getInput();
    }

    /**
     * 入力値を保持したデータモデルを設定します。
     * 
     * @param property
     */
    public void setInputData(Property property) {
        propertySheet.setInput(property);
    }

    /**
     * @return Returns the uid.
     */
    public String getUid() {
        return this.id;
    }

    /**
     * ＯＫボタンのテキストを返します。
     * 
     * @return ＯＫボタンのテキスト
     */
    protected String getOkButtonText() {
        if (this.mode == PropertyConstant.MODE_MODIFY) {
            return Messages.getString("modify");
        }
        else if (this.mode == PropertyConstant.MODE_ADD) {
            return Messages.getString("register");
        }
        else{
            return Messages.getString("ok");
        }
    }

    /**
     * キャンセルボタンのテキストを返します。
     * 
     * @return キャンセルボタンのテキスト
     */
    protected String getCancelButtonText() {
        return Messages.getString("cancel");
    }

	/**
	 * OKボタンを押されたときの処理
	 * 
	 * @see org.eclipse.jface.dialogs.Dialog#okPressed()
	 */
	@Override
	protected void okPressed() {
        ValidateResult result = this.validate();

        if (result == null || result.isValid()) {
        	
    		//作成時
        	if(this.mode == PropertyConstant.MODE_ADD) {
        		this.action();
        	}
        	//変更時、表示時
        	else{
        		super.okPressed();
        	}
        	
        } else {
            this.displayError(result);
        }
	}
}