/*
 * $Id: hfstick.c,v 1.8 2006/10/14 04:47:47 afujita Exp $
 *
 *  Copyright (c) 2006 Ariya Fujita <afujita@sourceforge.jp>
 *
 */

/*
 *  HORI USB Flightsticks.
 */

/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/input.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/usb.h>
#include <linux/config.h>

#include <linux/spinlock.h>
#include <linux/errno.h>
#include <linux/ioport.h>
#include "./hfstick.h"
#include "./hfstick-compat.h"

#define FSTICK_MAX_LENGTH_IR   16
#define FSTICK_MAX_LENGTH_VR00 16
#define FSTICK_MAX_LENGTH_VR01 16
#define FSTICK_USB 2

#ifndef BTN_BASE7
#define BTN_BASE7 (BTN_BASE6+1)
#define BTN_BASE8 (BTN_BASE6+2)
#define BTN_BASE9 (BTN_BASE6+3)
#endif

static signed short btn_joystick_hf1[] = { BTN_SELECT, BTN_X, BTN_TOP, BTN_START,
                                       BTN_BASE, BTN_BASE2, BTN_BASE3, BTN_BASE4,
                                           BTN_THUMB, BTN_TRIGGER, -1 };

static signed short btn_joystick_hf2[] = { BTN_SELECT, BTN_X, BTN_TOP, BTN_START,
                                       BTN_BASE, BTN_BASE2, BTN_BASE3, BTN_BASE4,
                                       BTN_THUMB, BTN_TRIGGER,
                                       
                                       BTN_Y,
                                       BTN_MODE, BTN_THUMBL, BTN_THUMBR,
                                       BTN_A, BTN_B, BTN_C,
                                       BTN_DEAD,
                                       BTN_Z,
                                       BTN_BASE5, BTN_BASE6, BTN_BASE7, BTN_BASE8,
                                       -1 };

static signed short abs_joystick[] = { ABS_X, ABS_Y,
                                       ABS_RUDDER, ABS_THROTTLE,
                                       ABS_HAT0X, ABS_HAT0Y,
                                       ABS_DISTANCE, ABS_PRESSURE, -1 };

static void fstick_cycle_vr01(unsigned long data);
static void fstick_cycle_vr00(unsigned long data);
typedef void (*fstick_cycle_t)(unsigned long);
typedef void (*fstick_vr_call)(unsigned long);
struct fstick_device {
    u16 idvendor;
    u16 idproduct;
    char *name;
    signed short *btn;
    signed short *abs;
    signed short *ff;
    fstick_vr_call vr_next;
} fstick_device[] = {
    { 0x06d3, 0x0f10, "HORI Flightstick1", btn_joystick_hf1, abs_joystick, NULL, fstick_cycle_vr00 },
    { 0x06d3, 0x0f10, "HORI Flightstick2", btn_joystick_hf2, abs_joystick, NULL, fstick_cycle_vr01 },
    { }
};

struct fstick {
    struct input_dev *dev; /* Input device interface */
    struct fstick_device *type;
    
    char name[64];
    int open;
    int bus;
    struct usb_endpoint_descriptor *epirq;
    
    unsigned char data[FSTICK_MAX_LENGTH_IR];
    unsigned char edata0[FSTICK_MAX_LENGTH_VR00];
    unsigned char edata1[FSTICK_MAX_LENGTH_VR01];
    u16 ecmd;
    u16 expect_packet;

    unsigned int ifnum;
    struct usb_device *usbdev; /* USB transfer */
    struct urb *irq, *ctrl_vr00, *ctrl_vr01;
    struct usb_ctrlrequest dr_vr00, dr_vr01;

    fstick_vr_call vr_next;
    
    struct timer_list timer_list;
};


static int usb_control_msg_async( struct urb * urb,
    struct usb_device *dev, unsigned int pipe,
    __u8 request, __u8 requesttype,
    __u16 value, __u16 index,
    void *data,
    __u16 size)
{
    struct usb_ctrlrequest *dr = (struct usb_ctrlrequest*) urb->setup_packet;
    
    if (!urb || !dr) return -EINVAL;

    dr->bRequestType = requesttype;
    dr->bRequest = request;
    dr->wValue = cpu_to_le16p(&value);
    dr->wIndex = cpu_to_le16p(&index);
    dr->wLength = cpu_to_le16p(&size);

    FILL_CONTROL_URB( urb, dev, pipe,
                      urb->setup_packet,
                      data, size,
                      urb->complete, urb->context);
    
    return usb_submit_urb_with_atomic(urb);
}


inline
static void fstick_process_packet(struct fstick *fstick, u16 cmd, unsigned char *data)
{
    struct input_dev *dev = fstick->dev;
    struct controller_raw_input *raw_in;

    raw_in = (struct controller_raw_input *)data;

    input_report_abs(dev, ABS_X, raw_in->ir.stick_x);
    input_report_abs(dev, ABS_Y, raw_in->ir.stick_y);
    input_report_abs(dev, ABS_RUDDER, raw_in->ir.rudder);
    input_report_abs(dev, ABS_THROTTLE, 0xFF - raw_in->ir.throttle);

    input_report_abs(dev, ABS_HAT0X, raw_in->ir.hat_x);
    input_report_abs(dev, ABS_HAT0Y, raw_in->ir.hat_y);

    input_report_abs(dev, ABS_DISTANCE, fix_abutton_value(raw_in->ir.button_a, 0x80));
    input_report_abs(dev, ABS_PRESSURE, fix_abutton_value(raw_in->ir.button_b, 0x80));
}

#define DVAL(__v__) ((!(__v__)) & 0x1)     
inline
static void fstick_process_vr00(struct fstick *fstick, u16 cmd, unsigned char *edata)
{
    struct input_dev *dev = fstick->dev;
    struct controller_raw_input_vr_00 *raw_vr00;

    raw_vr00 = (struct controller_raw_input_vr_00 *)edata;
    
    input_report_key(dev, BTN_SELECT, DVAL(raw_vr00->fire_c));
    input_report_key(dev, BTN_X, DVAL(raw_vr00->button_d));
    input_report_key(dev, BTN_TOP, DVAL(raw_vr00->hat));
    input_report_key(dev, BTN_START, DVAL(raw_vr00->button_st));
    
    input_report_key(dev, BTN_BASE, DVAL(raw_vr00->dpad1_top));
    input_report_key(dev, BTN_BASE2, DVAL(raw_vr00->dpad1_right));
    input_report_key(dev, BTN_BASE3, DVAL(raw_vr00->dpad1_bottom));
    input_report_key(dev, BTN_BASE4, DVAL(raw_vr00->dpad1_left));

    input_report_key(dev, BTN_THUMB, DVAL(raw_vr00->launch));
    input_report_key(dev, BTN_TRIGGER, DVAL(raw_vr00->trigger));    
}

inline
static void fstick_process_vr01(struct fstick *fstick, u16 cmd, unsigned char *edata)
{
    struct input_dev *dev = fstick->dev;
    struct controller_raw_input_vr_01 *raw_vr01;

    raw_vr01 = (struct controller_raw_input_vr_01 *)edata;

    input_report_key(dev, BTN_THUMBR, DVAL(raw_vr01->dpad3_right));
    input_report_key(dev, BTN_MODE, DVAL(raw_vr01->dpad3_middle));
    input_report_key(dev, BTN_THUMBL, DVAL(raw_vr01->dpad3_left));

#define MVAL(__v__,__m__)    ((__v__) == (__m__) ? 1 : 0)   
    input_report_key(dev, BTN_A, MVAL(raw_vr01->mode_select,0x2));
    input_report_key(dev, BTN_B, MVAL(raw_vr01->mode_select,0x1));
    input_report_key(dev, BTN_C, MVAL(raw_vr01->mode_select,0x3));
#undef MVAL    
    
//    input_report_key(dev, BTN_DEAD, raw_vr01->reserved3);
    input_report_key(dev, BTN_Z, DVAL(raw_vr01->button_sw1));
    
    input_report_key(dev, BTN_BASE5, DVAL(raw_vr01->dpad2_top));
    input_report_key(dev, BTN_BASE6, DVAL(raw_vr01->dpad2_right));
    input_report_key(dev, BTN_BASE7, DVAL(raw_vr01->dpad2_bottom));
    input_report_key(dev, BTN_BASE8, DVAL(raw_vr01->dpad2_left));
    
}
#undef DVAL

#define CYCLE (HZ/50)
static void fstick_cycle_vr01(unsigned long data);
static void fstick_cycle_vr00(unsigned long data)
{
    struct fstick *fstick;
    int ret;

    fstick = (struct fstick*)data;

    if (unlikely(fstick==NULL)) return;

    ret = usb_control_msg_async( fstick->ctrl_vr00,
                                 fstick->usbdev,
                                 usb_rcvctrlpipe(fstick->usbdev,0),
                                 VR00_REQUEST, VR00_REQTYPE,
                                 VR00_VALUE, VR00_INDEX,
                                 fstick->edata0, sizeof(struct controller_raw_input_vr_00));
    
    if (unlikely(ret)) {
        ERR("%s - usb_control_msg_async(vr00) failed with result %d.",
            __FUNCTION__, ret);
    }

    fstick->timer_list.function = fstick->vr_next;
    fstick->timer_list.data = (unsigned long)fstick;
    fstick->timer_list.expires = jiffies + CYCLE;
    add_timer (&fstick->timer_list);
}

static void fstick_cycle_vr01(unsigned long data)
{
    struct fstick *fstick;
    int ret;

    fstick = (struct fstick*)data;

    if (unlikely(fstick==NULL)) return;

    ret = usb_control_msg_async( fstick->ctrl_vr01,
                                 fstick->usbdev,
                                 usb_rcvctrlpipe(fstick->usbdev,0),
                                 VR01_REQUEST, VR01_REQTYPE,
                                 VR01_VALUE, VR01_INDEX,
                                 fstick->edata1, sizeof(struct controller_raw_input_vr_01));
    
    if (unlikely(ret)) {
        ERR("%s - usb_control_msg_async(vr01) failed with result %d",
            __FUNCTION__, ret);
    }
        
    fstick->timer_list.function = fstick_cycle_vr00;
    fstick->timer_list.data = (unsigned long)fstick;
    fstick->timer_list.expires = jiffies + CYCLE;
    add_timer (&fstick->timer_list);
}

static inline void fstick_usb_irq_body(struct urb *urb)
{
	struct fstick *fstick = urb->context;
    int ret;

    switch(urb->status) {
        case 0:                 /* success */
            break;
        case -ECONNRESET:       /* unlink */
        case -ENOENT:
        case -ESHUTDOWN:
            DBG("%s - urb shutting down with status: %d",
                __FUNCTION__, urb->status);
            return;
        default:               
            ERR("%s - urb has status of: %d", __FUNCTION__, urb->status);
            goto resubmit;
    }
    
    fstick_process_packet(fstick, 0, fstick->data );
        
  resubmit:
    ret = usb_resubmit_urb(urb);
    if (ret) ERR("%s - usb_resubmit_urb failed with result %d", __FUNCTION__, ret);
        
}

static inline void fstick_usb_ctrl_vr00_body(struct urb *urb)
{
	struct fstick *fstick = urb->context;

	if (unlikely(urb->status)) {
        DBG("%s - urb has status of: %d", __FUNCTION__, urb->status);
        return;
    }
    
    fstick_process_vr00(fstick, 0, fstick->edata0 );
}

static inline void fstick_usb_ctrl_vr01_body(struct urb *urb)
{
	struct fstick *fstick = urb->context;

    if (unlikely(urb->status)) {
        DBG("%s - urb has status of: %d", __FUNCTION__, urb->status);
        return;
    }
    
    fstick_process_vr01(fstick, 0, fstick->edata1 );
}

static int fstick_open(struct input_dev *dev)
{
    struct fstick *fstick = dev->private;
    int i;

    DBG("%s - dev(%d) opened.", __FUNCTION__, (fstick->open+1));    
    if (fstick->open++) return -1;

    init_timer(&fstick->timer_list);
    fstick->timer_list.function = fstick_cycle_vr00;
    fstick->timer_list.data = (unsigned long)fstick;
    fstick->timer_list.expires = jiffies + HZ/4;
    add_timer (&fstick->timer_list);

    fstick->irq->dev = fstick->usbdev;
    if (usb_submit_urb_with_kernel(fstick->irq)) return -EIO;

    return 0;
}

static void fstick_close(struct input_dev *dev)
{
    struct fstick *fstick = dev->private;

    DBG("%s - dev(%d) closed.",__FUNCTION__,fstick->open);
    
    if (!--fstick->open) {
        del_timer_sync(&fstick->timer_list);
        usb_kill_urb(fstick->irq);
    }        
}


static int fstick_init_device(struct fstick *fstick)
{
#define COUNTMAX 10
    int i, vr01_success, fstick_ver;
    
/*
 * Input device fields.
 */

    INPUTDEV_IDBUSTYPE(fstick->dev) =BUS_USB;
    fstick->dev->private = fstick;
    fstick->dev->name = fstick->name;
    fstick->dev->open = fstick_open;
    fstick->dev->close = fstick_close;
    fstick->dev->event = NULL;
    fstick->dev->upload_effect = NULL;
    fstick->dev->erase_effect = NULL;
    
/*
 * Wait until device ready - until it sends its first response.
 */
    
    for(i=0;i<COUNTMAX;i++) {
        usb_control_msg( fstick->usbdev, usb_rcvctrlpipe(fstick->usbdev, 0),
                         VR00_REQUEST, VR00_REQTYPE,
                         VR00_VALUE, VR00_INDEX,
                         fstick->edata0, sizeof(struct controller_raw_input_vr_00),
                         HZ/COUNTMAX);
    }

/*
 * Get device info.
 */

    vr01_success = 0;
    for(i=0;i<COUNTMAX;i++) {
        if (usb_control_msg( fstick->usbdev, usb_rcvctrlpipe(fstick->usbdev, 0),
                         VR01_REQUEST, VR01_REQTYPE,
                         VR01_VALUE, VR01_INDEX,
                         fstick->edata1, sizeof(struct controller_raw_input_vr_01),
                             HZ/COUNTMAX)>0) vr01_success++;
    }

/*
 * Find appropriate device entry
 */

    fstick_ver = sizeof(fstick_device)+1;
    if(vr01_success>COUNTMAX/2) {
        fstick_ver = 2;
        DBG("HORI Flightstick2 (vr01 success=%3d)", vr01_success);
    } else {
        fstick_ver = 1;
        DBG("HORI Flightstick1 (vr01 success=%3d)", vr01_success);
    }

    if (fstick_ver>sizeof(fstick_device)) {
        ERR("%s - Unknown device (Prod/Vend)x%x:x%x.",
            __FUNCTION__,
            fstick->usbdev->descriptor.idProduct,
            fstick->usbdev->descriptor.idVendor);
        
        fstick_close(fstick->dev);
        return -1;
    }
    
    fstick->type = fstick_device + fstick_ver - 1;
    fstick->vr_next = fstick->type->vr_next;
    
    sprintf( fstick->name, "%s", fstick->type->name);

/*
 * Set input device bitfields and ranges.
 */

    fstick->dev->evbit[0] = BIT(EV_KEY) | BIT(EV_ABS);

    for (i = 0; fstick->type->btn[i] >= 0; i++) {
        signed short t = fstick->type->btn[i];
        set_bit(t, fstick->dev->keybit);
    }

    for (i = 0; fstick->type->abs[i] >= 0; i++) {
        signed short t = fstick->type->abs[i];
        set_bit(t, fstick->dev->absbit);
        switch (t) {
            case ABS_X:
            case ABS_Y:
                fstick->dev->absmax[t] = 255;
                fstick->dev->absmin[t] = 0;
                fstick->dev->absflat[t] = 0;
                fstick->dev->absfuzz[t] = 4;
                break;
                
            case ABS_RUDDER:
                fstick->dev->absmax[t] = 255;
                fstick->dev->absmin[t] = 0;
                fstick->dev->absflat[t] = 0;
                fstick->dev->absfuzz[t] = 4;
                break;
                
            case ABS_THROTTLE:
                fstick->dev->absmax[t] = 255;
                fstick->dev->absmin[t] = 0;
                fstick->dev->absflat[t] = 0;
                fstick->dev->absfuzz[t] = 1;
                break;

            case ABS_HAT0X:
            case ABS_HAT0Y:
                fstick->dev->absmax[t] = 255;
                fstick->dev->absmin[t] = 0;
                fstick->dev->absflat[t] = 0;
                fstick->dev->absfuzz[t] = 4;
                break;
            case ABS_DISTANCE:
            case ABS_PRESSURE:
                fstick->dev->absmax[t] = 255;
                fstick->dev->absmin[t] = 0;
                fstick->dev->absflat[t] = 0;
                fstick->dev->absfuzz[t] = 5;
                break;
        }
    }

/*
 * Register input device.
 */

    input_register_device(fstick->dev);

    return 0;
}

static int fstick_alloc_mem(struct fstick *fstick)
{
    if (!(fstick->irq=usb_alloc_urb_with_kernel(0))) 
        return -1;
    if (!(fstick->ctrl_vr00=usb_alloc_urb_with_atomic(0))) 
        return -1;
    if (!(fstick->ctrl_vr01=usb_alloc_urb_with_atomic(0))) 
        return -1;

    return 0;
}

static void fstick_free_mem(struct fstick *fstick)
{
    if (!fstick) return;

    if (fstick->irq)
        usb_free_urb(fstick->irq);
    if (fstick->ctrl_vr00)
        usb_free_urb(fstick->ctrl_vr00);
    if (fstick->ctrl_vr01)
        usb_free_urb(fstick->ctrl_vr01);
}


/* called by fsitck_usb_probe */
struct usb_probe_status fstick_usb_probe_body( struct usb_device *udev,
                                               struct usb_interface *intf,
                                               const struct usb_device_id *id)
{
    struct usb_endpoint_descriptor *epirq;
    struct fstick *fstick;
    struct input_dev *input_dev;
    struct usb_probe_status ret = { .errnum = -ENOMEM, .data = NULL };

    epirq = usb_get_first_ep(udev,intf);

    fstick = kzalloc( sizeof(struct fstick) + 32, GFP_KERNEL);
    input_dev = input_allocate_device();

    if (!fstick || !input_dev)
        goto fail1;

    if (fstick_alloc_mem(fstick))
        goto fail2;
    
    fstick->bus = FSTICK_USB;
    fstick->usbdev = udev;
    fstick->dev = input_dev;
    fstick->epirq = epirq;

/* usb_interrupt_read */
    FILL_INT_URB( fstick->irq,
                  fstick->usbdev,
                  usb_rcvintpipe(fstick->usbdev, fstick->epirq->bEndpointAddress),
                  fstick->data, sizeof(struct controller_raw_input_ir),
                  fstick_usb_irq,
                  fstick, fstick->epirq->bInterval);
    fstick->irq->dev = fstick->usbdev;

/* usb_control_msg Vendor Request 00 */    
    fstick->dr_vr00.bRequestType = VR00_REQUEST;
    fstick->dr_vr00.bRequest = VR00_REQTYPE;
    fstick->dr_vr00.wValue = VR00_VALUE;
    fstick->dr_vr00.wIndex = VR00_INDEX;
    fstick->dr_vr00.wLength = sizeof(struct controller_raw_input_vr_00);
 
    FILL_CONTROL_URB( fstick->ctrl_vr00,
                      fstick->usbdev, usb_rcvctrlpipe(fstick->usbdev, 0),
                      (void*) &fstick->dr_vr00,
                      fstick->edata0, sizeof(struct controller_raw_input_vr_00
),
                      fstick_usb_ctrl_vr00, fstick);

/* usb_control_msg Vendor Request 01 */    
    fstick->dr_vr01.bRequestType = VR01_REQUEST;
    fstick->dr_vr01.bRequest = VR01_REQTYPE;
    fstick->dr_vr01.wValue = VR01_VALUE;
    fstick->dr_vr01.wIndex = VR01_INDEX;
    fstick->dr_vr01.wLength = sizeof(struct controller_raw_input_vr_01);
 
    FILL_CONTROL_URB( fstick->ctrl_vr01,
                      fstick->usbdev, usb_rcvctrlpipe(fstick->usbdev, 0),
                      (void*) &fstick->dr_vr01,
                      fstick->edata1, sizeof(struct controller_raw_input_vr_01
),
                      fstick_usb_ctrl_vr01, fstick);
    
/* device initialize */
    if ((ret.errnum = fstick_init_device(fstick))) goto fail2;

    INFO( "%s input%d: %s [%d effects, %ld bytes memory %d interval] on usb%d:%d",
          __FUNCTION__,
           INPUTDEV_NUMBER(fstick->dev), fstick->dev->name,
           fstick->dev->ff_effects_max,(long)0,
           epirq->bInterval, udev->bus->busnum, udev->devnum);

    ret.data = fstick;
    /* ret.errnum = 0; */
    return ret;

  fail2:
    fstick_free_mem(fstick);
    ERR("%s - nullpo fstick_alloc_mem().", __FUNCTION__);
    
  fail1:
    input_free_device(input_dev);
    kfree(fstick);
    ret.data = NULL;
    ERR("%s - nullpo fstick or input_dev.", __FUNCTION__);
    
    return ret; /* -ENOMEM */
}

/* called by fstick_usb_disconnect */
static void fstick_usb_disconnect_body(struct usb_device *udev, struct usb_interface *intf, void *ptr)
{
    struct fstick *fstick = ptr;

    if (fstick) {
        fstick->usbdev = NULL;
        del_timer_sync(&fstick->timer_list);

        usb_kill_urb(fstick->irq);
        usb_kill_urb(fstick->ctrl_vr00);
        usb_kill_urb(fstick->ctrl_vr01);
        
        input_unregister_device(fstick->dev);
        
        fstick_free_mem(fstick);    
        kfree(fstick);
    }
}

static struct usb_device_id fstick_usb_ids[] = {
    { USB_DEVICE(0x06d3, 0x0f10) }, /* HORI Flightstick 1/2 */
    { }
};

MODULE_DEVICE_TABLE (usb, fstick_usb_ids);

static struct usb_driver fstick_usb_driver = {
    .name = "hfstick",
    .probe = fstick_usb_probe,
    .disconnect = fstick_usb_disconnect,
    .id_table = fstick_usb_ids
};

static int __init fstick_init(void)
{
    usb_register(&fstick_usb_driver);
    return 0;
}

static void __exit fstick_exit(void)
{
    usb_deregister(&fstick_usb_driver);
}


module_init(fstick_init);
module_exit(fstick_exit);

MODULE_AUTHOR("Ariya Fujita");
MODULE_DESCRIPTION("USB Hori Flightstick driver");
MODULE_LICENSE("GPL");

      
