﻿/*
Copyright (c) 2013, KAKUMOTO Masayuki
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

Redistributions of source code must retain the above copyright notice, this
list of conditions and the following disclaimer.

Redistributions in binary form must reproduce the above copyright notice,
this list of conditions and the following disclaimer in the documentation
and/or other materials provided with the distribution.

Neither the name of the outher KAKUMOTO Masayuki nor the names of its
contributors may be used to endorse or promote products derived from this
software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.
*/

using System;
using System.Collections;
using System.Collections.Generic;
using System.IO;
using System.Text;
using System.Text.RegularExpressions;
using System.Globalization;
using System.Reflection;

namespace GustFront
{
    /// <summary>
    /// データを管理する。値はスクリプトから取得、設定可能である。
    /// </summary>
    public abstract class Element : IHasMember
    {
        private string myTag = String.Empty;
        private string myType = null;
        private List<string> myKeyOrder = new List<string>();
        private Dictionary<string, object> myData = CollectionUtil.CreateCaseInsensitiveDictionary<object>();

        /// <summary>
        /// エレメントを作成する
        /// </summary>
        /// <param name="tagName">エレメントを識別するタグ</param>
        protected Element(string tagName)
            : base()
        {
            myTag = tagName;
        }

        /// <summary>
        /// エレメントを識別するタグ
        /// </summary>
        public string Tag
        {
            get { return myTag; }
        }

        /// <summary>
        /// エレメントの型名
        /// </summary>
        public string Type
        {
            get { return myType; }
        }

        /// <summary>
        /// Object.ToString のオーバーライド
        /// </summary>
        /// <returns>エレメントを識別するタグ</returns>
        public override string ToString()
        {
            return myTag;
        }

        /// <summary>
        /// エレメントの内容をコピーする
        /// </summary>
        /// <param name="dest">コピー先のエレメント</param>
        protected virtual void CopyTo(Element dest)
        {
            dest.myType = myType;
            dest.myKeyOrder.AddRange(myKeyOrder);
            foreach (string key in myKeyOrder) {
                object value = myData[key];
                if (value is IList) {
                    dest.myData[key] = CollectionUtil.CreateStdList((IList)value);
                } else if (value is IDictionary) {
                    dest.myData[key] = CollectionUtil.CreateStdDictionary((IDictionary)value);
                } else {
                    dest.myData[key] = value;
                }
            }
        }

        /// <summary>
        /// エレメントを複製する
        /// </summary>
        /// <param name="newTag">新しいエレメントを識別するタグ</param>
        /// <returns>複製されたエレメント</returns>
        public abstract Element Duplicate(string newTag);

        /// <summary>
        /// メンバの値を取得する。
        /// </summary>
        /// <param name="name">メンバの名前</param>
        /// <returns>メンバの値</returns>
        public virtual ScriptValue GetData(string name)
        {
            if (myData.ContainsKey(name)) {
                return new ScriptValue(myData[name]);
            } else if (Util.CaseInsensitiveEquals(name, "Tag")) {
                return Tag;
            } else {
                throw new GustFrontException(String.Format("プロパティ {0} は宣言されていません。", name));
            }
        }

        private void ValidateValue(object obj)
        {
            if (obj == null) {
                return;
            } else if (obj is IList) {
                foreach (object v in (IList)obj) {
                    ValidateValue(v);
                }
            } else if (obj is IDictionary) {
                foreach (object v in ((IDictionary)obj).Keys) {
                    ValidateValue(v);
                }
                foreach (object v in ((IDictionary)obj).Values) {
                    ValidateValue(v);
                }
            } else if (obj is bool) {
                return;
            } else if (obj is byte) {
                return;
            } else if (obj is short) {
                return;
            } else if (obj is long) {
                return;
            } else if (obj is int) {
                return;
            } else if (obj is float) {
                return;
            } else if (obj is decimal) {
                return;
            } else if (obj is double) {
                return;
            } else if (obj is string) {
                return;
            } else if (obj is Element) {
                return;
            } else {
                throw new GustFrontException("設定できない値が指定されました。");
            }
        }

        /// <summary>
        /// メンバに値を設定する。
        /// </summary>
        /// <param name="name">メンバの名前</param>
        /// <param name="value">設定する値</param>
        public virtual void SetData(string name, ScriptValue value)
        {
            if (myData.ContainsKey(name)) {
                ValidateValue(value.Instance);
                object current = myData[name];
                if (value.Instance == null) {
                    myData[name] = null;
                } else if (current is IList) {
                    IList c = (IList)current;
                    c.Clear();
                    foreach (object v in (IList)value.Instance) {
                        c.Add(v);
                    }
                } else if (current is IDictionary) {
                    IDictionary c = (IDictionary)current;
                    IDictionary v = (IDictionary)value.Instance;
                    c.Clear();
                    foreach (object key in v.Keys) {
                        c.Add(key, v[key]);
                    }
                } else if (current is bool) {
                    myData[name] = Convert.ToBoolean(value.Instance);
                } else if (current is byte) {
                    myData[name] = Convert.ToByte(value.Instance);
                } else if (current is short) {
                    myData[name] = Convert.ToInt16(value.Instance);
                } else if (current is long) {
                    myData[name] = Convert.ToInt64(value.Instance);
                } else if (current is int) {
                    myData[name] = Convert.ToInt32(value.Instance);
                } else if (current is float) {
                    myData[name] = Convert.ToSingle(value.Instance);
                } else if (current is decimal) {
                    myData[name] = Convert.ToDecimal(value.Instance);
                } else if (current is double) {
                    myData[name] = Convert.ToDouble(value.Instance);
                } else if (current is string) {
                    myData[name] = Convert.ToString(value.Instance);
                } else { // current is Element
                    if (value.Instance is Element) {
                        myData[name] = value.Instance;
                    } else if (value.Instance is string && myElementDic.ContainsKey((string)value.Instance)) {
                        myData[name] = myElementDic[(string)value.Instance];
                    } else {
                        throw new GustFrontException(String.Format("指定された値はこのプロパティに対して設定できません。", name));
                    }
                }
            } else {
                throw new GustFrontException(String.Format("プロパティ {0} は宣言されていません。", name));
            }
        }

        /// <summary>
        /// スクリプトからメンバの値を取得する。
        /// </summary>
        /// <param name="name">メンバの名前</param>
        /// <param name="params">無視される</param>
        /// <param name="result">値を代入する変数。値を取得できなかった場合は null が代入される。</param>
        /// <returns>取得できた場合は True を返し、それ以外の場合は False を返す。</returns>
        public virtual bool GetMember(string name, ScriptValue[] @params, out ScriptValue result)
        {
            if (myData.ContainsKey(name)) {
                result = GetData(name);
                return true;
            } else if (Util.CaseInsensitiveEquals(name, "Tag")) {
                result = Tag;
                return true;
            } else {
                result = null;
                return false;
            }
        }

        /// <summary>
        /// スクリプトでメンバに値を設定する。
        /// </summary>
        /// <param name="name">メンバの名前</param>
        /// <param name="params">設定する値（配列の最後の要素のみ参照）</param>
        /// <returns>設定できた場合は True を返し、それ以外の場合は False を返す。</returns>
        public virtual bool SetMember(string name, ScriptValue[] @params)
        {
            if (myData.ContainsKey(name)) {
                SetData(name, @params[@params.Length - 1]);
                return true;
            } else {
                return false;
            }
        }

        private static Dictionary<string, Element> myElementDic =
            CollectionUtil.CreateCaseInsensitiveDictionary<Element>();
        private static Dictionary<string, List<string>> myElementTagList =
            CollectionUtil.CreateCaseInsensitiveDictionary<List<string>>();

        /// <summary>
        /// 指定されたタグを持つエレメントを取得する
        /// </summary>
        /// <param name="tagName">タグ</param>
        /// <returns>エレメントが存在する場合、それを返す。それ以外の場合、null を返す。</returns>
        public static Element GetElement(string tagName)
        {
            if (myElementDic.ContainsKey(tagName)) {
                return myElementDic[tagName];
            } else {
                return null;
            }
        }

        /// <summary>
        /// 指定された型のエレメントを全て取得する
        /// </summary>
        /// <param name="typeName">型名</param>
        /// <returns>型が存在する場合、その全エレメントを配列として返す。それ以外の場合、null を返す。</returns>
        public static Element[] GetElements(string typeName)
        {
            if (myElementTagList.ContainsKey(typeName)) {
                List<string> tagList = myElementTagList[typeName];
                List<Element> elms = new List<Element>(tagList.Count);
                foreach (string tag in tagList) {
                    elms.Add(myElementDic[tag]);
                }
                return elms.ToArray();
            } else {
                return null;
            }
        }

        private static string EncodeString(string text)
        {
            StringBuilder sb = new StringBuilder(text.Length);

            int start = -1;
            for (int i = 0; i < text.Length; i++) {
                if (start != -1) {
                    if (text[i] == '"') {
                        if (i == text.Length - 1 || text[i + 1] != '"') {
                            sb.Append(text[i]);
                            start = -1;
                        } else {
                            i++;
                            sb.Append(((int)text[i]).ToString("X4"));
                        }
                    } else {
                        sb.Append(((int)text[i]).ToString("X4"));
                    }
                } else {
                    if (text[i] == '"') {
                        start = i;
                    }
                    sb.Append(text[i]);
                }
            }

            return sb.ToString();
        }

        private static string DecodeString(string text)
        {
            StringBuilder sb = new StringBuilder(text.Length);

            for (int i = 0; i < text.Length; i += 4) {
                sb.Append((char)Int32.Parse(text.Substring(i, 4), NumberStyles.HexNumber));
            }

            return sb.ToString();
        }

        private static string ToString(object obj)
        {
            if (obj == null) {
                return "Nothing";
            } else if (obj is IList) {
                StringBuilder sb = new StringBuilder();
                sb.Append("{");
                foreach (object v in (IList)obj) {
                    sb.Append(ToString(v));
                    sb.Append(',');
                    sb.Append(' ');
                }
                if (sb.Length > 1) {
                    sb.Remove(sb.Length - 2, 2);
                }
                sb.Append("}");
                return sb.ToString();
            } else if (obj is IDictionary) {
                IDictionary dic = (IDictionary)obj;
                StringBuilder sb = new StringBuilder();
                sb.Append("<");
                foreach (object key in dic.Keys) {
                    sb.Append(ToString(key));
                    sb.Append(':');
                    sb.Append(ToString(dic[key]));
                    sb.Append(',');
                    sb.Append(' ');
                }
                if (sb.Length > 1) {
                    sb.Remove(sb.Length - 2, 2);
                }
                sb.Append(">");
                return sb.ToString();
            } else if (obj is bool) {
                return ((bool)obj).ToString(CultureInfo.InvariantCulture);
            } else if (obj is byte) {
                return String.Concat(((byte)obj).ToString(CultureInfo.InvariantCulture), "B");
            } else if (obj is short) {
                return String.Concat(((short)obj).ToString(CultureInfo.InvariantCulture), "S");
            } else if (obj is int) {
                return ((int)obj).ToString(CultureInfo.InvariantCulture);
            } else if (obj is long) {
                return String.Concat(((long)obj).ToString(CultureInfo.InvariantCulture), "L");
            } else if (obj is decimal) {
                return String.Concat(((decimal)obj).ToString(CultureInfo.InvariantCulture), "D");
            } else if (obj is float) {
                float v = (float)obj;
                if (Single.MinValue <= v && v <= Single.MaxValue) {
                    return String.Concat(v.ToString(CultureInfo.InvariantCulture), "F");
                } else {
                    return v.ToString(CultureInfo.InvariantCulture);
                }
            } else if (obj is float || obj is double) {
                double v = (double)obj;
                if (Double.MinValue <= v && v <= Double.MaxValue) {
                    int t_int;
                    string s = v.ToString(CultureInfo.InvariantCulture);
                    if (!Int32.TryParse(s, NumberStyles.Integer, CultureInfo.InvariantCulture, out t_int)) {
                        return s;
                    } else {
                        return String.Concat(s, ".", "0");
                    }
                } else {
                    return v.ToString(CultureInfo.InvariantCulture);
                }
            } else if (obj is string) {
                return String.Concat('"', ((string)obj).Replace("\"", "\"\""), '"');
            } else if (obj is Element) {
                return String.Concat("[", ((Element)obj).Tag, "]");
            } else {
                throw new GustFrontException("セーブできない値が設定されています。");
            }
        }

        private static object Parse(string value)
        {
            value = value.Trim();

            char c_first = '\0';
            char c_last = '\0';
            if (value.Length >= 2) {
                c_first = Char.ToUpperInvariant(value[0]);
                c_last = Char.ToUpperInvariant(value[value.Length - 1]);
            }

            bool t_bool;
            byte t_byte;
            short t_short;
            int t_int;
            long t_long;
            decimal t_decimal;
            float t_float;
            double t_double;
            CultureInfo ivc = CultureInfo.InvariantCulture;

            if (Util.CaseInsensitiveEquals(value, "Nothing")) {
                return null;
            } else if (Boolean.TryParse(value, out t_bool)) {
                return t_bool;
            } else if (Int32.TryParse(value, NumberStyles.Integer, ivc, out t_int)) {
                return t_int;
            } else if (Double.TryParse(value, NumberStyles.Float, ivc, out t_double)) {
                return t_double;
            } else if (c_first == '"' && c_last == '"') {
                return DecodeString(value.Substring(1, value.Length - 2));
            } else if (c_first == '<' && c_last == '>') {
                int n = 0;
                int key = 0;
                int prev = 1;
                IDictionary result = CollectionUtil.CreateStdDictionary();
                for (int i = 0; i < value.Length; i++) {
                    switch (value[i]) {
                        case '{':
                            n++;
                            break;
                        case '<':
                            n++;
                            break;
                        case '}':
                            n--;
                            break;
                        case ':':
                            key = i;
                            break;
                        case '>':
                            n--;
                            if (n == 0) {
                                result.Add(
                                    Parse(value.Substring(prev, key - prev)),
                                    Parse(value.Substring(key + 1, i - key - 1)));
                                prev = i + 1;
                            }
                            break;
                        case ',':
                            if (n == 1) {
                                result.Add(
                                    Parse(value.Substring(prev, key - prev)),
                                    Parse(value.Substring(key, i - key)));
                                prev = i + 1;
                            }
                            break;
                    }
                }
                return result;
            } else if (c_first == '{' && c_last == '}') {
                int n = 0;
                int prev = 1;
                IList result = CollectionUtil.CreateStdList();
                for (int i = 0; i < value.Length; i++) {
                    switch (value[i]) {
                        case '{':
                            n++;
                            break;
                        case '<':
                            n++;
                            break;
                        case '}':
                            n--;
                            if (n == 0) {
                                string s = value.Substring(prev, i - prev);
                                if (result.Count > 0 || !String.IsNullOrEmpty(s.Trim())) {
                                    result.Add(Parse(value.Substring(prev, i - prev)));
                                    prev = i + 1;
                                }
                            }
                            break;
                        case '>':
                            n--;
                            break;
                        case ',':
                            if (n == 1) {
                                result.Add(Parse(value.Substring(prev, i - prev)));
                                prev = i + 1;
                            }
                            break;
                    }
                }
                return result;
            } else if (c_first == '[' && c_last == ']') {
                string theTag = value.Substring(1, value.Length - 2);
                if (myElementDic.ContainsKey(theTag)) {
                    return myElementDic[theTag];
                } else {
                    throw new GustFrontException(String.Format("タグ {0} は未定義です。", theTag));
                }
            } else {
                string v = value.Substring(0, value.Length - 1);
                switch (c_last) {
                    case 'B':
                        if (Byte.TryParse(v, NumberStyles.Integer, ivc, out t_byte)) return t_byte;
                        break;
                    case 'D':
                        if (Decimal.TryParse(v, NumberStyles.Number, ivc, out t_decimal)) return t_decimal;
                        break;
                    case 'F':
                        if (Single.TryParse(v, NumberStyles.Float, ivc, out t_float)) return t_float;
                        break;
                    case 'I':
                        if (Int32.TryParse(v, NumberStyles.Integer, ivc, out t_int)) return t_int;
                        break;
                    case 'L':
                        if (Int64.TryParse(v, NumberStyles.Integer, ivc, out t_long)) return t_long;
                        break;
                    case 'R':
                        if (Double.TryParse(v, NumberStyles.Float, ivc, out t_double)) return t_double;
                        break;
                    case 'S':
                        if (Int16.TryParse(v, NumberStyles.Integer, ivc, out t_short)) return t_short;
                        break;
                }
                throw new GustFrontException(String.Format("値 {0} を解析できませんでした。", value));
            }
        }

        /// <summary>
        /// 指定された型の全エレメントの内容を、ストリームに書き込む。
        /// </summary>
        /// <param name="writer">書き込みに用いる TextWriter のインスタンス</param>
        /// <param name="typeName">型名</param>
        public static void Save(TextWriter writer, string typeName)
        {
            foreach (string tag in myElementTagList[typeName]) {
                writer.Write('[');
                writer.Write(tag);
                writer.WriteLine(']');
                Element elm = myElementDic[tag];
                foreach (string key in elm.myKeyOrder) {
                    writer.Write(key);
                    writer.Write('=');
                    writer.WriteLine(ToString(elm.myData[key]));
                }
                writer.WriteLine();
            }
        }

        /// <summary>
        /// ストリームから値を読み込み、指定された型のエレメントとして初期化する
        /// </summary>
        /// <param name="reader">読み込みに用いる TextReader のインスタンス</param>
        /// <param name="typeName">型名</param>
        /// <param name="t">Element を継承する非抽象型</param>
        /// <returns></returns>
        public static Element[] Load(TextReader reader, string typeName, Type t)
        {
            Element elm = null;
            List<Element> elms = new List<Element>();
            List<string> tags = new List<string>();
            bool init_now = !myElementTagList.ContainsKey(typeName);
            if (init_now) myElementTagList.Add(typeName, tags);

            string line = reader.ReadLine();
            while (line != null) {
                if (line.Length > 0 && !line.StartsWith(";")) {
                    Match mkv = Regex.Match(line, "\\s*(.+)\\s*=\\s*(.*)\\s*", RegexOptions.Compiled);
                    if (mkv.Success) {
                        string key = mkv.Groups[1].Value;
                        string value = mkv.Groups[2].Value;
                        if (init_now) {
                            if (!elm.myData.ContainsKey(key)) {
                                elm.myData[key] = Parse(EncodeString(value));
                                elm.myKeyOrder.Add(key);
                            } else {
                                throw new GustFrontException(String.Format("キー {0} は既に使用されています。", key));
                            }
                        } else {
                            if (elm.myData.ContainsKey(key)) {
                                elm.myData[key] = Parse(EncodeString(value));
                            } else {
                                throw new GustFrontException(String.Format("キー {0} は存在しません。", key));
                            }
                        }
                    } else {
                        Match mtag = Regex.Match(line, "\\[\\s*(\\D.*)\\s*\\]", RegexOptions.Compiled);
                        if (mtag.Success) {
                            string tagName = mtag.Groups[1].Value;
                            if (init_now) {
                                if (!myElementDic.ContainsKey(tagName)) {
                                    elm = (Element)t.InvokeMember(null, BindingFlags.CreateInstance,
                                        null, null, new object[] { tagName });
                                    elm.myType = typeName;
                                    myElementDic.Add(tagName, elm);
                                    tags.Add(tagName);
                                } else {
                                    throw new GustFrontException(String.Format("タグ {0} は既に使用されています。", tagName));
                                }
                            } else {
                                if (myElementDic.ContainsKey(tagName)) {
                                    elm = myElementDic[tagName];
                                } else {
                                    throw new GustFrontException(String.Format("タグ {0} は存在しません。", tagName));
                                }
                            }
                            elms.Add(elm);
                        } else {
                            throw new GustFrontException("設定ファイルに不正な行があります。");
                        }
                    }
                }
                line = reader.ReadLine();
            }

            return elms.ToArray();
        }
    }

    /// <summary>
    /// Element の単純な実装
    /// </summary>
    public class UserElement : Element
    {
        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="tagName">エレメントを識別するタグ</param>
        public UserElement(string tagName)
            : base(tagName)
        {
        }

        /// <summary>
        /// エレメントを複製する
        /// </summary>
        /// <param name="newTag">新しいエレメントを識別するタグ</param>
        /// <returns>複製されたエレメント</returns>
        public override Element Duplicate(string newTag)
        {
            UserElement obj = new UserElement(newTag);
            CopyTo(obj);
            return obj;
        }

        /// <summary>
        /// 汎用データ
        /// </summary>
        public object Data { get; set; }
    }
}