/*
    wcommon
    copyright (c) 1998-2018 Kazuki Iwamoto https://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "wcommon.h"
#include <shlwapi.h>


/******************************************************************************
*                                                                             *
******************************************************************************/
/*  文字が数字であるかどうかを判断する
     ch,文字
    RET,TRUE:数字,FALSE:その他                                              */
BOOL WINAPI
IsCharNumericW (WCHAR ch)
{
  return IsCharAlphaNumericW (ch) && !IsCharAlphaW (ch);
}


/*  文字が数字であるかどうかを判断する
     ch,文字
    RET,TRUE:数字,FALSE:その他                                              */
BOOL WINAPI
IsCharNumericA (CHAR ch)
{
  return IsCharAlphaNumericA (ch) && !IsCharAlphaA (ch);
}


/*  文字がコントロールコードであるかどうかを判断する
     ch,文字
    RET,TRUE:コントロールコード,FALSE:その他                                */
BOOL WINAPI
IsCharControlW (WCHAR ch)
{
  WORD ct;

  return !GetStringTypeExW (LOCALE_USER_DEFAULT, CT_CTYPE1, &ch, 1, &ct)
                                                || (ct & C1_CNTRL) || ct == 0;
}


/*  文字がコントロールコードであるかどうかを判断する
     ch,文字
    RET,TRUE:コントロールコード,FALSE:その他                                */
BOOL WINAPI
IsCharControlA (CHAR ch)
{
  WORD ct;

  return !GetStringTypeExA (LOCALE_USER_DEFAULT, CT_CTYPE1, &ch, 1, &ct)
                                                || (ct & C1_CNTRL) || ct == 0;
}


/******************************************************************************
*                                                                             *
******************************************************************************/
/*  8進数文字列を数値に変換する
    lpszText,文字列
         RET,数値                                                           */
DWORD WINAPI
OctW (LPCWSTR lpszText)
{
  DWORD dwResult;

  if (lpszText)
    {
      LPCWSTR p;

      dwResult = 0;
      for (p = lpszText; *p != '\0'; p++)
        {
          int i;
          const static WCHAR cOct[] = {'0', '1', '2', '3', '4', '5', '6', '7'};

          for (i = 0; i < n_elements (cOct); i++)
            if (*p == cOct[i])
              {
                dwResult *= n_elements (cOct);
                dwResult += i;
                break;
              }
          if (i >= n_elements (cOct))
            {
              dwResult = (DWORD)-1;
              break;
            }
        }
    }
  else
    {
      dwResult = (DWORD)-1;
    }
  return dwResult;
}


/*  8進数文字列を数値に変換する
    lpszText,文字列
         RET,数値                                                           */
DWORD WINAPI
OctA (LPCSTR lpszText)
{
  DWORD dwResult;

  if (lpszText)
    {
      LPCSTR p;

      dwResult = 0;
      for (p = lpszText; *p != '\0'; p++)
        {
          int i;
          const static CHAR cOct[] = {'0', '1', '2', '3', '4', '5', '6', '7'};

          for (i = 0; i < n_elements (cOct); i++)
            if (*p == cOct[i])
              {
                dwResult *= n_elements (cOct);
                dwResult += i;
                break;
              }
          if (i >= n_elements (cOct))
            {
              dwResult = (DWORD)-1;
              break;
            }
        }
    }
  else
    {
      dwResult = (DWORD)-1;
    }
  return dwResult;
}


/*  10進数文字列を数値に変換する
    lpszText,文字列
         RET,数値                                                           */
DWORD WINAPI
DecW (LPCWSTR lpszText)
{
  DWORD dwResult;

  if (lpszText)
    {
      LPCWSTR p;

      dwResult = 0;
      for (p = lpszText; *p != '\0'; p++)
        {
          int i;
          const static WCHAR cDec[] = {'0', '1', '2', '3', '4',
                                       '5', '6', '7', '8', '9'};

          for (i = 0; i < n_elements (cDec); i++)
            if (*p == cDec[i])
              {
                dwResult *= n_elements (cDec);
                dwResult += i;
                break;
              }
          if (i >= n_elements (cDec))
            {
              dwResult = (DWORD)-1;
              break;
            }
        }
    }
  else
    {
      dwResult = (DWORD)-1;
    }
  return dwResult;
}


/*  10進数文字列を数値に変換する
    lpszText,文字列
         RET,数値                                                           */
DWORD WINAPI
DecA (LPCSTR lpszText)
{
  DWORD dwResult;

  if (lpszText)
    {
      LPCSTR p;

      dwResult = 0;
      for (p = lpszText; *p != '\0'; p++)
        {
          int i;
          const static CHAR cDec[] = {'0', '1', '2', '3', '4',
                                      '5', '6', '7', '8', '9'};

          for (i = 0; i < n_elements (cDec); i++)
            if (*p == cDec[i])
              {
                dwResult *= n_elements (cDec);
                dwResult += i;
                break;
              }
          if (i >= n_elements (cDec))
            {
              dwResult = (DWORD)-1;
              break;
            }
        }
    }
  else
    {
      dwResult = (DWORD)-1;
    }
  return dwResult;
}


/*  16進数文字列を数値に変換する
    lpszText,文字列
         RET,数値                                                           */
DWORD WINAPI
HexW (LPCWSTR lpszText)
{
  DWORD dwResult;

  if (lpszText)
    {
      LPCWSTR p;

      dwResult = 0;
      for (p = lpszText; *p != '\0'; p++)
        {
          int i;
          WCHAR c;
          const static WCHAR cHex[] = {'0', '1', '2', '3', '4', '5', '6', '7',
                                       '8', '9', 'a', 'b', 'c', 'd', 'e', 'f'};

          c = (WCHAR)CharLowerW ((LPWSTR)*p);
          for (i = 0; i < n_elements (cHex); i++)
            if (c == cHex[i])
              {
                dwResult <<= 4;
                dwResult += i;
                break;
              }
          if (i >= n_elements (cHex))
            {
              dwResult = (DWORD)-1;
              break;
            }
        }
    }
  else
    {
      dwResult = (DWORD)-1;
    }
  return dwResult;
}


/*  16進数文字列を数値に変換する
    lpszText,文字列
         RET,数値                                                           */
DWORD WINAPI
HexA (LPCSTR lpszText)
{
  DWORD dwResult;

  if (lpszText)
    {
      LPCSTR p;

      dwResult = 0;
      for (p = lpszText; *p != '\0'; p++)
        {
          int i;
          CHAR c;
          const static CHAR cHex[] = {'0', '1', '2', '3', '4', '5', '6', '7',
                                      '8', '9', 'a', 'b', 'c', 'd', 'e', 'f'};

          c = (CHAR)CharLowerA ((LPSTR)*p);
          for (i = 0; i < n_elements (cHex); i++)
            if (c == cHex[i])
              {
                dwResult <<= 4;
                dwResult += i;
                break;
              }
          if (i >= n_elements (cHex))
            {
              dwResult = (DWORD)-1;
              break;
            }
        }
    }
  else
    {
      dwResult = (DWORD)-1;
    }
  return dwResult;
}


/*  ja:数値→文字列
        nValue,数値
    lpszString,文字列
        nRadix,基数
         nWide,桁数(正:右詰め空白,負:右詰め0)
       fSigned,TRUE:符号あり,FALSE:符号なし                                 */
VOID WINAPI
ValStrW (int    nValue,
         LPWSTR lpszString,
         int    nRadix,
         int    nWide,
         BOOL   fSigned)
{
  if (lpszString && 2 <= nRadix && nRadix <= 16)
    {
      int i = 0, j;
      const static WCHAR cHex[] = {'0', '1', '2', '3', '4', '5', '6', '7',
                                   '8', '9', 'A', 'B', 'C', 'D', 'E', 'F'};

      if (nValue == 0)
        {
          lpszString[i++] = cHex[0];
        }
      else
        {
          int t;

          t = nValue;
          if (fSigned && nValue < 0)
            {
              while (t != 0)
                {
                  lpszString[i++] = cHex[abs (t % nRadix)];
                  t /= nRadix;
                }
            }
          else
            {
              while (t != 0)
                {
                  lpszString[i++] = cHex[(unsigned)t % nRadix];
                  t = (unsigned)t / nRadix;
                }
            }
        }
      if (fSigned && nValue < 0)
        lpszString[i++] = '-';
      if (nWide > 0)
        {
          while (i < nWide)
            lpszString[i++] = ' ';
        }
      else if (nWide < 0)
        {
          for (j = i; j < -nWide; j++)
            lpszString[j] = cHex[0];
          if (lpszString[i - 1] == '-')
            {
              lpszString[i - 1] = cHex[0];
              lpszString[j - 1] = '-';
            }
          i = j;
        }
      lpszString[i] = '\0';
      for (j = 0; j < i / 2; j++)
        {
          WCHAR c;

          c = lpszString[j];
          lpszString[j] = lpszString[i - j - 1];
          lpszString[i - j - 1] = c;
        }
    }
}


/*  ja:数値→文字列
        nValue,数値
    lpszString,文字列
        nRadix,基数
         nWide,桁数(正:右詰め空白,負:右詰め0)
       fSigned,TRUE:符号あり,FALSE:符号なし                                 */
VOID WINAPI
ValStrA (int   nValue,
         LPSTR lpszString,
         int   nRadix,
         int   nWide,
         BOOL  fSigned)
{
  if (lpszString && 2 <= nRadix && nRadix <= 16)
    {
      int i = 0, j;
      const static CHAR cHex[] = {'0', '1', '2', '3', '4', '5', '6', '7',
                                  '8', '9', 'A', 'B', 'C', 'D', 'E', 'F'};

      if (nValue == 0)
        {
          lpszString[i++] = cHex[0];
        }
      else
        {
          int t;

          t = nValue;
          if (fSigned && nValue < 0)
            {
              while (t != 0)
                {
                  lpszString[i++] = cHex[abs (t % nRadix)];
                  t /= nRadix;
                }
            }
          else
            {
              while (t != 0)
                {
                  lpszString[i++] = cHex[(unsigned)t % nRadix];
                  t = (unsigned)t / nRadix;
                }
            }
        }
      if (fSigned && nValue < 0)
        lpszString[i++] = '-';
      if (nWide > 0)
        {
          while (i < nWide)
            lpszString[i++] = ' ';
        }
      else if (nWide < 0)
        {
          for (j = i; j < -nWide; j++)
            lpszString[j] = cHex[0];
          if (lpszString[i - 1] == '-')
            {
              lpszString[i - 1] = cHex[0];
              lpszString[j - 1] = '-';
            }
          i = j;
        }
      lpszString[i] = '\0';
      for (j = 0; j < i / 2; j++)
        {
          CHAR c;

          c = lpszString[j];
          lpszString[j] = lpszString[i - j - 1];
          lpszString[i - j - 1] = c;
        }
    }
}


/*  ja:文字列→数値
        nValue,数値
    lpszString,文字列
        nRadix,基数
           RET,TRUE:正常終了,FALSE:エラー                                   */
BOOL WINAPI
StrValW (int     *nValue,
         LPCWSTR  lpszString,
         int      nRadix)
{
  BOOL fResult = FALSE;

  if (nValue && lpszString && 2 <= nRadix && nRadix <= 16)
    {
      unsigned s = 0;
      BOOL fSign;
      LPCWSTR p, lpszHex = L"0123456789abcdef";

      for (p = lpszString; *p != '\0' && *p != '-'; p++)
        {
          LPWSTR q;

          q = StrChrIW (lpszHex, *p);
          if (q && q - lpszHex < nRadix)
            break;
        }
      if (fSign = (*p == '-'))
        p++;
      while (*p != '\0')
        {
          int n;
          LPWSTR q;

          s *= nRadix;
          q = StrChrIW (lpszHex, *p);
          if (!q || (n = q - lpszHex) >= nRadix)
            break;
          s += n;
          p++;
          fResult = TRUE;
        }
      if (fResult)
        *nValue = fSign ? -s : s;
    }
  return fResult;
}


/*  ja:文字列→数値
        nValue,数値
    lpszString,文字列
        nRadix,基数
           RET,TRUE:正常終了,FALSE:エラー                                   */
BOOL WINAPI
StrValA (int    *nValue,
         LPCSTR  lpszString,
         int     nRadix)
{
  BOOL fResult = FALSE;

  if (nValue && lpszString && 2 <= nRadix && nRadix <= 16)
    {
      unsigned s = 0;
      BOOL fSign;
      LPCSTR p, lpszHex = "0123456789abcdef";

      for (p = lpszString; *p != '\0' && *p != '-'; p++)
        {
          LPSTR q;

          q = StrChrIA (lpszHex, *p);
          if (q && q - lpszHex < nRadix)
            break;
        }
      if (fSign = (*p == '-'))
        p++;
      while (*p != '\0')
        {
          int n;
          LPSTR q;

          s *= nRadix;
          q = StrChrIA (lpszHex, *p);
          if (!q || (n = q - lpszHex) >= nRadix)
            break;
          s += n;
          p++;
          fResult = TRUE;
        }
      if (fResult)
        *nValue = fSign ? -s : s;
    }
  return fResult;
}


/******************************************************************************
*                                                                             *
******************************************************************************/
/*  文字列を書式化する
    ppszString,文字列
    lpszFormat,書式文字列
       arglist,引数
           RET,文字数(終端のNULLを含まない)                                 */
SSIZE_T WINAPI
wvasprintfW (LPWSTR  *ppszString,
             LPCWSTR  lpszFormat,
             va_list  arglist)
{
  SSIZE_T nResult = -1;

  if (lpszFormat && arglist)
    {
      LPWSTR lpszText = NULL;
      SSIZE_T nSize = 0;

      do
        {
          nSize += 4096;
          lpszText = MemoryReAlloc (lpszText, nSize * sizeof (WCHAR));
          nResult = wvnsprintfW (lpszText, nSize, lpszFormat, arglist);
        }
      while (nResult < 0 || nResult >= nSize - 1);
      if (ppszString)
        *ppszString = MemoryReAlloc (lpszText, (nResult + 1) * sizeof (WCHAR));
      else
        MemoryFree (lpszText);
    }
  return nResult;
}


/*  文字列を書式化する
    ppszString,文字列
    lpszFormat,書式文字列
       arglist,引数
           RET,文字数(終端のNULLを含まない)                                 */
SSIZE_T WINAPI
wvasprintfA (LPSTR  *ppszString,
             LPCSTR  lpszFormat,
             va_list arglist)
{
  SSIZE_T nResult = -1;

  if (lpszFormat && arglist)
    {
      LPSTR lpszText = NULL;
      SSIZE_T nSize = 0;

      do
        {
          nSize += 4096;
          lpszText = MemoryReAlloc (lpszText, nSize * sizeof (CHAR));
          nResult = wvnsprintfA (lpszText, nSize, lpszFormat, arglist);
        }
      while (nResult < 0 || nResult >= nSize - 1);
      if (ppszString)
        *ppszString = MemoryReAlloc (lpszText, (nResult + 1) * sizeof (CHAR));
      else
        MemoryFree (lpszText);
    }
  return nResult;
}


/*  文字列を書式化する
    ppszString,文字列
      lpFormat,書式文字列
           RET,文字数(終端のNULLを含まない)                                 */
SSIZE_T WINAPI
wasprintfW (LPWSTR  *ppszString,
            LPCWSTR  lpszFormat, ...)
{
  int nResult;
  va_list arglist;

  va_start (arglist, lpszFormat);
  nResult = wvasprintfW (ppszString, lpszFormat, arglist);
  va_end (arglist);
  return nResult;
}


/*  文字列を書式化する
    ppszString,文字列
      lpFormat,書式文字列
           RET,文字数(終端のNULLを含まない)                                 */
SSIZE_T WINAPI
wasprintfA (LPSTR  *ppszString,
            LPCSTR  lpszFormat, ...)
{
  int nResult;
  va_list arglist;

  va_start (arglist, lpszFormat);
  nResult = wvasprintfA (ppszString, lpszFormat, arglist);
  va_end (arglist);
  return nResult;
}


/*  文字列を複製する
    lpszText,文字列
       nSize,文字数(負:NULL終端文字列)
         RET,複製したメモリ,NULL:エラー                                     */
LPWSTR WINAPI
StringDuplicateExW (LPCWSTR lpszText,
                    SSIZE_T nSize)
{
  LPWSTR lpszResult = NULL;

  if (lpszText)
    {
      if (nSize > 0 && lpszText[nSize - 1] != '\0')
        nSize++;
      else if (nSize < 0)
        nSize = lstrlenW (lpszText) + 1;
      if (nSize > 0)
        {
          lpszResult = MemoryAlloc (nSize * sizeof (WCHAR));
          MemoryCopy (lpszResult, lpszText, (nSize - 1) * sizeof (WCHAR));
        }
    }
  return lpszResult;
}


/*  文字列を複製する
    lpszText,文字列
       nSize,文字数(負:NULL終端文字列)
         RET,複製したメモリ,NULL:エラー                                     */
LPSTR WINAPI
StringDuplicateExA (LPCSTR  lpszText,
                    SSIZE_T nSize)
{
  LPSTR lpszResult = NULL;

  if (lpszText)
    {
      if (nSize > 0 && lpszText[nSize - 1] != '\0')
        nSize++;
      else if (nSize < 0)
        nSize = lstrlenA (lpszText) + 1;
      if (nSize > 0)
        {
          lpszResult = MemoryAlloc (nSize * sizeof (CHAR));
          MemoryCopy (lpszResult, lpszText, (nSize - 1) * sizeof (CHAR));
        }
    }
  return lpszResult;
}


/*  文字列を複製する
    lpszText,文字列
         RET,複製したメモリ,NULL:エラー                                     */
LPWSTR WINAPI
StringDuplicateW (LPCWSTR lpszText)
{
  return StringDuplicateExW (lpszText, -1);
}


/*  文字列を複製する
    lpszText,文字列
         RET,複製したメモリ,NULL:エラー                                     */
LPSTR WINAPI
StringDuplicateA (LPCSTR lpszText)
{
  return StringDuplicateExA (lpszText, -1);
}


/******************************************************************************
*                                                                             *
******************************************************************************/
/*  文字列をエスケープする
    lpszString,文字列
         nSize,文字数(負:NULL終端文字列)
           RET,エスケープされた文字列                                       */
LPWSTR WINAPI
StringEscapeExW (LPCWSTR lpszString,
                 SSIZE_T nSize)
{
  LPWSTR lpszResult;

  if (lpszString)
    {
      int nLen;
      LPCWSTR p, r;
      LPWSTR q;

      nLen = nSize >= 0 ? nSize : lstrlenW (lpszString);
      p = lpszString;
      r = p + nLen;
      q = lpszResult = MemoryAlloc ((nLen * 6 + 1) * sizeof (WCHAR));
      while (p < r)
        {
          switch (*p)
            {
              case '\a': *q++ = '\\'; *q++ = 'a';  break;
              case '\b': *q++ = '\\'; *q++ = 'b';  break;
              case '\n': *q++ = '\\'; *q++ = 'n';  break;
              case '\r': *q++ = '\\'; *q++ = 'r';  break;
              case '\f': *q++ = '\\'; *q++ = 'f';  break;
              case '\t': *q++ = '\\'; *q++ = 't';  break;
              case '\v': *q++ = '\\'; *q++ = 'v';  break;
              case '\\': *q++ = '\\'; *q++ = '\\'; break;
              case '\?': *q++ = '\\'; *q++ = '\?'; break;
              case '\'': *q++ = '\\'; *q++ = '\''; break;
              case '\"': *q++ = '\\'; *q++ = '\"'; break;
              default:
                if (0x20 <= *p && *p <= 0x7e)
                  {
                    *q++ = *p;
                  }
                else
                  {
                    wsprintfW (q, L"\\u%04X", *p);
                    q += 6;
                  }
            }
          p++;
        }
    }
  else
    {
      lpszResult = NULL;
    }
  return lpszResult;
}


/*  文字列をエスケープする
    lpszString,文字列
         nSize,文字数(負:NULL終端文字列)
           RET,エスケープされた文字列                                       */
LPSTR WINAPI
StringEscapeExA (LPCSTR  lpszString,
                 SSIZE_T nSize)
{
  LPSTR lpszResult;

  if (lpszString)
    {
      int nLen;
      LPCSTR p, r;
      LPSTR q;

      nLen = nSize >= 0 ? nSize : lstrlenA (lpszString);
      p = lpszString;
      r = p + nLen;
      q = lpszResult = MemoryAlloc ((nLen * 4 + 1) * sizeof (CHAR));
      while (p < r)
        {
          switch (*p)
            {
              case '\a': *q++ = '\\'; *q++ = 'a';  break;
              case '\b': *q++ = '\\'; *q++ = 'b';  break;
              case '\n': *q++ = '\\'; *q++ = 'n';  break;
              case '\r': *q++ = '\\'; *q++ = 'r';  break;
              case '\f': *q++ = '\\'; *q++ = 'f';  break;
              case '\t': *q++ = '\\'; *q++ = 't';  break;
              case '\v': *q++ = '\\'; *q++ = 'v';  break;
              case '\\': *q++ = '\\'; *q++ = '\\'; break;
              case '\?': *q++ = '\\'; *q++ = '\?'; break;
              case '\'': *q++ = '\\'; *q++ = '\''; break;
              case '\"': *q++ = '\\'; *q++ = '\"'; break;
              default:
                if (0x20 <= *p && *p <= 0x7e)
                  {
                    *q++ = *p;
                  }
                else
                  {
                    wsprintfA (q, "\\x%02X", (BYTE)*p);
                    q += 4;
                  }
            }
          p++;
        }
    }
  else
    {
      lpszResult = NULL;
    }
  return lpszResult;
}


/*  文字列をエスケープする
    lpszString,文字列
           RET,エスケープされた文字列                                       */
LPWSTR WINAPI
StringEscapeW (LPCWSTR lpszString)
{
  return StringEscapeExW (lpszString, -1);
}


/*  文字列をエスケープする
    lpszString,文字列
           RET,エスケープされた文字列                                       */
LPSTR WINAPI
StringEscapeA (LPCSTR lpszString)
{
  return StringEscapeExA (lpszString, -1);
}


/*  文字列をエスケープしない
    lpszString,エスケープされた文字列
         nSize,文字数(負:NULL終端文字列)
           RET,文字列                                                       */
LPWSTR WINAPI
StringUnescapeExW (LPCWSTR lpszString,
                   SSIZE_T nSize)
{
  LPWSTR lpszResult;

  if (lpszString)
    {
      int nLen;
      LPCWSTR p, r;
      LPWSTR q;

      nLen = nSize >= 0 ? nSize : lstrlenW (lpszString);
      p = lpszString;
      r = p + nLen;
      q = lpszResult = MemoryAlloc ((nLen + 1) * sizeof (WCHAR));
      while (p < r)
        if (*p == '\\')
          {
            switch (*++p)
              {
                case 'a':  *q++ = '\a';  p++; break;
                case 'b':  *q++ = '\b';  p++; break;
                case 'n':  *q++ = '\n';  p++; break;
                case 'r':  *q++ = '\r';  p++; break;
                case 'f':  *q++ = '\f';  p++; break;
                case 't':  *q++ = '\t';  p++; break;
                case 'v':  *q++ = '\v';  p++; break;
                case '\\': *q++ = '\\';  p++; break;
                case '\?': *q++ = '\?';  p++; break;
                case '\'': *q++ = '\'';  p++; break;
                case '\"': *q++ = '\"';  p++; break;
                case 'x':
                  {
                    int v;
                    WCHAR szString[3];

                    lstrcpynW (szString, p + 1, n_elements (szString));
                    if (lstrlenW (szString) == n_elements (szString) - 1
                                                && (v = HexW (szString)) >= 0)
                      {
                        *q++ = v;
                        p += n_elements (szString);
                      }
                    else
                      {
                        *q++ = '\\';
                      }
                  }
                  break;
                case 'u':
                  {
                    int v;
                    WCHAR szString[5];

                    lstrcpynW (szString, p + 1, n_elements (szString));
                    if (lstrlenW (szString) == n_elements (szString) - 1
                                                && (v = HexW (szString)) >= 0)
                      {
                        *q++ = v;
                        p += n_elements (szString);
                      }
                    else
                      {
                        *q++ = '\\';
                      }
                  }
                  break;
                default:
                  {
                    int v;
                    WCHAR szString[4];

                    lstrcpynW (szString, p, n_elements (szString));
                    if (lstrlenW (szString) == n_elements (szString) - 1
                                                && (v = OctW (szString)) >= 0)
                      {
                        *q++ = v;
                        p += n_elements (szString);
                      }
                    else if (*p == '0')
                      {
                        *q++ = '\0';
                        p++;
                      }
                    else
                      {
                        *q++ = '\\';
                      }
                  }
              }
          }
        else
          {
            *q++ = *p++;
          }
    }
  else
    {
      lpszResult = NULL;
    }
  return lpszResult;
}


/*  文字列をエスケープしない
    lpszString,エスケープされた文字列
         nSize,文字数(負:NULL終端文字列)
           RET,文字列                                                       */
LPSTR WINAPI
StringUnescapeExA (LPCSTR  lpszString,
                   SSIZE_T nSize)
{
  LPSTR lpszResult;

  if (lpszString)
    {
      int nLen;
      LPCSTR p, r;
      LPSTR q;

      nLen = nSize >= 0 ? nSize : lstrlenA (lpszString);
      p = lpszString;
      r = p + nLen;
      q = lpszResult = MemoryAlloc ((nLen + 1) * sizeof (CHAR));
      while (p < r)
        if (*p == '\\')
          {
            switch (*++p)
              {
                case 'a':  *q++ = '\a';  p++; break;
                case 'b':  *q++ = '\b';  p++; break;
                case 'n':  *q++ = '\n';  p++; break;
                case 'r':  *q++ = '\r';  p++; break;
                case 'f':  *q++ = '\f';  p++; break;
                case 't':  *q++ = '\t';  p++; break;
                case 'v':  *q++ = '\v';  p++; break;
                case '\\': *q++ = '\\';  p++; break;
                case '\?': *q++ = '\?';  p++; break;
                case '\'': *q++ = '\'';  p++; break;
                case '\"': *q++ = '\"';  p++; break;
                case 'x':
                  {
                    int v;
                    CHAR szString[3];

                    lstrcpynA (szString, p + 1, n_elements (szString));
                    if (lstrlenA (szString) == n_elements (szString) - 1
                                                && (v = HexA (szString)) >= 0)
                      {
                        *q++ = v;
                        p += n_elements (szString);
                      }
                    else
                      {
                        *q++ = '\\';
                      }
                  }
                  break;
                default:
                  {
                    int v;
                    CHAR szString[4];

                    lstrcpynA (szString, p, n_elements (szString));
                    if (lstrlenA (szString) == n_elements (szString) - 1
                                                && (v = OctA (szString)) >= 0)
                      {
                        *q++ = v;
                        p += n_elements (szString);
                      }
                    else if (*p == '0')
                      {
                        *q++ = '\0';
                        p++;
                      }
                    else
                      {
                        *q++ = '\\';
                      }
                  }
              }
          }
        else
          {
            if (IsDBCSLeadByte (*p))
              *q++ = *p++;
            *q++ = *p++;
          }
    }
  else
    {
      lpszResult = NULL;
    }
  return lpszResult;
}


/*  文字列をエスケープしない
    lpszString,エスケープされた文字列
           RET,文字列                                                       */
LPWSTR WINAPI
StringUnescapeW (LPCWSTR lpszString)
{
  return StringUnescapeExW (lpszString, -1);
}


/*  文字列をエスケープしない
    lpszString,エスケープされた文字列
           RET,文字列                                                       */
LPSTR WINAPI
StringUnescapeA (LPCSTR lpszString)
{
  return StringUnescapeExA (lpszString, -1);
}


/*  文字列を正規化する
    lpszString,文字列
         nSize,文字数(負:NULL終端文字列)
           RET,正規化された文字列                                           */
LPWSTR WINAPI
StringCanonicalizeExW (LPCWSTR lpszString,
                       SSIZE_T nSize)
{
  LPWSTR lpszResult;

  lpszResult = StringEscapeExW (lpszString, nSize);
  return lpszResult ? lpszResult : StringDuplicateW (L"NULL");
}


/*  文字列を正規化する
    lpszString,文字列
         nSize,文字数(負:NULL終端文字列)
           RET,正規化された文字列                                           */
LPSTR WINAPI
StringCanonicalizeExA (LPCSTR  lpszString,
                       SSIZE_T nSize)
{
  LPSTR lpszResult;

  lpszResult = StringEscapeExA (lpszString, nSize);
  return lpszResult ? lpszResult : StringDuplicateA ("NULL");
}


/*  文字列を正規化する
    lpszString,文字列
           RET,正規化された文字列                                           */
LPWSTR WINAPI
StringCanonicalizeW (LPCWSTR lpszString)
{
  return StringCanonicalizeExW (lpszString, -1);
}


/*  文字列を正規化する
    lpszString,文字列
           RET,正規化された文字列                                           */
LPSTR WINAPI
StringCanonicalizeA (LPCSTR lpszString)
{
  return StringCanonicalizeExA (lpszString, -1);
}
