/*
    IA32
    copyright (c) 1998-2011 Kazuki Iwamoto http://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "process.h"
#include "cpu.h"
#include "memory.h"
#include "message.h"
#include "module.h"
#include "stack.h"
#include "system.h"
#include "misc/peimage.h"


/******************************************************************************
*                                                                             *
* ja:プロセス/スレッド関数群                                                  *
*                                                                             *
******************************************************************************/
/*  ja: スレッド作成
    process,プロセス構造体
        RET,スレッド構造体                                                  */
Ia32Thread *
ia32_create_thread (Ia32Process *process)
{
  static guint id = 1;
  Ia32Thread *thread = NULL;

  if (process)
    {
      thread = g_malloc0 (sizeof (Ia32Thread));
      thread->id = id++;
      thread->process = process;
      thread->reg.esp = ia32_memory_alloc (process, 0x60000,
                                    IA32_STACK_SIZE, IA32_MEMORY_ATTR_STACK);
      if (thread->reg.esp == IA32_MEMORY_NULL)
        thread->reg.esp = ia32_memory_alloc (process, IA32_MEMORY_NULL,
                                    IA32_STACK_SIZE, IA32_MEMORY_ATTR_STACK);
      thread->reg.esp += IA32_STACK_SIZE;
      thread->reg.eflags = 0x00000246;
      process->thread = g_list_append (process->thread, thread);
    }
  return thread;
}


/*  ja: スレッド破棄
    thread,スレッド構造体
       RET,TRUE:正常終了,FALSE:エラー                                       */
gboolean
ia32_destroy_thread (Ia32Thread *thread)
{
  if (!thread)
    return FALSE;
  if (thread->process->current == thread)
    thread->process->current = NULL;
  thread->process->thread = g_list_remove (thread->process->thread, thread);
  while (thread->err)
    {
      g_free (thread->err->data);
      thread->err = g_list_delete_link (thread->err, thread->err);
    }
  while (thread->api)
    {
      g_free (thread->api->data);
      thread->api = g_list_delete_link (thread->api, thread->api);
    }
  while (thread->trace)
    {
      g_free (thread->trace->data);
      thread->trace = g_list_delete_link (thread->trace, thread->trace);
    }
  g_free (thread);
  return TRUE;
}


/*  ja:プロセス作成
    RET,プロセス                                                            */
static Ia32Process *
ia32_create_process (void)
{
  static guint id = 1;
  gint i;
  guint8 *p;
  guint32 address, fs;
  Ia32Process *process;
  Ia32CpuDesc *desc;
  ThreadEnvironmentBlock *teb;
  NtThreadInformationBlock *nttib;
  ProcessEnvironmentBlock *peb;
  ProcessEnvironmentBlockLdrData *pebld;
  ListEntry *lste0, *lste1;

  process = g_malloc0 (sizeof (Ia32Process));
  process->id = id++;
  process->address = IA32_MEMORY_NULL;
  process->winmain = IA32_MEMORY_NULL;
  process->export = g_hash_table_new_full (NULL, NULL, NULL, g_free);
  process->system = g_hash_table_new (g_str_hash, g_str_equal);
  process->current = ia32_create_thread (process);

  /* ja:システム情報 */
  for (i = 0; ia32_system_exports[i].func; i++);
  fs = address = ia32_memory_alloc (process, IA32_MEMORY_NULL,
                                    sizeof (ThreadEnvironmentBlock)
                                    + sizeof (ProcessEnvironmentBlock)
                                    + sizeof (ProcessEnvironmentBlockLdrData)
                                    + sizeof (ListEntry) + sizeof (guint32)
                                    + sizeof (ListEntry) + sizeof (guint32)
                                    + sizeof (Ia32CpuDesc) * 6
                                    + i * 2,
                                    IA32_MEMORY_ATTR_SYSTEM);
  teb = ia32_memory_real_address (process, address);
  /* ja:ebxにはPEBのアドレスが入る */
  process->current->reg.ebx = address + sizeof (ThreadEnvironmentBlock);
  /* TEB */
  nttib = teb_get_tib (teb);
  nttib_set_self (nttib, address);
  address += sizeof (ThreadEnvironmentBlock);
  teb_set_peb (teb, address);
  /* PEB (+fb8h) */
  peb = (ProcessEnvironmentBlock *)(teb + 1);
  address += sizeof (ProcessEnvironmentBlock);
  peb_set_ldr_data (peb, address);
  /* PEB_LDR_DATA (+11e4h) */
  pebld = (ProcessEnvironmentBlockLdrData *)(peb + 1);
  lste0 = pebld_get_in_initialization_order_module_list (pebld);
  address += sizeof (ProcessEnvironmentBlockLdrData);
  lste_set_flink (lste0, address);
  /* LIST_ENTRY (+1208h) */
  lste0 = (ListEntry *)(pebld + 1);
  address += sizeof (ListEntry) + sizeof (guint32);
  lste_set_flink (lste0, address);
  /* LIST_ENTRY (+1214h) */
  lste1 = (ListEntry *)((guint8 *)(lste0 + 1) + sizeof (guint32));
  /* ja:ディスクリプタ (+1220h) */
  process->current->reg.gdtr.limit = sizeof (Ia32CpuDesc) * 6;
  address += sizeof (ListEntry) + sizeof (guint32);
  process->current->reg.gdtr.base = address;
  desc = (Ia32CpuDesc *)((guint8 *)(lste1 + 1) + sizeof (guint32));
  /* cs */
  desc++;
  ia32_cpu_desc_set_limit (desc, 0xfff);
  ia32_cpu_desc_set_base (desc, 0);
  ia32_cpu_desc_set_a (desc,1);
  ia32_cpu_desc_set_type (desc, IA32_CPU_DESC_TYPE_EXECREAD);
  ia32_cpu_desc_set_s (desc, 1);
  ia32_cpu_desc_set_dpl (desc, 3);
  ia32_cpu_desc_set_p (desc, 1);
  ia32_cpu_desc_set_avl (desc, 0);
  ia32_cpu_desc_set_db (desc, 1);
  ia32_cpu_desc_set_g (desc, 1);
  /* ds */
  desc++;
  ia32_cpu_desc_set_limit (desc, 0xfff);
  ia32_cpu_desc_set_base (desc, 0);
  ia32_cpu_desc_set_a (desc,1);
  ia32_cpu_desc_set_type (desc, IA32_CPU_DESC_TYPE_READWRITE);
  ia32_cpu_desc_set_s (desc, 1);
  ia32_cpu_desc_set_dpl (desc, 3);
  ia32_cpu_desc_set_p (desc, 1);
  ia32_cpu_desc_set_avl (desc, 0);
  ia32_cpu_desc_set_db (desc, 1);
  ia32_cpu_desc_set_g (desc, 1);
  /* es */
  desc++;
  ia32_cpu_desc_set_limit (desc, 0xfff);
  ia32_cpu_desc_set_base (desc, 0);
  ia32_cpu_desc_set_a (desc,1);
  ia32_cpu_desc_set_type (desc, IA32_CPU_DESC_TYPE_READWRITE);
  ia32_cpu_desc_set_s (desc, 1);
  ia32_cpu_desc_set_dpl (desc, 3);
  ia32_cpu_desc_set_p (desc, 1);
  ia32_cpu_desc_set_avl (desc, 0);
  ia32_cpu_desc_set_db (desc, 1);
  ia32_cpu_desc_set_g (desc, 1);
  /* fs */
  desc++;
  ia32_cpu_desc_set_limit (desc, 0xfff);
  ia32_cpu_desc_set_base (desc, fs);
  ia32_cpu_desc_set_a (desc,1);
  ia32_cpu_desc_set_type (desc, IA32_CPU_DESC_TYPE_READWRITE);
  ia32_cpu_desc_set_s (desc, 1);
  ia32_cpu_desc_set_dpl (desc, 3);
  ia32_cpu_desc_set_p (desc, 1);
  ia32_cpu_desc_set_avl (desc, 0);
  ia32_cpu_desc_set_db (desc, 1);
  ia32_cpu_desc_set_g (desc, 1);
  /* ss */
  desc++;
  ia32_cpu_desc_set_limit (desc, 0xfff);
  ia32_cpu_desc_set_base (desc, 0);
  ia32_cpu_desc_set_a (desc,1);
  ia32_cpu_desc_set_type (desc, IA32_CPU_DESC_TYPE_STACK);
  ia32_cpu_desc_set_s (desc, 1);
  ia32_cpu_desc_set_dpl (desc, 3);
  ia32_cpu_desc_set_p (desc, 1);
  ia32_cpu_desc_set_avl (desc, 0);
  ia32_cpu_desc_set_db (desc, 1);
  ia32_cpu_desc_set_g (desc, 1);
  ia32_cpu_seg_set_rpl (process->current->reg.cs, 3);
  ia32_cpu_seg_set_ti (process->current->reg.cs, 0);
  ia32_cpu_seg_set_index (process->current->reg.cs, 1);
  ia32_cpu_seg_set_rpl (process->current->reg.ds, 3);
  ia32_cpu_seg_set_ti (process->current->reg.ds, 0);
  ia32_cpu_seg_set_index (process->current->reg.ds, 2);
  ia32_cpu_seg_set_rpl (process->current->reg.es, 3);
  ia32_cpu_seg_set_ti (process->current->reg.es, 0);
  ia32_cpu_seg_set_index (process->current->reg.es, 3);
  ia32_cpu_seg_set_rpl (process->current->reg.fs, 3);
  ia32_cpu_seg_set_ti (process->current->reg.fs, 0);
  ia32_cpu_seg_set_index (process->current->reg.fs, 4);
  ia32_cpu_seg_set_rpl (process->current->reg.ss, 3);
  ia32_cpu_seg_set_ti (process->current->reg.ss, 0);
  ia32_cpu_seg_set_index (process->current->reg.ss, 5);
  /* ja:システム関数 (+1250h) */
  p = (guint8 *)(desc + 1);
  address += sizeof (Ia32CpuDesc) * 6;
  for (i = 0; ia32_system_exports[i].func; i++)
    {
      Ia32Api *api;

      api = g_malloc0 (sizeof (Ia32Api));
      api->func = ia32_system_exports[i].func;
      g_hash_table_insert (process->export, GUINT_TO_POINTER (address), api);
      g_hash_table_insert (process->system, ia32_system_exports[i].name,
                                            GUINT_TO_POINTER (address));
      *p++ = 0xeb;
      *p++ = 0xfe;
      address += 2;
    }
  /* ja:例外処理 */
  ia32_stack_push32 (process->current,
        GPOINTER_TO_UINT (g_hash_table_lookup (process->system, "Terminate")));
  ia32_stack_push32 (process->current, (guint32)-1);
  nttib_set_exception_list (nttib, process->current->reg.esp);
  /* ja:モジュール */
  lste_set_extra32 (lste0, ia32_module_load_library (process, "ntdll.dll"));
  lste_set_extra32 (lste1, ia32_module_load_library (process, "kernel32.dll"));

  return process;
}


/*  ja:PEファイルからプロセス作成
    argv,ファイル名
     RET,プロセス                                                           */
Ia32Process *
ia32_create_process_from_argv (gchar *argv[])
{
  gsize len = 0, leng;
  gchar *str;
  gint i;
  guint8 *image;
  guint32 address;
  Ia32Process *process;

  process = ia32_create_process ();
  /* ja:イメージ読み込み */
  process->argv = argv;
  process->address = ia32_module_load_library (process, argv[0]);
  if (process->address == IA32_MEMORY_NULL)
    {
      if (!process->current->err)
        {
          gchar *str;

          str = g_strdup_printf ("\'%s\' load error.", argv[0]);
          ia32_message_record_error (process->current, str);
          g_free (str);
        }
      process->current->terminate = TRUE;
      return process;
    }
  /* ja:ヘッダ */
  image = ia32_memory_real_address (process, process->address);
  leng = ia32_memory_real_size (process, process->address);
  if (image && peimage_file_is_valid (image, leng))
    {
      gsize len;

      len = peimage_file_header_bytes (image);
      if (len <= leng)
        process->image = g_memdup (image, len);
    }
  /* ja:コマンドライン */
  for (i = 1; argv[i]; i++)
    if (g_strchr (argv[i], ' ') || g_strchr (argv[i], '\"')
                                || g_strchr (argv[i], '<')
                                || g_strchr (argv[i], '>')
                                || g_strchr (argv[i], '|'))
      {
        gchar *p;

        for (p = argv[i]; *p != '\0'; p++)
          len += *p == '\"' || *p == '%' || *p == '\\' || *p == '^' ? 2 : 1;
        len += 3;
      }
    else
      {
        gchar *p;

        for (p = argv[i]; *p != '\0'; p++)
          len += *p == '%' || *p == '^' ? 2 : 1;
        len++;
      }
  leng = (len + 3) / 4 * 4;
  str = g_malloc (leng);
  len = 0;
  for (i = 1; argv[i]; i++)
    {
      if (g_strchr (argv[i], ' ') || g_strchr (argv[i], '\"')
                                  || g_strchr (argv[i], '<')
                                  || g_strchr (argv[i], '>')
                                  || g_strchr (argv[i], '|'))
        {
          gchar *p;

          str[len++] = '\"';
          for (p = argv[i]; *p != '\0'; p++)
            {
              if (*p == '\"' || *p == '\\')
                str[len++] = '\\';
              else if (*p == '%' || *p == '^')
                str[len++] = '^';
              str[len++] = *p;
            }
          str[len++] = '\"';
        }
      else
        {
          gchar *p;

          for (p = argv[i]; *p != '\0'; p++)
            {
              if (*p == '%' || *p == '^')
                str[len++] = '^';
              str[len++] = *p;
            }
        }
      str[len++] = argv[i + 1] ? ' ' : '\0';
    }
  process->current->reg.esp -= leng;
  address = process->current->reg.esp;
  ia32_memory_write (process, process->current->reg.esp, str, len, -1);
  g_free (str);
  /* ja:実行 */
  ia32_stack_push32 (process->current, 0);                  /* uExitCode */
  ia32_stack_push32 (process->current, 0);                  /* nCmdShow */
  ia32_stack_push32 (process->current, address);            /* lpCmdLine */
  ia32_stack_push32 (process->current, 0);                  /* hPrevInstance */
  ia32_stack_push32 (process->current, process->address);   /* hInstance */
  ia32_stack_push32 (process->current, ia32_memory_virtual_address (process,
        peimage_file_get_func (ia32_memory_real_address (process,
        ia32_module_get_library (process, "kernel32.dll")), "ExitProcess")));
  process->current->reg.eip = process->address
                    + pe_ioh_get_address_of_entry_point
                        (ia32_memory_real_address (process, process->address));
  return process;
}


/*  ja:メモリイメージからプロセス作成
     image,ファイル名
    length,サイズ
      base,ベースアドレス
     entry,開始アドレス
       RET,プロセス                                                         */
Ia32Process *
ia32_create_process_from_image (const guint8  *image,
                                const gsize    length,
                                const guint32  base,
                                const guint32  entry)
{
  guint32 address;
  Ia32Process *process;

  process = ia32_create_process ();
  address = ia32_memory_alloc (process, base, length, IA32_MEMORY_ATTR_USER);
  if (address == IA32_MEMORY_NULL)
    {
      ia32_message_record_error (process->current, "Memory allocation error.");
      process->current->terminate = TRUE;
      return process;
    }
  if (ia32_memory_write (process, base, image, length, -1) < 0)
    {
      ia32_message_record_error (process->current, "Memory write error.");
      process->current->terminate = TRUE;
      return process;
    }
  /* ja:実行 */
  ia32_stack_push32 (process->current, 0);                  /* uExitCode */
  ia32_stack_push32 (process->current, 0);                  /* nCmdShow */
  ia32_stack_push32 (process->current, 0);                  /* lpCmdLine */
  ia32_stack_push32 (process->current, 0);                  /* hPrevInstance */
  ia32_stack_push32 (process->current, process->address);   /* hInstance */
  ia32_stack_push32 (process->current, ia32_memory_virtual_address (process,
        peimage_file_get_func (ia32_memory_real_address (process,
        ia32_module_get_library (process, "kernel32.dll")), "ExitProcess")));
  process->current->reg.eip = base + entry;
  return process;
}


/*  ja:プロセス破棄
    process,プロセス構造体
        RET,TRUE:正常終了,FALSE:エラー                                      */
gboolean
ia32_destroy_process (Ia32Process *process)
{
  gboolean result = TRUE;

  if (!process)
    return FALSE;
  /* ja:解放 */
  ia32_module_free_library (process, process->address);
  if (!ia32_destroy_thread (process->thread->data))
      result = FALSE;
  while (process->thread)
    if (!ia32_destroy_thread (process->thread->data))
      result = FALSE;
  while (process->memory)
    if (!ia32_memory_free (process,
                            ((Ia32Memory *)process->memory->data)->address))
      result = FALSE;
  if (process->export)
    g_hash_table_destroy (process->export);
  if (process->system)
    g_hash_table_destroy (process->system);
  g_free (process->image);
  g_free (process->table);
  g_free (process);
  return result;
}
