/*
    IA32
    copyright (c) 1998-2011 Kazuki Iwamoto http://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "memory.h"


/******************************************************************************
*                                                                             *
* ja:メモリ関数群                                                             *
*                                                                             *
******************************************************************************/
/*  ja:仮想メモリ比較
    a,仮想メモリ構造体
    b,仮想メモリ構造体
    RET,負:a<b,0:a=b,正:a>b                                                 */
static gint
ia32_memory_compare (gconstpointer a,
                     gconstpointer b)
{
  return ((Ia32Memory *)a)->address - ((Ia32Memory *)b)->address;
}


/*  ja:仮想メモリ登録
      process,プロセス構造体
      address,仮想アドレス
       length,サイズ
    attribute,属性
          RET,メモリ構造体,NULL:エラー                                      */
static Ia32Memory *
ia32_memory_register_internal (Ia32Process   *process,
                               const guint32  address,
                               const gsize    length,
                               const guint    attribute)
{
  Ia32Memory *memory = NULL;

  if (process && length > 0)
    {
      gint block = -1;
      gsize page;

      if (address == IA32_MEMORY_NULL)
        {
          page = (length + IA32_SYSTEM_PAGESIZE - 1) / IA32_SYSTEM_PAGESIZE;
          switch (attribute)
            {
              case IA32_MEMORY_ATTR_USER:
                {
                  gint i;

                  for (i = IA32_MEMORY_BASE_IMAGE / IA32_SYSTEM_PAGESIZE;
                    i < IA32_MEMORY_BASE_SYSTEM / IA32_SYSTEM_PAGESIZE; i++)
                    {
                      gint j;

                      for (j = i; j < i + page; j++)
                        if (process->map[j] != 0)
                          break;
                      if (j >= i + page)
                        {
                          block = i;
                          break;
                        }
                    }
                }
                break;
              case IA32_MEMORY_ATTR_SYSTEM:
                {
                  gint i;

                  for (i = IA32_MEMORY_BASE_SYSTEM / IA32_SYSTEM_PAGESIZE;
                                        i < G_N_ELEMENTS (process->map); i++)
                    {
                      gint j;

                      for (j = i; j < i + page; j++)
                        if (process->map[j] != 0)
                          break;
                      if (j >= i + page)
                        {
                          block = i;
                          break;
                        }
                    }
                }
                break;
              case IA32_MEMORY_ATTR_STACK:
                {
                  gint i;

                  for (i = IA32_MEMORY_BASE_SYSTEM / IA32_SYSTEM_PAGESIZE
                                                        - page; i > 0 ; i--)
                    {
                      gint j;

                      for (j = i; j < i + page; j++)
                        if (process->map[j] != 0)
                          break;
                      if (j >= i + page)
                        {
                          block = i;
                          break;
                        }
                    }
                }
            }
        }
      else
        {
          gint i;

          block = address / IA32_SYSTEM_PAGESIZE;
          page = (address + length + IA32_SYSTEM_PAGESIZE - 1)
                                                / IA32_SYSTEM_PAGESIZE - block;
          for (i = block; i < page; i++)
            if (process->map[i] != 0)
              {
                block = -1;
                break;
              }
        }
      if (block >= 0)
        {
          g_memset (process->map + block, attribute, page);
          memory = g_malloc (sizeof (Ia32Memory));
          memory->data = NULL;
          memory->level = g_malloc0 (length);
          memory->address = address == IA32_MEMORY_NULL
                                    ? block * IA32_SYSTEM_PAGESIZE : address;
          memory->length = length;
          memory->attribute = attribute;
          process->memory = g_list_insert_sorted (process->memory, memory,
                                                        ia32_memory_compare);
        }
    }
  return memory;
}


/*  ja:仮想メモリ登録
      process,プロセス構造体
      address,仮想アドレス
       length,サイズ
    attribute,属性
         data,実アドレス
          RET,TRUE:正常終了,FALSE:エラー                                    */
gboolean
ia32_memory_register (Ia32Process   *process,
                      const guint32  address,
                      const gsize    length,
                      const guint    attribute,
                      gpointer       data)
{
  Ia32Memory *memory;

  memory = ia32_memory_register_internal (process, address, length, attribute);
  if (!memory)
    return FALSE;
  memory->data = data;
  return TRUE;
}


/*  ja:仮想メモリ抹消
    process,プロセス構造体
    address,仮想アドレス
        RET,実アドレス,NULL:エラー                                          */
static gpointer
ia32_memory_unregister (Ia32Process   *process,
                        const guint32  address)
{
  if (process)
    {
      GList *glist;

      for (glist = g_list_first (process->memory);
                                            glist; glist = g_list_next (glist))
        {
          Ia32Memory *memory;

          memory = glist->data;
          if (memory->address == address)
            {
              gpointer data;
              gint block;
              gsize page;

              data = memory->data;
              block = memory->address / IA32_SYSTEM_PAGESIZE;
              page = (memory->address + memory->length
                    + IA32_SYSTEM_PAGESIZE - 1) / IA32_SYSTEM_PAGESIZE - block;
              g_memset (process->map + block, 0, page);
              process->memory = g_list_remove (process->memory, memory);
              g_free (memory->level);
              g_free (memory);
              return data;
            }
        }
    }
  return NULL;
}


/*  ja:仮想メモリ確保
      process,プロセス構造体
      address,仮想アドレス
       length,サイズ
    attribute,属性
          RET,仮想アドレス,IA32_MEMORY_NULL:エラー                          */
guint32
ia32_memory_alloc (Ia32Process   *process,
                   const guint32  address,
                   const gsize    length,
                   const guint    attribute)
{
  Ia32Memory *memory;

  memory = ia32_memory_register_internal (process, address, length, attribute);
  if (!memory)
    return IA32_MEMORY_NULL;
  memory->data = g_malloc0 (length);
  return memory->address;
}


/*  ja:仮想メモリ解放
    process,プロセス構造体
    address,仮想アドレス
        RET,TRUE:正常終了,FALSE:エラー                                      */
gboolean
ia32_memory_free (Ia32Process   *process,
                  const guint32  address)
{
  gpointer data;

  data = ia32_memory_unregister (process, address);
  if (!data)
    return FALSE;
  g_free (data);
  return TRUE;
}


/*  ja:実アドレスから仮想アドレスを求める
    process,プロセス構造体
       data,実アドレス
        RET,仮想アドレス,IA32_MEMORY_NULL:エラー                            */
guint32
ia32_memory_virtual_address (Ia32Process   *process,
                             gconstpointer  data)
{
  if (process && data)
    {
      GList *glist;

      for (glist = g_list_first (process->memory);
                                            glist; glist = g_list_next (glist))
        {
          Ia32Memory *memory;

          memory = glist->data;
          if (memory->data <= data
            && data < (gconstpointer)((guint8 *)memory->data + memory->length))
            return memory->address + GPOINTER_TO_UINT (data)
                                            - GPOINTER_TO_UINT (memory->data);
        }
    }
  return IA32_MEMORY_NULL;
}


/*  ja:仮想アドレスから実アドレスを求める
    process,プロセス構造体
    address,仮想アドレス
        RET,実アドレス,NULL:エラー                                          */
gpointer
ia32_memory_real_address (Ia32Process   *process,
                          const guint32  address)
{
  if (process && address != IA32_MEMORY_NULL)
    {
      GList *glist;

      for (glist = g_list_first (process->memory);
                                            glist; glist = g_list_next (glist))
        {
          Ia32Memory *memory;

          memory = glist->data;
          if (memory->address <= address
                                && address < memory->address + memory->length)
            return (guint8 *)memory->data + address - memory->address;
        }
    }
  return NULL;
}


/*  ja:仮想アドレスから実サイズを求める
    process,プロセス構造体
    address,仮想アドレス
        RET,実サイズ,0:エラー                                               */
gsize
ia32_memory_real_size (Ia32Process   *process,
                       const guint32  address)
{
  if (process && address != IA32_MEMORY_NULL)
    {
      GList *glist;

      for (glist = g_list_first (process->memory);
                                            glist; glist = g_list_next (glist))
        {
          Ia32Memory *memory;

          memory = glist->data;
          if (memory->address <= address
                                && address < memory->address + memory->length)
            return memory->length - (address - memory->address);
        }
    }
  return 0;
}


/******************************************************************************
*                                                                             *
* ja:メモリ文字列関数群                                                       *
*                                                                             *
******************************************************************************/
/*  ja:仮想アドレスが文字列か判定する
    process,プロセス構造体
    address,仮想アドレス
        RET,TRUE:文字列,FALSE:エラー                                        */
gboolean
ia32_memory_is_string (Ia32Process   *process,
                       const guint32  address)
{
  gint i = 0;
  guint8 c;

  do
    {
      if (ia32_memory_read_byte (process, address + i, &c) < 0)
        return FALSE;
      i++;
    }
  while (c != '\0');
  return TRUE;
}


/*  ja:仮想アドレスがワイド文字列か判定する
    process,プロセス構造体
    address,仮想アドレス
        RET,TRUE:ワイド文字列,FALSE:エラー                                  */
gboolean
ia32_memory_is_stringw (Ia32Process   *process,
                        const guint32  address)
{
  gint i = 0;
  guint16 c;

  do
    {
      if (ia32_memory_read_word (process, address + i, &c) < 0)
        return FALSE;
      i += 2;
    }
  while (c != '\0');
  return TRUE;
}


/******************************************************************************
*                                                                             *
* ja:メモリI/O関数群                                                          *
*                                                                             *
******************************************************************************/
/*  ja:仮想アドレスからデータを読み込む
    process,プロセス構造体
    address,仮想アドレス
     buffer,データ
     length,サイズ
        RET,レベル,-1:エラー                                                */
gint8
ia32_memory_read (Ia32Process   *process,
                  const guint32  address,
                  gpointer       buffer,
                  const gsize    length)
{
  gint8 level = -1;
  guint32 addr;
  guint8 *buf;
  gssize leng;
  GList *glist;

  if (!process)
    return -1;
  addr = address;
  buf = buffer;
  leng = length;
  for (glist = g_list_first (process->memory);
                                            glist; glist = g_list_next (glist))
    {
      Ia32Memory *memory;

      memory = glist->data;
      if (memory->address <= addr && addr < memory->address + memory->length)
        {
          gsize d, l;

          d = addr - memory->address;
          l = MIN (memory->length - d, leng);
          if (buffer)
            g_memmove (buf, (guint8 *)memory->data + d, l);
          if (level < 0)
            level = *((gint8 *)memory->level + d);
          addr += l;
          buf += l;
          leng -= l;
          if (leng <= 0)
            break;
        }
    }
  return leng <=0 ? level : -1;
}


/*  ja:仮想アドレスから1バイトデータを読み込む
    process,プロセス構造体
    address,仮想アドレス
      value,データ
        RET,レベル,-1:エラー                                                */
gint8
ia32_memory_read_byte (Ia32Process   *process,
                       const guint32  address,
                       guint8        *value)
{
  return ia32_memory_read (process, address, value, 1);
}


/*  ja:仮想アドレスから2バイトデータを読み込む
    process,プロセス構造体
    address,仮想アドレス
      value,データ
        RET,レベル,-1:エラー                                                */
gint8
ia32_memory_read_word (Ia32Process   *process,
                       const guint32  address,
                       guint16       *value)
{
  gint8 level;
  guint16 buf;

  level = ia32_memory_read (process, address, &buf, 2);
  if (level >= 0 && value)
    *value = GUINT16_FROM_LE (buf);
  return level;
}


/*  ja:仮想アドレスから2バイトデータを読み込む
    process,プロセス構造体
    address,仮想アドレス
      value,データ
        RET,レベル,-1:エラー                                                */
gint8
ia32_memory_read_dword (Ia32Process   *process,
                        const guint32  address,
                        guint32       *value)
{
  gint8 level;
  guint32 buf;

  level = ia32_memory_read (process, address, &buf, 4);
  if (level >= 0 && value)
    *value = GUINT32_FROM_LE (buf);
  return level;
}


/*  ja:仮想アドレスにデータを書き込む
    process,プロセス構造体
    address,仮想アドレス
     buffer,データ
     length,サイズ
      level,レベル(-1:変更なし)
        RET,レベル,-1:エラー                                                */
gint8
ia32_memory_write (Ia32Process   *process,
                   const guint32  address,
                   gconstpointer  buffer,
                   const gsize    length,
                   const gint8    level)
{
  guint32 addr;
  const guint8 *buf;
  gssize leng;
  GList *glist;

  if (!process)
    return -1;
  addr = address;
  buf = buffer;
  leng = length;
  for (glist = g_list_first (process->memory);
                                            glist; glist = g_list_next (glist))
    {
      Ia32Memory *memory;

      memory = glist->data;
      if (memory->address <= addr && addr < memory->address + memory->length)
        {
          gsize d, l;

          d = addr - memory->address;
          l = MIN (memory->length - d, leng);
          g_memmove ((guint8 *)memory->data + d, buf, l);
          if (level >= 0)
            {
              gint i;

              for (i = 0; i < l; i++)
                if (*((gint8 *)memory->level + d + i) < level + 1)
                  *((gint8 *)memory->level + d + i) = level + 1;
            }
          addr += l;
          buf += l;
          leng -= l;
          if (leng <= 0)
            break;
        }
    }
  return leng <= 0 ? MAX (level, 0) : -1;
}


/*  ja:仮想アドレスに1バイトデータを書き込む
    process,プロセス構造体
    address,仮想アドレス
      value,データ
      level,レベル(-1:変更なし)
        RET,レベル,-1:エラー                                                */
gint8
ia32_memory_write_byte (Ia32Process   *process,
                        const guint32  address,
                        const guint8   value,
                        const gint8    level)
{
  return ia32_memory_write (process, address, &value, 1, level);
}


/*  ja:仮想アドレスに2バイトデータを書き込む
    process,プロセス構造体
    address,仮想アドレス
      value,データ
      level,レベル(-1:変更なし)
        RET,レベル,-1:エラー                                                */
gint8
ia32_memory_write_word (Ia32Process   *process,
                        const guint32  address,
                        const guint16  value,
                        const gint8    level)
{
  guint16 buf;

  buf = GUINT16_TO_LE (value);
  return ia32_memory_write (process, address, &buf, 2, level);
}


/*  ja:仮想アドレスに4バイトデータを書き込む
    process,プロセス構造体
    address,仮想アドレス
      value,データ
      level,レベル(-1:変更なし)
        RET,レベル,-1:エラー                                                */
gint8
ia32_memory_write_dword (Ia32Process   *process,
                         const guint32  address,
                         const guint32  value,
                         const gint8    level)
{
  guint32 buf;

  buf = GUINT32_TO_LE (value);
  return ia32_memory_write (process, address, &buf, 4, level);
}


/*  ja:仮想アドレスのレベルを求める
    process,プロセス構造体
    address,仮想アドレス
        RET,レベル,-1:エラー                                                */
gint8
ia32_memory_get_level (Ia32Process   *process,
                       const guint32  address)
{
  gint8 level = -1;
  GList *glist;

  if (!process)
    return -1;
  for (glist = g_list_first (process->memory);
                                            glist; glist = g_list_next (glist))
    {
      Ia32Memory *memory;

      memory = glist->data;
      if (memory->address <= address
                                && address < memory->address + memory->length)
        {
          level = *((gint8 *)memory->level + address - memory->address);
          break;
        }
    }
  return level;
}
