/*
    gcommon
    copyright (c) 1998-2013 Kazuki Iwamoto http://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "gcommon.h"


/******************************************************************************
* URI Utilities                                                               *
******************************************************************************/
#if ! GLIB_CHECK_VERSION(2,16,0)
gchar *
g_uri_parse_scheme (const gchar *uri)
{
  gchar *scheme = NULL;

  if (uri && g_ascii_isalpha (*uri))
    {
      const gchar *p;

      for (p = uri + 1; g_ascii_isalnum (*p)
                                || *p == '+' || *p == '-' || *p == '.'; p++);
      if (*p == ':')
        scheme = g_strndup (uri, (p - uri) * sizeof (gchar));
    }
  return scheme;
}


gchar *
g_uri_escape_string (const gchar *unescaped,
                     const gchar *reserved_chars_allowed,
                     gboolean     allow_utf8)
{
  gchar *ret = NULL;

  if (unescaped)
    {
      const gchar *p;
      gchar *q, *allowed;

      allowed = g_strconcat (
        "-.0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ_abcdefghijklmnopqrstuvwxyz~",
                                                reserved_chars_allowed, NULL);
      p = unescaped;
      q = ret = g_malloc ((g_strlen (unescaped) * 3 + 1) * sizeof (gchar));
      while (*p != '\0')
        {
          gunichar c;
          gchar *r;

          if (allow_utf8
                && (c = g_utf8_get_char_validated (p, -1)) != (gunichar)-1
                && c != (gunichar)-2
                && (r = g_utf8_next_char (p)) - p > 1)
            {
              while (p < r)
                *q++ = *p++;
            }
          else if (g_strchr (allowed, *p))
            {
              *q++ = *p++;
            }
          else
            {
              const static gchar hex[16] = {'0', '1', '2', '3',
                                            '4', '5', '6', '7',
                                            '8', '9', 'A', 'B',
                                            'C', 'D', 'E', 'F'};

              *q++ = '%';
              *q++ = hex[((guchar)*p) >> 4];
              *q++ = hex[((guchar)*p++) & 15];
            }
        }
      g_free (allowed);
    }
  return ret;
}


gchar *
g_uri_unescape_string (const gchar *escaped_string,
                       const gchar *illegal_characters)
{
  return g_uri_unescape_segment (escaped_string, NULL, illegal_characters);
}


gchar *
g_uri_unescape_segment (const gchar *escaped_string,
                        const gchar *escaped_string_end,
                        const gchar *illegal_characters)
{
  gchar *ret = NULL;

  if (escaped_string)
    {
      const gchar *end, *p;
      gchar *q;

      p = escaped_string;
      end = escaped_string_end ? escaped_string_end : p + g_strlen (p);
      q = ret = g_malloc ((end - escaped_string + 1) * sizeof (gchar));
      while (p < end)
        {
          gchar c;

          c = *p++;
          if (c == '%')
            {
              gint c1, c2;

              c = (c1 = g_ascii_xdigit_value (*p++)) >= 0
               && (c2 = g_ascii_xdigit_value (*p++)) >= 0
                                                    ? (c1 << 4) | c2 : '\0';
              if (c == '\0' || g_strchr (illegal_characters, c))
                {
                  g_free (ret);
                  ret = NULL;
                  break;
                }
            }
          *q++ = c;
        }
    }
  return ret;
}
#endif /* not GLIB_CHECK_VERSION(2,16,0) */
