/*
    gcommon
    copyright (c) 1998-2013 Kazuki Iwamoto http://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "gcommon.h"


/******************************************************************************
* String Utility Functions                                                    *
******************************************************************************/
#ifdef USE_GTK_EMULATE
gchar*
g_strdup (const gchar *str)
{
  gchar *s = NULL;

  if (str)
    {
      s = g_malloc ((g_strlen (str) + 1) * sizeof (gchar));
      g_strcpy (s, str);
    }
  return s;
}


gchar *
g_strndup (const gchar *str,
           gsize        n)
{
  gchar *s = NULL;

  if (str)
    {
      s = g_malloc ((n + 1) * sizeof (gchar));
      g_strncpy (s, str, n);
      s[n] = '\0';
    }
  return s;
}


gchar **
g_strdupv (gchar **str_array)
{
  gchar **str = NULL;

  if (str_array)
    {
      gint i;

      for (i = 0; str_array[i]; i++);
      str = g_malloc ((i + 1) * sizeof (gchar *));
      for (i = 0; str_array[i]; i++)
        str[i] = g_strdup (str_array[i]);
      str[i] = NULL;
    }
  return str;
}


gchar *
g_strnfill (gsize length,
            gchar fill_char)
{
  gchar *str;

  str = g_malloc ((length + 1) * sizeof (gchar));
  g_memset (str, fill_char, length * sizeof (gchar));
  str[length] = '\0';
  return str;
}


gchar *
g_stpcpy (gchar       *dest,
          const gchar *src)
{
  gchar *d;
  const gchar *s;

  if (!dest || !src)
    return NULL;
  d = dest;
  s = src;
  do
    *d++ = *s;
  while (*s++ != '\0');
  return d - 1;
}


gchar *
g_strstr_len (const gchar *haystack,
              gssize       haystack_len,
              const gchar *needle)
{
  gsize needle_len;
  const gchar *p;

  if (!haystack || !needle)
    return NULL;
  if (haystack_len < 0)
    haystack_len = g_strlen (haystack);
  needle_len = g_strlen (needle);
  if (needle_len < 1)
    return (gchar *)haystack;
  if (haystack_len < needle_len)
    return NULL;
  for (p = haystack; *p != '\0'; p++)
    if (g_strncmp (p, needle, needle_len) == 0)
      return (gchar *)p;
  return NULL;
}


gchar *
g_strrstr (const gchar *haystack,
           const gchar *needle)
{
  return g_strrstr_len (haystack, -1, needle);
}


gchar *
g_strrstr_len (const gchar *haystack,
               gssize       haystack_len,
               const gchar *needle)
{
  gsize needle_len;
  const gchar *p;

  if (!haystack || !needle)
    return NULL;
  needle_len = g_strlen (needle);
  if (needle_len < 1)
    return (gchar *)haystack;
  if (haystack_len < 0)
    haystack_len = g_strlen (haystack);
  if (haystack_len < needle_len)
    return NULL;
  for (p = haystack + haystack_len; p >= haystack; p--)
    if (g_strncmp (p, needle, needle_len) == 0)
      return (gchar *)p;
  return NULL;
}
#endif /* USE_GTK_EMULATE */


#if ! GLIB_CHECK_VERSION(2,2,0)
gboolean
g_str_has_prefix (const gchar *str,
                  const gchar *prefix)
{
  return str && prefix && g_strncmp (str, prefix, g_strlen (prefix)) == 0;
}


gboolean
g_str_has_suffix (const gchar *str,
                  const gchar *suffix)
{
  if (str && suffix)
    {
      gsize len_str, len_suffix;

      len_str = g_strlen (str);
      len_suffix = g_strlen (suffix);
      if (len_str > len_suffix)
        return g_strcmp (str + len_str - len_suffix, suffix) == 0;
    }
  return FALSE;
}
#endif /* GLIB_CHECK_VERSION(2,2,0) */


#if ! GLIB_CHECK_VERSION(2,16,0)
int
g_strcmp0 (const char *str1,
           const char *str2)
{
  return g_strcmp (str1, str2);
}
#endif /* GLIB_CHECK_VERSION(2,16,0) */


#ifdef USE_GTK_EMULATE
gsize
g_strlcpy (gchar       *dest,
           const gchar *src,
           gsize        dest_size)
{
  gint i;

  if (!dest || !src)
    return 0;
  for (i = 0; i < dest_size - 1 && src[i] != '\0'; i++)
    dest[i] = src[i];
  while (i < dest_size)
    dest[i++] = '\0';
  return g_strlen (src);
}


gsize
g_strlcat (gchar       *dest,
           const gchar *src,
           gsize        dest_size)
{
  gsize len;
  gint i, j;

  if (!dest || !src)
    return 0;
  len = g_strlen (dest);
  for (i = len, j = 0; i < dest_size - 1 && src[j] != '\0'; i++, j++)
    dest[i] = src[j];
  while (i < dest_size)
    dest[i++] = '\0';
  return len + g_strlen (src);
}
#endif /* USE_GTK_EMULATE */


#ifdef USE_GTK_EMULATE
gchar *
g_strdup_printf (const gchar *format,
                 ...)
{
  gchar *str;
  va_list args;

  va_start (args, format);
  str = g_strdup_vprintf (format, args);
  va_end (args);
  return str;
}


gchar *
g_strdup_vprintf (const char *format,
                  va_list     args)
{
  gchar *str;

  return g_vasprintf (&str, format, args) >= 0 ? str : NULL;
}
#endif /* USE_GTK_EMULATE */


#if ! GLIB_CHECK_VERSION(2,4,0)
gint
g_vasprintf (gchar       **string,
             gchar const  *format,
             va_list       args)
{
  if (string)
    {
# ifdef HAVE_VASPRINTF
      gchar *str;

      if (vasprintf (&str, format, args) >= 0)
        {
          *string = g_strdup (str);
          free (str);
        }
      else
        {
          *string = NULL;
        }
# else /* not HAVE_VASPRINTF */
      gchar *str = NULL;
      gsize len, length = 0;

      do
        {
          length += 1024;
          if (length > 16416)
            {
              free (str);
              str = NULL;
              break;
            }
          str = g_realloc (str, length * sizeof (gchar));
#  ifdef _MSC_VER
          len = _vsnprintf (str, length, format, args);
#  else /* not _MSC_VER */
          len = vsnprintf (str, length, format, args);
#  endif /* not _MSC_VER */
        }
      while (len < 0 || length <= len);
      *string = str;
# endif /* not HAVE_VASPRINTF */
    }
  return string && *string ? g_strlen (*string) : -1;
}
#endif /* ! GLIB_CHECK_VERSION(2,4,0) */


#ifdef USE_GTK_EMULATE
gsize
g_printf_string_upper_bound (const gchar *format,
                             va_list      args)
{
  gsize len;
  gchar *str;

  str = g_strdup_vprintf (format, args);
  if (!str)
    return 0;
  len = g_strlen (str) + 1;
  g_free (str);
  return len;
}


gint
g_ascii_digit_value (gchar c)
{
  return g_ascii_isdigit (c) ? c - '0' : -1;
}


gint
g_ascii_xdigit_value (gchar c)
{
  if ('A' <= c && c <= 'F')
    return c - 'A' + 10;
  if ('a' <= c && c <= 'f')
    return c - 'a' + 10;
  return g_ascii_digit_value (c);
}


gint
g_ascii_strcasecmp (const gchar *s1,
                    const gchar *s2)
{
  if (!s1 || !s2)
    return 0;
  while (*s1 && *s2)
    {
      gint c1, c2;

      c1 = (gint)(guchar)g_ascii_tolower (*s1);
      c2 = (gint)(guchar)g_ascii_tolower (*s2);
      if (c1 != c2)
        return c1 - c2;
      s1++;
      s2++;
    }
  return (((gint)(guchar)*s1) - ((gint)(guchar)*s2));
}


gint
g_ascii_strncasecmp (const gchar *s1,
                     const gchar *s2,
                     gsize        n)
{
  if (!s1 || !s2)
    return 0;
  while (n > 0 && *s1 && *s2)
    {
      gint c1, c2;

      n--;
      c1 = (gint)(guchar)g_ascii_tolower (*s1);
      c2 = (gint)(guchar)g_ascii_tolower (*s2);
      if (c1 != c2)
        return c1 - c2;
      s1++;
      s2++;
    }
  return n > 0 ? (((gint)(guchar)*s1) - ((gint)(guchar)*s2)) : 0;
}


gchar *
g_ascii_strup (const gchar *str,
               gssize       len)
{
  gchar *ret = NULL;

  if (str)
    {
      gint i;

      if (len < 0)
        len = g_strlen (str);
      ret = g_malloc ((len + 1) * sizeof (gchar));
      g_memmove (ret, str, len);
      ret[len] = '\0';
      for (i = 0; i < len; i++)
        ret[i] = g_ascii_toupper (ret[i]);
    }
  return ret;
}


gchar *
g_ascii_strdown (const gchar *str,
                 gssize       len)
{
  gchar *ret = NULL;

  if (str)
    {
      gint i;

      if (len < 0)
        len = g_strlen (str);
      ret = g_malloc ((len + 1) * sizeof (gchar));
      g_memmove (ret, str, len);
      ret[len] = '\0';
      for (i = 0; i < len; i++)
        ret[i] = g_ascii_tolower (ret[i]);
    }
  return ret;
}


gchar *
g_strreverse (gchar *string)
{
  if (string)
    {
      gchar *p, *q;

      p = string;
      q = string + g_strlen (string) - 1;
      while (p < q)
        {
          gchar c;

          c = *p;
          *p = *q;
          *q = c;
          p++;
          q--;
        }
    }
  return string;
}
#endif /* USE_GTK_EMULATE */


#if ! GLIB_CHECK_VERSION(2,12,0)
gint64
g_ascii_strtoll (const gchar  *nptr,
                 gchar       **endptr,
                 guint         base)
{
  gint64 ret = 0;

  if (nptr)
    {
      gchar *p;

      ret = g_strtol (nptr, &p, base);
      if (endptr)
        *endptr = p;
    }
  return ret;
}
#endif /* not GLIB_CHECK_VERSION(2,12,0) */


#if ! GLIB_CHECK_VERSION(2,2,0)
guint64
g_ascii_strtoull (const gchar  *nptr,
                  gchar       **endptr,
                  guint         base)
{
  guint64 ret = 0;

  if (nptr)
    {
      gchar *p;

      ret = g_strtoul (nptr, &p, base);
      if (endptr)
        *endptr = p;
    }
  return ret;
}
#endif /* not GLIB_CHECK_VERSION(2,2,0) */


#ifdef USE_GTK_EMULATE
gchar *
g_strchug (gchar *string)
{
  gchar *p;

  if (string)
    {
      for (p = string; *p != '\0' && g_ascii_isspace (*p); p++);
      g_memmove (string, p, (g_strlen (p) + 1) * sizeof (gchar));
    }
  return string;
}


gchar *
g_strchomp (gchar *string)
{
  if (string)
    {
      gsize len;

      len = g_strlen (string);
      while (len-- > 0)
        if (g_ascii_isspace (string[len]))
          string[len] = '\0';
        else
          break;
    }
  return string;
}


gchar *
g_strescape (const gchar *source,
             const gchar *exceptions)
{
  const guchar *p;
  gchar *escape;
  gchar *q;
  guchar excmap[256];
  gint i;

  if (!source)
    return NULL;
  for (i = 0; i < 256; i++)
    excmap[i] = g_ascii_isprint (i) ? 1 : 0;
  if (exceptions)
    {
      guchar *e;

      e = (guchar *)exceptions;
      while (*e != '\0')
        {
          excmap[*e] = 1;
          e++;
        }
    }
  q = escape = g_malloc ((g_strlen (source) * 4 + 1) * sizeof (gchar));
  for (p = (guchar *)source; *p != '\0'; p++)
    if (excmap[*p] == 1)
      *q++ = *p;
    else
      switch (*p)
        {
          case '\b': *q++ = '\\'; *q++ = 'b';  break;
          case '\f': *q++ = '\\'; *q++ = 'f';  break;
          case '\n': *q++ = '\\'; *q++ = 'n';  break;
          case '\r': *q++ = '\\'; *q++ = 'r';  break;
          case '\t': *q++ = '\\'; *q++ = 't';  break;
          case '\\': *q++ = '\\'; *q++ = '\\'; break;
          case '"':  *q++ = '\\'; *q++ = '"';  break;
          default:
            *q++ = '\\';
            *q++ = '0' + (((*p) >> 6) & 7);
            *q++ = '0' + (((*p) >> 3) & 7);
            *q++ = '0' + ((*p) & 7);
        }
  *q = '\0';
  return escape;
}


gchar *
g_strcompress (const gchar *source)
{
  gchar *p, *q, *str;

  if (!source)
    return NULL;
  p = (gchar *)source;
  q = str = g_malloc ((g_strlen (source) + 1) * sizeof (gchar));
  while (*p != '\0')
    if (*p == '\\')
      switch (*(++p))
        {
          case '\0': break;
          case 'b' : p++; *q++ = '\b'; break;
          case 'f' : p++; *q++ = '\f'; break;
          case 'n' : p++; *q++ = '\n'; break;
          case 'r' : p++; *q++ = '\r'; break;
          case 't' : p++; *q++ = '\t'; break;
          case '0': case '1': case '2': case '3':
          case '4': case '5': case '6': case '7':
            {
              gint i;

              *q = 0;
              for (i = 0; i < 3 && '0' <= p[i] && p[i] <= '7'; i++)
                *q = *q * 8 + p[i] - '0';
              p += i;
              q++;
            }
            break;
          default:
            *q++ = *p++;
        }
    else
      *q++ = *p++;
  *q = '\0';
  return str;
}


gchar **
g_strsplit (const gchar *string,
            const gchar *delimiter,
            gint         max_tokens)
{
  GList *glist = NULL;
  gchar **ret, *s;
  guint n = 0;
  const gchar *remainder;

  if (!string || !delimiter || delimiter[0] == '\0')
    return NULL;
  if (max_tokens < 1)
    max_tokens = G_MAXINT;
  remainder = string;
  s = g_strstr (remainder, delimiter);
  if (s)
    {
      gsize delimiter_len;

      delimiter_len = g_strlen (delimiter);
      while (--max_tokens && s)
        {
          gsize len;

          len = s - remainder;
          glist = g_list_append (glist, g_strndup (remainder, len));
          n++;
          remainder = s + delimiter_len;
          s = g_strstr (remainder, delimiter);
        }
    }
  if (*string)
    {
      n++;
      glist = g_list_append (glist, g_strdup (remainder));
    }

  ret = g_malloc ((n + 1) * sizeof (gchar *));
  ret[n] = NULL;
  n = 0;
  while (glist)
    {
      ret[n++] = glist->data;
      glist = g_list_delete_link (glist, glist);
    }

  return ret;
}


void
g_strfreev (gchar **str_array)
{
  if (str_array)
    {
      gint i;

      for (i = 0; str_array[i]; i++)
        g_free (str_array[i]);
      g_free (str_array);
    }
}


gchar *
g_strconcat (const gchar *str,
             ...)
{
  gchar *concat = NULL;

  if (str)
    {
      gsize len;
      gchar *s;
      va_list arg;

      len = strlen (str) + 1;
      va_start (arg, str);
      s = va_arg (arg, gchar *);
      while (s)
        {
          len += g_strlen (s);
          s = va_arg (arg, gchar *);
        }
      va_end (arg);
      concat = g_malloc (len * sizeof (gchar));
      g_strcpy (concat, str);
      va_start (arg, str);
      s = va_arg (arg, gchar *);
      while (s)
        {
          g_strcat (concat, s);
          s = va_arg (arg, gchar*);
        }
      va_end (arg);
    }
  return concat;
}
#endif /* USE_GTK_EMULATE */


#if ! GLIB_CHECK_VERSION(2,6,0)
guint
g_strv_length (gchar **str_array)
{
  guint i = 0;

  if (str_array)
    while (str_array[i])
      i++;
  return i;
}
#endif /* not GLIB_CHECK_VERSION(2,6,0) */
