/*
    gcommon
    copyright (c) 1998-2017 Kazuki Iwamoto https://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "gcommon.h"
#ifdef G_OS_WIN32
# include <windows.h>
# include <shlwapi.h>
# include <shlobj.h>
#endif /* G_OS_WIN32 */


/******************************************************************************
* Miscellaneous Utility Functions                                             *
******************************************************************************/
#ifdef USE_GTK_EMULATE
static gchar *g_prgname = NULL;


gchar *
g_get_prgname (void)
{
  return g_prgname;
}


void g_set_prgname (const gchar *prgname)
{
  g_free (g_prgname);
  g_prgname = g_strdup (prgname);
}


# ifdef G_OS_WIN32
static GHashTable *ghash_env = NULL;


gboolean
g_strcase_equal (gconstpointer v1,
                 gconstpointer v2)
{
  return g_ascii_strcasecmp ((const gchar *)v1, (const gchar *)v2) == 0;
}


guint
g_strcase_hash (gconstpointer v)
{
  const gchar *p;
  guint32 h = 5381;

  if (v)
    for (p = v; *p != '\0'; p++)
      h = (h << 5) + h + g_ascii_tolower (*p);
  return h;
}
# endif /* G_OS_WIN32 */


const gchar *
g_getenv (const gchar *variable)
{
# ifdef G_OS_WIN32
  gchar *env = NULL;

  if (!ghash_env)
    ghash_env = g_hash_table_new_full (g_strcase_hash, g_strcase_equal,
                                                            g_free, g_free);
  if (variable && !(env = g_hash_table_lookup (ghash_env, variable)))
    {
      DWORD dwSize, dwResult;
#  ifdef UNICODE
      LPWSTR lpszName, lpszBuffer;
      gchar *utf8str;

      utf8str = g_locale_to_utf8 (variable, -1, NULL, NULL, NULL);
      lpszName = g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL);
      g_free (utf8str);
      dwSize = GetEnvironmentVariableW (lpszName, NULL, 0);
      lpszBuffer = g_malloc (dwSize * sizeof (WCHAR));
      dwResult = GetEnvironmentVariableW (lpszName, lpszBuffer, dwSize);
      g_free (lpszName);
      if (dwResult == dwSize - 1)
        {
          utf8str = g_utf16_to_utf8 (lpszBuffer, -1, NULL, NULL, NULL);
          env = g_locale_from_utf8 (utf8str, -1, NULL, NULL, NULL);
          g_free (utf8str);
        }
      g_free (lpszBuffer);
#  else /* not UNICODE */
      dwSize = GetEnvironmentVariableA (variable, NULL, 0);
      env = g_malloc (dwSize * sizeof (CHAR));
      dwResult = GetEnvironmentVariableA (variable, env, dwSize);
      if (dwResult != dwSize - 1)
        {
          g_free (env);
          env = NULL;
        }
#  endif /* not UNICODE */
      if (env)
        g_hash_table_insert (ghash_env, g_strdup (variable), env);
    }
  return env;
# else /* not G_OS_WIN32 */
  return variable ? getenv (variable) : NULL;
# endif /* not G_OS_WIN32 */
}
#endif /* USE_GTK_EMULATE */


#if ! GLIB_CHECK_VERSION(2,4,0)
gboolean
g_setenv (const gchar *variable,
          const gchar *value,
          gboolean     overwrite)
{
# ifdef G_OS_WIN32
  gboolean ret = FALSE;

  if (variable)
    {
      if (overwrite || !g_getenv (variable))
        {
#  ifdef UNICODE
          LPWSTR lpszName, lpszValue;
          gchar *utf8str;

          utf8str = g_locale_to_utf8 (variable, -1, NULL, NULL, NULL);
          lpszName = g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL);
          g_free (utf8str);
          utf8str = g_locale_to_utf8 (value, -1, NULL, NULL, NULL);
          lpszValue = g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL);
          g_free (utf8str);
          ret = SetEnvironmentVariableW (lpszName, lpszValue);
          g_free (lpszName);
          g_free (lpszValue);
#  else /* not UNICODE */
          ret = SetEnvironmentVariableA (variable, value);
#  endif /* not UNICODE */
          if (ghash_env)
            g_hash_table_remove (ghash_env, variable);
        }
      else
        {
          ret = TRUE;
        }
    }
  return ret;
# else /* not G_OS_WIN32 */
#  ifdef HAVE_SETENV
  return variable ? setenv (variable, value, overwrite) : FALSE;
#  else /* not HAVE_SETENV */
  gboolean ret = FALSE;

  if (variable)
    {
      if (overwrite || !g_getenv (variable))
        {
          gchar *string;

          string = g_strconcat (variable, "=", value, NULL);
          ret = putenv (string);
          g_free (string);
        }
      else
        {
          ret = TRUE;
        }
    }
  return ret;
#  endif /* not HAVE_SETENV */
# endif /* not G_OS_WIN32 */
}


void
g_unsetenv (const gchar *variable)
{
  if (variable)
    {
# ifdef G_OS_WIN32
#  ifdef UNICODE
      LPWSTR lpszName;
      gchar *utf8str;

      utf8str = g_locale_to_utf8 (variable, -1, NULL, NULL, NULL);
      lpszName = g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL);
      g_free (utf8str);
      SetEnvironmentVariableW (lpszName, NULL);
      g_free (lpszName);
#  else /* not UNICODE */
      SetEnvironmentVariableA (variable, NULL);
#  endif /* not UNICODE */
      if (ghash_env)
        g_hash_table_remove (ghash_env, variable);
# else /* not G_OS_WIN32 */
#  ifdef HAVE_UNSETENV
      unsetenv (variable);
#  else /* not HAVE_UNSETENV */
      gsize len;
      gchar **p, **q;

      len = g_strlen (variable);
      for (p = q = environ; *p; p++)
        if (g_strncmp (*p, variable, len) != 0 || (*p)[len] != '=')
          {
            *q = *p;
            q++;
          }
      *q = NULL;
#  endif /* not HAVE_UNSETENV */
# endif /* not G_OS_WIN32 */
    }
}
#endif /* not GLIB_CHECK_VERSION(2,4,0) */


#ifdef USE_GTK_EMULATE
const gchar *
g_get_user_name (void)
{
  static const gchar *name = NULL;

  if (!name)
    {
# ifdef G_OS_WIN32
#  ifdef UNICODE
  DWORD dwSize = 0;
  LPWSTR lpszName;

  GetUserNameW (NULL, &dwSize);
  lpszName = g_malloc (dwSize * sizeof (WCHAR));
  if (GetUserNameW (lpszName, &dwSize))
    {
      gchar *utf8str;

      utf8str = g_utf16_to_utf8 (lpszName, -1, NULL, NULL, NULL);
      name = g_locale_from_utf8 (utf8str, -1, NULL, NULL, NULL);
      g_free (utf8str);
    }
  g_free (lpszName);
#  else /* not UNICODE */
  DWORD dwSize = 0;

  GetUserNameA (NULL, &dwSize);
  name = g_malloc (dwSize * sizeof (CHAR));
  if (!GetUserNameA ((LPSTR)name, &dwSize))
    {
      g_free ((gpointer)name);
      name = NULL;
    }
#  endif /* not UNICODE */
# else /* not G_OS_WIN32 */
      name = g_getenv ("LOGNAME");
# endif /* not G_OS_WIN32 */
    }
  return name;
}
#endif /* USE_GTK_EMULATE */


#if ! GLIB_CHECK_VERSION(2,6,0)
# ifdef G_OS_WIN32
static gchar *
g_get_special_folder (INT nFolder)
{
  gchar *path = NULL;
  LPITEMIDLIST pidl = NULL;

  if (SHGetSpecialFolderLocation (NULL, nFolder, &pidl) == S_OK)
    {
      TCHAR szPath[MAX_PATH + 1];

      if (SHGetPathFromIDList (pidl, szPath))
        {
          gchar *utf8str;

#  ifdef UNICODE
          utf8str = g_utf16_to_utf8 (szPath, -1, NULL, NULL, NULL);
#  else /* not UNICODE */
          utf8str = g_locale_to_utf8 (szPath, -1, NULL, NULL, NULL);
#  endif /* not UNICODE */
          if (utf8str)
            {
              path = g_filename_from_utf8 (utf8str, -1, NULL, NULL, NULL);
              g_free (utf8str);
            }
        }
      CoTaskMemFree (pidl);
    }
  return path;
}
# endif /* G_OS_WIN32 */


const gchar *
g_get_user_cache_dir (void)
{
  static const gchar *cache = NULL;

  if (!cache)
    {
# ifdef G_OS_WIN32
      cache = g_get_special_folder (CSIDL_INTERNET_CACHE);
# else /* not G_OS_WIN32 */
      cache = g_getenv ("XDG_CACHE_HOME");
# endif /* not G_OS_WIN32 */
      if (!cache)
        {
          cache = g_build_filename (g_get_home_dir (), ".cache", NULL);
          if (!cache)
            cache = g_build_filename (g_get_tmp_dir (),
                                      g_get_user_name (), ".cache", NULL);
        }
    }
  return cache;
}


const gchar *
g_get_user_data_dir (void)
{
  static const gchar *data = NULL;

  if (!data)
    {
# ifdef G_OS_WIN32
      data = g_get_special_folder (CSIDL_LOCAL_APPDATA);
# else /* not G_OS_WIN32 */
      data = g_getenv ("XDG_DATA_HOME");
# endif /* not G_OS_WIN32 */
      if (!data)
        {
          data = g_build_filename (g_get_home_dir (), ".local", "share", NULL);
          if (!data)
            data = g_build_filename (g_get_tmp_dir (),
                                g_get_user_name (), ".local", "share", NULL);
        }
    }
  return data;
}


const gchar *
g_get_user_config_dir (void)
{
  static const gchar *config = NULL;

  if (!config)
    {
# ifdef G_OS_WIN32
      config = g_get_special_folder (CSIDL_LOCAL_APPDATA);
# else /* not G_OS_WIN32 */
      config = g_getenv ("XDG_CONFIG_HOME");
# endif /* not G_OS_WIN32 */
      if (!config)
        {
          config = g_build_filename (g_get_home_dir (), ".config", NULL);
          if (!config)
            config = g_build_filename (g_get_tmp_dir (),
                                       g_get_user_name (), ".config", NULL);
        }
    }
  return config;
}
#endif /* not GLIB_CHECK_VERSION(2,6,0) */


#ifdef USE_GTK_EMULATE
const gchar *
g_get_home_dir (void)
{
  static const gchar *home = NULL;

  if (!home)
    {
# ifdef G_OS_WIN32
      home = g_get_special_folder (CSIDL_PROFILE);
      if (!home)
        home = g_getenv ("USERPROFILE");
# else /* not G_OS_WIN32 */
      home = g_getenv ("HOME");
# endif /* not G_OS_WIN32 */
    }
  return home;
}


const gchar *
g_get_tmp_dir (void)
{
  static const gchar *tmp = NULL;

  if (!tmp)
    {
      tmp = g_getenv ("TMPDIR");
      if (!tmp)
        {
          tmp = g_getenv ("TMP");
          if (!tmp)
            tmp = g_getenv ("TEMP");
        }
    }
  return tmp;
}


gchar *
g_get_current_dir (void)
{
  gchar *dir = NULL;
# ifdef G_OS_WIN32
#  ifdef UNICODE
  DWORD dwLength;
  LPWSTR lpszDir;

  dwLength = GetCurrentDirectoryW (0, NULL);
  lpszDir = g_malloc (dwLength * sizeof (WCHAR));
  if (GetCurrentDirectoryW (dwLength, lpszDir) == dwLength - 1)
    {
      gchar *utf8str;

      utf8str = g_utf16_to_utf8 (lpszDir, -1, NULL, NULL, NULL);
      dir = g_filename_from_utf8 (utf8str, -1, NULL, NULL, NULL);
      g_free (utf8str);
    }
  g_free (lpszDir);
#  else /* not UNICODE */
  DWORD dwLength;

  dwLength = GetCurrentDirectoryA (0, NULL);
  dir = g_malloc (dwLength * sizeof (CHAR));
  if (GetCurrentDirectoryA (dwLength, dir) != dwLength - 1)
    {
      g_free (dir);
      dir = NULL;
    }
#  endif /* not UNICODE */
# else /* not G_OS_WIN32 */
  gchar *buf;
#  ifdef HAVE_GETCWD
  buf = getcwd (NULL, 0);
  dir = g_strdup (buf);
  free (buf);
#  else /* not HAVE_GETCWD */
  buf = g_malloc (MAXPATHLEN * sizeof (gchar));
  dir = getwd (buf);
  if (!dir)
    g_free (buf);
#  endif /* not HAVE_GETCWD */
# endif /* not G_OS_WIN32 */
  if (!dir)
    dir = g_strdup (G_DIR_SEPARATOR_S);
  return dir;
}


gboolean
g_path_is_absolute (const gchar *file)
{
  if (file)
    {
# ifdef G_OS_WIN32
#  ifdef UNICODE
      gboolean ret = FALSE;
      gchar *utf8str;

      utf8str = g_filename_to_utf8 (file, -1, NULL, NULL, NULL);
      if (utf8str)
        {
          LPWSTR lpszPath;

          lpszPath = g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL);
          if (lpszPath)
            {
              ret = ! PathIsRelativeW (lpszPath);
              g_free (lpszPath);
            }
          g_free (utf8str);
        }
      return ret;
#  else /* not UNICODE */
      return ! PathIsRelativeA (file);
#  endif /* not UNICODE */
# else /* not G_OS_WIN32 */
      return G_IS_DIR_SEPARATOR (file[0]);
# endif /* not G_OS_WIN32 */
    }
  return FALSE;
}


gchar *
g_path_get_basename (const gchar *file)
{
  gchar *name = NULL;

  if (file)
    {
# ifdef G_OS_WIN32
#  ifdef UNICODE
      gchar *utf8str;

      utf8str = g_filename_to_utf8 (file, -1, NULL, NULL, NULL);
      if (utf8str)
        {
          LPWSTR lpszFile;

          lpszFile = g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL);
          g_free (utf8str);
          if (lpszFile)
            {
              LPWSTR lpszName;

              lpszName = PathFindFileNameW (lpszFile);
              if (lpszName)
                {
                  utf8str = g_utf16_to_utf8 (lpszName, -1, NULL, NULL, NULL);
                  if (utf8str)
                    {
                      name = g_filename_from_utf8 (utf8str, -1,
                                                            NULL, NULL, NULL);
                      g_free (utf8str);
                    }
                }
              g_free (lpszFile);
            }
        }
#  else /* not UNICODE */
      LPSTR lpszName;

      lpszName = PathFindFileNameA (file);
      if (lpszName)
        name = g_strdup (lpszName);
#  endif /* not UNICODE */
# else /* not G_OS_WIN32 */
      gchar *base;

      base = g_strrchr (file, G_DIR_SEPARATOR);
      if (base)
        name = g_strdup (base + 1);
# endif /* not G_OS_WIN32 */
      if (!name)
        name = g_strdup (*file != '\0' ? file : ".");
    }
  return name;
}


gchar *
g_path_get_dirname (const gchar *file)
{
  gchar *path = NULL;

  if (file)
    {
# ifdef G_OS_WIN32
#  ifdef UNICODE
      gchar *utf8str;

      utf8str = g_filename_to_utf8 (file, -1, NULL, NULL, NULL);
      if (utf8str)
        {
          LPWSTR lpszFile;

          lpszFile = g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL);
          g_free (utf8str);
          if (lpszFile)
            {
              LPWSTR lpszName;

              lpszName = PathFindFileNameW (lpszFile);
              if (lpszName > lpszFile + 1)
                {
                  utf8str = g_utf16_to_utf8 (lpszFile, lpszName - lpszFile - 1,
                                                            NULL, NULL, NULL);
                  if (utf8str)
                    {
                      path = g_filename_from_utf8 (utf8str, -1,
                                                            NULL, NULL, NULL);
                      g_free (utf8str);
                    }
                }
            }
        }
#  else /* not UNICODE */
      LPSTR lpszName;

      lpszName = PathFindFileNameA (file);
      if (lpszName > file + 1)
        path = g_strndup (file, lpszName - file - 1);
#  endif /* not UNICODE */
# else /* not G_OS_WIN32 */
      gchar *base;

      base = g_strrchr (file, G_DIR_SEPARATOR);
      if (base)
        {
          while (base > file && G_IS_DIR_SEPARATOR (*base))
            base--;
          path = g_strndup (file, base - file + 1);
        }
# endif /* not G_OS_WIN32 */
    }
  return path ? path : g_strdup (".");
}


gchar *
g_build_filename (const gchar *first_element,
                  ...)
{
  gchar *str = NULL;

  if (first_element)
    {
      gchar **a, *s;
      gint i = 0;
      va_list args;

      va_start (args, first_element);
      while (va_arg (args, gchar *))
        i++;
      va_end (args);
      a = g_malloc ((i + 2) * sizeof (gchar *));
      a[0] = g_strdup (first_element);
      i = 1;
      va_start (args, first_element);
      while ((s = va_arg (args, gchar *)))
        a[i++] = g_strdup (s);
      va_end (args);
      a[i] = NULL;
      str = g_build_pathv (G_DIR_SEPARATOR_S, a);
      g_strfreev (a);
    }
  return str;
}
#endif /* USE_GTK_EMULATE */


#if ! GLIB_CHECK_VERSION(2,8,0)
gchar *
g_build_filenamev (gchar **args)
{
  return g_build_pathv (G_DIR_SEPARATOR_S, args);
}
#endif /* not GLIB_CHECK_VERSION(2,8,0) */


#ifdef USE_GTK_EMULATE
gchar *
g_build_path (const gchar *separator,
              const gchar *first_element,
              ...)
{
  gchar *str = NULL;

  if (first_element)
    {
      gchar **a, *s;
      gint i = 0;
      va_list args;

      va_start (args, first_element);
      while (va_arg (args, gchar *))
        i++;
      va_end (args);
      a = g_malloc ((i + 2) * sizeof (gchar *));
      a[0] = g_strdup (first_element);
      i = 1;
      va_start (args, first_element);
      while ((s = va_arg (args, gchar *)))
        a[i++] = g_strdup (s);
      va_end (args);
      a[i] = NULL;
      str = g_build_pathv (separator, a);
      g_strfreev (a);
    }
  return str;
}
#endif /* USE_GTK_EMULATE */


#if ! GLIB_CHECK_VERSION(2,8,0)
gchar *
g_build_pathv (const gchar  *separator,
               gchar       **args)
{
  gsize len, leng = 0;
  gchar **str, *path;
  gint i, j = 0;
  guint length;

  length = g_strv_length (args);
  str = g_malloc ((length + 1) * sizeof (gchar *));
  len = g_strlen (separator);
  for (i = 0; args[i]; i++)
    {
      gchar *p, *q;

      p = args[i];
      while (g_strncmp (p, separator, len) == 0)
        p += len;
      q = args[i] + g_strlen (args[i]) - len;
      while (p < q && g_strncmp (q, separator, len) == 0)
        q -= len;
      q += len;
      if (p < q)
        {
          str[j] = g_strndup (p, q - p);
          leng += g_strlen (str[j]);
          j++;
        }
    }
  str[j] = NULL;
  path = g_malloc ((leng + j * len + 1) * sizeof (gchar));
  path[0] = '\0';
  for (i = 0; str[i]; i++)
    {
# ifdef G_OS_WIN32
      if (!(i == 0 && g_ascii_isalpha ((str[0])[0]) && (str[0])[1] == ':'))
# endif /* G_OS_WIN32 */
      g_strcat (path, separator);
      g_strcat (path, str[i]);
    }
  g_strfreev (str);
  return path;
}
#endif /* not GLIB_CHECK_VERSION(2,8,0) */
