/*
    gcommon
    copyright (c) 1998-2013 Kazuki Iwamoto http://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "gcommon.h"
#ifdef G_OS_WIN32
# include <windows.h>
#endif /* G_OS_WIN32 */


/******************************************************************************
* Memory Allocation                                                           *
******************************************************************************/
#ifdef USE_GTK_EMULATE
# if ! defined (G_OS_WIN32) && defined (_SC_PAGESIZE) && defined (_SC_PHYS_PAGES)
static gulonglong g_memory_max_size = 0;
static gulonglong g_memory_used_size = 0;
# endif /* ! defined (G_OS_WIN32) && defined (_SC_PAGESIZE) && defined (_SC_AVPHYS_PAGES) */


gpointer
g_malloc (gsize n_bytes)
{
  gpointer mem = NULL;

  if (n_bytes > 0)
    {
      mem = g_try_malloc (n_bytes);
      if (!mem)
        {
          g_fprintf (stderr,
                    "Failed to allocate %"G_GSIZE_FORMAT" bytes\n", n_bytes);
          abort ();
        }
    }
  return mem;
}


gpointer
g_malloc0 (gsize n_bytes)
{
  gpointer mem;

  mem = g_malloc (n_bytes);
  g_memset (mem, 0, n_bytes);
  return mem;
}


gpointer
g_realloc (gpointer mem,
           gsize    n_bytes)
{
  if (n_bytes > 0)
    {
      mem = g_try_realloc (mem, n_bytes);
      if (!mem)
        {
          g_fprintf (stderr,
                    "Failed to allocate %"G_GSIZE_FORMAT" bytes\n", n_bytes);
          abort ();
        }
    }
  else
    {
      g_free (mem);
      mem = NULL;
    }
  return mem;
}


gpointer
g_try_malloc (gsize n_bytes)
{
  gpointer mem = NULL;

  if (n_bytes > 0)
    {
# if defined (G_OS_WIN32)
      MEMORYSTATUSEX ms;

      ms.dwLength = sizeof (MEMORYSTATUSEX);
      GlobalMemoryStatusEx (&ms);
      if (ms.ullAvailPhys >= n_bytes)
        mem = malloc (n_bytes);
# elif defined (_SC_PAGESIZE) && defined (_SC_PHYS_PAGES)
      if (g_memory_max_size == 0)
        g_memory_max_size = (gulonglong)sysconf (_SC_PAGESIZE)
                                        * (gulonglong)sysconf (_SC_PHYS_PAGES);
      if (g_memory_max_size >= g_memory_used_size + (gulonglong)n_bytes)
        mem = malloc (n_bytes + sizeof (gsize));
      if (mem)
        {
          g_memory_used_size += n_bytes;
          *(gsize *)mem = n_bytes;
          mem = (gsize *)mem + 1;
        }
# else /* ! defined (G_OS_WIN32) && defined (_SC_PAGESIZE) && defined (_SC_AVPHYS_PAGES) */
      mem = malloc (n_bytes);
# endif /* ! defined (G_OS_WIN32) && defined (_SC_PAGESIZE) && defined (_SC_AVPHYS_PAGES) */
    }
  return mem;
}
#endif /* USE_GTK_EMULATE */


#if ! GLIB_CHECK_VERSION(2,8,0)
gpointer
g_try_malloc0 (gsize n_bytes)
{
  gpointer mem;

  mem = g_try_malloc (n_bytes);
  g_memset (mem, 0, n_bytes);
  return mem;
}
#endif /* not GLIB_CHECK_VERSION(2,8,0) */


#ifdef USE_GTK_EMULATE
gpointer
g_try_realloc (gpointer mem,
               gsize    n_bytes)
{
  gpointer ret = NULL;
# ifdef G_OS_WIN32
  MEMORYSTATUSEX ms;
# endif /* G_OS_WIN32 */

  if (!mem)
    return g_try_malloc (n_bytes);
  if (n_bytes <= 0)
    {
      g_free (mem);
      return NULL;
    }
# if defined (G_OS_WIN32)
  ms.dwLength = sizeof (MEMORYSTATUSEX);
  GlobalMemoryStatusEx (&ms);
  if (ms.ullAvailPhys >= n_bytes)
    ret = realloc (mem, n_bytes);
# elif defined (_SC_PAGESIZE) && defined (_SC_PHYS_PAGES)
  if (g_memory_max_size >= g_memory_used_size + (gulonglong)n_bytes)
    ret = realloc ((gsize *)mem - 1, n_bytes + sizeof (gsize));
  if (ret)
    {
      g_memory_used_size = g_memory_used_size - *(gsize *)ret + n_bytes;
      *(gsize *)ret = n_bytes;
      ret = (gsize *)ret + 1;
    }
# else /* ! defined (G_OS_WIN32) && defined (_SC_PAGESIZE) && defined (_SC_AVPHYS_PAGES) */
  ret = realloc (mem, n_bytes);
# endif /* ! defined (G_OS_WIN32) && defined (_SC_PAGESIZE) && defined (_SC_AVPHYS_PAGES) */
  return ret;
}
#endif /* USE_GTK_EMULATE */


#if ! GLIB_CHECK_VERSION(2,24,0)
gpointer
g_malloc_n (gsize n_blocks,
            gsize n_block_bytes)
{
  return g_malloc (n_blocks * n_block_bytes);
}


gpointer
g_malloc0_n (gsize n_blocks,
             gsize n_block_bytes)
{
  return g_malloc0 (n_blocks * n_block_bytes);
}


gpointer
g_realloc_n (gpointer mem,
             gsize    n_blocks,
             gsize    n_block_bytes)
{
  return g_realloc (mem, n_blocks * n_block_bytes);
}


gpointer
g_try_malloc_n (gsize n_blocks,
                gsize n_block_bytes)
{
  return g_try_malloc (n_blocks * n_block_bytes);
}


gpointer
g_try_malloc0_n (gsize n_blocks,
                 gsize n_block_bytes)
{
  return g_try_malloc0 (n_blocks * n_block_bytes);
}


gpointer
g_try_realloc_n (gpointer mem,
                 gsize    n_blocks,
                 gsize    n_block_bytes)
{
  return g_try_realloc (mem, n_blocks * n_block_bytes);
}
#endif /* not GLIB_CHECK_VERSION(2,24,0) */


#ifdef USE_GTK_EMULATE
void
g_free (gpointer mem)
{
  if (mem)
    {
# if ! defined (G_OS_WIN32) && defined (_SC_PAGESIZE) && defined (_SC_PHYS_PAGES)
      mem = (gsize *)mem - 1;
      g_memory_used_size -= *(gsize *)mem;
# endif /* ! defined (G_OS_WIN32) && defined (_SC_PAGESIZE) && defined (_SC_AVPHYS_PAGES) */
      free (mem);
    }
}


gpointer
g_memmove (gpointer      dest,
           gconstpointer src,
           gsize         len)
{
  return dest && src ? memmove (dest, src, len) : NULL;
}


gpointer
g_memdup (gconstpointer mem,
          guint         byte_size)
{
  gpointer ret = NULL;

  if (mem)
    {
      ret = g_malloc (byte_size);
      g_memmove (ret, mem, byte_size);
    }
  return ret;
}
#endif /* USE_GTK_EMULATE */
