/*
    gcommon
    copyright (c) 1998-2013 Kazuki Iwamoto http://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "gcommon.h"


/******************************************************************************
* GDateTime                                                                   *
******************************************************************************/
#if ! GLIB_CHECK_VERSION(2,26,0)
struct _GDateTime
{
  time_t t;
  struct tm tm;
  gint ref;
};


void
g_date_time_unref (GDateTime *datetime)
{
  if (datetime && --datetime->ref <= 0)
    g_free (datetime);
}


GDateTime *
g_date_time_ref (GDateTime *datetime)
{
  if (datetime)
    datetime->ref++;
  return datetime;
}


GDateTime *
g_date_time_new_now_utc (void)
{
  return g_date_time_new_from_unix_utc (g_get_real_time () / G_USEC_PER_SEC);
}


static void
g_date_time_gmtime (GDateTime *datetime)
{
  if (datetime)
    {
# ifdef HAVE_GMTIME_R
      gmtime_r (&datetime->t, &datetime->tm);
# else /* not HAVE_GMTIME_R */
      struct tm *tm;

      tm = gmtime (&datetime->t);
      if (tm)
        datetime->tm = *tm;
# endif /* not HAVE_GMTIME_R */
    }
}


static void
g_date_time_mktime (GDateTime *datetime)
{
  if (datetime)
    {
# if defined (G_OS_WIN32)
      datetime->tm.tm_isdst = 0;
      datetime->t = _mkgmtime (&datetime->tm);
# elif defined (HAVE_TIMEGM)
      datetime->tm.tm_isdst = 0;
      datetime->t = timegm (&datetime->tm);
# else /* not HAVE_TIMEGM */
      const gchar *tz;

      tz = g_getenv ("TZ");
      g_setenv ("TZ", "", TRUE);
#  ifdef HAVE_TZSET
      tzset ();
#  endif /* HAVE_TZSET */
      datetime->tm.tm_isdst = 0;
      datetime->t = mktime (&datetime->tm);
      if (tz)
        g_setenv ("TZ", tz, TRUE);
      else
        g_unsetenv ("TZ");
#  ifdef HAVE_TZSET
      tzset ();
#  endif /* HAVE_TZSET */
# endif /* not HAVE_TIMEGM */
    }
}


GDateTime *
g_date_time_new_from_unix_utc (gint64 t)
{
  GDateTime *datetime;

  datetime = g_malloc (sizeof (GDateTime));
  datetime->ref = 1;
  datetime->t = t;
  g_date_time_gmtime (datetime);
  return datetime;
}


GDateTime *
g_date_time_new_from_timeval_utc (const GTimeVal *tv)
{
  return tv ? g_date_time_new_from_unix_utc (tv->tv_sec) : NULL;
}


GDateTime *
g_date_time_new_utc (gint    year,
                     gint    month,
                     gint    day,
                     gint    hour,
                     gint    minute,
                     gdouble seconds)
{
  GDateTime *datetime;

  datetime = g_malloc (sizeof (GDateTime));
  datetime->ref = 1;
  datetime->tm.tm_year = year - 1900;
  datetime->tm.tm_mon = month - 1;
  datetime->tm.tm_mday = day;
  datetime->tm.tm_hour = hour;
  datetime->tm.tm_min = minute;
  datetime->tm.tm_sec = seconds;
  g_date_time_mktime (datetime);
  return datetime;
}


GDateTime *
g_date_time_add (GDateTime *datetime,
                 GTimeSpan  timespan)
{
  GDateTime *dt = NULL;

  if (datetime)
    {
      dt = g_malloc (sizeof (GDateTime));
      dt->ref = 1;
      dt->t = datetime->t + timespan / G_USEC_PER_SEC;
      g_date_time_gmtime (dt);
    }
  return dt;
}


GDateTime *
g_date_time_add_years (GDateTime *datetime,
                       gint       years)
{
  return g_date_time_add_full (datetime, years, 0, 0, 0, 0, 0);
}


GDateTime *
g_date_time_add_months (GDateTime *datetime,
                        gint       months)
{
  return g_date_time_add_full (datetime, 0, months, 0, 0, 0, 0);
}


GDateTime *
g_date_time_add_weeks (GDateTime *datetime,
                       gint       weeks)
{
  return g_date_time_add_days (datetime, weeks * 7);
}


GDateTime *
g_date_time_add_days (GDateTime *datetime,
                      gint       days)
{
  return g_date_time_add_full (datetime, 0, 0, days, 0, 0, 0);
}


GDateTime *
g_date_time_add_hours (GDateTime *datetime,
                       gint       hours)
{
  return g_date_time_add (datetime, (gint64)hours * 60 * 60 * G_USEC_PER_SEC);
}


GDateTime *
g_date_time_add_minutes (GDateTime *datetime,
                         gint       minutes)
{
  return g_date_time_add (datetime, (gint64)minutes * 60 * G_USEC_PER_SEC);
}


GDateTime *
g_date_time_add_seconds (GDateTime *datetime,
                         gdouble    seconds)
{
  return g_date_time_add (datetime, seconds * G_USEC_PER_SEC);
}


GDateTime *
g_date_time_add_full (GDateTime *datetime,
                      gint       years,
                      gint       months,
                      gint       days,
                      gint       hours,
                      gint       minutes,
                      gdouble    seconds)
{
  GDateTime *dt = NULL;

  if (datetime)
    {
      gint leap;
      const static guint16 days_in_months[2][12] = {
                            {31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31},
                            {31, 29, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31}};

      dt = g_malloc (sizeof (GDateTime));
      dt->ref = 1;
      dt->tm.tm_year = datetime->tm.tm_year + (years * 12 + months) / 12;
      dt->tm.tm_mon = datetime->tm.tm_mon + (years * 12 + months) % 12;
      if (dt->tm.tm_mon < 0)
        {
          dt->tm.tm_mon += 12;
          dt->tm.tm_year--;
        }
      else if (dt->tm.tm_mon > 12)
        {
          dt->tm.tm_mon -= 12;
          dt->tm.tm_year++;
        }
      leap = dt->tm.tm_year % 400 == 0
            || (dt->tm.tm_year % 100 != 0 && dt->tm.tm_year % 4 == 0) ? 1 : 0;
      dt->tm.tm_mday = MIN (datetime->tm.tm_mday,
                                        days_in_months[leap][dt->tm.tm_mon]);
      dt->tm.tm_hour = datetime->tm.tm_hour;
      dt->tm.tm_min = datetime->tm.tm_min;
      dt->tm.tm_sec = datetime->tm.tm_sec;
      g_date_time_mktime (dt);
      dt->t += (((days * 24) + hours) * 60 + minutes) * 60 + seconds;
      g_date_time_gmtime (dt);
    }
  return dt;
}


gint
g_date_time_compare (gconstpointer dt1,
                     gconstpointer dt2)
{
  GTimeSpan difference;

  difference = g_date_time_difference ((GDateTime *)dt1, (GDateTime *)dt2);
  return difference < 0 ? -1 : difference > 0 ? 1 : 0;
}


GTimeSpan
g_date_time_difference (GDateTime *end,
                        GDateTime *begin)
{
  return end && begin ? (GTimeSpan)(end->t - begin->t) * G_USEC_PER_SEC : 0;
}


guint
g_date_time_hash (gconstpointer datetime)
{
  return datetime ? ((GDateTime *)datetime)->t : 0;
}


gboolean
g_date_time_equal (gconstpointer dt1,
                   gconstpointer dt2)
{
  return g_date_time_compare (dt1, dt2) == 0;
}


void
g_date_time_get_ymd (GDateTime *datetime,
                     gint      *year,
                     gint      *month,
                     gint      *day)
{
  if (datetime)
    {
      if (year)
        *year = datetime->tm.tm_year + 1900;
      if (month)
        *month = datetime->tm.tm_mon + 1;
      if (day)
        *day = datetime->tm.tm_mday;
    }
}


gint
g_date_time_get_year (GDateTime *datetime)
{
  return datetime ? datetime->tm.tm_year + 1900 : 0;
}


gint
g_date_time_get_month (GDateTime *datetime)
{
  return datetime ? datetime->tm.tm_mon + 1 : 0;
}


gint
g_date_time_get_day_of_month (GDateTime *datetime)
{
  return datetime ? datetime->tm.tm_mday : 0;
}


gint
g_date_time_get_hour (GDateTime *datetime)
{
  return datetime ? datetime->tm.tm_hour : 0;
}


gint
g_date_time_get_minute (GDateTime *datetime)
{
  return datetime ? datetime->tm.tm_min : 0;
}


gint
g_date_time_get_second (GDateTime *datetime)
{
  return datetime ? datetime->tm.tm_sec : 0;
}

gint
g_date_time_get_microsecond (GDateTime *datetime)
{
  return 0;
}


gdouble
g_date_time_get_seconds (GDateTime *datetime)
{
  return g_date_time_get_second (datetime);
}


gint64
g_date_time_to_unix (GDateTime *datetime)
{
  return datetime ? datetime->t : 0;
}


gboolean
g_date_time_to_timeval (GDateTime *datetime,
                        GTimeVal  *tv)
{
  if (datetime && tv)
    {
      tv->tv_sec = datetime->t;
      tv->tv_usec = 0;
      return TRUE;
    }
  return FALSE;
}
#endif /* not GLIB_CHECK_VERSION(2,26,0) */
