/*
    IA32
    copyright (c) 1998-2011 Kazuki Iwamoto http://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "stack.h"
#include "message.h"


/******************************************************************************
*                                                                             *
* ja:スタック関数群                                                           *
*                                                                             *
******************************************************************************/
/*  ja:スタック格納(32ビット)
    thread,スレッド構造体
     value,値
       RET,TRUE:正常終了,FALSE:エラー                                       */
gboolean
ia32_stack_push32 (Ia32Thread    *thread,
                   const guint32  value)
{
  thread->reg.esp -= 4;
  return ia32_memory_write_dword (thread->process,
                                            thread->reg.esp, value, -1) >= 0;
}


/*  ja:スタック搬出(32ビット)
    thread,スレッド構造体
     value,値
       RET,TRUE:正常終了,FALSE:エラー                                       */
gboolean
ia32_stack_pop32 (Ia32Thread *thread,
                  guint32    *value)
{
  guint32 v = 0;

  if (ia32_memory_read_dword (thread->process, thread->reg.esp, &v) < 0)
    return FALSE;
  thread->reg.esp += 4;
  if (value)
    *value = v;
  return TRUE;
}


/*  ja:スタック格納(16ビット)
    thread,スレッド構造体
     value,値
       RET,TRUE:正常終了,FALSE:エラー                                       */
gboolean
ia32_stack_push16 (Ia32Thread    *thread,
                   const guint16  value)
{
  thread->reg.esp -= 2;
  return ia32_memory_write_word (thread->process,
                                            thread->reg.esp, value, -1) >= 0;
}


/*  ja:スタック搬出(16ビット)
    thread,スレッド構造体
     value,値
       RET,TRUE:正常終了,FALSE:エラー                                       */
gboolean
ia32_stack_pop16 (Ia32Thread *thread,
                  guint16    *value)
{
  guint16 v = 0;

  if (ia32_memory_read_word (thread->process, thread->reg.esp, &v) < 0)
    return FALSE;
  thread->reg.esp += 2;
  if (value)
    *value = v;
  return TRUE;
}


/******************************************************************************
*                                                                             *
* ja:引数関数群                                                               *
*                                                                             *
******************************************************************************/
/*  ja:スタックから引数の値を取得する
    thread,スレッド構造体
      argv,引数の位置(0:戻りアドレス,1:最左引数,...)
       RET,値                                                               */
guint32
ia32_stack_value (Ia32Thread *thread,
                  const gint  argv)
{
  guint32 value = 0;

  if (ia32_memory_read_dword (thread->process,
                                    thread->reg.esp + argv * 4, &value) < 0)
    ia32_message_record_exception (thread, IA32_ERR_SS);
  return value;
}


/*  ja:スタックから引数のポインタを取得する
    thread,スレッド構造体
      argv,引数の位置(0:戻りアドレス,1:最左引数,...)
       RET,実アドレス(NULL:エラー)                                          */
gpointer
ia32_stack_pointer (Ia32Thread *thread,
                    const gint  argv)
{
  guint32 address;

  return ia32_memory_read_dword (thread->process,
                                    thread->reg.esp + argv * 4, &address) >= 0
                ? ia32_memory_real_address (thread->process, address) : NULL;
}


/*  ja:スタックから引数の文字列を取得する
    thread,スレッド構造体
      argv,引数の位置(0:戻りアドレス,1:最左引数,...)
       RET,実アドレス(NULL:エラー)                                          */
gpointer
ia32_stack_string (Ia32Thread *thread,
                   const gint  argv)
{
  gint i = 0;
  guint8 c;
  guint32 address;

  if (ia32_memory_read_dword (thread->process,
                                    thread->reg.esp + argv * 4, &address) < 0)
    return NULL;
  do
    {
      if (ia32_memory_read_byte (thread->process, address + i, &c) < 0)
        return NULL;
      i++;
    }
  while (c != '\0');
  return ia32_memory_real_address (thread->process, address);
}


/*  ja:スタックから引数のワイド文字列を取得する
    thread,スレッド構造体
      argv,引数の位置(0:戻りアドレス,1:最左引数,...)
       RET,実アドレス(NULL:エラー)                                          */
gpointer
ia32_stack_stringw (Ia32Thread *thread,
                    const gint  argv)
{
  gint i = 0;
  guint16 c;
  guint32 address;

  if (ia32_memory_read_dword (thread->process,
                                    thread->reg.esp + argv * 4, &address) < 0)
    return NULL;
  do
    {
      if (ia32_memory_read_word (thread->process, address + i, &c) < 0)
        return NULL;
      i += 2;
    }
  while (c != '\0');
  return ia32_memory_real_address (thread->process, address);
}
