// <!-- Encoding Fixer: 厄 虫 退 散
// $Id: Reflection.java 132 2008-07-02 13:51:18Z yo-zi $
// Copyright 2007 Yo-zi.
// Licensed under the Apache License, Version 2.0 (the "License"); you may
// not use this file except in compliance with the License. You may obtain
// a copy of the License at http://www.apache.org/licenses/LICENSE-2.0
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//-->

package jp.sourceforge.greflect;

import java.lang.reflect.*;
import jp.sourceforge.greflect.impl.*;

/**
 * [concurrent][facade]
 * <BR/> A <EM>facade of the reflection API</EM> supports generic features 
 * <BR/> ジェネリクスをサポートする<EM>リフレクションＡＰＩのfacade</EM>。
 * .
 * 
 * <DT><B> Responsibilities &amp; Collaborations: </B></DT>
 * 
 * <OL><LI> 
 * <EM>Create ungeneric instance</EM> type-safely <B>with</B> {@link Instantiator}
 * 
 * </LI><LI> 
 * <EM>Invoke method</EM> type-safely<B>with</B> {@link Invoker}
 * 
 * </LI><LI> 
 * Cache the reflection results <B>with</B> 
 * 
 * </LI></OL>
 *
 * 
 * <DT><B> Notes &amp; Descriptions: </B></DT>
 * <DD>
 * 
 * See
 * <A href="../../../../basicuse.html">
 * How To Invoke Method</A>
 * to know the basic use of this class.
 * <P> @lang ja このクラスの基本的な使用方法を知るには
 * <A href="../../../../basicuse.html">
 * How To Invoke Method</A>
 * を見てください。
 * </P>
 * 
 * </DD>
 * <DT><B> SourceInfo: </B></DT>
 * <DD> $Id: Reflection.java 132 2008-07-02 13:51:18Z yo-zi $
 * </DD>
 * 
 * @author yo-zi
 *
 */
public class Reflection
{
    TypeConversionStrategy strategy = new DefaultTypeConversionStrategy();

    BytecodeGeneratorClassLoader classLoader = new BytecodeGeneratorClassLoader(
            Instantiator.class.getClassLoader());

    private final Field invokerReturnValueField;

    private final Field instantiatorReturnValueField;

    private final Field casterReturnValueField;

    {
        Field fi = null;
        Field fv = null;
        Field fc = null;
        try{
            fi = Invoker.class.getDeclaredField("returnValue");
            fv = Instantiator.class.getDeclaredField("returnValue");
            fc = Caster.class.getDeclaredField("returnValue");
        }catch(Exception ex){
            Error e = new ExceptionInInitializerError(ex);
            throw e;
        }finally{
            invokerReturnValueField = fi;
            instantiatorReturnValueField = fv;
            casterReturnValueField = fc;
        }
    }

    /**
     * Creates a facade of reflection with the current {@link ClassLoader}
     * <BR/> 現在のクラスローダでリフレクションのfacadeを生成します。
     * .
     * <DL>
     * <DT><B> Pre-conditions : </B></DT>
     * <DD>
     * </DD>
     *
     * <DT><B> Post-conditions : </B></DT>
     * <DD>
     * </DD>
     *
     * </DL>
     * @param class_loader
     */
    public Reflection(){
        this(Reflection.class.getClassLoader());
    }

    /**
     * Creates a facade of reflection with the specified {@link ClassLoader}
     * <BR/> 指定したクラスローダでリフレクションのfacadeを生成します。
     * .
     * <DL>
     * <DT><B> Pre-conditions : </B></DT>
     * <DD>
     * </DD>
     *
     * <DT><B> Post-conditions : </B></DT>
     * <DD>
     * </DD>
     *
     * </DL>
     * @param class_loader
     */
    public Reflection(ClassLoader class_loader){
        classLoader = new BytecodeGeneratorClassLoader(class_loader);
    }

    /**
     * [ungeneric]
     * <BR/> Instantiator <EM>creates a ungeneric object</EM> of the specified type
     * <BR/> Instantiatorは指定された型の<EM>アンジェネリックオブジェクトを生成</EM>します。
     * .
     * <DT><B> Type Variable {@code T}: </B></DT>
     * <DD> 
     * The type of instantiate() return.
     * 
     * </DD>
     * <DT><B> Responsibilities &amp; Collaborations: </B></DT>
     * 
     * <OL><LI> 
     * Generate the bytecode and create its instance <B>with</B> {@link BytecodeGeneratorCloassLoader}
     * <P>@lang ja バイトコード生成することでアンジェネリックなクラスを作成し、そのインスタンスを生成する。
     * </P>
     *
     * </LI><LI> 
     * Check the generic types of parameters <B>with</B> {@link Param}
     * <P>@lang ja コンストラクタにパラメータを指定する。
     * </P>
     * 
     * </LI></OL>
     *
     * 
     * <DT><B> Notes &amp; Descriptions: </B></DT>
     * <DD>
     * See
     * <A href="../../../../basicuse.html#How To Create Instance">
     * How To Create Instance</A>
     * to know the basic use of this class.
     * <P> @lang ja このクラスの基本的な使用方法を知るには
     * <A href="../../../../basicuse.html#How To Create Instance">
     * How To Create Instance</A>
     * を見てください。
     * </P>
     * 
     * 
     * </DD>
     * <DT><B> SourceInfo: </B></DT>
     * <DD> $Id: Reflection.java 132 2008-07-02 13:51:18Z yo-zi $
     * </DD>
     * 
     * @author yo-zi
     */
    public abstract class Instantiator<T>
    {
        private final Object[] constructorArgs;

        private final Constructor<?> constructor;

        private T returnValue = null;

        /**
         * Prepares to generate the instance by checking the number of parameters and the validity of the generic types
         * <BR/> パラメタ数とジェネリックタイプの妥当性をチェックする
         * ことによって、インスタンスを生成する準備をします。 
         * .
         * 
         * <DL>
         * <DT><B> Pre-conditions : </B></DT>
         * <DD>
         * Each instype and args must be satisfy one of following conditions.<ol>
         * <li> An ungeneric instance of {@link Param}. </li>
         * <li> The instance of {@link UngenericParam} of ungeneric object.</li></ol>
         * When both of above conditions are not satisfied.
         * {@link UnresolvedTypeVariableException} will be thrown.
         * <P/>@lang ja
         * instypeおよびargsは以下のいずれかである必要があります。<ol>
         * <li>{@liink Param}のアンジェネリックインスタンスである</li>
         * <li>アンジェネリックをオブジェクトを指定した{@link UngenericParam}
         * インスタンスである</li></ol>
         * もしいずれでもないなら、{@link UnresolvedTypeVariableException}が
         * スローされます。
         * </DD>
         *
         * <DT><B> Post-conditions : </B></DT>
         * <DD> 
         * The target constructor is executable.
         * <P/>@lang ja 対象のコンストラクタは実行可能である。
         * </DD>
         *
         * </DL>
         * @param instype the type of instance to create.
         * <BR/>@lang ja 生成するオブジェクトの型。
         * @param args the parameters of constructor.
         * <BR/>@lang ja コンストラクタに渡す引数。
         * @throws SecurityException
         * @throws ClassNotFoundException
         * @throws TypeViolationException
         * @throws IllegalArgumentException
         * @throws InstantiationException
         * @throws IllegalAccessException
         */
        public Instantiator(Param<?> instype, Param<?>... args)
            throws SecurityException, ClassNotFoundException,
            TypeViolationException, IllegalArgumentException,
            NoSuchMethodException, IllegalAccessException
        {
            this(instype, findConstructor(instype.getType(), instype
                    .getTypeVarScope(), args), args);
        }

        /**
         * Prepares to generate the instance by checking the number of parameters and the validity of the generic types
         * <BR/> パラメタ数とジェネリックタイプの妥当性をチェックすることによって、
         * インスタンスを生成する準備をします。 
         * .
         * Type variable {@code T} is used to fix generic type.
         * このコンストラクタでは型変数{@code T}を使ってジェネリック型を決定します。
         * 
         * <DL>
         * <DT><B> Pre-conditions : </B></DT>
         * <DD>
         * Each instype and args must be satisfy one of following conditions.<ol>
         * <li> An ungeneric instance of {@link Param}. </li>
         * <li> The instance of {@link UngenericParam} of ungeneric object.</li></ol>
         * When both of above conditions are not satisfied.
         * {@link UnresolvedTypeVariableException} will be thrown.
         * <P/>@lang ja
         * instypeおよびargsは以下のいずれかである必要があります。<ol>
         * <li>{@link Param}のアンジェネリックインスタンスである</li>
         * <li>アンジェネリックをオブジェクトを指定した{@link UngenericParam}
         * インスタンスである</li></ol>
         * もしいずれでもないなら、{@link UnresolvedTypeVariableException}が
         * スローされます。
         * </DD>
         *
         * <DT><B> Post-conditions : </B></DT>
         * <DD>
         * The target constructor is executable.
         * <P/>@lang ja 対象のコンストラクタは実行可能である。
         * </DD>
         *
         * </DL>
         * @param instype the type of instance to create.
         * <BR/>@lang ja 生成するオブジェクトの型。
         * @param cst the constructor to invoke.
         * <BR/>@lang ja オブジェクトの生成に使用するコンストラクタ。
         * @param args the parameters of constructor.
         * <BR/>@lang ja コンストラクタに渡す引数。
         * @throws SecurityException
         * @throws ClassNotFoundException
         * @throws TypeViolationException
         * @throws IllegalArgumentException
         * @throws InstantiationException
         * @throws IllegalAccessException
         */
        public Instantiator(Param<?> instype, Constructor<?> cst,
                Param<?>... args)
            throws SecurityException, ClassNotFoundException,
            TypeViolationException, IllegalArgumentException,
            IllegalAccessException
        {
            Type type = instype.getType();
            TypeVarScope iscope = instype.getTypeVarScope();
            // get ungeneric info
            final Class<?> cclass = cst.getDeclaringClass();
            final TypeVarScope cscope;
            type = strategy.resolveTypeVariable(type, iscope);
            {
                if(type instanceof ParameterizedType){
                    ParameterizedType pt = (ParameterizedType) type;
                    if(cclass.equals(pt.getRawType())){
                        Type[] tps = pt.getActualTypeArguments();
                        cscope = ChainedTypeVarScope.chain(iscope, cclass, tps);
                    }else{
                        String src = strategy.getTypeDescriptionFor(type,
                                iscope, true);
                        throw new TypeViolationException(
                                "The specified type ''{0}'' is not declaring the constructor ''{1}''. Revise the specification.",
                                src, cst);
                    }
                }else{
                    cscope = iscope;
                }
            }

            // check constructor and return type.
            {
                Type t = Instantiator.class.getTypeParameters()[0];
                TypeVarScope s = ClassTypeVarScope
                        .getTypeScopeWithEnclosings(getClass());
                if(strategy.isAssignableFromTo(cclass, cscope, t, s)){
                    //pass
                }else{
                    String src = strategy.getTypeDescriptionFor(t, s, true);
                    String dst = strategy.getTypeDescriptionFor(cclass, cscope,
                            true);
                    throw new TypeViolationException(
                            "The specified type ''{0}'' is not suitable to set the created object of ''{1}''. Revise the specification.",
                            src, dst);
                }
            }

            // check constructor type.
            if(strategy.isAssignableFromTo(type, iscope, cclass, cscope)){
                //pass
            }else{
                String src = strategy.getTypeDescriptionFor(type, iscope, true);
                throw new TypeViolationException(
                        "The specified type ''{0}'' is not declaring the constructor ''{1}''. Revise the specification.",
                        src, cst);
            }

            // check parameter types.
            final Type[] iprmtps = cst.getGenericParameterTypes();
            if(iprmtps.length != args.length){
                throw new IllegalArgumentException(
                        "The specified parameter has '" + args.length
                                + "' parameters though the constructor has '"
                                + iprmtps.length
                                + "' parameters. Revise the specification.");
            }
            for(int i = args.length; --i >= 0;){
                Type itype = iprmtps[i]; // defined parameter type.
                Type atype = args[i].getType();
                //final TypeVarScope ascope = ClassTypeVarScope
                //        .getTypeScopeWithEnclosings(args[i].getClass());
                final TypeVarScope ascope = args[i].getTypeVarScope();
                if(strategy.isAssignableFromTo(atype, ascope, itype, cscope)){
                    // pass
                }else{
                    String src = strategy.getTypeDescriptionFor(atype, ascope,
                            true);
                    String dst = strategy.getTypeDescriptionFor(itype, cscope,
                            true);
                    throw new TypeViolationException(
                            "The specified type ''{0}'' is not applicable for the param{1} type ''{2}'' of the constructor {3}. Revise the specification.",
                            src, i, dst, cst);
                }
            }

            // generate bytecode.
            String gsig = strategy.getTypeSignatureTo(type, iscope);
            Constructor<?> icst = classLoader
                    .generateClassAndGetConstructorFor(cst, gsig);

            Object[] values = new Object[args.length];
            for(int i = args.length; --i >= 0;){
                final Object av = args[i].getValue();
                values[i] = av;
            }
            constructor = icst;
            constructorArgs = values;

        }

        /**
         * Create a instance
         * <BR/> インスタンスを生成します。
         * .
         * The created instance is sure to be ungeneric object.
         * <P/> @lang ja 作られたインスタンスは必ずアンジェネリックオブジェクトになります。
         * <DL>
         * <DT><B> Pre-conditions : </B></DT>
         * <DD>
         * </DD>
         *
         * <DT><B> Post-conditions : </B></DT>
         * <DD>
         * </DD>
         *
         * </DL>
         * @return
         * @throws InvocationTargetException
         */
        public T instantiate()
            throws InvocationTargetException
        {
            try{
                Object r = constructor.newInstance(constructorArgs);
                instantiatorReturnValueField.set(this, r);
                return returnValue;
            }catch(InstantiationException ex){
                throw new ReflectionError(ex,
                        "InstantiationException has not prechecked. Report as a bug.");
            }catch(IllegalAccessException ex){
                throw new ReflectionError(ex,
                        "IllegalAccessException has not prechecked. Report as a bug.");
            }
        }

    }

    /**
     * [ungeneric]
     * <BR/> Invoker <EM>invoke the method </EM> of the specified instance
     * <BR/> Invokerは指定されたインスタンスの<EM>メソッドを呼び出し</EM>ます。
     * .
     * <DT><B> Type Variable {@code T}: </B></DT>
     * <DD> 
     * The type of invoke() return.
     * 
     * </DD>
     * <DT><B> Responsibilities &amp; Collaborations: </B></DT>
     * 
     * <OL><LI> 
     * Check the generic types of parameters <B>with</B> {@link Param}
     * <P>@lang ja @lang ja コンストラクタにパラメータを指定する。
     * </P>
     * </LI><LI> 
     * <B>with</B> {@link }
     * 
     * </LI><LI> 
     * <B>with</B> {@link }
     *
     * </LI><LI> 
     * <B>with</B> {@link }
     * 
     * </LI></OL>
     *
     * 
     * <DT><B> Notes &amp; Descriptions: </B></DT>
     * <DD>
     * See
     * <A href="../../../../basicuse.html">
     * How To Invoke Method</A>
     * to know the basic use of this class.
     * <P> @lang ja このクラスの基本的な使用方法を知るには
     * <A href="../../../../basicuse.html">
     * How To Invoke Method</A>
     * を見てください。
     * </P>
     * 
     * 
     * </DD>
     * <DT><B> SourceInfo: </B></DT>
     * <DD> $Id: Reflection.java 132 2008-07-02 13:51:18Z yo-zi $
     * </DD>
     * 
     * @author yo-zi
     *
     */
    public abstract class Invoker<T>
    {
        private final Object instance;

        private final Method method;

        private final Object[] methodArgs;

        private T returnValue = null;

        /**
         * Prepares to execute the method by checking the number of parameters and the validity of the generic types
         * <BR/> パラメタ数とジェネリックタイプの妥当性をチェックする
         * ことによって、メソッドを実行する準備をします。 
         * .
         * 
         * <DL>
         * <DT><B> Pre-conditions : </B></DT>
         * <DD>
         * Each instance and args must be satisfy one of following conditions.<ol>
         * <li> An ungeneric instance of {@link Param}. </li>
         * <li> The instance of {@link UngenericParam} of ungeneric object.</li></ol>
         * When both of above conditions are not satisfied.
         * {@link UnresolvedTypeVariableException} will be thrown.
         * <P/>@lang ja
         * instanceおよびargsは以下のいずれかである必要があります。<ol>
         * <li>{@link Param}のアンジェネリックインスタンスである</li>
         * <li>アンジェネリックをオブジェクトを指定した{@link UngenericParam}
         * インスタンスである</li></ol>
         * もしいずれでもないなら、{@link UnresolvedTypeVariableException}が
         * スローされます。
         * </DD>
         *
         * <DT><B> Post-conditions : </B></DT>
         * <DD>
         * The target method is executable.
         * <P/>@lang ja 対象のメソッドは実行可能である。
         * </DD>
         *
         * </DL>
         * @param instance instance to invoke the method.
         * <BR/>@lang ja メソッドを実行するインスタンス。
         * @param methodName method name to invoke.
         * <BR/>@lang ja 実行するメソッドの名前。
         * @param args The parametes of the method.
         * <BR/>@lang ja 実行するメソッドのパラメータ。
         * @throws TypeViolationException
         * @throws NoSuchMethodException
         * @throws SecurityException
         * @throws IllegalAccessException
         */
        public Invoker(Param<?> instance, String methodName, Param<?>... args)
            throws TypeViolationException, NoSuchMethodException,
            SecurityException, IllegalAccessException
        {
            this(instance, findMethod(instance, methodName, args), args);
        }

        /**
         * Prepares to execute the method by checking the number of parameters and the validity of the generic types
         * <BR/> パラメタ数とジェネリックタイプの妥当性をチェックする
         * ことによって、メソッドを実行する準備をします。 
         * .
         * <DL>
         * <DT><B> Pre-conditions : </B></DT>
         * <DD>
         * Each instance and args must be satisfy one of following conditions.<ol>
         * <li> An ungeneric instance of {@link Param}. </li>
         * <li> The instance of {@link UngenericParam} of ungeneric object.</li></ol>
         * When both of above conditions are not satisfied.
         * {@link UnresolvedTypeVariableException} will be thrown.
         * <P/>@lang ja
         * instanceおよびargsは以下のいずれかである必要があります。<ol>
         * <li>{@link Param}のアンジェネリックインスタンスである</li>
         * <li>アンジェネリックをオブジェクトを指定した{@link UngenericParam}
         * インスタンスである</li></ol>
         * もしいずれでもないなら、{@link UnresolvedTypeVariableException}が
         * スローされます。
         * </DD>
         *
         * <DT><B> Post-conditions : </B></DT>
         * <DD>
         * The target method is executable.
         * <P/>@lang ja 対象のメソッドは実行可能である。
         * </DD>
         *
         * </DL>
         * @param instance instance to invoke the method.
         * <BR/>@lang ja メソッドを実行するインスタンス。
         * @param method method to invoke.
         * <BR/>@lang ja 実行するメソッド。
         * @param args The parametes of the method.
         * <BR/>@lang ja 実行するメソッドのパラメータ。
         * @throws TypeViolationException
         * @throws IllegalArgumentException
         * @throws IllegalAccessException
         * When the class declaring the static method is specified with type parameters.
         * <BR/>staticメソッドを宣言したクラスが型パラメータとともに指定されたとき。
         */
        public Invoker(Param<?> instance, Method method, Param<?>... args)
            throws TypeViolationException, IllegalArgumentException,
            IllegalAccessException
        {
            this.instance = instance.getValue();
            this.method = method;

            //final TypeVarScope iscope = new ClassTypeVarScope(instance
            //        .getClass());
            //final TypeVarScope iscope = new ClassTypeVarScope(instance.getValue().getClass());
            final Type[] iprmtps = method.getGenericParameterTypes();
            if(iprmtps.length != args.length){
                throw new IllegalArgumentException(
                        "The specified parameter has '" + args.length
                                + "' parameters though the constructor has '"
                                + iprmtps.length
                                + "' parameters. Revise the specification.");
            }

            final TypeVarScope cscope;
            {
                final TypeVarScope iscope = instance.getTypeVarScope();
                final Class<?> cclass = method.getDeclaringClass();
                Type type = strategy.resolveTypeVariable(instance.getType(),
                        iscope);
                if(Modifier.isStatic(method.getModifiers())){
                    if(type instanceof Class){
                        cscope = iscope;
                    }else{
                        throw new IllegalAccessException(
                                "The class that declare the static method must be specified without type variables. Revise the specification.");
                    }
                }else if(type instanceof ParameterizedType){
                    ParameterizedType pt = (ParameterizedType) type;
                    if(cclass.equals(pt.getRawType())){
                        Type[] tps = pt.getActualTypeArguments();
                        cscope = ChainedTypeVarScope.chain(iscope, cclass, tps);
                    }else{
                        String src = strategy.getTypeDescriptionFor(type,
                                iscope, true);
                        throw new TypeViolationException(
                                "The specified type ''{0}'' is not declaring the method ''{1}''. Revise the specification.",
                                src, method);
                    }
                }else{
                    cscope = iscope;
                }
            }

            TypeVarScope irscope = cscope;
            for(int i = iprmtps.length; --i >= 0;){
                Type itype = iprmtps[i]; // defined parameter type.
                Type atype = args[i].getType();
                //final TypeVarScope ascope = ClassTypeVarScope
                //        .getTypeScopeWithEnclosings(args[i].getClass());
                final TypeVarScope ascope = args[i].getTypeVarScope();
                TypeVarScope tv = strategy.getUnresolvedTypeVarsFromTo(atype,
                        ascope, itype, cscope);
                if(tv != null){
                    if(tv.isEmpty()){
                        // pass
                    }else{
                        for(TypeVariable<? extends GenericDeclaration> t : tv
                                .getTypeVariableSet()){
                            if(method.equals(t.getGenericDeclaration())){
                                // pass
                            }else{
                                throw new UnresolvedTypeVariableException(
                                        "The specified type variable ''{0}'' is not declared in method ''{1}''. Report as a bug.",
                                        t, method);
                            }
                        }

                        irscope = new BranchedTypeVarScope(irscope, tv);
                    }
                }else{
                    String src = strategy.getTypeDescriptionFor(atype, ascope,
                            true);
                    String dst = strategy.getTypeDescriptionFor(itype, cscope,
                            true);
                    throw new TypeViolationException(
                            "The specified type ''{0}'' is not applicable for the param{1} type ''{2}'' of the method {3}. Revise the specification.",
                            src, i, dst, method);
                }
            }

            //final TypeVarScope rscope
            //    = new BranchedTypeVarScope(new ClassTypeVarScope(getClass()), iscope);
            // check return type
            TypeVarScope rscope = ClassTypeVarScope
                    .getTypeScopeWithEnclosings(getClass());
            final Type rtype = getType();
            final Type irtype = method.getGenericReturnType();
            if(void.class.equals(irtype)){
                // ignore
            }else if(strategy
                    .isAssignableFromTo(irtype, irscope, rtype, rscope)){
                // pass
            }else{
                String src = strategy.getTypeDescriptionFor(irtype, cscope,
                        true);
                String dst = strategy
                        .getTypeDescriptionFor(rtype, rscope, true);
                throw new TypeViolationException(
                        "The specified type ''{0}'' is not applicable for the return type ''{1}'' of the method {2}. Revise the type.",
                        dst, src, method);
            }

            Object[] values = new Object[iprmtps.length];
            for(int i = iprmtps.length; --i >= 0;){
                final Object av = args[i].getValue();
                values[i] = av;
            }
            methodArgs = values;
        }

        /**
         * Invoke the method
         * <BR/> メソッドを実行します。
         * .
         * <DL>
         * <DT><B> Pre-conditions : </B></DT>
         * <DD>
         * </DD>
         *
         * <DT><B> Post-conditions : </B></DT>
         * <DD>
         * </DD>
         *
         * </DL>
         * @return
         * @throws InvocationTargetException
         */
        //@SuppressWarnings("unckecked")
        public T invoke()
            throws InvocationTargetException
        {
            try{
                Object r = method.invoke(instance, methodArgs);
                invokerReturnValueField.set(this, r);
                return returnValue;
            }catch(IllegalAccessException ex){
                throw new ReflectionError(ex,
                        "IllegalAccessException has not prechecked. Report as a bug.");
            }
        }

        /**
         * 
         * .
         * <DL>
         * <DT><B> Pre-conditions : </B></DT>
         * <DD>
         * </DD>
         *
         * <DT><B> Post-conditions : </B></DT>
         * <DD>
         * </DD>
         *
         * </DL>
         * @return
         */
        Type getType()
            throws TypeViolationException
        {
            return Invoker.class.getTypeParameters()[0];
        }
    }

    /**
     * [ungeneric]
     * <BR/> Caster cast ungeneric object to specified generic type 
     * <BR/> Casterはアンジェネリックオブジェクトを指定された型にキャストします。
     * .
     * <DT><B> Type Variable {@code T}: </B></DT>
     * <DD> 
     * The type of cast() return.
     * 
     * </DD>
     * <DT><B> Responsibilities &amp; Collaborations: </B></DT>
     * 
     * <OL><LI> 
     * 
     * </LI></OL>
     *
     * 
     * <DT><B> Notes &amp; Descriptions: </B></DT>
     * <DD>
     * See
     * <A href="../../../../basicuse.html#How To Create Instance">
     * How To Create Instance</A>
     * to know the basic use of this class.
     * <P> @lang ja このクラスの基本的な使用方法を知るには
     * <A href="../../../../basicuse.html#How To Create Instance">
     * How To Create Instance</A>
     * を見てください。
     * </P>
     * 
     * 
     * </DD>
     * <DT><B> SourceInfo: </B></DT>
     * <DD> $Id: Reflection.java 132 2008-07-02 13:51:18Z yo-zi $
     * </DD>
     * 
     * @author yo-zi
     *
     */
    public abstract class Caster<T>
    {
        private T returnValue = null;

        /**
         * Prepares to execute the method by checking the number of parameters and the validity of the generic types
         * <BR/> パラメタ数とジェネリックタイプの妥当性をチェックし、
         * インスタンスをキャストします。 
         * .
         * <DL>
         * <DT><B> Pre-conditions : </B></DT>
         * <DD>
         * The instance must be ungeneric.
         * When not, {@link UnresolvedTypeVariableException} will be thrown.
         * <P/>@lang ja
         * instanceはアンジェネリックインスタンスである必要があります。
         * もしそうでないなら、{@link UnresolvedTypeVariableException}が
         * スローされます。
         * </DD>
         *
         * <DT><B> Post-conditions : </B></DT>
         * <DD>
         * The target method is executable.
         * <P/>@lang ja 対象のメソッドは実行可能である。
         * </DD>
         *
         * </DL>
         * @param instance Ungeneric instance to be casted.
         * <BR/>@lang ja キャストするアンジェネリックインスタンス。
         * @throws TypeViolationException
         * @throws NullPointerException
         */
        public Caster(Object instance)
            throws TypeViolationException, NullPointerException
        {
            if(instance == null){
                throw new NullPointerException(
                        "The instance is null. Revise specification.");
            }
            Type itype = instance.getClass();
            TypeVarScope iscope = ClassTypeVarScope
                    .getTypeScopeWithEnclosings(instance.getClass());
            Type rtype = Caster.class.getTypeParameters()[0];
            TypeVarScope rscope = ClassTypeVarScope
                    .getTypeScopeWithEnclosings(getClass());
            if(strategy.isAssignableFromTo(itype, iscope, rtype, rscope)){
                // pass
            }else{
                String dst = strategy
                        .getTypeDescriptionFor(rtype, rscope, true);
                throw new TypeViolationException(
                        "The instance cannot cast to ''{0}''. Revise specification.",
                        dst);
            }
            try{
                casterReturnValueField.set(this, instance);
            }catch(IllegalAccessException ex){
                throw new ReflectionError(ex,
                        "IllegalAccessException should not be thrown. Report as a bug.");
            }
        }

        /**
         * Gets caster instance
         * <BR/> キャストしたインスタンスを取得します。
         * .
         * <DL>
         * <DT><B> Pre-conditions : </B></DT>
         * <DD>
         * </DD>
         *
         * <DT><B> Post-conditions : </B></DT>
         * <DD>
         * </DD>
         *
         * </DL>
         * @return
         */
        public T cast(){
            return returnValue;
        }
    }

    /**
     * Generate the Param object from the type variable and ungeneric class
     * .
     * <P> @lang ja 型変数とアンジェネリッククラスからCasterオブジェクトを生成します。
     * </P>
     * <DL>
     * <DT><B> Pre-conditions : </B></DT>
     * <DD>
     * </DD>
     *
     * <DT><B> Post-conditions : </B></DT>
     * <DD>
     * </DD>
     *
     * </DL>
     * @return Param object.
     */
    /*
     private Param<?> createCaster(final Type type, final Class<?> ungeneric)
     throws TypeViolationException
     {
     if(type == null){
     throw new NullPointerException("type");
     }
     final TypeVarScope scope;
     if(ungeneric == null){
     scope = TypeVarScope.NULL;
     }else{
     scope = ExtractingTypeVarScope.makeMargedScope(type,
     new ClassTypeVarScope(ungeneric));
     }

     return new Param<Object>(null){
     @Override
     Type getType()
     throws TypeViolationException
     {
     return type;
     }

     @Override
     TypeVarScope getTypeVarScope()
     throws TypeViolationException
     {
     return scope;
     }
     };

     }
     */

    /**
     * Generate the Param object from the class
     * .
     * <P> @lang ja クラスからCasterオブジェクトを生成します。
     * </P>
     * <DL>
     * <DT><B> Pre-conditions : </B></DT>
     * <DD>
     * </DD>
     *
     * <DT><B> Post-conditions : </B></DT>
     * <DD>
     * </DD>
     *
     * </DL>
     * @return Param object.
     */
    /*
     private Param<?> createCaster(final Class<?> cls){
     return new Param<Object>(null){

     @Override
     Type getType()
     throws TypeViolationException
     {
     return cls;
     }

     @Override
     TypeVarScope getTypeVarScope()
     throws TypeViolationException
     {
     return TypeVarScope.NULL;
     }
     };
     }
     */

    /**
     * Resolve specified type variable and get the resolved type.
     * .
     * <P> @lang ja 型変数を解決し、解決した型を返します。
     * </P>
     * 
     * <DL>
     * <DT><B> Pre-conditions : </B></DT>
     * <DD>
     * </DD>
     *
     * <DT><B> Post-conditions : </B></DT>
     * <DD>
     * </DD>
     *
     * </DL>
     * @return The resolved type.
     * <P> @lang ja 解決された型。
     * </P>
     */
    /*
     public Type resolve(Param<?> from)
     throws TypeViolationException
     {
     return strategy.resolveTypeVariable(from.getType(), from
     .getTypeVarScope());
     }
     */

    /**
     * Judge whether substitution is possible
     * <BR/> Paramオブジェクトの保持する型同士で代入が可能か判定します。
     * .
     * 
     * <DL>
     * <DT><B> Pre-conditions : </B></DT>
     * <DD>
     * </DD>
     *
     * <DT><B> Post-conditions : </B></DT>
     * <DD>
     * </DD>
     *
     * </DL>
     * @return True when {@code from} can be substituted for {@code to}.
     * <P> @lang ja from をtoに代入可能なときtrue。
     * </P>
     */
    public boolean isAssignableFromTo(Param<?> from, Param<?> to)
        throws TypeViolationException
    {
        return strategy.isAssignableFromTo(from.getType(), from
                .getTypeVarScope(), to.getType(), to.getTypeVarScope());
    }

    /**
     * Return the string description of the specified type
     * <BR/> 指定された型のString表記を返します。
     * .
     * 
     * <DL>
     * <DT><B> Pre-conditions : </B></DT>
     * <DD>
     * </DD>
     *
     * <DT><B> Post-conditions : </B></DT>
     * <DD>
     * </DD>
     *
     * </DL>
     * @return 
     */
    public String getTypeDescriptionFor(Param<?> caster)
        throws TypeViolationException
    {
        return strategy.getTypeDescriptionFor(caster.getType(), caster
                .getTypeVarScope(), true);
    }

    private Method findMethod(Param<?> instance, String methodName,
            Param<?>... args)
        throws TypeViolationException, NoSuchMethodException, SecurityException
    {
        Class<?> cls = strategy.getRawClassOf(instance.getType(), instance
                .getTypeVarScope());
        Method[] ms = cls.getMethods();
        Class<?>[][] msp = new Class<?>[ms.length][];
        for(int i = ms.length; --i >= 0;){
            if(!ms[i].getName().equals(methodName)) continue;
            msp[i] = ms[i].getParameterTypes();
        }
        return findMostSpecificT(cls, methodName, args, ms, msp);
    }

    private Constructor findConstructor(Type type, TypeVarScope scope,
            Param<?>[] args)
        throws TypeViolationException, NoSuchMethodException, SecurityException
    {
        Class<?> rc = strategy.getRawClassOf(type, scope);
        Constructor[] ms = rc.getConstructors();
        Class<?>[][] msp = new Class<?>[ms.length][];
        for(int i = ms.length; --i >= 0;){
            msp[i] = ms[i].getParameterTypes();
        }
        return findMostSpecificT(rc, "<init>", args, ms, msp);
    }

    private <T> T findMostSpecificT(Class iclass, String methodName,
            Param<?>[] args, T[] ms, Class<?>[][] msp)
        throws TypeViolationException, NoSuchMethodException, SecurityException
    {
        Class<?>[] clss = new Class[args.length];
        for(int i = args.length; --i >= 0;){
            Type atype = args[i].getType();
            final TypeVarScope ascope = ClassTypeVarScope
                    .getTypeScopeWithEnclosings(args[i].getClass());
            clss[i] = strategy.getRawClassOf(atype, ascope);
        }

        T msm = null; // most specific method
        Class<?>[] mpts = null; // the parameter of most specific method
        for(int i = ms.length; --i >= 0;){
            T m = ms[i];
            Class<?>[] pts = msp[i];
            if(pts == null) continue;
            if(!isAssignable(pts, clss)) continue;
            if(msm == null || isAssignable(mpts, pts)){
                msm = m;
                mpts = pts;
            }else{
                throw new TypeViolationException(
                        "The methods/constructors ''{0}'' and ''{1}'' is matched. Fix the signature to invoke.",
                        msm, m);
            }
        }
        if(msm != null){
            return msm;
        }else{
            StringBuilder mes = new StringBuilder();
            mes.append("There is no suitable method to ");
            mes.append(iclass);
            mes.append(" ");
            mes.append(methodName);
            mes.append("(");
            for(Class c : clss){
                mes.append(c.getName());
            }
            mes.append(").");
            throw new NoSuchMethodException(mes.toString());
        }
    }

    private boolean isAssignable(Class<?>[] to, Class<?>[] from){
        if(to.length != from.length) return false;
        for(int i = to.length; --i >= 0;){
            if(strategy.isAssignableFromTo(from[i], to[i])){
                // match
            }else{
                return false;
            }
        }
        return true;
    }

}
