#!/usr/bin/perl
#------------------------------------------------------------------------------
#    59bbs, blog like bulletin board system.
#    Copyright (C) 2007-2010 Kaga, Hiroaki
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#------------------------------------------------------------------------------

use strict;
use warnings;

use Lib::Logger;
use Lib::Error;
use Lib::User;
use Lib::Ping;
use Lib::Filter;
use Lib::Ads;
use Lib::Topic;
use Lib::Archive;
use Lib::Search;
use Lib::Conf;
use Lib::Language;

my $logger = Lib::Logger->new();
my $error = Lib::Error->new();
my $conf = Lib::Conf->new();

sub act() {
    my ($act, $cgi) = @_;

    my $user = Lib::User->new();
    my $sessionid = $cgi->cookie('cookie59b');
    if ($sessionid eq '') {
        $error->print_error('00003e');
    }
    my $session_userid = $user->check_session($sessionid);
    if ($session_userid eq '') {
        $error->print_error('00003e');
    }

    if (!$user->is_admin($session_userid)) {
        $error->print_error('00004e');
    }

    if ($act eq 'exec_manage') {
        _exec_manage($cgi);
    }
    else {
        _view_manage($cgi);
    }
}

sub _view_manage() {
    my ($cgi) = @_;

    # func: rebuild, newuser, userlist, account, ads, template, ping, filter, htaccess
    my $func = $cgi->param('func');
    if ($func eq 'config') {
        _input_config($func);
    }
    elsif ($func eq 'newuser') {
        _input_user($func);
    }
    elsif ($func eq 'userlist') {
        _view_userlist($func, $cgi);
    }
    elsif ($func eq 'account') {
        _edit_user($func, $cgi);
    }
    elsif ($func eq 'ads') {
        _edit_ads($func);
    }
    elsif ($func eq 'template') {
        _edit_template($func);
    }
    elsif ($func eq 'ping') {
        _edit_ping($func);
    }
    elsif ($func eq 'filter') {
        _edit_filter($func);
    }
    elsif ($func eq 'htaccess') {
        _edit_htaccess($func);
    }
    else {
        _input_rebuild('rebuild');
    }
}


sub _exec_manage() {
    my ($cgi) = @_;

    # func: rebuild, newuser, account, ads, template,  ping, filter
    my $func = $cgi->param('func');
    if ($func eq '') {
        $error->print_error('00002e');
    }

    if ($func eq 'rebuild') {
        _exec_rebuild($cgi);
    }
    elsif ($func eq 'config') {
        _update_config($cgi);
    }
    elsif ($func eq 'newuser') {
        _add_user($cgi);
    }
    elsif ($func eq 'account') {
        _update_user($cgi);
    }
    elsif ($func eq 'ads') {
        _update_ads($cgi);
    }
    elsif ($func eq 'template') {
        _update_template($cgi);
    }
    elsif ($func eq 'ping') {
        _update_ping($cgi);
    }
    elsif ($func eq 'filter') {
        _update_filter($cgi);
    }
    elsif ($func eq 'htaccess') {
        _update_htaccess($cgi);
    }
}

sub _show_form {
    my ($func, $data, $msg, $q, $page) = @_;

    my $language = Lib::Language->new();
    my $caption = $language->get_statictext('manage') . ':';
    my $mainform = '';
    if ($func eq 'rebuild') {
        $caption .= $language->get_statictext('rebuild');
        $mainform = _rebuild_form($data);
    }
    elsif ($func eq 'config') {
        $caption .= $language->get_statictext('config');
        $mainform = _config_form($data);
    }
    elsif ($func eq 'newuser') {
        $caption .= $language->get_statictext('add-user');
        $mainform = _user_form($func, $data, '', '');
    }
    elsif ($func eq 'userlist') {
        $caption .= $language->get_statictext('manage-user');
        $mainform = _userlist_form($q, $page);
    }
    elsif ($func eq 'account') {
        $caption .= $language->get_statictext('edit-account');
        $mainform = _user_form($func, $data, $q, $page);
    }
    elsif ($func eq 'ads') {
        $caption .= $language->get_statictext('manage-ad');
        $mainform = _ads_form();
    }
    elsif ($func eq 'template') {
        $caption .= $language->get_statictext('edit-template');
        $mainform = _template_form();
    }
    elsif ($func eq 'ping') {
        $caption .= $language->get_statictext('set-ping');
        $mainform = _ping_form();
    }
    elsif ($func eq 'filter') {
        $caption .= $language->get_statictext('set-filter');
        $mainform = _filter_form();
    }
    elsif ($func eq 'htaccess') {
        $caption .= $language->get_statictext('edit-htaccess');
        $mainform = _htaccess_form();
    }

    my $docrootdir = $conf->get_docroot_dir();
    my $sitetitle = $conf->get_site_title();
    my $subtitle = $conf->get_sub_title();

    my $homelabel = $language->get_statictext('home');

    my $path = <<"END_PATH";
<a class="path" href="./">$homelabel</a>&nbsp;
&gt;&nbsp;
$caption
END_PATH

    if ($msg ne '') {
        $msg = "<p><div id=\"message\">$msg</div></p>";
    }

    my $systemdir = $conf->get_system_dir();

    my $config = $language->get_statictext('config');
    my $rebuild = $language->get_statictext('rebuild');
    my $manageuser = $language->get_statictext('manage-user');
    my $adduser = $language->get_statictext('add-user');
    my $managead = $language->get_statictext('manage-ad');
    my $edittemplate = $language->get_statictext('edit-template');
    my $setping = $language->get_statictext('set-ping');
    my $setfilter = $language->get_statictext('set-filter');
    my $edithtaccess = $language->get_statictext('edit-htaccess');

    open my $templatefh, '<', "$systemdir/tmpl/manage.tmpl";
    my $template = do { local $/; <$templatefh> };
    close $templatefh;

    $template =~ s/\$CAPTION\$/$caption/g;
    $template =~ s/\$PATH\$/$path/g;
    $template =~ s/\$SITETITLE\$/$sitetitle/g;
    $template =~ s/\$SUBTITLE\$/$subtitle/g;
    $template =~ s/\$MSG\$/$msg/g;
    $template =~ s/\$CONFIG\$/$config/g;
    $template =~ s/\$REBUILD\$/$rebuild/g;
    $template =~ s/\$MANAGEUSER\$/$manageuser/g;
    $template =~ s/\$ADDUSER\$/$adduser/g;
    $template =~ s/\$MANAGEAD\$/$managead/g;
    $template =~ s/\$EDITTEMPLATE\$/$edittemplate/g;
    $template =~ s/\$SETPING\$/$setping/g;
    $template =~ s/\$SETFILTER\$/$setfilter/g;
    $template =~ s/\$EDITHTACCESS\$/$edithtaccess/g;
    $template =~ s/\$MAINFORM\$/$mainform/g;

    print "Content-Type: text/html\n\n";
    print $template;

    exit;
}

sub _view_userlist() {
    my ($func, $cgi) = @_;

    my $page = $cgi->param('page');
    $page = ($page eq '') ? 1 : $page;
    my $q = $cgi->param('q');

    my $data;
    _show_form($func, $data, '', $q, $page);
}

sub _userlist_form() {
    my ($q, $page) = @_;

    my $pattern = '';
    if ($q ne '') {
        $pattern = "$q.*";
    }

    my $systemdir = $conf->get_system_dir();

    my @result = ();
    opendir my $dirh, "$systemdir/user/";
    while (defined(my $dir = readdir $dirh)) {
        next if ($dir =~ /\A[.]+\z/);
        next if ($pattern ne '' && $dir !~ /\A$pattern\z/);
        push @result, $dir;
    }
    closedir $dirh;

    my $language = Lib::Language->new();
    my $userlabel = $language->get_statictext('user');

    @result = sort @result;
    my $usernum = scalar(@result);
    my $usercount = "$usernum $userlabel";

    my $prevlist = $language->get_statictext('prev-list');
    my $nextlist = $language->get_statictext('next-list');

    my $resultlist = '';
    my $pagenavi = '';
    if (scalar(@result)) {
        my $itemnum = $conf->get_result_count();
        my $pagenum = int(scalar(@result) / $itemnum); 
        if (scalar(@result) / $itemnum > $pagenum) {
            $pagenum++;
        }

        my @items = ();
        for my $i (0 .. $#result) {
            if (($i >= ($page - 1) * $itemnum) && ($i < $page * $itemnum)) {
                push @items, $result[$i];
            }
        }

        $resultlist = _get_userlist($q, $page, @items);

        my $base = 1;
        my $currentnum = ($pagenum < 5) ? $pagenum : 5;
        if ($page > 5) {
            $base = $page - 4;
        }
        my @pagelist = ();
        for my $pagecount ($base .. ($base + $currentnum - 1)) {
            push @pagelist, $pagecount;
        }
        my $current = '';
        foreach my $pagecount (@pagelist) {
            if ($pagecount != $page) {
                $current .= "<a href=\"./?act=view_manage&func=userlist&q=$q&page=$pagecount\">$pagecount</a>&nbsp;";
            }
            else {
                $current .= "$pagecount&nbsp;";
            }
        }
        my $prev = '';
        if ($page > 1) {
            my $prevpage = $page - 1;
            $prev = "<a href=\"./?act=view_manage&func=userlist&q=$q&page=$prevpage\">$prevlist</a>";
        }
        my $next = '';
        if ($page < $pagenum) {
            my $nextpage = $page + 1;
            $next = "<a href=\"./?act=view_manage&func=userlist&q=$q&page=$nextpage\">$nextlist</a>";
        }
        $pagenavi = <<"END_PAGENAVI";
<strong>$prev&nbsp;&nbsp;$current&nbsp;$next</strong>
END_PAGENAVI
    }

    my $button = $language->get_statictext('search');
    my $userid = $language->get_statictext('userid');
    my $mailaddr = $language->get_statictext('mail-address');
    my $status = $language->get_statictext('status');
    my $auth = $language->get_statictext('auth');

    my $form = <<"END_FORM";
<form method="post" action="./">
  <input type="hidden" name="act" value="view_manage">
  <input type="hidden" name="func" value="userlist">
  <input type="text" name="q" size="60" value="$q">&nbsp;
  <input type="submit" value="$button">
</form>

<table border="0" width="640" cellpadding="0" cellspacing="0">
<tr>
  <td valign="top" align="right" colspan="5">$usercount</td>
</tr>
<tr>
  <td><b>$userid</b></td>
  <td width="200"><b>$mailaddr</b></td>
  <td width="60" align="center"><b>$auth</b></td>
  <td width="100" align="center"><b>$status</b></td>
  <td width="40" align="center">&nbsp;</td>
</tr>
$resultlist
<tr>
  <td valign="top" align="center" colspan="5">$pagenavi</td>
</tr>
</table>
END_FORM

    return $form;
}

sub _get_userlist() {
    my $q = shift;
    my $page = shift;
    my (@useridlist) = @_;

    my $language = Lib::Language->new();
    my $valid = $language->get_statictext('valid');
    my $invalid = $language->get_statictext('invalid');
    my $privilege = $language->get_statictext('privilege');
    my $general = $language->get_statictext('general');
    my $editlink = $language->get_statictext('edit');

    my $user = Lib::User->new();
    my $list = '';
    foreach my $userid (@useridlist) {
        $user->load($userid);
        my $mailaddr = $user->get_mailaddr();
        my $status = $user->get_status();
        my $statusname = $valid;
        if ($status == 0) {
            $statusname = $invalid;
        }
        my $auth     = $user->get_auth();
        my $authname = $general;
        if ($auth == 0) {
            $authname = $privilege;
        }

        $list .= <<"END_ROW";
<tr>
  <td nowrap >$userid</td>
  <td width="200" nowrap>$mailaddr</td>
  <td width="60" nowrap align="center">$authname</td>
  <td width="100" nowrap align="center">$statusname</td>
  <td width="40" nowrap align="center"><a href="./?act=view_manage&func=account&userid=$userid&q=$q&page=$page">$editlink</a></td>
</tr>
END_ROW
    }

    return $list;
}

sub _input_user() {
    my ($func) = @_;

    my $userdata;
    $userdata->{userid}   = '';
    $userdata->{password} = '';
    $userdata->{mailaddr} = '';
    $userdata->{hpurl}    = '';
    $userdata->{username} = '';
    $userdata->{auth}     = 1;
    $userdata->{status}   = 1;
    $userdata->{ad1}      = '';
    $userdata->{ad2}      = '';
    $userdata->{ad3}      = '';
    $userdata->{parts}    = '';

    _show_form($func, $userdata, '', '', '');
}

sub _add_user() {
    my ($cgi) = @_;

    my $userdata;
    $userdata->{userid}   = $cgi->param('userid');
    $userdata->{password} = $cgi->param('password');
    $userdata->{mailaddr} = $cgi->param('mailaddr');
    $userdata->{hpurl}    = $cgi->param('hpurl');
    $userdata->{username} = $cgi->param('username');
    $userdata->{auth}     = $cgi->param('auth');
    $userdata->{status}   = $cgi->param('status');
    $userdata->{ad1}      = $cgi->param('ad1');
    $userdata->{ad2}      = $cgi->param('ad2');
    $userdata->{ad3}      = $cgi->param('ad3');
    $userdata->{parts}    = $cgi->param('parts');

    # 入力チェック
    _check_user('newuser', $userdata);

    my $user = Lib::User->new();
    $user->set_userid($userdata->{userid});
    $user->set_password($userdata->{password});
    $user->set_mailaddr($userdata->{mailaddr});
    $user->set_hpurl($userdata->{hpurl});
    $user->set_username($userdata->{username});
    $user->set_auth($userdata->{auth});
    $user->set_status($userdata->{status});
    $user->set_ad1($userdata->{ad1});
    $user->set_ad2($userdata->{ad2});
    $user->set_ad3($userdata->{ad3});
    $user->set_parts($userdata->{parts});
    $user->add();

    print "Location: ./?act=view_manage&func=newuser", "\n\n";
}

sub _edit_user() {
    my ($func, $cgi) = @_;

    my $userid = $cgi->param('userid');
    if ($userid eq '') {
        $error->print_error('00002e');
    }

    my $q = $cgi->param('q');
    my $page = $cgi->param('page');

    my $user = Lib::User->new();
    $user->load($userid);

    my $userdata;
    $userdata->{userid}   = $userid;
    $userdata->{password} = '********';
    $userdata->{mailaddr} = $user->get_mailaddr();
    $userdata->{hpurl}    = $user->get_hpurl();
    $userdata->{username} = $user->get_username();
    $userdata->{auth}     = $user->get_auth();
    $userdata->{status}   = $user->get_status();
    $userdata->{ad1}      = $user->get_ad1();
    $userdata->{ad2}      = $user->get_ad2();
    $userdata->{ad3}      = $user->get_ad3();
    $userdata->{parts}    = $user->get_parts();

    _show_form($func, $userdata, '', $q, $page);
}

sub _update_user() {
    my ($cgi) = @_;

    my $q = $cgi->param('q');
    my $page = $cgi->param('page');

    my $userdata;
    $userdata->{userid}   = $cgi->param('userid');
    $userdata->{password} = $cgi->param('password');
    $userdata->{mailaddr} = $cgi->param('mailaddr');
    $userdata->{hpurl}    = $cgi->param('hpurl');
    $userdata->{username} = $cgi->param('username');
    $userdata->{auth}     = $cgi->param('auth');
    $userdata->{status}   = $cgi->param('status');
    $userdata->{ad1}      = $cgi->param('ad1');
    $userdata->{ad2}      = $cgi->param('ad2');
    $userdata->{ad3}      = $cgi->param('ad3');
    $userdata->{parts}    = $cgi->param('parts');

    # 入力チェック
    _check_user('account', $userdata);

    my $user = Lib::User->new();
    $user->load($userdata->{userid});

    $user->set_mailaddr($userdata->{mailaddr});
    $user->set_hpurl($userdata->{hpurl});
    $user->set_username($userdata->{username});
    $user->set_auth($userdata->{auth});
    $user->set_status($userdata->{status});
    $user->set_ad1($userdata->{ad1});
    $user->set_ad2($userdata->{ad2});
    $user->set_ad3($userdata->{ad3});
    $user->set_parts($userdata->{parts});
    $user->update();

    print "Location: ./?act=view_manage&func=userlist&q=$q&page=$page", "\n\n";
}

sub _check_user() {
    my ($func, $userdata, $q, $page) = @_;

    my $user = Lib::User->new();
    my $language = Lib::Language->new();

    if ($userdata->{userid} eq '') {
        _show_error($func, $userdata, $language->get_usererror('input-userid'), $q, $page);
    }
    if ($userdata->{userid} =~ /[^0-9a-zA-Z]/) {
        _show_error($func, $userdata, $language->get_usererror('userid-format'), $q, $page);
    }

    if ($func eq 'newuser') {
        # ユーザーIDの重複チェック
        if (!$user->check_valid($userdata->{userid})) {
            _show_error($func, $userdata, $language->get_usererror('user-exist'), $q, $page);
        }
        # パスワード
        if ($userdata->{password} eq '') {
            _show_error($func, $userdata, $language->get_usererror('input-password'), $q, $page);
        }
        # パスワードの長さをチェック
        my $strlen = length($userdata->{password});
        if ($strlen < 4 || $strlen > 9) {
            _show_error($func, $userdata, $language->get_usererror('password-length'), $q, $page);
        }
        # パスワードが英数字かチェック
        if ($userdata->{password} =~ /[^0-9a-zA-Z]/) {
            _show_error($func, $userdata, $language->get_usererror('password-format'), $q, $page);
        }
    }

    if ($userdata->{mailaddr} eq '') {
        _show_error($func, $userdata, $language->get_usererror('input-mailaddr'), $q, $page);
    }
    if ($userdata->{mailaddr} !~ /[-_0-9a-z]+\@[-_0-9a-z]+\.[-_0-9a-z.]+/) {
        _show_error($func, $userdata, $language->get_usererror('mailaddr-format'), $q, $page);
    }

    if ($userdata->{username} eq '') {
        _show_error($func, $userdata, $language->get_usererror('input-name'), $q, $page);
    }

    if ($userdata->{auth} eq '') {
        _show_error($func, $userdata, $language->get_usererror('select-auth'), $q, $page);
    }

    if ($userdata->{status} eq '') {
        _show_error($func, $userdata, $language->get_usererror('select-status'), $q, $page);
    }
}

sub _input_config() {
    my ($func) = @_;

    my $confdata;
    $confdata->{site_title}    = $conf->get_site_title();
    $confdata->{sub_title}     = $conf->get_sub_title();
    $confdata->{admin_address} = $conf->get_admin_address();
    $confdata->{anonymous}     = $conf->get_anonymous();
    $confdata->{result_count}  = $conf->get_result_count();
    $confdata->{language}      = $conf->get_language();
    $confdata->{static_html}   = $conf->get_static_html();
    $confdata->{free_signup}   = $conf->get_free_signup();

    _show_form($func, $confdata, '', '', '');
}

sub _update_config() {
    my ($cgi) = @_;
#    $logger->write("_update_config : start");  # for debug

    my $confdata;
    $confdata->{site_title}    = $cgi->param('sitetitle');
    $confdata->{sub_title}     = $cgi->param('subtitle');
    $confdata->{admin_address} = $cgi->param('adminaddress');
    $confdata->{anonymous}     = $cgi->param('anonymous');
    $confdata->{result_count}  = $cgi->param('resultcount');
    $confdata->{language}      = $cgi->param('language');
    $confdata->{static_html}    = ($cgi->param('statichtml') eq '') ? 0 : 1;
    $confdata->{free_signup}    = ($cgi->param('freesignup') eq '') ? 0 : 1 ;

    my $language = Lib::Language->new();
    if (!$confdata->{result_count}) {
        _show_error('config', $confdata, $language->get_usererror('input-listitem'));
    }
    if ($confdata->{result_count} =~ /\A[^0-9]+\z/) {
        _show_error('config', $confdata, $language->get_usererror('listitem-format'));
    }

    # config.cgi を更新
    $conf->set_site_title($confdata->{site_title});
    $conf->set_sub_title($confdata->{sub_title});
    $conf->set_admin_address($confdata->{admin_address});
    $conf->set_anonymous($confdata->{anonymous});
    $conf->set_result_count($confdata->{result_count});
    $conf->set_language($confdata->{language});
    $conf->set_static_html($confdata->{static_html});
    $conf->set_free_signup($confdata->{free_signup});
    $conf->update_conf();

    print "Location: ./?act=view_manage&func=config", "\n\n";
}

sub _input_rebuild() {
    my ($func) = @_;

    my $optiondata;
    $optiondata->{htmlcb}     = 0;
    $optiondata->{indexcb}    = 0;
    $optiondata->{rsscb}      = 0;
    $optiondata->{archivecb}  = 0;
    $optiondata->{sitemapcb}  = 0;
    $optiondata->{htmlstart}  = '';
    $optiondata->{htmlend}    = '';
    $optiondata->{indexstart} = '';
    $optiondata->{indexend}   = '';

    _show_form($func, $optiondata, '');
}

sub _exec_rebuild() {
    my ($cgi) = @_;

    my $optiondata;
    $optiondata->{htmlcb}     = ($cgi->param('htmlcb') eq '') ? 0 : 1;
    $optiondata->{htmlstart}  = $cgi->param('htmlstart');
    $optiondata->{htmlend}    = $cgi->param('htmlend');
    $optiondata->{indexcb}    = ($cgi->param('indexcb') eq '') ? 0 : 1;
    $optiondata->{indexstart} = $cgi->param('indexstart');
    $optiondata->{indexend}   = $cgi->param('indexend');
    $optiondata->{rsscb}      = ($cgi->param('rsscb') eq '') ? 0 : 1;
    $optiondata->{archivecb}  = ($cgi->param('archivecb') eq '') ? 0 : 1;
    $optiondata->{sitemapcb}  = ($cgi->param('sitemapcb') eq '') ? 0 : 1;

    my $language = Lib::Language->new();

    # 入力チェック
    if ($optiondata->{htmlstart} ne '') {
        if ($optiondata->{htmlstart} =~ /\A[^0-9]+\z/) {
            _show_error('rebuild', $optiondata, $language->get_usererror('start-id'));
        }
    }
    if ($optiondata->{htmlend} ne '') {
        if ($optiondata->{htmlend} =~ /\A[^0-9]+\z/) {
            _show_error('rebuild', $optiondata, $language->get_usererror('end-id'));
        }
    }
    if ($optiondata->{indexstart} ne '') {
        if ($optiondata->{indexstart} =~ /\A[^0-9]+\z/) {
            _show_error('rebuild', $optiondata, $language->get_usererror('start-id'));
        }
    }
    if ($optiondata->{indexend} ne '') {
        if ($optiondata->{indexend} =~ /\A[^0-9]+\z/) {
            _show_error('rebuild', $optiondata, $language->get_usererror('end-id'));
        }
    }

    if ($optiondata->{htmlcb}) {
        _output_html($optiondata->{htmlstart}, $optiondata->{htmlend});
    }

    if ($optiondata->{indexcb}) {
        _output_index($optiondata->{indexstart}, $optiondata->{indexend});
    }

    if ($optiondata->{rsscb}) {
        _output_rss();
    }

    if ($optiondata->{archivecb}) {
        _output_archive();
    }

    if ($optiondata->{sitemapcb}) {
        _output_sitemap();
    }

    print "Location: ./?act=view_manage&func=rebuild", "\n\n";
}

sub _user_form() {
    my ($func, $userdata, $q, $page) = @_;

    my $language = Lib::Language->new();

    my $readonly = '';
    my $button = $language->get_statictext('add');
    if ($func eq 'account') {
        $readonly = 'readonly';
        $button = $language->get_statictext('update');
    }

    my $privilege = $language->get_statictext('privilege');
    my $general = $language->get_statictext('general');
    my $valid = $language->get_statictext('valid');
    my $invalid = $language->get_statictext('invalid');

    my $authlist = '';
    if ($userdata->{auth} == 0) {
        $authlist = <<"END_AUTH";
<select name="auth">
<option selected value="0">$privilege
<option value="1">$general
</select>
END_AUTH
    }
    else {
        $authlist = <<"END_AUTH";
<select name="auth">
<option value="0">$privilege
<option selected value="1">$general
</select>
END_AUTH
    }

    my $statuslist = '';
    if ($userdata->{status} == 1) {
        $statuslist = <<"END_STATUS";
<select name="status">
<option selected value="1">$valid
<option value="0">$invalid
</select>
END_STATUS
    }
    else {
        $statuslist = <<"END_STATUS";
<select name="status">
<option value="1">$valid
<option selected value="0">$invalid
</select>
END_STATUS
    }

    my $userid = $language->get_statictext('userid');
    my $password = $language->get_statictext('password');
    my $mailaddr = $language->get_statictext('mail-address');
    my $homepageurl = $language->get_statictext('homepage-url');
    my $username = $language->get_statictext('user-name');
    my $auth = $language->get_statictext('auth');
    my $status = $language->get_statictext('status');
    my $adcode1 = $language->get_statictext('ad-code1');
    my $adcode2 = $language->get_statictext('ad-code2');
    my $adcode3 = $language->get_statictext('ad-code3');
    my $blogparts = $language->get_statictext('blogparts');

    my $form = <<"END_FORM";
<form method="post" action="./">
<input type="hidden" name="act" value="exec_manage">
<input type="hidden" name="func" value="$func">
<input type="hidden" name="q" value="$q">
<input type="hidden" name="page" value="$page">
$userid :<br />
<input type="text" name="userid" size="50" value="$userdata->{userid}" $readonly><br />
$password :<br />
<input type="password" name="password" size="50" value="$userdata->{password}" $readonly><br />
$mailaddr :<br />
<input type="text" name="mailaddr" size="90" value="$userdata->{mailaddr}"><br />
$homepageurl :<br />
<input type="text" name="hpurl" size="90" value="$userdata->{hpurl}"><br />
$username :<br />
<input type="text" name="username" size="90" value="$userdata->{username}"><br />
$auth :<br />
$authlist<br />
$status :<br />
$statuslist<br />
$adcode1 :<br />
<textarea name="ad1" rows="3" cols="75">$userdata->{ad1}</textarea><br />
$adcode2 :<br />
<textarea name="ad2" rows="3" cols="75">$userdata->{ad2}</textarea><br />
$adcode3 :<br />
<textarea name="ad3" rows="3" cols="75">$userdata->{ad3}</textarea><br />
$blogparts :<br />
<textarea name="parts" rows="3" cols="75">$userdata->{parts}</textarea><br />
<p>
<input type="submit" value="$button">
</p>
</form>
END_FORM

    return $form;
}

sub _config_form() {
    my ($confdata) = @_;

    my $language = Lib::Language->new();
    my $sitetitle = $language->get_statictext('site-title');
    my $subtitle = $language->get_statictext('sub-title');
    my $adminaddress = $language->get_statictext('admin-address');
    my $anonymous = $language->get_statictext('anonymous');
    my $listitemnum = $language->get_statictext('list-itemnum');
    my $languagecode = $language->get_statictext('language-code');
    my $statichtml = $language->get_statictext('static-html');
    my $freesignup = $language->get_statictext('free-signup');
    my $updatebutton = $language->get_statictext('update');

    my $statichtmlcb = <<"END_CHECKBOX";
<input type="checkbox" name="statichtml" value="1">$statichtml
END_CHECKBOX
    if ($confdata->{static_html}) {
        $statichtmlcb = <<"END_CHECKBOX";
<input type="checkbox" name="statichtml" value="1" checked>$statichtml
END_CHECKBOX
    }

    my $freesignupcb = <<"END_CHECKBOX";
<input type="checkbox" name="freesignup" value="1">$freesignup
END_CHECKBOX
    if ($confdata->{free_signup}) {
        $freesignupcb = <<"END_CHECKBOX";
<input type="checkbox" name="freesignup" value="1" checked>$freesignup
END_CHECKBOX
    }

    my $form = <<"END_FORM";
<form method="post" action="./">
<input type="hidden" name="act" value="exec_manage">
<input type="hidden" name="func" value="config">
$sitetitle :<br />
<input type="text" name="sitetitle" size="100" value="$confdata->{site_title}"><br />
$subtitle :<br />
<textarea name="subtitle" rows="2" cols="75">$confdata->{sub_title}</textarea><br />
$adminaddress :<br />
<input type="text" name="adminaddress" size="100" value="$confdata->{admin_address}"><br />
$anonymous :<br />
<input type="text" name="anonymous" size="100" value="$confdata->{anonymous}"><br />
$listitemnum :<br />
<input type="text" name="resultcount" size="20" value="$confdata->{result_count}"><br />
$languagecode :<br />
<input type="text" name="language" size="20" value="$confdata->{language}"><br />
<br />
$statichtmlcb &nbsp;&nbsp;$freesignupcb<br />
<p>
<input type="submit" value="$updatebutton">
</p>
</form>
END_FORM

    return $form;
}

sub _rebuild_form() {
    my ($optiondata) = @_;

    my $language = Lib::Language->new();
    my $htmlfile = $language->get_statictext('html-file');
    my $searchindex = $language->get_statictext('search-index');
    my $rssfeed = $language->get_statictext('rss-feed');
    my $archivelog = $language->get_statictext('archive-log');
    my $sitemap = $language->get_statictext('sitemap');
    my $button = $language->get_statictext('rebuild');

    my $checked = '';
    if ($optiondata->{htmlcb}) {
        $checked = 'checked';
    }
    my $htmlcb = <<"END_OPTION";
<input type="checkbox" name="htmlcb" value="1" $checked>$htmlfile
END_OPTION

    $checked = '';
    if ($optiondata->{indexcb}) {
        $checked = 'checked';
    }
    my $indexcb = <<"END_OPTION";
<input type="checkbox" name="indexcb" value="1" $checked>$searchindex
END_OPTION

    $checked = '';
    if ($optiondata->{rsscb}) {
        $checked = 'checked';
    }
    my $rsscb = <<"END_OPTION";
<input type="checkbox" name="rsscb" value="1" $checked>$rssfeed
END_OPTION

    $checked = '';
    if ($optiondata->{archivecb}) {
        $checked = 'checked';
    }
    my $archivecb = <<"END_OPTION";
<input type="checkbox" name="archivecb" value="1" $checked>$archivelog
END_OPTION

    $checked = '';
    if ($optiondata->{sitemapcb}) {
        $checked = 'checked';
    }
    my $sitemapcb = <<"END_OPTION";
<input type="checkbox" name="sitemapcb" value="1" $checked>$sitemap
END_OPTION

    my $form = <<"END_FORM";
<table border="0" width="600" cellpadding="0" cellspacing="0">
<form name="hideform" method="post" action="./">
<input type="hidden" name="act" value="exec_manage">
<input type="hidden" name="func" value="rebuild">
<tr>
  <td width="200">$htmlcb</td>
  <td>ID : <input type="text" name="htmlstart" size="10" value="$optiondata->{htmlstart}">&nbsp; - &nbsp;<input type="text" name="htmlend" size="10" value="$optiondata->{htmlend}"></td>
</tr>
<tr>
  <td width="200">$indexcb</td>
  <td>ID : <input type="text" name="indexstart" size="10" value="$optiondata->{indexstart}">&nbsp; - &nbsp;<input type="text" name="indexend" size="10" value="$optiondata->{indexend}"></td>
</tr>
<tr>
  <td width="200">$rsscb</td>
  <td>&nbsp;</td>
</tr>
<tr>
  <td width="200">$archivecb</td>
  <td>&nbsp;</td>
</tr>
<tr>
  <td width="200">$sitemapcb</td>
  <td>&nbsp;</td>
</tr>
<tr>
  <td><br /><input type="submit" value="$button"></td>
  <td>&nbsp;</td>
</tr>
</form>
</table>
<br />
END_FORM

    return $form;
}

sub _output_html {
    my $startpos = shift;
    if ($startpos eq '') {
        $startpos = 0;
    }
    my $endpos = 0;
    if (@_) {
        $endpos = shift;
    }

    my $static_html = $conf->get_static_html();

    my $archive = Lib::Archive->new();
    my $counter = $archive->get_counter();

    my $topicnum = $counter;
    if ($topicnum < $endpos || $endpos == 0) {
        $endpos = $topicnum;
    }

    my $topic = Lib::Topic->new();
    for my $topicid ($startpos .. $endpos) {
        my $ret = $topic->load($topicid, 1);
        next if (!$ret);
        my $status = $topic->get_status();
        my $htmlfile = "detail" . "$topicid.html";
        if ($status == 1 || $status == 2 || $status == 3) {
            if ($static_html) {
                $topic->output_html();
            }
            else {
                unlink $htmlfile;
            }
        }
        else {
            if ($static_html) {
                unlink $htmlfile;
            }
        }
    }
}

sub _output_index {
    my $startpos = shift;
    if ($startpos eq '') {
        $startpos = 0;
    }
    my $endpos = 0;
    if (@_) {
        $endpos = shift;
    }

    my $archive = Lib::Archive->new();
    my $counter = $archive->get_counter();

    my $topicnum = $counter;
    if ($topicnum < $endpos || $endpos == 0) {
        $endpos = $topicnum;
    }

    my $datadir = $conf->get_data_dir();

    if (($startpos == 0) && ($endpos == $counter)) {
        my $dir = "$datadir/search/*.data";
        my @files = <${dir}>;
        foreach my $fname (@files) {
            unlink $fname;
        }
    }

    my $search = Lib::Search->new();
    my $topic = Lib::Topic->new();
    for my $topicid ($startpos .. $endpos) {
        my $ret = $topic->load($topicid, 0);
        next if (!$ret);
        my $title = $topic->get_title();
        my $text = $topic->get_text();
        my $editor = $topic->get_editor();
        my $status = $topic->get_status();
        my $keyword = $topic->get_keyword();
        my $str = "$title $text $editor $keyword";
        if ($status != 9) {
            $search->addindex($topicid, $str);
        }
    }
}

sub _output_rss {
    my $archive = Lib::Archive->new();
    $archive->outputrss2();
}

sub _output_archive {
    my $datadir = $conf->get_data_dir();

    my $topic = Lib::Topic->new();

    my $dir = "$datadir/*ar.data";
    my @files = <${dir}>;
    foreach my $fname (@files) {
        unlink $fname;
    }

    my $archive = Lib::Archive->new();
    my $counter = $archive->get_counter();

    for my $topicid (0 .. $counter) {
        my $ret = $topic->load($topicid, 0);
        next if (!$ret);
        my $lastdate = $topic->get_lastdate();
        my $status = $topic->get_status();
        if ($status == 1 || $status == 2 || $status == 3) {
            $archive->add_archive($topicid, $lastdate);
        }
    }
}

sub _output_sitemap {
    my $docrootdir = $conf->get_docroot_dir();
    my $systemdir = $conf->get_system_dir();
    my $datadir = $conf->get_data_dir();
    my $static_html = $conf->get_static_html();

    my $archive = Lib::Archive->new();
    my $counter = $archive->get_counter();

    my $topic = Lib::Topic->new();

    my $urllist = <<"END_ITEM";
    <url>
    <loc>$docrootdir/</loc>
    <changefreq>daily</changefreq>
    <priority>0.9</priority>
    </url>
END_ITEM

    for my $topicid (0 .. $counter) {
        my $ret = $topic->load($topicid, 0);
        next if (!$ret);
        my $status = $topic->get_status();
        if ($status == 1 || $status == 2 || $status == 3) {
            my $htmlurl = "$docrootdir/?func=detail&amp;id=$topicid";
            if ($static_html) {
                $htmlurl = "$docrootdir/detail" . "$topicid.html";
            }
            $urllist .= <<"END_ITEM";
   <url>
    <loc>$htmlurl</loc>
    <changefreq>daily</changefreq>
    <priority>0.8</priority>
   </url>
END_ITEM
        }
    }

    open my $templatefh, '<', "$systemdir/tmpl/sitemap.tmpl";
    my $template = do { local $/; <$templatefh> };
    close $templatefh;

    $template =~ s/\$URLLIST\$/$urllist/g;

    my $sitemapfile = "sitemap.xml";
    open my $sitemapfh, '>', $sitemapfile;
    print {$sitemapfh} $template;
    close $sitemapfh;

    chmod 0766, $sitemapfile;
}

sub _edit_ads() {
    my ($func) = @_;
    my $data;
    _show_form($func, $data, '');
}

sub _ads_form() {
    my $ads = Lib::Ads->new();
    my $ad1 = $ads->get_ad1();
    my $ad2 = $ads->get_ad2();
    my $ad3 = $ads->get_ad3();
    my $adlist = $ads->get_adlist();

    my $language = Lib::Language->new();
    my $adcode1 = $language->get_statictext('ad-code1');
    my $adcode2 = $language->get_statictext('ad-code2');
    my $adcode3 = $language->get_statictext('ad-code3');
    my $notice = $language->get_statictext('notice');
    my $button = $language->get_statictext('update');

    my $form = <<"END_FORM";
<form method="post" action="./">
<input type="hidden" name="act" value="exec_manage">
<input type="hidden" name="func" value="ads">
$adcode1 :<br />
<textarea name="ad1" rows="3" cols="75">$ad1</textarea><br />
$adcode2 :<br />
<textarea name="ad2" rows="3" cols="75">$ad2</textarea><br />
$adcode3 :<br />
<textarea name="ad3" rows="3" cols="75">$ad3</textarea><br />
$notice :<br />
<textarea name="adlist" rows="12" cols="75">$adlist</textarea><br />
<p>
<input type="submit" value="$button">
</p>
</form>
END_FORM

    return $form;
}

sub _update_ads() {
    my ($cgi) = @_;

    my $ad1 = $cgi->param('ad1');
    my $ad2 = $cgi->param('ad2');
    my $ad3 = $cgi->param('ad3');
    my $adlist = $cgi->param('adlist');

    my $ads = Lib::Ads->new();
    $ads->set_ad1($ad1);
    $ads->set_ad2($ad2);
    $ads->set_ad3($ad3);
    $ads->set_adlist($adlist);
    $ads->update();

    print "Location: ./?act=view_manage&func=ads", "\n\n";
}

sub _edit_template() {
    my ($func) = @_;
    my $data;
    _show_form($func, $data, '');
}

sub _template_form() {
    my $systemdir = $conf->get_system_dir();
    my $filepath = "$systemdir/tmpl/main.tmpl";

    open my $datafh, '<', $filepath;
    my $filedata = do { local $/; <$datafh> };
    close $datafh;

    my $language = Lib::Language->new();
    my $templatefile = $language->get_statictext('template-file');
    my $button = $language->get_statictext('save');

    my $form = <<"END_FORM";
$templatefile :
<form method="post" action="./">
<input type="hidden" name="act" value="exec_manage">
<input type="hidden" name="func" value="template">
<textarea name="filedata" rows="20" cols="75">$filedata</textarea><br />
<p>
<input type="submit" value="$button">
</p>
</form>
END_FORM

    return $form;
}

sub _update_template() {
    my ($cgi) = @_;

    my $filedata = $cgi->param('filedata');
    if ($filedata eq '') {
        my $language = Lib::Language->new();
        _show_error('template', $filedata, $language->get_usererror('input-template'));
    }

    my $systemdir = $conf->get_system_dir();
    my $filepath = "$systemdir/tmpl/main.tmpl";
    $filedata =~ s/\r?\n/\n/g;
    open my $datafh, '>', $filepath;
    print {$datafh} $filedata;
    close $datafh;

    print "Location: ./?act=view_manage&func=template", "\n\n";
}

sub _edit_ping() {
    my ($func) = @_;
    my $data;
    _show_form($func, $data, '');
}

sub _ping_form() {
    my $ping = Lib::Ping->new();
    $ping->load();
    my @items = $ping->get_pinglist();

    my $itemlist = '';
    foreach my $item (@items) {
        $itemlist .= "$item\n";
    }

    my $language = Lib::Language->new();
    my $pinglist = $language->get_statictext('ping-list');
    my $button = $language->get_statictext('update');

    my $form = <<"END_FORM";
$pinglist :
<form method="post" action="./">
<input type="hidden" name="act" value="exec_manage">
<input type="hidden" name="func" value="ping">
<textarea name="itemlist" rows="12" cols="75">$itemlist</textarea><br />
<p>
<input type="submit" value="$button">
</p>
</form>
END_FORM

    return $form;
}

sub _update_ping() {
    my ($cgi) = @_;

    my $itemlist = $cgi->param('itemlist');
    $itemlist =~ s/\r?\n/ /g;

    my @items = split(/ /, $itemlist);

    my $ping = Lib::Ping->new();
    $ping->set_pinglist(@items);
    $ping->update();

    print "Location: ./?act=view_manage&func=ping", "\n\n";
}

sub _edit_filter() {
    my ($func) = @_;
    my $data;
    _show_form($func, $data, '');
}

sub _filter_form() {
    my $filter = Lib::Filter->new();
    my @items = $filter->get_filter();

    my $itemlist = '';
    foreach my $item (@items) {
        $itemlist .= "$item\n";
    }

    my $language = Lib::Language->new();
    my $filterlist = $language->get_statictext('filter');
    my $button = $language->get_statictext('update');

    my $form = <<"END_FORM";
$filterlist :
<form method="post" action="./">
<input type="hidden" name="act" value="exec_manage">
<input type="hidden" name="func" value="filter">
<textarea name="itemlist" rows="12" cols="75">$itemlist</textarea><br />
<p>
<input type="submit" value="$button">
</p>
</form>
END_FORM

    return $form;
}

sub _update_filter() {
    my ($cgi) = @_;

    my $itemlist = $cgi->param('itemlist');
    $itemlist =~ s/\r?\n/ /g;

    my @items = split(/ /, $itemlist);

    my $filter = Lib::Filter->new();
    $filter->set_filter(@items);
    $filter->update();

    print "Location: ./?act=view_manage&func=filter", "\n\n";
}

sub _edit_htaccess() {
    my ($func) = @_;
    my $data;
    _show_form($func, $data, '');
}

sub _htaccess_form() {
    my $systemdir = $conf->get_system_dir();
    my $filepath = ".htaccess";
    open my $datafh, '<', $filepath;
    my $filedata = do { local $/; <$datafh> };
    close $datafh;

    my $language = Lib::Language->new();
    my $htaccess = $language->get_statictext('htaccess');
    my $button = $language->get_statictext('save');

    my $form = <<"END_FORM";
$htaccess :
<form method="post" action="./">
<input type="hidden" name="act" value="exec_manage">
<input type="hidden" name="func" value="htaccess">
<textarea name="filedata" rows="20" cols="75">$filedata</textarea><br />
<p>
<input type="submit" value="$button">
</p>
</form>
END_FORM

    return $form;
}

sub _update_htaccess() {
    my ($cgi) = @_;

    my $filedata = $cgi->param('filedata');

    my $systemdir = $conf->get_system_dir();
    my $filepath = ".htaccess";
    $filedata =~ s/\r?\n/\n/g;
    open my $datafh, '>', $filepath;
    print {$datafh} $filedata;
    close $datafh;

    print "Location: ./?act=view_manage&func=htaccess", "\n\n";
}

1;
