#!/usr/bin/perl
#------------------------------------------------------------------------------
#    59Tracker, weblog software for personal publisher.
#    Copyright (C) 2004-2010 Kaga, Hiroaki
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#------------------------------------------------------------------------------

use strict;
use warnings;

use XMLRPC::Transport::HTTP;

use Lib::User;
use Lib::DateTime;
use Lib::Logger;
use Lib::Topic;
use Lib::Archive;
use Lib::Search;
use Lib::Conf;

my $logger = Lib::Logger->new();
my $user = Lib::User->new();
my $topic = Lib::Topic->new();
my $archive = Lib::Archive->new();
my $conf = Lib::Conf->new();

my $server = XMLRPC::Transport::HTTP::CGI->dispatch_to('metaWeblog', 'blogger', 'mt')->handle;

package metaWeblog;

# metaWeblog.newPost (blogid, username, password, struct, publish) returns string
sub newPost {
    my ($self, $blogid, $username, $password, $content, $publish) = @_;

    my $result = $user->check_account($username, $password);
    if (!$result) {
        my $res;
        $res->{faultCode} = 'fc1';
        $res->{faultString} = "authentication error";
        return $res;
    }

    my $newid = $archive->get_counter() + 1;
    $archive->update_counter($newid);
    my $dt = Lib::DateTime->new();
    my $lastdate = $dt->local_datetime(0);

    if ($publish) {
        $topic->set_status(1);
    }
    else {
        $topic->set_status(0);
    }
    my $title = $content->{title};
    my $text = $content->{description};
    my $keywords = $content->{mt_keywords};

#    $logger->write("newPost : title = $title");   # for debug

    $topic->set_topicid($newid);
    $topic->set_lastdate($lastdate);
    $topic->set_editor($username);
    $topic->set_title($title);
    $topic->set_text($text);
    $topic->set_keyword($keywords);
    $topic->update();

    if ($publish) {
        $archive->add_archive($newid, $lastdate);
        $archive->add($newid);
        my $search = Lib::Search->new();
        my $str = "$title $text $username";
        $search->addindex($newid, $str);
    }

    return 1;
}

# metaWeblog.editPost (postid, username, password, struct, publish) returns true
sub editPost {
    my ($self, $postid, $username, $password, $content, $publish) = @_;

    my $result = $user->check_account($username, $password);
    if (!$result) {
        my $res;
        $res->{faultCode} = 'fc1';
        $res->{faultString} = "authentication error";
        return $res;
    }

    $topic->load($postid, 1);
    if ($publish) {
        $topic->set_status(1);
    }
    else {
        $topic->set_status(0);
    }
    my $title = $content->{title};
    my $text = $content->{description};
    my $keywords = $content->{mt_keywords};

    $topic->set_title($title);
    $topic->set_text($text);
    $topic->set_keyword($keywords);
    $topic->update();

    return 1;
}

# metaWeblog.getPost (postid, username, password) returns struct
sub getPost {
    my ($self, $postid, $username, $password) = @_;

    my $result = $user->check_account($username, $password);
    if (!$result) {
        my $res;
        $res->{faultCode} = 'fc1';
        $res->{faultString} = "authentication error";
        return $res;
    }

    my $dt = Lib::DateTime->new();
    $topic->load($postid, 0);

    my $res;
    $res->{userid} = $username;
    my $title = $topic->get_title();
    my $text = $topic->get_text();

    my $keyword = $topic->get_keyword();
    my $lastdate = $topic->get_lastdate();
    $dt->set_datetime($lastdate);
    $lastdate = $dt->format_datetime(4);
    $res->{dateCreated} = SOAP::Data->type(dateTime => $lastdate);
    $res->{postid} = SOAP::Data->type(string => $postid);
    $res->{description} = SOAP::Data->type(string => $text);
    $res->{title} = SOAP::Data->type(string => $title);
    my $htmlurl = $conf->get_docroot_dir() . "/?func=detail&amp;id=$postid";
    $res->{link} = SOAP::Data->type(string => $htmlurl);
    $res->{permaLink} = SOAP::Data->type(string => $htmlurl);
    $res->{mt_keywords} = SOAP::Data->type(string => $keyword);
    return $res;
}

# metaWeblog.getCategories (blogid, username, password) returns struct
sub getCategories {
    my ($self, $blogid, $username, $password) = @_;

    my $result = $user->check_account($username, $password);
    if (!$result) {
        my $res;
        $res->{faultCode} = 'fc1';
        $res->{faultString} = "authentication error";
        return $res;
    }

    my @items = ();
    my $item;
    my $lastdate = $topic->get_lastdate();
    $item->{description} = 'Unclassified';
    $item->{title} = 'Unclassified';
    push(@items, $item);
    my $res = \@items;
    return $res;
}

# metaWeblog.getRecentPosts (blogid, username, password, numberOfPosts) returns array of structs
sub getRecentPosts {
    my ($self, $blogid, $username, $password, $numberOfPosts) = @_;

    my $result = $user->check_account($username, $password);
    if (!$result) {
        my $res;
        $res->{faultCode} = 'fc1';
        $res->{faultString} = "authentication error";
        return $res;
    }

    my $dt = Lib::DateTime->new();
    my $lastcount = $archive->get_counter();
    my @items = ();
    my $count = 0;
#    $logger->write("before loop");	# for debug
    for (my $i = $lastcount; ($i >= 0) && ($count < $numberOfPosts); $i--) {
        my $topicid = $i;
        $topic->load($topicid, 0);
        my $status = $topic->get_status();
        if (($topic->get_editor() eq $username) && ($status != 9)) {
            my $item;
            my $title = $topic->get_title();
            my $text = $topic->get_text();

            my $keyword = $topic->get_keyword();
            my $lastdate = $topic->get_lastdate();
            $dt->set_datetime($lastdate);
            $lastdate = $dt->format_datetime(4);
            $item->{dateCreated} = SOAP::Data->type(dateTime => $lastdate);
            $item->{userid} = SOAP::Data->type(string => $username);
            $item->{postid} = SOAP::Data->type(string => $topicid);
            $item->{description} = SOAP::Data->type(string => $text);
            $item->{title} = SOAP::Data->type(string => $title);
            my $htmlurl = $conf->get_docroot_dir() . "/?func=detail&amp;id=$topicid";
            $item->{link} = SOAP::Data->type(string => $htmlurl);
            $item->{permaLink} = $htmlurl;
            $item->{mt_keywords} = SOAP::Data->type(string => $keyword);
            push(@items, $item);
            $count++;
        }
    }

    my $res = \@items;
    return $res;
}

# metaWeblog.newMediaObject (blogid, username, password, struct) returns struct
sub newMediaObject {
    my ($self, $blogid, $username, $password, $file) = @_;

    my $result = $user->check_account($username, $password);
    if (!$result) {
        my $res;
        $res->{faultCode} = 'fc1';
        $res->{faultString} = "authentication error";
        return $res;
    }

    use MIME::Base64;
    my $bits = $file->{bits};
    $bits = decode_base64($bits);

    my $name = $file->{name};
    my $filename = "files/" . $username . "_" . $name;
    my $url = $conf->get_docroot_dir() . "/$filename";

    open my $outfh, '>', $filename; 
    binmode $outfh; 
    print {$outfh} $file; 
    close $outfh;

    return $url;
}

package blogger;

# blogger.getUsersBlogs (appkey, username, password) returns struct
sub getUsersBlogs {
    my ($self, $appkey, $username, $password) = @_;

    my $result = $user->check_account($username, $password);
    if ($result) {
        my @items = ();
        my $item;
        my $sitetitle = $conf->get_site_title();
        $item->{url} = $conf->get_docroot_dir() . "/";
        $item->{blogid} = '59Tracker';
        $item->{blogName} = SOAP::Data->type(string => $sitetitle);
        push(@items, $item);

        my $res = \@items;
        return $res;
    }

    my $res;
    $res->{faultCode} = 'fc1';
    $res->{faultString} = "authentication error";

    return $res;
}

# blogger.getUserInfo (appkey, username, password) returns struct
sub getUserInfo {
    my ($self, $appkey, $username, $password) = @_;

    my $result = $user->check_account($username, $password);
    if ($result) {
        my $item;
        $item->{userid} = $user->get_userid();
        $item->{nickname} = '';
        $item->{email} = $user->get_mailaddr();
        $item->{url} = $conf->get_docroot_dir() . "/";
        return $item;
    }

    my $res;
    $res->{faultCode} = 'fc1';
    $res->{faultString} = "authentication error";

    return $res;
}

# blogger.deletePost (appkey, postid, username, password, publish) returns true
sub deletePost {
    my ($self, $appkey, $postid, $username, $password, $publish) = @_;

    my $result = $user->check_account($username, $password);
    if (!$result) {
        my $res;
        $res->{faultCode} = 'fc1';
        $res->{faultString} = "authentication error";
        return $res;
    }

    $topic->load($postid, 1);
    $topic->set_status(9);
    $topic->update();

    if ($conf->get_static_html()) {
        my $htmlfile = "detail$postid.html";
        unlink $htmlfile;
    }

    return 1;
}

package mt;

# mt.getCategoryList (blogid, username, password) returns struct
sub getCategoryList {
    my ($self, $blogid, $username, $password) = @_;

    my $result = $user->check_account($username, $password);
    if (!$result) {
        my $res;
        $res->{faultCode} = 'fc1';
        $res->{faultString} = "authentication error";
        return $res;
    }

    my @items = ();
    my $item;
    $item->{categoryId} = 'abcdef';
    $item->{categoryName} = 'Unclassified';
    push(@items, $item);
    my $res = \@items;
    return $res;
}

# mt.getPostCategories (blogid, username, password) returns struct
sub getPostCategories {
    my ($self, $blogid, $username, $password) = @_;

    my $result = $user->check_account($username, $password);
    if (!$result) {
        my $res;
        $res->{faultCode} = 'fc1';
        $res->{faultString} = "authentication error";
        return $res;
    }


    my @items = ();
    my $item;
    $item->{categoryId} = 'id';
    $item->{categoryName} = 'Unclassified';
    $item->{isPrimary} = 1;
    push(@items, $item);
    my $res = \@items;
    return $res;
}

# mt.setPostCategories (blogid, username, password, categories) returns struct
sub setPostCategories {
    my ($self, $blogid, $username, $password, $categories) = @_;

    my $result = $user->check_account($username, $password);
    if (!$result) {
        my $res;
        $res->{faultCode} = 'fc1';
        $res->{faultString} = "authentication error";
        return $res;
    }
    return 1;
}

# mt.publishPost (postid, username, password) returns struct
sub publishPost {
    my ($self, $postid, $username, $password) = @_;

    my $result = $user->check_account($username, $password);
    if (!$result) {
        my $res;
        $res->{faultCode} = 'fc1';
        $res->{faultString} = "authentication error";
        return $res;
    }
    return 1;
}

