#!/usr/bin/perl
#------------------------------------------------------------------------------
#    59Tracker, weblog software for personal publisher.
#    Copyright (C) 2004-2010 Kaga, Hiroaki
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#------------------------------------------------------------------------------

use strict;
use warnings;

use Lib::Logger;
use Lib::Error;
use Lib::User;
use Lib::DateTime;
use Lib::String;
use Lib::Ping;
use Lib::Filter;
use Lib::Topic;
use Lib::Archive;
use Lib::Search;
use Lib::Keyword;
use Lib::Util;
use Lib::Conf;
use Lib::Language;

my $logger = Lib::Logger->new();
my $error = Lib::Error->new();
my $conf = Lib::Conf->new();

sub act() {
    my ($act, $cgi) = @_;

    my $user = Lib::User->new();
    my $sessionid = $cgi->cookie('cookie59t');
    if ($sessionid eq '') {
        $error->print_error('00003e');
    }
    my $session_userid = $user->check_session($sessionid);
    if ($session_userid eq '') {
        my $func = $cgi->param('func');
        if ($func eq 'newtopic') {
            print "Location: ./?func=login", "\n\n";
            exit;
        }
        else {
            $error->print_error('00003e');
        }
    }

    if ($act eq 'exec_editor') {
        _exec_editor($session_userid, $cgi);
    }
    else {
        _view_editor($session_userid, $cgi);
    }
}

sub _view_editor() {
    my ($session_userid, $cgi) = @_;

    # func: newtopic, edittopic, newps, editps, delps, upload, delfile
    #       newlink, editlink, dellink, delcomment, deltb
    my $func = $cgi->param('func');
    if ($func eq 'edittopic') {
        _edit_topic($func, $session_userid, $cgi);
    }
    elsif ($func eq 'newps') {
        _input_ps($func, $session_userid, $cgi);
    }
    elsif ($func eq 'editps') {
        _edit_ps($func, $session_userid, $cgi);
    }
    elsif ($func eq 'delps') {
        _confirm_ps($func,  $session_userid, $cgi);
    }
    elsif ($func eq 'upload') {
        _upload_file($func, $session_userid, $cgi);
    }
    elsif ($func eq 'delfile') {
        _confirm_file($func, $session_userid, $cgi);
    }
    elsif ($func eq 'newlink') {
        _input_link($func, $session_userid, $cgi);
    }
    elsif ($func eq 'editlink') {
        _edit_link($func, $session_userid, $cgi);
    }
    elsif ($func eq 'dellink') {
        _confirm_link($func, $session_userid, $cgi);
    }
    elsif ($func eq 'delcomment') {
        _confirm_comment($func, $session_userid, $cgi);
    }
    elsif ($func eq 'deltb') {
        _confirm_trackback($func, $session_userid, $cgi);
    }
    else {
        _input_topic('newtopic', $session_userid, $cgi);
    }
}

sub _exec_editor() {
    my ($session_userid, $cgi) = @_;

    # func: newtopic, edittopic, newps, editps, delps, upload, delfile
    #       newlink, editlink, dellink, delcomment, admitcomment, deltb, admittb
    my $func = $cgi->param('func');
    if ($func eq '') {
        $error->print_error('00002e');
    }

    if ($func eq 'newtopic') {
        _add_topic($session_userid, $cgi);
    }
    elsif ($func eq 'edittopic') {
        _update_topic($session_userid, $cgi);
    }
    elsif ($func eq 'newps') {
        _add_ps($session_userid, $cgi);
    }
    elsif ($func eq 'editps') {
        _update_ps($session_userid, $cgi);
    }
    elsif ($func eq 'delps') {
        _delete_ps($session_userid, $cgi);
    }
    elsif ($func eq 'upload') {
        _add_file($session_userid, $cgi);
    }
    elsif ($func eq 'delfile') {
        _delete_file($session_userid, $cgi);
    }
    elsif ($func eq 'newlink') {
        _add_link($session_userid, $cgi);
    }
    elsif ($func eq 'editlink') {
        _update_link($session_userid, $cgi);
    }
    elsif ($func eq 'dellink') {
        _delete_link($session_userid, $cgi);
    }
    elsif ($func eq 'delcomment') {
        _delete_comment($session_userid, $cgi);
    }
    elsif ($func eq 'admitcomment') {
        _admit_comment($session_userid, $cgi);
    }
    elsif ($func eq 'deltb') {
        _delete_trackback($session_userid, $cgi);
    }
    elsif ($func eq 'admittb') {
        _admit_trackeback($session_userid, $cgi);
    }
}

sub _input_topic() {
    my ($func, $session_userid, $cgi) = @_;

    my $topic = Lib::Topic->new();

    my $ptopicid = $cgi->param('topicid');
    my $keyword = '';
    my ($ad1, $ad2, $ad3) = ('', '', '');
    if ($ptopicid ne '') {
        $topic->load($ptopicid, 0);
        $keyword = $topic->get_keyword();
        my $editor = $topic->get_editor();
        if ($editor eq $session_userid) {
            $ad1 = $topic->get_ad1;
            $ad2 = $topic->get_ad2;
            $ad3 = $topic->get_ad3;
        }
    }

    my $util = Lib::Util->new();
    my $url = $cgi->param('url');
    my $status = $conf->get_default_status();
    my $convcrlf = 1;
    my $title = '';
    my $text = '';
    if ($url ne '') {
        my $language = Lib::Language->new();
        my $evallabel = $language->get_statictext('evaluate');
        my $bookmarklink = $language->get_statictext('bookmark-link');

        my $user = Lib::User->new();
        $user->load($session_userid);
        my $username = $user->get_username();
        my $hpurl = $user->get_hpurl();
        my $dt = Lib::DateTime->new();
        my $today = substr($dt->local_datetime(0), 0, 10);
        my $dtreviewed = substr($today, 0, 4) . substr($today, 5, 2) . substr($today, 8, 2);

        my $vcard = <<"END_VCARD";
(<span class="reviewer vcard"><a class="url fn" href="$hpurl">$username</a> <abbr class="dtreviewed" title="$dtreviewed">$today</abbr></span>)
END_VCARD

        ($title, $text) = $util->get_blockquote($url);
        $text = "<div class=\"description\">\n<p>$text</p>\n</div>\n";
        $text .= "<small>$evallabel : </small><abbr class=\"rating\" title=\"5\">★★★★★</abbr> $vcard<br /><br />\n";
        $text .= "<a href=\"$url\" target=\"_blank\">$bookmarklink</a>\n";
        $convcrlf = 0;
    }

    $title = $util->tag_invalidate($title);

    my $topicdata;
    $topicdata->{topicid}    = $ptopicid;
    $topicdata->{title}      = $title;
    $topicdata->{status}     = $conf->get_default_status();
    $topicdata->{updateinfo} = 1;
    $topicdata->{convcrlf}   = $convcrlf;
    $topicdata->{updatedate} = 1;
    $topicdata->{text}       = $text;
    $topicdata->{keyword}    = $keyword;
    $topicdata->{url}        = $url;
    $topicdata->{trackback}  = '';
    $topicdata->{ad1}        = $ad1;
    $topicdata->{ad2}        = $ad2;
    $topicdata->{ad3}        = $ad3;

    _show_form($func, $topicdata, '');
}

sub _add_topic() {
    my ($session_userid, $cgi) = @_;

    if ($session_userid eq '') {
        $error->print_error('00003e');
    }

    my $topicdata;
    $topicdata->{topicid}    = $cgi->param('topicid');
    $topicdata->{title}      = $cgi->param('title');
    $topicdata->{status}     = $cgi->param('status');
    $topicdata->{updateinfo} = ($cgi->param('updateinfo') eq '') ? 0 : 1;
    $topicdata->{convcrlf}   = ($cgi->param('convcrlf') eq '') ? 0 : 1;
    $topicdata->{updatedate} = 1;
    $topicdata->{text}       = $cgi->param('text');
    $topicdata->{keyword}    = $cgi->param('keyword');
    $topicdata->{url}        = $cgi->param('url');
    $topicdata->{trackback}  = $cgi->param('trackback');
    $topicdata->{ad1}        = $cgi->param('ad1');
    $topicdata->{ad2}        = $cgi->param('ad2');
    $topicdata->{ad3}        = $cgi->param('ad3');

    $topicdata->{keyword} =~ s/\A[ ]+//g;
    $topicdata->{keyword} =~ s/[ ]+\z//g;
    $topicdata->{keyword} =~ s/[ ]+/ /g;

    my $util = Lib::Util->new();
    $topicdata->{title} = $util->tag_invalidate($topicdata->{title});

    my $topic = Lib::Topic->new();
    $topic->load($topicdata->{topicid}, 1);

    # 入力チェック
    _check_topic('newtopic', $topicdata);

    $topicdata->{editor} = $session_userid;

    my $ptopicid = $topicdata->{topicid}; 

    # 新規トピックIDを取得
    my $archive = Lib::Archive->new();
    my $newtopicid = $archive->get_counter() + 1;
    $archive->update_counter($newtopicid);

    my $topicurl = "./?func=detail&id=$newtopicid";
    my $htmlurl = $topicurl;

    my $string = Lib::String->new();
    my $linknum = 0;
    if ($ptopicid ne '') {
        my @links = $topic->get_links();
        my $linkid = (split(/\,/, $links[$#links]))[0];
        $linkid++;
        my $title = $topicdata->{title};
        my $summary = $string->cut($topicdata->{text}, $conf->get_summary_length());
        $title =~ s/,/enc_conma/g;
        $summary =~ s/,/enc_conma/g;
        my $linkdata = "$linkid,$title,$htmlurl,$summary";
        push @links, $linkdata;
        $topic->set_linknum(scalar(@links));
        $topic->set_links(@links);
        $topic->update();
    }

    my $ipaddr = $ENV{'REMOTE_ADDR'}; # 投稿者のIPアドレス
    my $dt = Lib::DateTime->new();
    my $lastdate = $dt->local_datetime(0);

    my $newtopic = Lib::Topic->new();

    if ($ptopicid ne '') {
        # 新規トピックの関連リンクに親トピックを追加
        my $ptext = $topic->get_text();
        my $ptitle = $topic->get_title();
        my $peditor = $topic->get_editor();
        my $psummary = $string->cut($ptext, $conf->get_summary_length());
        my $ptopicurl = "./?func=detail&id=$ptopicid";
        $ptitle =~ s/,/enc_conma/g;
        $psummary =~ s/,/enc_conma/g;
        my $linkdata = "1,$ptitle,$ptopicurl,$psummary";
        $newtopic->set_links($linkdata);
        $linknum++;
        if ($session_userid eq $peditor) {
            # 親トピックの広告を引き継ぐ
            if ($topicdata->{ad1} eq 'hidden') {
                $topicdata->{ad1} = $topic->get_ad1();
            }
            if ($topicdata->{ad2} eq 'hidden') {
                $topicdata->{ad2} = $topic->get_ad2();
            }
            if ($topicdata->{ad3} eq 'hidden') {
                $topicdata->{ad3} = $topic->get_ad3();
            }
        }
    }

    # 改行の処理
    if ($topicdata->{convcrlf}) {
        $topicdata->{text} =~ s/\r?\n/<br \/>enc_crlf/g;
    }
    else {
        $topicdata->{text} =~ s/\r?\n/enc_crlf/g;
    }

    $newtopic->set_topicid($newtopicid);
    $newtopic->set_status($topicdata->{status});
    $newtopic->set_editor($topicdata->{editor});
    $newtopic->set_lastdate($lastdate);
    $newtopic->set_title($topicdata->{title});
    $newtopic->set_text($topicdata->{text});
    $newtopic->set_psnum(0);
    $newtopic->set_commentnum(0);
    $newtopic->set_trackbacknum(0);
    $newtopic->set_linknum($linknum);
    $newtopic->set_filenum(0);
    if ($topicdata->{ad1} ne 'hidden') {
        $newtopic->set_ad1($topicdata->{ad1});
    }
    if ($topicdata->{ad2} ne 'hidden') {
        $newtopic->set_ad2($topicdata->{ad2});
    }
    if ($topicdata->{ad3} ne 'hidden') {
        $newtopic->set_ad3($topicdata->{ad3});
    }
    $newtopic->set_point(0);
    $newtopic->set_keyword($topicdata->{keyword});
    $newtopic->set_url($topicdata->{url});
    $newtopic->set_ipaddr($ipaddr);
    $newtopic->update();

    # キーワードが設定されていある場合はキーワードデータに追加
    if ($topicdata->{keyword} ne '') {
        my @newarray = split(/ /, $topicdata->{keyword});
        my $keywordobj = Lib::Keyword->new();
        foreach my $keyworditem (@newarray) {
            $keywordobj->add($keyworditem, $newtopicid);
        }
    }

    my $str = "$topicdata->{title} $topicdata->{text} $topicdata->{editor} $topicdata->{keyword}";
    my $search = Lib::Search->new();
    $search->addindex($newtopicid, $str);

    my $ping = Lib::Ping->new();
    if ($topicdata->{updateinfo}) {
        $archive->add_archive($newtopicid, $lastdate);
        $archive->add($newtopicid);
        $ping->weblogupdates($conf->get_docroot_dir(), $conf->get_site_title());
    }

    # トラックバックの送信
    if ($topicdata->{trackback} ne '') {
        my $trackback = $topicdata->{trackback};
        $trackback =~ s/\r?\n/,/g;
        my @pinglist = split(/\,/, $trackback);
        my $string = Lib::String->new();
        my $pingdata;
        $pingdata->{url} = $conf->get_docroot_dir(). "/?func=detail&id=$newtopicid";
        $pingdata->{blog_name} = $conf->get_site_title();
        $pingdata->{title} = $topicdata->{title};
        $topicdata->{text} =~ s/enc_crlf//g;
        $pingdata->{excerpt} = $string->cut($topicdata->{text}, $conf->get_excerpt_length());
        $ping->trackback($pingdata, @pinglist);
    }

    print "Location: ./?func=detail&id=$newtopicid", "\n\n";
}

sub _edit_topic() {
    my ($func, $session_userid, $cgi) = @_;

    my $topicid = $cgi->param('topicid');
    if ($topicid eq '') {
        $error->print_error('00003e');
    }

    my $topic = Lib::Topic->new();
    $topic->load($topicid, 0);
    my $editor = $topic->get_editor();
    my $user = Lib::User->new();
    if (($editor ne $session_userid) && !$user->is_admin($session_userid)) {
        $error->print_error('00004e');
    }

    my $topicdata;
    $topicdata->{topicid}    = $topicid;
    $topicdata->{title}      = $topic->get_title();
    $topicdata->{status}     = $topic->get_status();
    $topicdata->{updateinfo} = 0;
    $topicdata->{convcrlf}   = 0;
    $topicdata->{updatedate} = $conf->get_date_update();
    $topicdata->{text}       = $topic->get_text();
    $topicdata->{keyword}    = $topic->get_keyword();
    $topicdata->{url}        = $topic->get_url();
    $topicdata->{trackback}  = '';
    $topicdata->{ad1}        = $topic->get_ad1();
    $topicdata->{ad2}        = $topic->get_ad2();
    $topicdata->{ad3}        = $topic->get_ad3();

    _show_form($func, $topicdata, '');
}

sub _update_topic() {
    my ($session_userid, $cgi) = @_;

    my $topicid = $cgi->param('topicid');
    if ($topicid eq '') {
        $error->print_error('00002e');
    }

    my $topic = Lib::Topic->new();
    $topic->load($topicid, 1);
    my $editor = $topic->get_editor();
    my $user = Lib::User->new();
    if (($editor ne $session_userid) && !$user->is_admin($session_userid)) {
        $error->print_error('00004e');
    }

    my $topicdata;
    $topicdata->{topicid}    = $topicid;
    $topicdata->{title}      = $cgi->param('title');
    $topicdata->{status}     = $cgi->param('status');
    $topicdata->{updateinfo} = ($cgi->param('updateinfo') eq '') ? 0 : 1;
    $topicdata->{convcrlf}   = ($cgi->param('convcrlf') eq '') ? 0 : 1;
    $topicdata->{updatedate} = ($cgi->param('updatedate') eq '') ? 0 : 1;
    $topicdata->{text}       = $cgi->param('text');
    $topicdata->{keyword}    = $cgi->param('keyword');
    $topicdata->{url}        = $cgi->param('url');
    $topicdata->{trackback}  = $cgi->param('trackback');
    $topicdata->{ad1}        = $cgi->param('ad1');
    $topicdata->{ad2}        = $cgi->param('ad2');
    $topicdata->{ad3}        = $cgi->param('ad3');

    $topicdata->{keyword} =~ s/　/ /g;
    $topicdata->{keyword} =~ s/\A[ ]+//g;
    $topicdata->{keyword} =~ s/[ ]+\z//g;
    $topicdata->{keyword} =~ s/[ ]+/ /g;

    my $util = Lib::Util->new();
    $topicdata->{title} = $util->tag_invalidate($topicdata->{title});

    my $language = Lib::Language->new();

    # 最初のトピックは削除不可
    if ($topicid == 0 && $topicdata->{status} == 9) {
        _show_form('edittopic', $topicdata, "[$topicdata->{title}]" .  $language->get_usererror('not-delete'));
    }

    # 入力チェック
    _check_topic('edittopic', $topicdata);

    my $oldlastdate = $topic->get_lastdate();
    my $oldkeyword = $topic->get_keyword();

    # 改行の処理
    if ($topicdata->{convcrlf}) {
        $topicdata->{text} =~ s/\r?\n/<br \/>enc_crlf/g;
    }
    else {
        $topicdata->{text} =~ s/\r?\n/enc_crlf/g;
    }

    my $dt = Lib::DateTime->new();
    my $lastdate = $dt->local_datetime(0);

    $topic->set_status($topicdata->{status});
    if ($topicdata->{updatedate}) {
        $topic->set_lastdate($lastdate);
    }
    $topic->set_title($topicdata->{title});
    $topic->set_text($topicdata->{text});

    # 広告が非表示の場合は更新しない
    if ($topicdata->{ad1} ne 'hidden') {
        $topic->set_ad1($topicdata->{ad1});
    }
    if ($topicdata->{ad2} ne 'hidden') {
        $topic->set_ad2($topicdata->{ad2});
    }
    if ($topicdata->{ad3} ne 'hidden') {
        $topic->set_ad3($topicdata->{ad3});
    }
    $topic->set_keyword($topicdata->{keyword});
    $topic->set_url($topicdata->{url});
    $topic->update();

    # キーワードが設定されていある場合はキーワードデータに追加
    if ($topicdata->{keyword} ne $oldkeyword) {
        my $newkeyword = $topicdata->{keyword};
        my @addarray = ();
        my @delarray = ();
        my @newarray = split(/ /, $newkeyword);
        my @oldarray = split(/ /, $oldkeyword);

        # 新しいキーワードに存在して古いキーワードに存在しないものは新規追加
        foreach my $new (@newarray) {
            my $found = 0;
            foreach my $old (@oldarray) {
                if ($new eq $old) {
                    $found = 1;
                    last;
                }
            }
            if (!$found) {
                push @addarray, $new;
            }
        }

        # 古いキーワードに存在して新しいキーワードに存在しないものは削除
        foreach my $old (@oldarray) {
            my $found = 0;
            foreach my $new (@newarray) {
                if ($old eq $new) {
                    $found = 1;
                    last;
                }
            }
            if (!$found) {
                push @delarray, $old;
            }
        }

        my $keywordobj = Lib::Keyword->new();
        foreach my $item (@addarray) {
            $keywordobj->add($item, $topicid);  # 追加
        }
        foreach my $item (@delarray) {
            $keywordobj->delete($item, $topicid); # 削除
        }
    }

    my $archive = Lib::Archive->new();
    my $ping = Lib::Ping->new();

    # ドラフトまたは削除の場合はアーカイブデータから削除
    if ($topicdata->{status} == 0 || $topicdata->{status} == 9) {
        my $detailhtml = "detail$topicid.html";
        if (-f $detailhtml) {
            unlink $detailhtml;
        }
        $archive->delete($topicid);
        $archive->delete_archive($topicid, $oldlastdate);
    }
    else {
        my $ping = Lib::Ping->new();
        if ($topicdata->{updateinfo}) {
            # アーカイブデータに追加
            $archive->add($topicid);
            $archive->add_archive($topicid, $lastdate);
            # 更新Ping送信
            $ping->weblogupdates($conf->get_docroot_dir(), $conf->get_site_title());
        }

        # トラックバックの送信
        if ($topicdata->{trackback} ne '') {
            my $trackback = $topicdata->{trackback};
            $trackback =~ s/\r?\n/,/g;
            my @pinglist = split(/\,/, $trackback);
            my $string = Lib::String->new();
            my $pingdata;
            $pingdata->{url} = $conf->get_docroot_dir(). "/?func=detail&id=$topicid";
            $pingdata->{blog_name} = $conf->get_site_title();
            $pingdata->{title} = $topicdata->{title};
            $topicdata->{text} =~ s/enc_crlf//g;
            $pingdata->{excerpt} = $string->cut($topicdata->{text}, $conf->get_excerpt_length());
            $ping->trackback($pingdata, @pinglist);
         }
    }

    print "Location: ./?func=detail&id=$topicid", "\n\n";
}

sub _check_topic() {
    my ($func, $topicdata) = @_;

    my $language = Lib::Language->new();

    # ステータス
    if ($topicdata->{status} eq '') {
        _show_form($func, $topicdata, $language->get_usererror('select-status'));
    }

    # トピックのタイトル
    if ($topicdata->{title} eq '') {
        _show_form($func, $topicdata, $language->get_usererror('input-title'));
    }

    # トピックの本文
    if ($topicdata->{text} eq '') {
        _show_form($func, $topicdata, $language->get_usererror('input-text'));
    }

    # キーワードのチェック
    my $keywordobj = Lib::Keyword->new();
    if ($keywordobj->check($topicdata->{keyword})) {
        _show_form($func, $topicdata, $language->get_usererror('keyword-format'));
    }
}

sub _show_form {
    my ($func, $data, $msg) = @_;

    my $language = Lib::Language->new();
    my $caption = '';
    my $mainform = '';
    if ($func eq 'newtopic') {
        $caption  = $language->get_statictext('post');
        $mainform = _topic_form($func, $data);
    }
    elsif ($func eq 'edittopic') {
        $caption  = $language->get_statictext('edit-topic');
        $mainform = _topic_form($func, $data);
    }
    elsif ($func eq 'newps') {
        $caption  = $language->get_statictext('ps');
        $mainform = _ps_form($func, $data);
    }
    elsif ($func eq 'editps') {
        $caption  = $language->get_statictext('edit-ps');
        $mainform = _ps_form($func, $data);
    }
    elsif ($func eq 'delps') {
        $caption  = $language->get_statictext('delete-ps');
        $mainform = _delps_form($data);
    }
    elsif ($func eq 'upload') {
        $caption  = $language->get_statictext('add-file');
        $mainform = _upload_form($data);
    }
    elsif ($func eq 'delfile') {
        $caption  = $language->get_statictext('delete-file');
        $mainform = _delfile_form($data);
    }
    elsif ($func eq 'newlink') {
        $caption  = $language->get_statictext('add-link');
        $mainform = _link_form($func, $data);
    }
    elsif ($func eq 'editlink') {
        $caption  = $language->get_statictext('edit-link');
        $mainform = _link_form($func, $data);
    }
    elsif ($func eq 'dellink') {
        $caption  = $language->get_statictext('delete-link');
        $mainform = _dellink_form($data);
    }
    elsif ($func eq 'delcomment') {
        $caption  = $language->get_statictext('delete-comment');
        $mainform = _delcomment_form($data);
    }
    elsif ($func eq 'deltb') {
        $caption  = $language->get_statictext('delete-trackback');
        $mainform = _deltb_form($data);
    }

    my $docrootdir = $conf->get_docroot_dir();
    my $sitetitle = $conf->get_site_title();
    my $subtitle = $conf->get_sub_title();

    my $topicid = $data->{topicid};
    my $topic = Lib::Topic->new();
    $topic->load($topicid, 0);
    my $topictitle = $topic->get_title();

    my $homelabel = $language->get_statictext('home');

    my $path = <<"END_PATH";
<a class="path" href="./">$homelabel</a>&nbsp;
&gt;&nbsp;
$caption
END_PATH
    if ($topicid ne '') {
        my $pathtitle = _get_pathtitle($topicid);
        $path = <<"END_PATH";
<a class="path" href="./">$homelabel</a>&nbsp;
&gt;&nbsp;
<a class="path" href="./?func=detail&id=$topicid">$pathtitle</a>
&gt;&nbsp;
$caption
END_PATH
    }

    if ($msg ne '') {
        $msg = "<p><div id=\"message\">$msg</div></p>";
    }

    my $systemdir = $conf->get_system_dir();

    open my $templatefh, '<', "$systemdir/tmpl/editor.tmpl";
    my $template = do { local $/; <$templatefh> };
    close $templatefh;

    $template =~ s/\$CAPTION\$/$caption/g;
    $template =~ s/\$SITETITLE\$/$sitetitle/g;
    $template =~ s/\$SUBTITLE\$/$subtitle/g;
    $template =~ s/\$PATH\$/$path/g;
    $template =~ s/\$MSG\$/$msg/g;
    $template =~ s/\$MAINFORM\$/$mainform/g;

    print "Content-Type: text/html\n\n";
    print $template;

    exit;
}

sub _get_pathtitle() {
    my ($topicid) = @_;
    my $string = Lib::String->new();
    my $topic = Lib::Topic->new();
    $topic->load($topicid, 0);
    my $topictitle = $topic->get_title();
    my $adjusttitle = $string->cut($topictitle, 50);
    return $adjusttitle;
}

sub _topic_form() {
    my ($func, $topicdata) = @_;

    my $language = Lib::Language->new();
    my $draftstatus  = $language->get_statictext('draft-status');
    my $openstatus  = $language->get_statictext('open-status');
    my $closestatus  = $language->get_statictext('close-status');
    my $limitstatus  = $language->get_statictext('limit-status');
    my $deletestatus  = $language->get_statictext('delete-status');

    # ステータスの設定
    my $statuslist = '';
    if ($topicdata->{status} == 0) {
        $statuslist = <<"END_STATUS";
<input type="radio" name="status" value="0" checked>$draftstatus
<input type="radio" name="status" value="1">$openstatus
<input type="radio" name="status" value="2">$closestatus
<input type="radio" name="status" value="3">$limitstatus
<input type="radio" name="status" value="9">$deletestatus
END_STATUS
    }
    elsif ($topicdata->{status} == 1) {
        $statuslist = <<"END_STATUS";
<input type="radio" name="status" value="0">$draftstatus
<input type="radio" name="status" value="1" checked>$openstatus
<input type="radio" name="status" value="2">$closestatus
<input type="radio" name="status" value="3">$limitstatus
<input type="radio" name="status" value="9">$deletestatus
END_STATUS
    }
    elsif ($topicdata->{status} == 2) {
        $statuslist = <<"END_STATUS";
<input type="radio" name="status" value="0">$draftstatus
<input type="radio" name="status" value="1">$openstatus
<input type="radio" name="status" value="2" checked>$closestatus
<input type="radio" name="status" value="3">$limitstatus
<input type="radio" name="status" value="9">$deletestatus
END_STATUS
    }
    elsif ($topicdata->{status} == 3) {
        $statuslist = <<"END_STATUS";
<input type="radio" name="status" value="0">$draftstatus
<input type="radio" name="status" value="1">$openstatus
<input type="radio" name="status" value="2">$closestatus
<input type="radio" name="status" value="3" checked>$limitstatus
<input type="radio" name="status" value="9">$deletestatus
END_STATUS
    }
    else {
        $statuslist = <<"END_STATUS";
<input type="radio" name="status" value="0">$draftstatus
<input type="radio" name="status" value="1">$openstatus
<input type="radio" name="status" value="2">$closestatus
<input type="radio" name="status" value="3">$limitstatus
<input type="radio" name="status" value="9" checked>$deletestatus
END_STATUS
    }

    my $publishoption = $language->get_statictext('publish-option');
    my $croption = $language->get_statictext('cr-option');
    my $datetimeoption = $language->get_statictext('datetime-option');

    # オプションの設定
    my $optioncb = '';
    if ($topicdata->{updateinfo} == 1) {
        $optioncb = <<"END_OPTION";
<input type="checkbox" name="updateinfo" value="1" checked>$publishoption
END_OPTION
    }
    else {
        $optioncb = <<"END_OPTION";
<input type="checkbox" name="updateinfo" value="1">$publishoption
END_OPTION
    }
    if ($topicdata->{convcrlf} == 1) {
        $optioncb .= <<"END_OPTION";
&nbsp;&nbsp;<input type="checkbox" name="convcrlf" value="1" checked>$croption
END_OPTION
    }
    else {
        $optioncb .= <<"END_OPTION";
&nbsp;&nbsp;<input type="checkbox" name="convcrlf" value="1">$croption
END_OPTION
    }
    if ($topicdata->{updatedate} == 1) {
        $optioncb .= <<"END_OPTION";
&nbsp;&nbsp;<input type="checkbox" name="updatedate" value="1" checked>$datetimeoption
END_OPTION
    }
    else {
        $optioncb .= <<"END_OPTION";
&nbsp;&nbsp;<input type="checkbox" name="updatedate" value="1">$datetimeoption
END_OPTION
    }

    my $refer = '';
    if ($topicdata->{url} ne '') {
        my $referlabel = $language->get_statictext('refer');
        $refer = "<a href=\"$topicdata->{url}\" target=\"_blank\">$referlabel</a>";
    }

    my $button = $language->get_statictext('post');
    if ($func eq 'edittopic') {
        $button = $language->get_statictext('update');
    }
    my $cancelbutton = $language->get_statictext('cancel');

    my $boldtitle = $language->get_statictext('bold');
    my $italictitle = $language->get_statictext('italic');
    my $linktitle = $language->get_statictext('link');
    my $quotetitle = $language->get_statictext('quote');
    my $striketitle = $language->get_statictext('strike');
    my $underlinetitle = $language->get_statictext('underline');

    my $titlelabel = $language->get_statictext('title');
    my $statuslabel = $language->get_statictext('status');
    my $keywordguide = $language->get_statictext('keyword-guide');
    my $bookmarkurl = $language->get_statictext('bookmark-url');
    my $showfield = $language->get_statictext('show-field');
    my $hidefield = $language->get_statictext('hide-field');
    my $trackbacklabel = $language->get_statictext('trackback');
    my $adcode1 = $language->get_statictext('ad-code1');
    my $adcode2 = $language->get_statictext('ad-code2');
    my $adcode3 = $language->get_statictext('ad-code3');

    my $formpart = <<"END_FORMPART";
<input type="hidden" name="topicid" value="$topicdata->{topicid}">
<input type="hidden" name="act" value="exec_editor">
<input type="hidden" name="func" value="$func">
$titlelabel :<br />
<input type="text" name="title" size="100" value="$topicdata->{title}"><br />
$statuslabel :<br />
$statuslist<br />
<br />
<a class="button" href="javascript:;" onmousedown="supportInput('bold');" title="$boldtitle"><font class="button"><b>b</b></font></a>
<a class="button" href="javascript:;" onmousedown="supportInput('italic');" title="$italictitle"><font class="button"><b>i</b></font></a>
<a class="button" href="javascript:;" onmousedown="supportInput('link');" title="$linktitle"><font class="button"><b>L</b></font></a>
<a class="button" href="javascript:;" onmousedown="supportInput('quote');" title="$quotetitle"><font class="button"><b>Q</b></font></a>
<a class="button" href="javascript:;" onmousedown="supportInput('strike');" title="$striketitle"><font class="button"><b>S</b></font></a>
<a class="button" href="javascript:;" onmousedown="supportInput('underline');" title="$underlinetitle"><font class="button"><b>U</b></font></a>
&nbsp;&nbsp;&nbsp;
$optioncb<br />
<textarea name="text" rows="12" cols="75">$topicdata->{text}</textarea><br />
$keywordguide :<br />
<input type="text" name="keyword" size="60" value="$topicdata->{keyword}"><br />
$bookmarkurl :<br />
<input type="text" name="url" size="100" value="$topicdata->{url}">&nbsp;&nbsp;$refer<br />
END_FORMPART

    my $form = <<"END_FORM";
<div id="toggleHide" style="display:block">
<form name="hideform" method="post" action="./">
$formpart
<p>
<a href="javascript:;" onmousedown="toggleInput('toggleShow','toggleHide');">$showfield</a><br />
</p>
<input type="hidden" name="ad1" value="hidden">
<input type="hidden" name="ad2" value="hidden">
<input type="hidden" name="ad3" value="hidden">
<input type="submit" value="$button"> &nbsp;<input type="button" value="$cancelbutton" onclick="history.back()">
</form>
</div>

<div id="toggleShow" style="display:none">
<form name="showform" method="post" action="./">
$formpart
<p>
<a href="javascript:;" onmousedown="toggleInput('toggleShow','toggleHide');">$hidefield</a><br />
$trackbacklabel :<br />
<textarea name="trackback" rows="3" cols="75">$topicdata->{trackback}</textarea><br />
$adcode1 :<br />
<textarea name="ad1" rows="3" cols="75">$topicdata->{ad1}</textarea><br />
$adcode2 :<br />
<textarea name="ad2" rows="3" cols="75">$topicdata->{ad2}</textarea><br />
$adcode3 :<br />
<textarea name="ad3" rows="3" cols="75">$topicdata->{ad3}</textarea><br />
</p>
<input type="submit" value="$button"> &nbsp;<input type="button" value="$cancelbutton" onclick="history.back()">
</form>
</div>
END_FORM

    return $form;
}

sub _input_ps() {
    my ($func, $session_userid, $cgi) = @_;

    my $psdata;
    $psdata->{topicid} = $cgi->param('topicid');
    $psdata->{psid}    = '';
    $psdata->{title}   = '';
    $psdata->{text}    = '';

    _show_form($func, $psdata, '');
}

sub _add_ps() {
    my ($session_userid, $cgi) = @_;

    my $psdata;
    $psdata->{topicid} = $cgi->param('topicid');
    $psdata->{psid}    = $cgi->param('psid');
    $psdata->{title}   = $cgi->param('title');
    $psdata->{text}    = $cgi->param('text');

    my $topic = Lib::Topic->new();
    $topic->load($psdata->{topicid}, 1);
    my $editor = $topic->get_editor();
    my $user = Lib::User->new();
    if (($editor ne $session_userid) && !$user->is_admin($session_userid)) {
        $error->print_error('00004e');
    }

    my @pss = $topic->get_pss();
    my $psid = 0;
    if (scalar(@pss) > 0) {
        $psid = (split(/\,/, $pss[$#pss]))[0];
    }
    $psid++;

    my $util = Lib::Util->new();
    $psdata->{title} = $util->tag_invalidate($psdata->{title});

    # 入力チェック
    _check_ps('newps', $psdata);

    my $dt = Lib::DateTime->new();
    $psdata->{lastdate} = $dt->local_datetime(0);
    $psdata->{editor}   = $session_userid;

    $psdata->{title} =~ s/\r?\n//g;
    $psdata->{text} =~ s/\r?\n/enc_crlf/g;

    $psdata->{title} =~ s/,/enc_conma/g; 
    $psdata->{text} =~ s/,/enc_conma/g;

    my $ps = "$psid,$psdata->{lastdate},$psdata->{editor},$psdata->{title},$psdata->{text}";
    push @pss, $ps;
    $topic->set_pss(@pss);
    $topic->update();

    my $archive = Lib::Archive->new();
    $archive->add($psdata->{topicid});

    my $str = "$psdata->{title} $psdata->{text} $psdata->{editor}";
    my $search = Lib::Search->new();
    $search->addindex(0, $psdata->{topicid}, $str);

    print "Location: ./?func=detail&id=$psdata->{topicid}", "\n\n";
}

sub _edit_ps() {
    my ($func, $session_userid, $cgi) = @_;

    my $psdata;
    $psdata->{topicid} = $cgi->param('topicid');
    $psdata->{psid}    = $cgi->param('psid');
    $psdata->{title}   = '';
    $psdata->{text}    = '';

    if (($psdata->{topicid} eq '') || ($psdata->{psid} eq '')) {
        $error->print_error('00002e');
    }

    my $topic = Lib::Topic->new();
    $topic->load($psdata->{topicid}, 1);
    my $editor = $topic->get_editor();
    my $user = Lib::User->new();
    if (($editor ne $session_userid) && !$user->is_admin($session_userid)) {
        $error->print_error('00004e');
    }

    my @pss = $topic->get_pss();
    foreach my $ps (@pss) {
        my ($psid, $lastdate, $editor, $title, $text) = split(/\,/, $ps);
        if ($psid eq $psdata->{psid}) {
            $psdata->{title} = $title;
            $psdata->{text}  = $text;
            $psdata->{text} =~ s/enc_crlf/\n/g;
            $psdata->{title} =~ s/enc_conma/,/g; 
            $psdata->{text} =~ s/enc_conma/,/g;
            last;
        }
    }

    _show_form($func, $psdata, '');
}

sub _update_ps() {
    my ($session_userid, $cgi) = @_;

    my $psdata;
    $psdata->{topicid} = $cgi->param('topicid');
    $psdata->{psid}    = $cgi->param('psid');
    $psdata->{title}   = $cgi->param('title');
    $psdata->{text}    = $cgi->param('text');

    if (($psdata->{topicid} eq '') || ($psdata->{psid} eq '')) {
        $error->print_error('00002e');
    }

    my $dt = Lib::DateTime->new();
    $psdata->{lastdate} = $dt->local_datetime(0);
    $psdata->{editor}   = $session_userid;

    my $topic = Lib::Topic->new();
    $topic->load($psdata->{topicid}, 1);
    my $editor = $topic->get_editor();
    my $user = Lib::User->new();
    if (($editor ne $session_userid) && !$user->is_admin($session_userid)) {
        $error->print_error('00004e');
    }

    my $util = Lib::Util->new();
    $psdata->{title} = $util->tag_invalidate($psdata->{title});

    # 入力チェック
    _check_ps('editps', $psdata);

    my @pss = $topic->get_pss();
    my @new_pss = ();
    foreach my $ps (@pss) {
        my ($psid, $lastdate, $editor, $title, $text) = split(/\,/, $ps);
        if ($psid eq $psdata->{psid}) {
            $psdata->{title} =~ s/\r?\n//g;
            $psdata->{text} =~ s/\r?\n/enc_crlf/g;
            $psdata->{title} =~ s/,/enc_conma/g; 
            $psdata->{text} =~ s/,/enc_conma/g;
            $ps = "$psid,$psdata->{lastdate},$psdata->{editor},$psdata->{title},$psdata->{text}";
        }
        push @new_pss, $ps;
    }
    $topic->set_pss(@new_pss);
    $topic->update();

    print "Location: ./?func=detail&id=$psdata->{topicid}", "\n\n";
}

sub _check_ps() {
    my ($func, $psdata) = @_;

    my $language = Lib::Language->new();
    if ($psdata->{title} eq '') {
        _show_form($func, $psdata, $language->get_usererror('input-title'));
    }
    if ($psdata->{text} eq '') {
        _show_form($func, $psdata, $language->get_usererror('input-ps'));
    }
}

sub _ps_form() {
    my ($func, $psdata) = @_;

    my $language = Lib::Language->new();
    my $titlelabel = $language->get_statictext('title');
    my $pslabel = $language->get_statictext('ps');

    my $button = $language->get_statictext('ps');
    if ($func eq 'editps') {
        $button = $language->get_statictext('update');
    }
    my $cancelbutton = $language->get_statictext('cancel');

    my $form = <<"END_FORM";
<form method="post" action="./">
<input type="hidden" name="topicid" value="$psdata->{topicid}">
<input type="hidden" name="psid" value="$psdata->{psid}">
<input type="hidden" name="act" value="exec_editor">
<input type="hidden" name="func" value="$func">
$titlelabel :<br />
<input type="text" name="title" size="100" value="$psdata->{title}"><br />
$pslabel :<br />
<textarea name="text" rows="12" cols="75">$psdata->{text}</textarea><br />
<p>
<input type="submit" value="$button"> &nbsp;<input type="button" value="$cancelbutton" onclick="history.back()">
</p>
</form>
END_FORM

    return $form;
}

sub _confirm_ps() {
    my ($func, $session_userid, $cgi) = @_;

    my $psdata;
    $psdata->{topicid}  = $cgi->param('topicid');
    $psdata->{psid}     = $cgi->param('id');
    $psdata->{lastdate} = '';
    $psdata->{editor}   = '';
    $psdata->{title}    = '';
    $psdata->{text}     = '';

    if (($psdata->{topicid} eq '') || ($psdata->{psid} eq '')) {
        $error->print_error('00002e');
    }

    my $topic = Lib::Topic->new();
    $topic->load($psdata->{topicid}, 1);
    my $editor = $topic->get_editor();
    my $user = Lib::User->new();
    if (($editor ne $session_userid) && !$user->is_admin($session_userid)) {
        $error->print_error('00004e');
    }

    my @pss = $topic->get_pss();
    foreach my $ps (@pss) {
        my ($psid, $lastdate, $editor, $title, $text) = split(/\,/, $ps);
        if ($psid eq $psdata->{psid}) {
            $psdata->{title}    = $title;
            $psdata->{text}     = $text;
            $psdata->{lastdate} = $lastdate;
            $psdata->{editor}   = $editor;

            $psdata->{text} =~ s/enc_crlf/<br \/>/g;
            $psdata->{title} =~ s/enc_conma/,/g; 
            $psdata->{text} =~ s/enc_conma/,/g;
            last;
        }
    }

    _show_form($func, $psdata, '');
}

sub _delps_form() {
    my ($psdata) = @_;

    my $language = Lib::Language->new();
    my $psquestion = $language->get_message('ps-question');
    my $deletebutton = $language->get_statictext('delete');
    my $cancelbutton = $language->get_statictext('cancel');

    my $form = <<"END_FORM";
<p>
<big>
<b>$psquestion</b>
</big>
</p>

<h3>$psdata->{title}</h3>
<p>
$psdata->{text}
</p>
Posted by $psdata->{editor} at $psdata->{lastdate}

<form method="post" action="./">
<input type="hidden" name="topicid" value="$psdata->{topicid}">
<input type="hidden" name="id" value="$psdata->{psid}">
<input type="hidden" name="act" value="exec_editor">
<input type="hidden" name="func" value="delps">
<p>
<input type="submit" value="$deletebutton"> &nbsp;<input type="button" value="$cancelbutton" onclick="history.back()">
</p>
</form>
END_FORM

    return $form;
}

sub _delete_ps() {
    my ($session_userid, $cgi) = @_;

    my $psdata;
    $psdata->{topicid} = $cgi->param('topicid');
    $psdata->{psid}    = $cgi->param('id');

    if (($psdata->{topicid} eq '') || ($psdata->{psid} eq '')) {
        $error->print_error('00002e');
    }

    my $topic = Lib::Topic->new();
    $topic->load($psdata->{topicid}, 1);
    my $editor = $topic->get_editor();
    my $user = Lib::User->new();
    if (($editor ne $session_userid) && !$user->is_admin($session_userid)) {
        $error->print_error('00004e');
    }

    my @pss = $topic->get_pss();
    my @new_pss = ();
    foreach my $ps (@pss) {
        my $psid = (split(/\,/, $ps))[0];
        next if ($psid eq $psdata->{psid});
        push @new_pss, $ps;
    }
    $topic->set_pss(@new_pss);
    $topic->update();

    print "Location: ./?func=detail&id=$psdata->{topicid}", "\n\n";
}

sub _upload_file() {
    my ($func, $session_userid, $cgi) = @_;
#    $logger->write("_upload_file");   # for debug

    my $filedata;
    $filedata->{topicid}  = $cgi->param('topicid');
    $filedata->{filename} = '';

    my $topic = Lib::Topic->new();
    $topic->load($filedata->{topicid}, 0);
    my $editor = $topic->get_editor();
    my $user = Lib::User->new();
    if (($editor ne $session_userid) && !$user->is_admin($session_userid)) {
        $error->print_error('00004e');
    }

    _show_form($func, $filedata, '');
}

sub _add_file() {
    my ($session_userid, $cgi) = @_;

    my $filedata;
    $filedata->{topicid}  = $cgi->param('topicid');
    $filedata->{filename} = $cgi->param('filename');

    my $topic = Lib::Topic->new();
    $topic->load($filedata->{topicid}, 1);
    my $editor = $topic->get_editor();
    my $user = Lib::User->new();
    if (($editor ne $session_userid) && !$user->is_admin($session_userid)) {
        $error->print_error('00004e');
    }

    my @files = $topic->get_files();
    my $fileid = 0;
    if (scalar(@files) > 0) {
        $fileid = (split(/\,/, $files[$#files]))[0];
    }
    $fileid++;

    # 入力チェック
    _check_upload($filedata);

    my $dt = Lib::DateTime->new();
    $filedata->{lastdate} = $dt->local_datetime(0);
    $filedata->{lastuser} = $session_userid;

    my @paths = split(/\//, $filedata->{filename});
    my ($basename, $ext) = split(/\./, $paths[-1]);
    my $newfile = $filedata->{topicid} . "_" . $fileid . "_$basename.$ext";
    my $filesdir = $conf->get_files_dir();
    my $filepath = "$filesdir/$newfile";

    my $data = '';
    my $buffer = '';
    while(read($filedata->{filename}, $buffer, 1024)) {
        $data .= $buffer;
    }

    open my $outfh, '>', $filepath; 
    binmode $outfh; 
    print {$outfh} $data; 
    close $outfh;

    my $filesize = (stat($filepath))[7];

    my $file = "$fileid,$newfile,$ext,$filesize,$filedata->{lastdate},$filedata->{lastuser}";
    push @files, $file;
    $topic->set_files(@files);
    $topic->update();

    print "Location: ./?func=detail&id=$filedata->{topicid}", "\n\n";
}

sub _check_upload() {
    my ($filedata) = @_;
    my $language = Lib::Language->new();
    if ($filedata->{filename} eq '') {
        _show_form('upload', $filedata, $language->get_usererror('input-file'));
    }
}

sub _upload_form() {
    my ($filedata) = @_;

    my $language = Lib::Language->new();
    my $filelabel = $language->get_statictext('file');
    my $uploadbutton = $language->get_statictext('upload');
    my $cancelbutton = $language->get_statictext('cancel');

    my $form = <<"END_FORM";
<form method="post" action="./" enctype="multipart/form-data">
<input type="hidden" name="topicid" value="$filedata->{topicid}">
<input type="hidden" name="act" value="exec_editor">
<input type="hidden" name="func" value="upload">
$filelabel :<br />
<input type="file" size="70" name="filename"><br />
<p>
<input type="submit" value="$uploadbutton"> &nbsp;<input type="button" value="$cancelbutton" onclick="history.back()">
</p>
</form>
END_FORM

    return $form;
}

sub _confirm_file() {
    my ($func, $session_userid, $cgi) = @_;

    my $filedata;
    $filedata->{topicid}  = $cgi->param('topicid');
    $filedata->{fileid}   = $cgi->param('fileid');
    $filedata->{filename}  = '';
    $filedata->{filesize}  = '';
    $filedata->{lastdate}  = '';
    $filedata->{lastuser}  = '';

    if (($filedata->{topicid} eq '') || ($filedata->{fileid} eq '')) {
        $error->print_error('00002e');
    }

    my $topic = Lib::Topic->new();
    $topic->load($filedata->{topicid}, 1);
    my $editor = $topic->get_editor();
    my $user = Lib::User->new();
    if (($editor ne $session_userid) && !$user->is_admin($session_userid)) {
        $error->print_error('00004e');
    }

    my @files = $topic->get_files();
    foreach my $file (@files) {
        my ($fileid, $filename, $fileext, $filesize, $lastdate, $lastuser) = split(/\,/, $file);
        if ($fileid eq $filedata->{fileid}) {
            $filedata->{filename} = $filename;
            $filedata->{filesize} = $filesize;
            $filedata->{lastdate} = $lastdate;
            $filedata->{lastuser} = $lastuser;
            last;
        }
    }

    _show_form($func, $filedata, '');
}

sub _delfile_form() {
    my ($filedata) = @_;

    my $language = Lib::Language->new();
    my $filequestion = $language->get_message('file-question');
    my $deletebutton = $language->get_statictext('delete');
    my $cancelbutton = $language->get_statictext('cancel');

    my $form = <<"END_FORM";
<p>
<big>
<b>$filequestion</b>
</big>
</p>
<p>
$filedata->{filename} &nbsp;&nbsp;$filedata->{filesize}byte&nbsp;&nbsp;$filedata->{lastdate}&nbsp;&nbsp;$filedata->{lastuser}
</p>

<form method="post" action="./">
<input type="hidden" name="topicid" value="$filedata->{topicid}">
<input type="hidden" name="id" value="$filedata->{fileid}">
<input type="hidden" name="act" value="exec_editor">
<input type="hidden" name="func" value="delfile">
<p>
<input type="submit" value="$deletebutton"> &nbsp;<input type="button" value="$cancelbutton" onclick="history.back()">
</p>
</form>
END_FORM

    return $form;
}

sub _delete_file {
    my ($session_userid, $cgi) = @_;

    my $filedata;
    $filedata->{topicid} = $cgi->param('topicid');
    $filedata->{fileid}  = $cgi->param('id');

    if (($filedata->{topicid} eq '') || ($filedata->{fileid} eq '')) {
        $error->print_error('00002e');
    }

    my $topic = Lib::Topic->new();
    $topic->load($filedata->{topicid}, 1);
    my $editor = $topic->get_editor();
    my $user = Lib::User->new();
    if (($editor ne $session_userid) && !$user->is_admin($session_userid)) {
        $error->print_error('00004e');
    }

    my @files = $topic->get_files();
    my @new_files = ();
    foreach my $file (@files) {
        my ($fileid, $filename) = (split(/\,/, $file))[0,1];
        if ($fileid eq $filedata->{fileid}) {
            my $filesdir = $conf->get_files_dir();
            my $filepath = "$filesdir/$filename";
            unlink $filepath;
            next;
        }
        push @new_files, $file;
    }
    $topic->set_files(@new_files);
    $topic->update();

    print "Location: ./?func=detail&id=$filedata->{topicid}", "\n\n";
}

sub _input_link() {
    my ($func, $session_userid, $cgi) = @_;
#    $logger->write("_input_link");   # for debug

    my $linkdata;
    $linkdata->{topicid}     = $cgi->param('topicid');
    $linkdata->{linkid}      = '';
    $linkdata->{linktitle}   = '';
    $linkdata->{linkurl}     = '';
    $linkdata->{linksummary} = '';

    my $topic = Lib::Topic->new();
    $topic->load($linkdata->{topicid}, 0);
    my $editor = $topic->get_editor();
    my $user = Lib::User->new();
    if (($editor ne $session_userid) && !$user->is_admin($session_userid)) {
        $error->print_error('00004e');
    }

   _show_form($func, $linkdata, '');
}

sub _add_link() {
    my ($session_userid, $cgi) = @_;

    my $linkdata;
    $linkdata->{topicid}     = $cgi->param('topicid');
    $linkdata->{linkid}      = $cgi->param('linkid');
    $linkdata->{linktitle}   = $cgi->param('linktitle');
    $linkdata->{linkurl}     = $cgi->param('linkurl');
    $linkdata->{linksummary} = $cgi->param('linksummary');

    my $topic = Lib::Topic->new();
    $topic->load($linkdata->{topicid}, 1);
    my $editor = $topic->get_editor();
    my $user = Lib::User->new();
    if (($editor ne $session_userid) && !$user->is_admin($session_userid)) {
        $error->print_error('00004e');
    }

    my $util = Lib::Util->new();
    $linkdata->{linktitle} = $util->tag_invalidate($linkdata->{linktitle});

    my @links = $topic->get_links();
    my $linkid = 0;
    if (scalar(@links) > 0) {
        $linkid = (split(/\,/, $links[$#links]))[0];
    }
    $linkid++;

    # 入力チェック
    _check_link('newlink', $linkdata);

    $linkdata->{linktitle} =~ s/\r?\n//g;
    $linkdata->{linkurl} =~ s/\r?\n//g;
    $linkdata->{linksummary} =~ s/\r?\n/enc_crlf/g;

    $linkdata->{linktitle} =~ s/,/enc_conma/g; 
    $linkdata->{linkurl} =~ s/,/enc_conma/g;
    $linkdata->{linksummary} =~ s/,/enc_conma/g;

    my $link = "$linkid,$linkdata->{linktitle},$linkdata->{linkurl},$linkdata->{linksummary}";
    push @links, $link;
    $topic->set_links(@links);
    $topic->update();

    print "Location: ./?func=detail&id=$linkdata->{topicid}", "\n\n";
}

sub _edit_link() {
    my ($func, $session_userid, $cgi) = @_;

    my $linkdata;
    $linkdata->{topicid}     = $cgi->param('topicid');
    $linkdata->{linkid}      = $cgi->param('linkid');
    $linkdata->{linktitle}   = '';
    $linkdata->{linkurl}     = '';
    $linkdata->{linksummary} = '';
    if (($linkdata->{topicid} eq '') || ($linkdata->{linkid} eq '')) {
        $error->print_error('00002e');
    }

    my $topic = Lib::Topic->new();
    $topic->load($linkdata->{topicid}, 1);
    my $editor = $topic->get_editor();
    my $user = Lib::User->new();
    if (($editor ne $session_userid) && !$user->is_admin($session_userid)) {
        $error->print_error('00004e');
    }

    my @links = $topic->get_links();
    foreach my $link (@links) {
        my ($linkid, $title, $url, $summary) = split(/\,/, $link);
        if ($linkid eq $linkdata->{linkid}) {
            $linkdata->{linktitle}   = $title;
            $linkdata->{linkurl}     = $url;
            $linkdata->{linksummary} = $summary;
            $linkdata->{linksummary} =~ s/enc_crlf/\n/g;
            $linkdata->{linktitle} =~ s/enc_conma/,/g; 
            $linkdata->{linkurl} =~ s/enc_conma/,/g;
            $linkdata->{linksummary} =~ s/enc_conma/,/g;
            last;
        }
    }

    _show_form($func, $linkdata, '');
}

sub _update_link() {
    my ($session_userid, $cgi) = @_;

    my $linkdata;
    $linkdata->{topicid}     = $cgi->param('topicid');
    $linkdata->{linkid}      = $cgi->param('linkid');
    $linkdata->{linktitle}   = $cgi->param('linktitle');
    $linkdata->{linkurl}     = $cgi->param('linkurl');
    $linkdata->{linksummary} = $cgi->param('linksummary');

    if (($linkdata->{topicid} eq '') || ($linkdata->{linkid} eq '')) {
        $error->print_error('00002e');
    }

    my $topic = Lib::Topic->new();
    $topic->load($linkdata->{topicid}, 1);
    my $editor = $topic->get_editor();
    my $user = Lib::User->new();
    if (($editor ne $session_userid) && !$user->is_admin($session_userid)) {
        $error->print_error('00004e');
    }

    my $util = Lib::Util->new();
    $linkdata->{linktitle} = $util->tag_invalidate($linkdata->{linktitle});

    # 入力チェック
    _check_link('editlink', $linkdata);

    my @links = $topic->get_links();
    my @new_links = ();
    foreach my $link (@links) {
        my ($linkid, $title, $url, $summary) = split(/\,/, $link);
        if ($linkid eq $linkdata->{linkid}) {

            $linkdata->{linktitle} =~ s/\r?\n//g;
            $linkdata->{linkurl} =~ s/\r?\n//g;
            $linkdata->{linksummary} =~ s/\r?\n/enc_crlf/g;

            $linkdata->{linktitle} =~ s/,/enc_conma/g; 
            $linkdata->{linkurl} =~ s/,/enc_conma/g;
            $linkdata->{linksummary} =~ s/,/enc_conma/g;

            $link = "$linkdata->{linkid},$linkdata->{linktitle},$linkdata->{linkurl},$linkdata->{linksummary}";
        }
        push @new_links, $link;
    }
    $topic->set_links(@new_links);
    $topic->update();

    print "Location: ./?func=detail&id=$linkdata->{topicid}", "\n\n";
}

sub _link_form() {
    my ($func, $linkdata) = @_;
#    $logger->write("_link_form : func = $func");   # for debug

    my $language = Lib::Language->new();
    my $titlelabel = $language->get_statictext('title');
    my $urllabel = $language->get_statictext('url');
    my $summarylabel = $language->get_statictext('summary');

    my $button = $language->get_statictext('add');
    if ($func eq 'editlink') {
        $button = $language->get_statictext('update');
    }
    my $cancelbutton = $language->get_statictext('cancel');

    my $form = <<"END_FORM";
<form method="post" action="./">
<input type="hidden" name="topicid" value="$linkdata->{topicid}">
<input type="hidden" name="linkid" value="$linkdata->{linkid}">
<input type="hidden" name="act" value="exec_editor">
<input type="hidden" name="func" value="$func">
$titlelabel :<br />
<input type="text" name="linktitle" size="100" value="$linkdata->{linktitle}"><br />
$urllabel :<br />
<input type="text" name="linkurl" size="100" value="$linkdata->{linkurl}"><br />
$summarylabel :<br />
<textarea name="linksummary" rows="12" cols="75">$linkdata->{linksummary}</textarea><br />
<p>
<input type="submit" value="$button"> &nbsp;<input type="button" value="$cancelbutton" onclick="history.back()">
</p>
</form>
END_FORM

    return $form;
}

sub _check_link() {
    my ($func, $linkdata) = @_;

    my $language = Lib::Language->new();
    if ($linkdata->{linktitle} eq '') {
        _show_form($func, $linkdata, $language->get_usererror('input-title'));
    }
    if ($linkdata->{linkurl} eq '') {
        _show_form($func, $linkdata, $language->get_usererror('input-url'));
    }
}

sub _confirm_link() {
    my ($func, $session_userid, $cgi) = @_;

    my $linkdata;
    $linkdata->{topicid}     = $cgi->param('topicid');
    $linkdata->{linkid}      = $cgi->param('id');
    $linkdata->{linktitle}   = '';
    $linkdata->{linkurl}     = '';
    $linkdata->{linksummary} = '';

    if (($linkdata->{topicid} eq '') || ($linkdata->{linkid} eq '')) {
        $error->print_error('00002e');
    }

    my $topic = Lib::Topic->new();
    $topic->load($linkdata->{topicid}, 1);
    my $editor = $topic->get_editor();
    my $user = Lib::User->new();
    if (($editor ne $session_userid) && !$user->is_admin($session_userid)) {
        $error->print_error('00004e');
    }

    my @links = $topic->get_links();
    foreach my $link (@links) {
        my ($linkid, $title, $url, $summary) = split(/\,/, $link);
        if ($linkid eq $linkdata->{linkid}) {
            $linkdata->{linktitle}   = $title;
            $linkdata->{linkurl}     = $url;
            $linkdata->{linksummary} = $summary;
            $linkdata->{linksummary} =~ s/enc_crlf/<br \/>/g;
            $linkdata->{linktitle} =~ s/enc_conma/,/g; 
            $linkdata->{linkurl} =~ s/enc_conma/,/g;
            $linkdata->{linksummary} =~ s/enc_conma/,/g;
            last;
        }
    }

    _show_form($func, $linkdata, '');
}

sub _dellink_form() {
    my ($linkdata) = @_;

    my $language = Lib::Language->new();
    my $linkquestion = $language->get_message('link-question');
    my $deletebutton = $language->get_statictext('delete');
    my $cancelbutton = $language->get_statictext('cancel');

    my $form = <<"END_FORM";
<p>
<big>
<b>$linkquestion</b>
</big>
</p>

<h3><a href="$linkdata->{linkurl}">$linkdata->{linktitle}</a></h3>

$linkdata->{linksummary}

<form method="post" action="./">
<input type="hidden" name="topicid" value="$linkdata->{topicid}">
<input type="hidden" name="id" value="$linkdata->{linkid}">
<input type="hidden" name="act" value="exec_editor">
<input type="hidden" name="func" value="dellink">
<p>
<input type="submit" value="$deletebutton"> &nbsp;<input type="button" value="$cancelbutton" onclick="history.back()">
</p>
</form>
END_FORM

    return $form;
}

sub _delete_link() {
    my ($session_userid, $cgi) = @_;

    my $linkdata;
    $linkdata->{topicid} = $cgi->param('topicid');
    $linkdata->{linkid}  = $cgi->param('id');
#    $logger->write("_delete_link : linkid = $linkdata->{linkid}");   # for debug

    if (($linkdata->{topicid} eq '') || ($linkdata->{linkid} eq '')) {
        $error->print_error('00002e');
    }

    my $topic = Lib::Topic->new();
    $topic->load($linkdata->{topicid}, 1);
    my $editor = $topic->get_editor();
    my $user = Lib::User->new();
    if (($editor ne $session_userid) && !$user->is_admin($session_userid)) {
        $error->print_error('00004e');
    }

    my @links = $topic->get_links();
    my @new_links = ();
    foreach my $link (@links) {
        my $linkid = (split(/\,/, $link))[0];
        next if ($linkid eq $linkdata->{linkid});
        push @new_links, $link;
    }
    $topic->set_links(@new_links);
    $topic->update();

    print "Location: ./?func=detail&id=$linkdata->{topicid}", "\n\n";
}

sub _confirm_comment() {
    my ($func, $session_userid, $cgi) = @_;

    my $commentdata;
    $commentdata->{topicid}   = $cgi->param('topicid');
    $commentdata->{commentid} = $cgi->param('id');
    $commentdata->{status}    = 0;
    $commentdata->{postdate}  = '';
    $commentdata->{author}    = '';
    $commentdata->{mailaddr}  = '';
    $commentdata->{hpurl}     = '';
    $commentdata->{text}      = '';
    $commentdata->{evaluate}  = '';
    $commentdata->{ipaddr}    = '';

    if (($commentdata->{topicid} eq '') || ($commentdata->{commentid} eq '')) {
        $error->print_error('00002e');
    }

    my $topic = Lib::Topic->new();
    $topic->load($commentdata->{topicid}, 1);
    my $editor = $topic->get_editor();
    my $user = Lib::User->new();
    if (($editor ne $session_userid) && !$user->is_admin($session_userid)) {
        $error->print_error('00004e');
    }

    my @comments = $topic->get_comments();
    foreach my $comment (@comments) {
        my ($commentid, $status, $postdate, $author, $mailaddr, $hpurl, $text, $evaluate, $ipaddr) = split(/\,/, $comment);
        if ($commentid eq $commentdata->{commentid}) {
            $commentdata->{status}   = $status;
            $commentdata->{postdate} = $postdate;
            $commentdata->{author}   = $author;
            $commentdata->{mailaddr} = $mailaddr;
            $commentdata->{hpurl}    = $hpurl;
            $commentdata->{text}     = $text;
            $commentdata->{evaluate} = $evaluate;
            $commentdata->{ipaddr}   = $ipaddr;

            $commentdata->{author} =~ s/enc_conma/,/g;
            $commentdata->{mailaddr} =~ s/enc_conma/,/g;
            $commentdata->{hpurl} =~ s/enc_conma/,/g;
            $commentdata->{text} =~ s/enc_crlf/<br \/>/g;
            $commentdata->{text} =~ s/enc_conma/,/g;
            last;
        }
    }

    _show_form($func, $commentdata, '');
}

sub _delcomment_form() {
    my ($commentdata) = @_;

    my $language = Lib::Language->new();
    my $commentquestion = $language->get_message('comment-question');
    my $statuslabel = $language->get_statictext('status');
    my $evaluatelabel = $language->get_statictext('evaluate');
    my $deletebutton = $language->get_statictext('delete');
    my $cancelbutton = $language->get_statictext('cancel');

    my $status = $language->get_statictext('invalid');
    if ($commentdata->{status} == 1) {
        $status = $language->get_statictext('valid');
    }
    elsif ($commentdata->{status} == 2) {
        $status = $language->get_statictext('waiting');
    } 

    my $evaluate = $language->get_statictext('nothing');
    if ($commentdata->{evaluate} == 1) {
        $evaluate = $language->get_statictext('recommend');
    }

    my $form = <<"END_FORM";
<p>
<big>
<b>$commentquestion</b>
</big>
</p>
<p>
[$commentdata->{commentid}] Posted by $commentdata->{author}（$commentdata->{ipaddr}）&nbsp;at $commentdata->{postdate} &nbsp;
$statuslabel : $status &nbsp;$evaluatelabel : $evaluate
</p>
<p>
$commentdata->{text}
</p>

<form method="post" action="./">
<input type="hidden" name="topicid" value="$commentdata->{topicid}">
<input type="hidden" name="id" value="$commentdata->{commentid}">
<input type="hidden" name="act" value="exec_editor">
<input type="hidden" name="func" value="delcomment">
<p>
<input type="submit" value="$deletebutton"> &nbsp;<input type="button" value="$cancelbutton" onclick="history.back()">
</p>
</form>
END_FORM

    return $form;
}

sub _delete_comment() {
    my ($session_userid, $cgi) = @_;

    my $commentdata;
    $commentdata->{topicid}   = $cgi->param('topicid');
    $commentdata->{commentid} = $cgi->param('id');

    if (($commentdata->{topicid} eq '') || ($commentdata->{commentid} eq '')) {
        $error->print_error('00002e');
    }

    my $topic = Lib::Topic->new();
    $topic->load($commentdata->{topicid}, 1);
    my $editor = $topic->get_editor();
    my $user = Lib::User->new();
    if (($editor ne $session_userid) && !$user->is_admin($session_userid)) {
        $error->print_error('00004e');
    }

    my $point = $topic->get_point();

    my @comments = $topic->get_comments();
    my @new_comments = ();
    foreach my $comment (@comments) {
        my ($commentid, $evaluate) = (split(/\,/, $comment))[0,7];
        if ($commentid eq $commentdata->{commentid}) {
            if ($evaluate > 0) {
                $point -= $conf->get_good_point();
            }
            else {
                $point -= $conf->get_normal_point();
            }
            next;
        }
        push @new_comments, $comment;
    }
    $topic->set_comments(@new_comments);
    $topic->set_point($point) ;
    $topic->update();
    
    use Lib::Comment;
    my $comment = Lib::Comment->new();
    $comment->delete($commentdata->{topicid}, $commentdata->{commentid});

    print "Location: ./?func=detail&id=$commentdata->{topicid}", "\n\n";
}

sub _admit_comment() {
    my ($session_userid, $cgi) = @_;
#    $logger->write("_admit_comment");   # for debug

    my $commentdata;
    $commentdata->{topicid}   = $cgi->param('topicid');
    $commentdata->{commentid} = $cgi->param('id');

    if (($commentdata->{topicid} eq '') || ($commentdata->{commentid} eq '')) {
        $error->print_error('00002e');
    }

    my $topic = Lib::Topic->new();
    $topic->load($commentdata->{topicid}, 1);
    my $editor = $topic->get_editor();
    my $user = Lib::User->new();
    if (($editor ne $session_userid) && !$user->is_admin($session_userid)) {
        $error->print_error('00004e');
    }

    my @comments = $topic->get_comments();
    my @new_comments = ();
    foreach my $comment (@comments) {
        my ($commentid, $status, $postdate, $author, $mailaddr, $hpurl, $text, $evaluate, $ipaddr) = split(/\,/, $comment);
        if ($commentid eq $commentdata->{commentid}) {
            $comment = "$commentid,1,$postdate,$author,$mailaddr,$hpurl,$text,$evaluate,$ipaddr";
        }
        push @new_comments, $comment;
    }
    $topic->set_comments(@new_comments);
    $topic->update();

    print "Location: ./?func=detail&id=$commentdata->{topicid}", "\n\n";
}

sub _confirm_trackback() {
    my ($func, $session_userid, $cgi) = @_;

    my $trackbackdata;
    $trackbackdata->{topicid}     = $cgi->param('topicid');
    $trackbackdata->{trackbackid} = $cgi->param('id');
    $trackbackdata->{status}      = '';
    $trackbackdata->{recvdate}    = '';
    $trackbackdata->{url}         = '';
    $trackbackdata->{blogname}    = '';
    $trackbackdata->{title}       = '';
    $trackbackdata->{excerpt}     = '';
    $trackbackdata->{ipaddr}      = '';

    if (($trackbackdata->{topicid} eq '') || ($trackbackdata->{trackbackid} eq '')) {
        $error->print_error('00002e');
    }

    my $topic = Lib::Topic->new();
    $topic->load($trackbackdata->{topicid}, 1);
    my $editor = $topic->get_editor();
    my $user = Lib::User->new();
    if (($editor ne $session_userid) && !$user->is_admin($session_userid)) {
        $error->print_error('00004e');
    }

    my @trackbacks = $topic->get_trackbacks();
    foreach my $trackback (@trackbacks) {
        my ($trackbackid, $status, $recvdate, $url, $blogname, $title, $excerpt, $ipaddr) = split(/\,/, $trackback);
        if ($trackbackid eq $trackbackdata->{trackbackid}) {
            $trackbackdata->{status}   = $status;
            $trackbackdata->{recvdate} = $recvdate;
            $trackbackdata->{url}      = $url;
            $trackbackdata->{blogname} = $blogname;
            $trackbackdata->{title}    = $title;
            $trackbackdata->{excerpt}  = $excerpt;
            $trackbackdata->{ipaddr}   = $ipaddr;

            $trackbackdata->{url} =~ s/enc_conma/,/g;
            $trackbackdata->{blogname} =~ s/enc_conma/,/g;
            $trackbackdata->{title} =~ s/enc_conma/,/g;
            $trackbackdata->{excerpt} =~ s/enc_conma/,/g;
            last;
        }
    }

    _show_form($func, $trackbackdata, '');
}

sub _deltb_form() {
    my ($trackbackdata) = @_;

    my $language = Lib::Language->new();
    my $trackbackquestion = $language->get_message('trackback-question');
    my $deletebutton = $language->get_statictext('delete');
    my $cancelbutton = $language->get_statictext('cancel');

    my $status = $language->get_statictext('invalid');
    if ($trackbackdata->{status} == 1) {
        $status = $language->get_statictext('valid');
    }
    elsif ($trackbackdata->{status} == 2) {
        $status = $language->get_statictext('waiting');
    } 

    my $trackback_target = $conf->get_trackback_target();

    my $form = <<"END_FORM";
<p>
<big>
<b>$trackbackquestion</b>
</big>
</p>
<p>
<a href="$trackbackdata->{url}" $trackback_target>$trackbackdata->{title}</a>：
$trackbackdata->{blogname} ($trackbackdata->{recvdate})
</p>
<p>
$trackbackdata->{excerpt}
</p>

<form method="post" action="./">
<input type="hidden" name="topicid" value="$trackbackdata->{topicid}">
<input type="hidden" name="id" value="$trackbackdata->{trackbackid}">
<input type="hidden" name="act" value="exec_editor">
<input type="hidden" name="func" value="deltb">
<p>
<input type="submit" value="$deletebutton"> &nbsp;<input type="button" value="$cancelbutton" onclick="history.back()">
</p>
</form>
END_FORM

    return $form;
}

sub _delete_trackback() {
    my ($session_userid, $cgi) = @_;

    my $trackbackdata;
    $trackbackdata->{topicid}     = $cgi->param('topicid');
    $trackbackdata->{trackbackid} = $cgi->param('id');

    if (($trackbackdata->{topicid} eq '') || ($trackbackdata->{trackbackid} eq '')) {
        $error->print_error('00002e');
    }

    my $topic = Lib::Topic->new();
    $topic->load($trackbackdata->{topicid}, 1);
    my $editor = $topic->get_editor();
    my $user = Lib::User->new();
    if (($editor ne $session_userid) && !$user->is_admin($session_userid)) {
        $error->print_error('00004e');
    }

    my $point = $topic->get_point();

    my @trackbacks = $topic->get_trackbacks();
    my @new_trackbacks = ();
    foreach my $trackback (@trackbacks) {
        my $trackbackid = (split(/\,/, $trackback))[0];
        if ($trackbackid eq $trackbackdata->{trackbackid}) {
            $point -= $conf->get_normal_point();
            next;
        }
        push @new_trackbacks, $trackback;
    }
    $topic->set_trackbacks(@new_trackbacks);
    $topic->set_point($point) ;
    $topic->update();

    use Lib::Trackback;
    my $trackback = Lib::Trackback->new();
    $trackback->delete($trackbackdata->{topicid}, $trackbackdata->{trackbackid});

    print "Location: ./?func=detail&id=$trackbackdata->{topicid}", "\n\n";
}

sub _admit_trackback() {
    my ($session_userid, $cgi) = @_;

    my $trackbackdata;
    $trackbackdata->{topicid}     = $cgi->param('topicid');
    $trackbackdata->{trackbackid} = $cgi->param('id');

    if (($trackbackdata->{topicid} eq '') || ($trackbackdata->{trackbackid} eq '')) {
        $error->print_error('00002e');
    }

    my $topic = Lib::Topic->new();
    $topic->load($trackbackdata->{topicid}, 1);
    my $editor = $topic->get_editor();
    my $user = Lib::User->new();
    if (($editor ne $session_userid) && !$user->is_admin($session_userid)) {
        $error->print_error('00004e');
    }

    my @trackbacks = $topic->get_trackbacks();
    my @new_trackbacks = ();
    foreach my $trackback (@trackbacks) {
        my ($trackbackid, $status, $recvdate, $url, $blogname, $title, $excerpt, $ipaddr) = split(/\,/, $trackback);
        if ($trackbackid eq $trackbackdata->{trackbackid}) {
            $trackback = "$trackbackid,1,$recvdate,$url,$blogname,$title,$excerpt,$ipaddr";
        }
        push @new_trackbacks, $trackback;
    }
    $topic->set_trackbacks(@new_trackbacks);
    $topic->update();

    print "Location: ./?func=detail&id=$trackbackdata->{topicid}", "\n\n";
}

1;
