#------------------------------------------------------------------------------
#    59bbs, blog like bulletin board system.
#    Copyright (C) 2007-2009 Kaga, Hiroaki
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#------------------------------------------------------------------------------

package Lib::String;

use strict;
use warnings;

use Jcode;

use Lib::Logger;

my $logger = Lib::Logger->new();

# コンストラクタ
sub new {
	my $self = {};

    use Lib::App::Conf;
    my $conf = Lib::App::Conf->new();

    $self->{system_dir} = $conf->get_system_dir(); # システムディレクトリ

    $self->{tail} = '...'; # 尻尾

    bless($self);
    return $self;
}

# 尻尾
sub set_tail {
    my $self = shift;
    $self->{tail} = shift;
}

# 文字列を一定の長さで切る(文字コードはシフトＪＩＳを前提)
sub cut {
    my $self = shift;
    my $srcstr = shift;	# 第一引数：文字列
    my $cutlen = shift;	# 第二引数：切る長さ

    $srcstr =~ s/\r?\n//g;  # 改行を取り除く
    $srcstr =~ s/<.*?>//g;  # HTMLタグを取り除く
    $srcstr = $self->tag_invalidate($srcstr);

    # シフトJISに変換
    $srcstr = $self->convert($srcstr, 'utf8', 'sjis');

    # 文字列が指定の長さと同じか短い場合はそのまま返す
    my $srclen = length($srcstr);
    my $deststr = '';
    if ($srclen <= $cutlen) {
        $deststr = $srcstr;
    }
    else {
        # 指定された長さで文字列を切る
        for (my $i = 0; $i < $cutlen; $i++) {
            my $ch = substr($srcstr, $i, 1);
            $deststr .= $ch;
            if ($ch =~ /[\x81-\x9F\xE0-\xEF]/) {
                $i++;
                $ch = substr($srcstr, $i, 1);
                $deststr .= $ch;
            }
        }
        # 文章を切った場合は末尾を付ける
        $deststr .= $self->{tail};
    }

    $deststr = $self->convert($deststr, 'sjis', 'utf8');

	return $deststr;
}

# HTMLタグを無効化
sub tag_invalidate {
    my $self = shift;
    my $text = shift;

    $text =~ s/</&lt;/g;
    $text =~ s/>/&gt;/g;

    return $text;
}

# 文字エンコードの取得(デフォルト UTF-8)
sub get_encode {
    my $self = shift;
    my $text = shift;

    # 文字コードの種類を取得
    my ($code, $nmatch) = Jcode::getcode(\$text); 
    return $code;
}

# 文字エンコードの変換(変換先エンコード, 変換文字列) 変換した文字列を返す
sub convert {
    my $self = shift;
    my ($text, $srccode, $destcode) = @_;

    if ($srccode ne $destcode) {
        if ($destcode eq 'euc') {
            $text = Jcode->new($text, $srccode)->euc;
        }
        elsif ($destcode eq 'sjis') {
            $text = Jcode->new($text, $srccode)->sjis;
        }
        elsif ($destcode eq 'utf8') {
            $text = Jcode->new($text, $srccode)->utf8;
        }
        elsif ($destcode eq 'jis') {
            $text = Jcode->new($text, $srccode)->jis;
        }
    }
    return $text;
}

# テキストをMIME-Base64形式に変換
sub encode_mime {
    my $self = shift;
    my $srctext = shift;

    # テキストをMIME-Base64形式に変換
    my $desttext = jcode($srctext)->mime_encode;

    return $desttext;
}

# 英数全角英数半角変換(文字コードはシフトJIS)
sub zen_han {
    my $self = shift;
    my $text = shift;
    my $from = '[Ａ-Ｚａ-ｚ０-９，-]';
    my $to = '[A-Za-z0-9]';

    $from = jcode($from, 'utf8')->euc;
    $to = jcode($to, 'utf8')->euc;
    $text = jcode($text, 'sjis')->euc;
    $text = jcode($text)->tr($from, $to)->sjis;

	return $text;
}

# カナの半角を全角に変換(文字コードはシフトJIS)
sub kana_han_zen {
    my $self = shift;
    my $text = shift;
    $text = Jcode->new($text, 'sjis')->h2z->sjis;
    return $text;
}

1;
# End of String.pm
