#:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
# This file is part of G-language Genome Analysis Environment package
#
#     Copyright (C) 2001-2011 Keio University
#:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
# 
# G-language GAE is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
# 
# G-language GAE is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
# See the GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public
# License along with G-language GAE -- see the file COPYING.
# If not, write to the Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
# 
#END_HEADER
#
# written by Kazuharu Arakawa <gaou@sfc.keio.ac.jp> at
# G-language Project, Institute for Advanced Biosciences, Keio University.
#

package Rcmd::Normality;

use strict;
use autouse 'Carp'=>qw(croak carp);

use SubOpt;
use G::Messenger;

#:::::::::::::::::::::::::::::::::
#       Perldoc
#:::::::::::::::::::::::::::::::::


=head1 NAME

  Rcmd::Normality - Interfaces to tests of normality of R language.

=head1 DESCRIPTION

    This class is a part of G-language Genome Analysis Environment, 
    collecting interfaces to tests of normality of R language.

=cut


#::::::::::::::::::::::::::::::
#    Let the code begin...
#::::::::::::::::::::::::::::::

sub sample_data_for_normality{
    my $rcmd = shift;

    $rcmd->exec(
		'rnormality<-rnorm(100,mean=5,sd=3)'
		);
}


=head2 $rcmd->normtest()

  Name: $rcmd->normtest()   -   performs test for normality

  Description:
    Performs test for normality of the given vector, using
    Anderson-Darling test, Kolmogorov-Smirnov Lilliefors test,
    or Shapiro-Wilk test.
    
    Anderson-Darling test gives more weight to the tails
    of the distribution than Kolmogorov-Smirnov Lilliefors test
    or Shapiro-Wilk test.

    Kolmogorov-Smirnov Lilliefors test is one of the most widely 
    used methods for the test of normality, but is known to 
    perform worse than Anderson-Darling test or Shapiro-Wilk test.

    Shapiro-Wilk test is limited to samples less than 5000.

    Installation of nortest library for R language is required.
        run R as a super user - sudo R - and type the following:
        install.packages('nortest')) 
    
  Usage:
    $rcmd = new Rcmd();
    ($statistic, $pvalue) = $rcmd->normtest(@data);
      or
    ($statistic, $pvalue) = $rcmd->normtest(-sampledata=>1);

  Options:
   -method       name of the test
                 "ad" for Anderson-Darling 
                 "ks" for Kolmogorov-Smirnov Lilliefors (default)
                 "sw" for Shapiro-Wilk
   -sampledata   use sample data (default: 0)

  References:
   Thode Jr., H.C. (2002) Testing for Normality, Marcel Dekker, NewYork

  Author: 
    Kazuharu Arakawa (gaou@sfc.keio.ac.jp)

  History:
   20070613-01 initial posting

=cut


sub normtest{
    my $rcmd = shift;
    $rcmd->set_mode('tmp');

    opt_default(method=>"ks", sampledata=>0);
    my @args = opt_get(@_);
    my $sampledata = opt_val("sampledata");
    my $method = opt_val("method");
    
    if($sampledata){
	$rcmd->sample_data_for_normality();
    }else{
	$rcmd->array('rnormality', @args);
	
	if(scalar(@args) > 5000 && $method eq 'sw'){
	    carp("Shapiro-Wilk test cannot handle samples >5000.\nUsing Kolmogorov-Smirnov Lilliefors instead.\n");
	    $method = "ks";
	}
    }
    
    unless($method eq 'sw'){
	my $message = $rcmd->exec(
				  'require(nortest)'
				  );
	
	croak(
	      "\n\nPackage nortest not installed.\nrun\n\tinstall.packages(\'nortest\')\n" . 
	      "in R as root.\n\n"
	      ) if ($message =~ /FALSE/);
    }	
    
    my @result;

    if($method eq 'sw'){
	@result = $rcmd->exec(
			      'result = shapiro.test(rnormality)',
			      'c(as.vector(result$statistic), result$p.value)'
			      );
    }elsif($method eq 'ad'){
	@result = $rcmd->exec(
			      'result = ad.test(rnormality)',
			      'c(as.vector(result$statistic), result$p.value)'
			      );
    }else{
	unless($method eq 'ks'){
	    carp("Unknown method specified.\nUsing Kolmogorov-Smirnov Lillifors instead.\n");
	}

	@result = $rcmd->exec(
			      'result = lillie.test(rnormality)',
			      'c(as.vector(result$statistic), result$p.value)'
			      );
    }

    $rcmd->set_mode();
    return @result;
}






1;

