#!/usr/bin/env perl

#:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
# This file is part of G-language Database Interface Library
#
#     Copyright (C) 2005 Keio University
#:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
# 
#   $Id: G.pm,v 1.4 2002/07/30 17:40:56 gaou Exp $
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
# 
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
# See the GNU Lesser General Public License for more details.
# 
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, 
# write to the Free Software Foundation, Inc.,
# 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
# 
#END_HEADER
#
# written by Kazuharu Arakawa <gaou@g-language.org> at
# G-language Project, Institute for Advanced Biosciences, Keio University.
#

package G::DB::GDBI;

use strict;

use DBI;

use G::DB::GDBAPI;

use vars qw($VERSION @ISA @INC $AUTOLOAD);

@ISA = qw(G::DB::GDBAPI);

$VERSION = '0.1';

#::::::::::::::::::::::::::::::
#          Variables
#::::::::::::::::::::::::::::::

my $defaultdb = "g-language";
my $defaultuser = $ENV{USER};
my $defaultpass = "";
my $defaulthost = "localhost";
my $defaultdbi = "mysql";

my $dbh;
my $infile = '';

#::::::::::::::::::::::::::::::
#    Let the code begin...
#::::::::::::::::::::::::::::::

sub new {
    my $pkg = shift;
    my %arg = @_;
    my $dbi = $defaultdbi;
    my $host = $defaulthost;
    my $user = $defaultuser;
    my $pass = $defaultpass;
    my $db = $defaultdb;

    $host = $arg{"-host"} if(length $arg{"-host"});
    $dbi = $arg{"-dbi"}   if(length $arg{"-dbi"});
    $user = $arg{"-user"} if(length $arg{"-user"});
    $pass = $arg{"-pass"} if(length $arg{"-pass"});
    $db = $arg{"-db"}     if(length $arg{"-db"});

    my $this = {};
    bless $this;
    return $this if ($_[0] eq "blessed");

    $dbh = DBI->connect("dbi:$dbi:$db:$host", $user, $pass) || warn $dbh->errstr;

    return $this;
}


sub sql {
    my $this = shift;
    my $sql = shift;

    my $sth = $dbh->prepare($sql);
    my $ref = $sth->execute();

    unless($ref){
	die $sth->errstr;
    }

    my $r = 0;
    my $f = 0;
    my @rows;
    my %name;
    while(@rows = $sth->fetchrow_array){
	for($f = 0; $f < $sth->{NUM_OF_FIELDS}; $f ++){
	    $this->{$r}->{$sth->{NAME_lc}->[$f]} = $rows[$f];
	    $name{$sth->{NAME_lc}->[$f]} ++;
	}
	$r ++;
    }

    $this->{COLS} = join("\t:\t", keys %name);
    $this->{SQL} = $sql;
    $this->{QUERY} = "SQL: $sql";

    return $sth;
}


sub qsql {
    my $this = shift;
    my $sql = $dbh->quote(shift);

    my $sth = $dbh->prepare($sql);
    my $ref = $sth->execute();

    unless($ref){
	die $sth->errstr;
    }

    return $sth;
}

sub load {
    my $this = shift;
    my $table = shift;

    my $sql = "select * from $table";
    my $sth = $dbh->prepare($sql);
    my $ref = $sth->execute;

    unless($ref){
	die $sth->errstr;
    }

    my $r = 0;
    my $f = 0;
    my @rows;
    my %name;
    while(@rows = $sth->fetchrow_array){
	for($f = 0; $f < $sth->{NUM_OF_FIELDS}; $f ++){
	    $this->{$r}->{$sth->{NAME_lc}->[$f]} = $rows[$f];
	    $name{$sth->{NAME_lc}->[$f]} ++;
	}
	$r ++;
    }

    $this->{COLS} = join("\t:\t", keys %name);
    $this->{SQL} = $sql;
    $this->{QUERY} = "Table: $table";

    return $sth;
}


sub output{
    my $this = shift;
    my $output = shift || 'out.csv';

    open(OUT, '>' . $output) || die $!;
    print OUT 'id,';
    foreach my $col ($this->cols()){
	next if ($col eq 'id');
	print OUT "$col,";
    }
    print OUT "\n";

    foreach my $row ($this->rows()){
	print OUT $this->{$row}->{id}, ",";
	foreach my $col ($this->cols()){
	    next if ($col eq 'id');
	    print OUT $this->{$row}->{$col}, ",";
	}
	print OUT "\n";
    }
    close(OUT);

    return $output;
}


sub print{
    my $this = shift;
    my $first = shift || 0;
    my $last = shift || 1e12;

    print 'id,';
    foreach my $col ($this->cols()){
	next if ($col eq 'id');
	print "$col,";
    }
    print "\n";

    foreach my $row ($this->rows()){
	next unless($row >= $first);
	last unless($row <= $last);
	print $this->{$row}->{id}, ",";
	foreach my $col ($this->cols()){
	    next if ($col eq 'id');
	    print $this->{$row}->{$col}, ",";
	}
	print "\n";
    }

    return 1;
}


sub view {
    my $this = shift;
    my $sql = shift;

    my $sth = $dbh->prepare($sql);
    my $ref = $sth->execute;

    unless($ref){
	die $sth->errstr;
    }

    my $r = 0;
    my $f = 0;
    my @rows;
    my %name;
    while(@rows = $sth->fetchrow_array){
	for($f = 0; $f < $sth->{NUM_OF_FIELDS}; $f ++){
	    $this->{$r}->{$sth->{NAME_lc}->[$f]} = $rows[$f];
	    $name{$sth->{NAME_lc}->[$f]} ++;
	}
	$r ++;
    }

    $this->{COLS} = join("\t:\t", keys %name);
    $this->{SQL} = $sql;
    $this->{QUERY} = $sql;

    return $sth;
}


sub save {
    my $this = shift;
    my $table = shift;
    my $force = shift;

    if($force eq 'force'){
	$dbh->do("drop table $table");
    }

    my $sql = "create table $table (id int, ";
    my @tmp;
    my @cols = $this->cols();
    foreach my $col (@cols){
	push(@tmp, $col . " text");
    }
    $sql .= join(', ', @tmp) . ')';

    my $sth = $dbh->prepare($sql);
    my $ref = $sth->execute;

    unless($ref){
	die("\n  SQL:  $sql\n\n" . $sth->errstr);
    }

    foreach my $row ($this->rows("all")){
	my @tmp = ();
	foreach my $col (@cols){
	    push(@tmp, $dbh->quote($this->{$row}->{$col}));
	}
	my $sql = "insert into $table values($row, " . join(', ', @tmp) . ")";
	my $sth = $dbh->prepare($sql);
	my $ref = $sth->execute();

	unless($ref){
	    die("\n  SQL:  $sql\n\n" . $sth->errstr);
	}
    }

    return 1;
}    



sub DESTROY {
    my $self = shift;

    $dbh->disconnect();
    undef %{$self};
}



1;

__END__

=head1 NAME

G::DB::GDBI

=head1 SYNOPSIS

 use G::DB::GDBI;
 @ISA = (G::DB::GDBI);
   
=head1 DESCRIPTION

 Inherits all necessary classes.
 Intended to be used as a library

=back

=head1 AUTHOR

Kazuharu Gaou Arakawa, gaou@g-language.org

=head1 SEE ALSO

perl(1).

=cut





