#include "stdafx.h"

#include "FileInfoCollectionImpl.hpp"
#include "ObjectLockUty.hpp"

using namespace FWatchCore2Lib;

//////////////////////////////////////////////////

HRESULT CFileInfoEnumerator::FinalConstruct() throw()
{
	pos_ = NULL;
	pParent_ = NULL;
	version_ = 0;

	return S_OK;
}

HRESULT __stdcall CFileInfoEnumerator::Reset() throw()
{
	if ( !pParent_) {
		ATLASSERT(false);
		return E_FAIL;
	}

	ThreadGuard<CFileInfoCollection> lock(pParent_);
	ThreadGuard<CFileInfoEnumerator> lock2(this);

	HRESULT hr = CheckVersion();
	if (FAILED(hr)) {
		return hr;
	}

	pos_ = pParent_->fileInfoMap_.GetStartPosition();

	return S_OK;
}

HRESULT __stdcall CFileInfoEnumerator::HasNext() throw()
{
	if ( !pParent_) {
		ATLASSERT(false);
		return E_FAIL;
	}

	ThreadGuard<CFileInfoCollection> lock(pParent_);
	ThreadGuard<CFileInfoEnumerator> lock2(this);

	HRESULT hr = CheckVersion();
	if (FAILED(hr)) {
		return hr;
	}

	return pos_ != NULL ? S_OK : S_FALSE;
}

HRESULT __stdcall CFileInfoEnumerator::Next(BSTR* v_pName, IFileInfo** v_ppFileInfo) throw()
{
	if ( !pParent_) {
		ATLASSERT(false);
		return E_FAIL;
	}

	ThreadGuard<CFileInfoCollection> lock(pParent_);
	ThreadGuard<CFileInfoEnumerator> lock2(this);

	if (v_ppFileInfo) {
		ATLASSERT(!*v_ppFileInfo);
		*v_ppFileInfo = NULL;
	}

	HRESULT hr = CheckVersion();
	if (FAILED(hr)) {
		return hr;
	}

	if (pos_ == NULL) {
		ATLASSERT(false);
		return E_FAIL;
	}

	FileInfoMap::CPair* pResult = pParent_->fileInfoMap_.GetNext(pos_);
	ATLASSERT(pResult);

	if (v_pName) {
		*v_pName = SysAllocString(pResult->m_key);
	}

	if (v_ppFileInfo) {
		IFileInfoPtr pFileInfo(pResult->m_value);
		ATLASSERT(pFileInfo);
		return pFileInfo.QueryInterface(v_ppFileInfo);
	}

	return S_OK;
}

HRESULT CFileInfoEnumerator::Init(CFileInfoCollection* v_pParent) throw()
{
	if ( !v_pParent) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	ThreadGuard<CFileInfoCollection> lock(v_pParent);
	ThreadGuard<CFileInfoEnumerator> lock2(this);

	pParent_ = v_pParent;
	pos_ = pParent_->fileInfoMap_.GetStartPosition();
	version_ = pParent_->version_;

	return S_OK;
}

HRESULT CFileInfoEnumerator::CheckVersion() const throw()
{
	if ( !pParent_) {
		ATLASSERT(false);
		return E_FAIL;
	}

	if (version_ != pParent_->version_) {
		return E_ABORT;
	}

	return S_OK;
}

OBJECT_ENTRY_NON_CREATEABLE_EX_AUTO(__uuidof(CFileInfoEnumerator), CFileInfoEnumerator)

///////////////////////////////////////////////////////////

/// PRIVATE

HRESULT CFileInfoCollection::FinalConstruct() throw()
{
	version_ = 0;

	return S_OK;
}

/// IFileInfoCollection

HRESULT __stdcall CFileInfoCollection::get_Length(ULONG* v_pLength) throw()
{
	if ( !v_pLength) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	ThreadGuard<CFileInfoCollection> lock(this);

	*v_pLength = static_cast<ULONG>(fileInfoMap_.GetCount());

	return S_OK;
}

HRESULT __stdcall CFileInfoCollection::new_Enum(IFileInfoEnumerator** v_ppFileInfoEnumerator) throw()
{
	if ( !v_ppFileInfoEnumerator) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	ThreadGuard<CFileInfoCollection> lock(this);

	ATLASSERT(!*v_ppFileInfoEnumerator);
	*v_ppFileInfoEnumerator = NULL;

	HRESULT hr;
	CComObject<CFileInfoEnumerator>* pFileInfoEnumerator = NULL;
	hr = CComObject<CFileInfoEnumerator>::CreateInstance(&pFileInfoEnumerator);
	if (FAILED(hr)) {
		return hr;
	}
	ATLASSERT(pFileInfoEnumerator);
	pFileInfoEnumerator->AddRef();
	
	hr = pFileInfoEnumerator->Init(this);
	if (FAILED(hr)) {
		pFileInfoEnumerator->Release();
		return hr;
	}

	*v_ppFileInfoEnumerator = pFileInfoEnumerator;

	return S_OK;
}

HRESULT __stdcall CFileInfoCollection::get_Item(LPCWSTR v_name, IFileInfo** v_ppFileInfo) throw()
{
	if ( !v_name) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	ThreadGuard<CFileInfoCollection> lock(this);

	FileInfoMap::CPair* pResult = fileInfoMap_.Lookup(v_name);
	if ( !pResult) {
		if (v_ppFileInfo) {
			ATLASSERT(!*v_ppFileInfo);
			*v_ppFileInfo = NULL;
		}
		return S_FALSE;
	}

	if (v_ppFileInfo) {
		IFileInfoPtr pFileInfo(pResult->m_value);
		ATLASSERT(pFileInfo);
		return pFileInfo.QueryInterface(v_ppFileInfo);
	}

	return S_OK;
}

HRESULT __stdcall CFileInfoCollection::set_Item(LPCWSTR v_pName, IFileInfo* v_pFileInfo) throw()
{
	if ( !v_pName || !v_pFileInfo) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	ThreadGuard<CFileInfoCollection> lock(this);

	try {
		version_ += 1;
		fileInfoMap_.SetAt(v_pName, v_pFileInfo);
	}
	catch (...) {
		return E_OUTOFMEMORY;
	}

	return S_OK;
}

HRESULT __stdcall CFileInfoCollection::remove_Item(LPCWSTR v_pName) throw()
{
	if ( !v_pName) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	ThreadGuard<CFileInfoCollection> lock(this);
	
	bool result = fileInfoMap_.RemoveKey(v_pName);

	return result ? S_OK : S_FALSE;
}

HRESULT __stdcall CFileInfoCollection::Clear() throw()
{
	ThreadGuard<CFileInfoCollection> lock(this);

	fileInfoMap_.RemoveAll();

	return S_OK;
}

HRESULT __stdcall CFileInfoCollection::Clone(IFileInfoCollection** v_ppClone) throw()
{
	if ( !v_ppClone) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	HRESULT hr;

	CComObject<CFileInfoCollection>* pClone = NULL;
	hr = CComObject<CFileInfoCollection>::CreateInstance(&pClone);
	if (FAILED(hr)) {
		return hr;
	}
	ATLASSERT(pClone);
	CComPtr<IFileInfoCollection> pFileInfoCollection(pClone);

	ThreadGuard<CFileInfoCollection> lock(this);

	ATLASSERT(!*v_ppClone);
	*v_ppClone = NULL;

	POSITION pos = fileInfoMap_.GetStartPosition();
	while (pos) {
		FileInfoMap::CPair* pResult = fileInfoMap_.GetNext(pos);
		ATLASSERT(pResult);

		CComBSTR name(pResult->m_key);
		IFileInfoPtr pFileInfo(pResult->m_value);

		hr = pClone->set_Item(name, pFileInfo);
		if (FAILED(hr)) {
			return hr;
		}
	}

	ATLASSERT(pFileInfoCollection);
	return pFileInfoCollection.QueryInterface(v_ppClone);
}

/// IFileInfoReceiver

HRESULT __stdcall CFileInfoCollection::ReceiveFileInfo(LPCWSTR v_pPath, IFileInfo* v_pFileInfo) throw()
{
	return set_Item(v_pPath, v_pFileInfo);
}

/// IFileInfoCollectionComparable

HRESULT __stdcall CFileInfoCollection::Compare(IFileInfoCollection* v_pFileInfoCollection, IFileInfoComparator* v_pComparator) throw()
{
	if ( !v_pFileInfoCollection || !v_pComparator) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	return Compare0(v_pFileInfoCollection, v_pComparator, true);
}

HRESULT __stdcall CFileInfoCollection::CompareAllItems(IFileInfoCollection* v_pFileInfoCollection, IFileInfoComparator* v_pComparator) throw()
{
	if ( !v_pFileInfoCollection || !v_pComparator) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	return Compare0(v_pFileInfoCollection, v_pComparator, false);
}

HRESULT CFileInfoCollection::Compare0(IFileInfoCollection* v_pFileInfoCollection, IFileInfoComparator* v_pComparator, bool v_fastTerminate) throw()
{

	ThreadGuard<CFileInfoCollection> lock(this);

	HRESULT hr;
	ItemExistsMap itemExistsMap;

	CComPtr<INamedFileInfoComparator> pNamedComparator;
	v_pComparator->QueryInterface(&pNamedComparator);

	hr = MakeItemExistsMap(itemExistsMap, v_pFileInfoCollection);
	if( FAILED(hr)) {
		return hr;
	}

	bool matchAll = true;

	POSITION pos = itemExistsMap.GetStartPosition();
	while (pos) {
		ItemExistsMap::CPair* pExistsMapItem = itemExistsMap.GetNext(pos);
		ATLASSERT(pExistsMapItem);

		CComBSTR name(pExistsMapItem->m_key);

		CComPtr<IFileInfo> pFileInfo1;
		CComPtr<IFileInfo> pFileInfo2;

		if (pExistsMapItem->m_value & 0x01) {
			FileInfoMap::CPair* pResult = fileInfoMap_.Lookup(name);
			ATLASSERT(pResult);
			ATLASSERT(pResult->m_value);

			pFileInfo1 = pResult->m_value;
		}
		if (pExistsMapItem->m_value & 0x02) {
			hr = v_pFileInfoCollection->get_Item(name, &pFileInfo2);
			if (FAILED(hr)) {
				return hr;
			}
			if (hr != S_OK) {
				// 폜ĂB
				ATLASSERT(false);
				return E_FAIL;
			}
		}

		if (pNamedComparator) {
			hr = pNamedComparator->IsEquals(name, pFileInfo1, pFileInfo2);
		}
		else {
			hr = v_pComparator->IsEquals(pFileInfo1, pFileInfo2);
		}

		if (FAILED(hr)) {
			return hr;
		}

		if (hr != S_OK) {
			matchAll = false;

			if (v_fastTerminate) {
				break;
			}
		}
	}

	return matchAll ? S_OK : S_FALSE;
}

HRESULT CFileInfoCollection::MakeItemExistsMap(ItemExistsMap& v_itemExistsMap, IFileInfoCollection* v_pFileInfoCollection) throw()
{
	ATLASSERT(v_pFileInfoCollection);

	// ̃RNV̈ꗗ쐬

	POSITION pos = fileInfoMap_.GetStartPosition();
	while (pos) {
		FileInfoMap::CPair* pResult = fileInfoMap_.GetNext(pos);
		ATLASSERT(pResult);

		CComBSTR name(pResult->m_key);
		ItemExistsMap::CPair* pExistsMapItem = v_itemExistsMap.Lookup(name);
		if (!pExistsMapItem) {
			try {
				v_itemExistsMap.SetAt(name, 1);
			}
			catch (...) {
				return E_OUTOFMEMORY;
			}
		}
	}

	// ̃RNV̈ꗗ쐬

	HRESULT hr;
	CComPtr<IFileInfoEnumerator> pEnumerator;
	hr = v_pFileInfoCollection->new_Enum(&pEnumerator);
	if (FAILED(hr)) {
		return hr;
	}

	for(;;) {
		hr = pEnumerator->HasNext();
		if (hr != S_OK) {
			if (FAILED(hr)) {
				return hr;
			}
			break;
		}

		CComBSTR name;
		hr = pEnumerator->Next(&name, NULL);
		if (FAILED(hr)) {
			return hr;
		}

		ItemExistsMap::CPair* pExistsMapItem = v_itemExistsMap.Lookup(name);

		DWORD flg = 2;
		if (pExistsMapItem) {
			flg = 3;
		}

		try {
			v_itemExistsMap.SetAt(name, flg);
		}
		catch (...) {
			return E_OUTOFMEMORY;
		}
	}

	return S_OK;
}

OBJECT_ENTRY_NON_CREATEABLE_EX_AUTO(__uuidof(CFileInfoCollection), CFileInfoCollection)

////////////////////////////////////////////////////

extern "C" HRESULT __stdcall CreateFileInfoCollectionObject(IFileInfoCollection** v_ppFileInfoMap) throw()
{
	if ( !v_ppFileInfoMap) {
		ATLASSERT(false);
		return E_INVALIDARG;
	}

	ATLASSERT(!*v_ppFileInfoMap);
	*v_ppFileInfoMap = NULL;

	HRESULT hr;

	CComObject<CFileInfoCollection>* pCollectionImpl = NULL;
	hr = CComObject<CFileInfoCollection>::CreateInstance(&pCollectionImpl);
	if (FAILED(hr)) {
		return hr;
	}
	ATLASSERT(pCollectionImpl);
	CComPtr<IFileInfoCollection> pCollection(pCollectionImpl);

	return pCollection.QueryInterface(v_ppFileInfoMap);
}
