/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2005 Masataka Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "futaba.h"
#include "futaba_ui.h"
#include "imageviewer.h"
#include "canvas.h"
#include "dirview.h"

#define FRAME 4
#define CURSOR_TIMEOUT 2000

gint viewer_width;
gint viewer_height;
gfloat zoom_scale;
gint zoom_width;
gint zoom_height;
gint open_way = RIGHT;

/*
 *-------------------------------------------------
 *       Utility
 *-------------------------------------------------
 */
gboolean cursor_unvisible(FbImageViewer *viewer);

gboolean
cursor_unvisible(FbImageViewer *viewer)
{
	GdkPixmap *src;
	GdkCursor *cursor;

	src = gdk_bitmap_create_from_data(NULL, "\0\0\0", 1, 1);

	cursor = gdk_cursor_new_from_pixmap(src, src,
					    &viewer->draw->style->fg[GTK_STATE_ACTIVE],
					    &viewer->draw->style->bg[GTK_STATE_ACTIVE],
					    0, 0);

	gdk_window_set_cursor(viewer->draw->window, cursor);

	gdk_cursor_unref(cursor);
	gdk_pixmap_unref(src);

	return FALSE;
}

/*
 *---------------------------------------------
 *       Zoom Window
 *---------------------------------------------
 */
void fb_zoom_window_init(FbImageViewer *viewer);
void fb_zoom_window_draw(FbImageCanvas *fic, gint x, gint y);
void fb_zoom_window_move(FbImageViewer *viewer, gint x, gint y);
void fb_zoom_window_leave(FbImageViewer *viewer);
void fb_zoom_window_destroy(FbImageViewer *viewer);

void
fb_zoom_window_init(FbImageViewer *viewer)
{
	GdkRectangle *zwin, *dummy;

	dummy = g_new0(GdkRectangle, 1);
	zwin = g_new0(GdkRectangle, 1);
	zwin->x = viewer->pointer_last_x - (zoom_width / 2);
	zwin->y = viewer->pointer_last_y - (zoom_height / 2);
	zwin->width = zoom_width;
	zwin->height = zoom_height;

	if (gdk_rectangle_intersect(viewer->first->canvas, zwin, dummy)) {
		fb_zoom_window_draw(viewer->first, viewer->pointer_last_x, viewer->pointer_last_y);
		if (!GDK_IS_PIXMAP(viewer->first->pixmap)) fb_imagecanvas_load_pixmap(viewer->first);
	}

	if (viewer->two_pages && gdk_rectangle_intersect(viewer->second->canvas, zwin, dummy)) {
		fb_zoom_window_draw(viewer->second, viewer->pointer_last_x, viewer->pointer_last_y);
		if (!GDK_IS_PIXMAP(viewer->second->pixmap)) fb_imagecanvas_load_pixmap(viewer->second);
	}

	cursor_unvisible(viewer);

	g_free(dummy);
	g_free(zwin);
}

void
fb_zoom_window_draw(FbImageCanvas *fic,
		    gint x,
		    gint y)
{
	gint src_x, src_y;
	GdkPoint points[5];
	GdkRectangle *zwin, *dest;

	if (!GDK_IS_PIXMAP(fic->zoom)) fb_imagecanvas_load_zoom(fic);

	dest = g_new0(GdkRectangle, 1);
	zwin = g_new0(GdkRectangle, 1);
	zwin->x = x - (zoom_width / 2);
	zwin->y = y - (zoom_height / 2);
	zwin->width = zoom_width;
	zwin->height = zoom_height;

	if (!gdk_rectangle_intersect(fic->canvas, zwin, dest)) {
		g_free(dest);
		g_free(zwin);
		return;
	}

	src_x = ((x + fic->x0 - fic->canvas->x) * zoom_scale) - (zoom_width / 2);
	src_y = ((y + fic->y0 - fic->canvas->y) * zoom_scale) - (zoom_height / 2);

	gdk_draw_drawable(fic->widget->window,
			  fic->widget->style->fg_gc[GTK_WIDGET_STATE(fic->widget)],
			  fic->zoom,
			  src_x, src_y, dest->x, dest->y,
			  dest->width, dest->height);

	/* zoom window frame */
	points[0].x = zwin->x + 1;
	points[0].y = zwin->y + 1;
	points[1].x = points[0].x;
	points[1].y = zwin->y + zoom_height - 2;
	points[2].x = zwin->x + zoom_width - 2;
	points[2].y = points[1].y;
	points[3].x = points[2].x;
	points[3].y = points[0].y;
	points[4].x = points[0].x;
	points[4].y = points[0].y;

	/*   points[0].x = zwin->x; */
	/*   points[0].y = zwin->y; */
	/*   points[1].x = points[0].x; */
	/*   points[1].y = zwin->y + zoom_height; */
	/*   points[2].x = zwin->x + zoom_width; */
	/*   points[2].y = points[1].y; */
	/*   points[3].x = points[2].x; */
	/*   points[3].y = points[0].y; */
	/*   points[4].x = points[0].x; */
	/*   points[4].y = points[0].y; */

	gdk_draw_lines(fic->widget->window,
		       fic->widget->style->white_gc,
		       points, 5);

	g_free(dest);
	g_free(zwin);
}

void
fb_zoom_window_move(FbImageViewer *viewer,
		    gint x,
		    gint y)
{
	GdkRectangle *zwin, *prezwin, *harea, *varea;

	zwin = g_new0(GdkRectangle, 1);
	prezwin = g_new0(GdkRectangle, 1);
	harea = g_new0(GdkRectangle, 1);
	varea = g_new0(GdkRectangle, 1);

	prezwin->x = viewer->pointer_last_x - (zoom_width / 2);
	prezwin->y = viewer->pointer_last_y - (zoom_height / 2);
	prezwin->width = zoom_width;
	prezwin->height = zoom_height;

	zwin->x = x - (zoom_width / 2);
	zwin->y = y - (zoom_height / 2);
	zwin->width = zoom_width;
	zwin->height = zoom_height;

	fb_rectangle_diff(zwin, prezwin, harea, varea);

	fb_zoom_window_draw(viewer->first, x, y);
	fb_imagecanvas_draw(viewer->first, harea);
	fb_imagecanvas_draw(viewer->first, varea);

	if (viewer->two_pages) {
		fb_zoom_window_draw(viewer->second, x, y);
		fb_imagecanvas_draw(viewer->second, harea);
		fb_imagecanvas_draw(viewer->second, varea);
	}

	g_free(zwin);
	g_free(prezwin);
	g_free(harea);
	g_free(varea);
}

void
fb_zoom_window_leave(FbImageViewer *viewer)
{
	GdkRectangle *zwin;

	zwin = g_new0(GdkRectangle, 1);
	zwin->x = viewer->pointer_last_x - (zoom_width / 2);
	zwin->y = viewer->pointer_last_y - (zoom_height / 2);
	zwin->width = zoom_width;
	zwin->height = zoom_height;

	fb_imagecanvas_draw(viewer->first, zwin);

	if (viewer->two_pages)
		fb_imagecanvas_draw(viewer->second, zwin);

	g_free(zwin);
}

void
fb_zoom_window_destroy(FbImageViewer *viewer)
{
	GdkRectangle *zwin;

	zwin = g_new0(GdkRectangle, 1);
	zwin->x = viewer->pointer_last_x - (zoom_width / 2);
	zwin->y = viewer->pointer_last_y - (zoom_height / 2);
	zwin->width = zoom_width;
	zwin->height = zoom_height;

	if (GDK_IS_PIXMAP(viewer->first->zoom)) g_object_unref(viewer->first->zoom);
	viewer->first->zoom = NULL;

	fb_imagecanvas_draw(viewer->first, zwin);

	if (viewer->two_pages) {
		if (GDK_IS_PIXMAP(viewer->second->zoom)) g_object_unref(viewer->second->zoom);
		viewer->second->zoom = NULL;

		fb_imagecanvas_draw(viewer->second, zwin);
	}

	gdk_window_set_cursor(viewer->draw->window, NULL);

	g_free(zwin);
}

/*
 *-----------------------------------------------
 *     Image Viewer
 *-----------------------------------------------
 */
static void cb_destroy(GtkObject *obj, gpointer data);
static void cb_size_allocate(GtkWidget *wg, GtkAllocation *allocation, gpointer data);
static gboolean cb_focus_in(GtkWidget *wg, GdkEventFocus *event, gpointer data);
static gboolean cb_focus_out(GtkWidget *wg, GdkEventFocus *event, gpointer data);
static gboolean cb_expose(GtkWidget *wg, GdkEventExpose *event, gpointer data);
static gboolean cb_key_press(GtkWidget *wg, GdkEventKey *event, gpointer data);
static gboolean cb_button_press(GtkWidget *wg, GdkEventButton *event, gpointer data);
static gboolean cb_button_release(GtkWidget *wg, GdkEventButton *event, gpointer data);
static gboolean cb_motion_notify(GtkWidget *wg, GdkEventMotion *event, gpointer data);
static gboolean cb_scroll(GtkWidget *wg, GdkEvent *event, gpointer data);
static gboolean cb_leave_notify(GtkWidget *wg, GdkEventCrossing *event, gpointer data);

static void
cb_destroy(GtkObject *obj,
	   gpointer data)
{
  Futaba *futaba = data;

  fb_imageviewer_end(futaba);
}

static gboolean
cb_focus_in(GtkWidget *wg,
	    GdkEventFocus *event,
	    gpointer data)
{
	gboolean flag;
	GtkWidget *item1, *item2;
	FbImageViewer *viewer;
	Futaba *futaba = data;

	futaba->viewer = NULL;

	viewer = g_object_get_data(G_OBJECT(wg), "Viewer");

	if (viewer) viewer->focus = TRUE;

	if (viewer->two_pages) {
		item1 = gtk_ui_manager_get_widget(futaba->ui_manager,
						  "/ViewerTwoPopup/TwoWindow");
		item2 = gtk_ui_manager_get_widget(futaba->ui_manager,
						  "/ViewerTwoPopup/FullScreen");
	}
	else {
		item1 = gtk_ui_manager_get_widget(futaba->ui_manager,
						  "/ViewerPopup/TwoWindow");
		item2 = gtk_ui_manager_get_widget(futaba->ui_manager,
						  "/ViewerPopup/FullScreen");
	}

	if (viewer->two_pages)
		flag = TRUE;
	else
		flag = FALSE;

	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item1), flag);

	if (viewer->size > WINDOW_FULL)
		flag = TRUE;
	else
		flag = FALSE;

	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item2), flag);

	futaba->viewer = viewer;

	return TRUE;
}

static gboolean
cb_focus_out(GtkWidget *wg,
	     GdkEventFocus *event,
	     gpointer data)
{
	Futaba *futaba = data;

	futaba->viewer->focus = FALSE;

	return TRUE;
}

static gboolean
cb_key_press(GtkWidget *wg,
	     GdkEventKey *event,
	     gpointer data)
{
	Futaba *futaba = data;

	/*   statusbar_pop(); */
	g_print("key press: %x\n",event->keyval);

	switch (event->keyval) {
	case GDK_q:
		g_signal_emit_by_name(G_OBJECT(wg), "destroy", futaba);
		break;
	case GDK_Home:
		fb_imageviewer_first_page(futaba->viewer);
		break;
	case GDK_Right:
	case GDK_n:
		fb_imageviewer_next_pages(futaba->viewer, 1);
		break;
	case GDK_Down:
		fb_imageviewer_next_pages(futaba->viewer, 10);
		break;
	case GDK_Page_Down:
		fb_imageviewer_next_pages(futaba->viewer, 50);
		break;
	case GDK_Left:
	case GDK_b:
		fb_imageviewer_prev_pages(futaba->viewer, 1);
		break;
	case GDK_Up:
		fb_imageviewer_prev_pages(futaba->viewer, 10);
		break;
	case GDK_Page_Up:
		fb_imageviewer_prev_pages(futaba->viewer, 50);
		break;
	case GDK_End:
	case GDK_e: /* temporary binding */
		fb_imageviewer_last_page(futaba->viewer);
		break;
	case GDK_h:
		gtk_window_iconify(GTK_WINDOW(wg));
		break;
	}

	return FALSE;
}

static gboolean
cb_button_press(GtkWidget *wg,
		GdkEventButton *event,
		gpointer data)
{
	FbImageViewer *viewer = data;

	/*   statusbar_pop(); */

	viewer->pointer_last_x = event->x;
	viewer->pointer_last_y = event->y;

	switch (event->button) {
	case 1: {
		GdkCursor *cursor;
	
		if (event->state == GDK_CONTROL_MASK)
			g_print("control mask\n");

		cursor = gdk_cursor_new(GDK_FLEUR);
		gdk_window_set_cursor(viewer->draw->window, cursor);
		gdk_cursor_unref(cursor);
	}
		break;
	case 2: {
		if (!viewer->zooming)
			fb_zoom_window_init(viewer);
		else
			fb_zoom_window_destroy(viewer);
		viewer->zooming = !viewer->zooming;
	}
		break;
	case 3: {
		GtkWidget *menu;

		if (viewer->two_pages)
			menu = viewer->twopopup;
		else
			menu = viewer->popup;

		gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL,
			       event->button, event->time);
	}
		break;
	}

	return TRUE;
}

static gboolean
cb_button_release(GtkWidget *wg,
		  GdkEventButton *event,
		  gpointer data)
{
	FbImageViewer *viewer = data;

	switch (event->button) {
	case 1: {
		if (viewer->zooming) {
			cursor_unvisible(viewer);
			fb_zoom_window_draw(viewer->first, event->x, event->y);
		}
		else if (viewer->two_pages && viewer->zooming)
			fb_zoom_window_draw(viewer->second, event->x, event->y);
		else
			gdk_window_set_cursor(viewer->draw->window, NULL);
	}
		break;
	case 2:
		break;
	case 3:
		break;
	}

	return TRUE;
}

static gboolean
cb_motion_notify(GtkWidget *wg,
		 GdkEventMotion *event,
		 gpointer data)
{
	gint dx, dy;
	FbImageViewer *viewer = data;

	dx = event->x - viewer->pointer_last_x;
	dy = event->y - viewer->pointer_last_y;

	if (viewer->timer_id)
		g_source_remove(viewer->timer_id);

	if (!viewer->zooming && event->state != GDK_BUTTON1_MASK)
		gdk_window_set_cursor(viewer->draw->window, NULL);

	/* drag motion */
	if (event->state == GDK_BUTTON1_MASK && 
	    (viewer->pointer_last_x > viewer->first->canvas->x))
		fb_imagecanvas_move_image(viewer->first, dx, dy);
	else if (viewer->two_pages && event->state == GDK_BUTTON1_MASK)
		fb_imagecanvas_move_image(viewer->second, dx, dy);

	/* zoom motion */
	if (viewer->zooming && event->state != GDK_BUTTON1_MASK)
		fb_zoom_window_move(viewer, event->x, event->y);

	/* update */
	viewer->pointer_last_x = event->x;
	viewer->pointer_last_y = event->y;

	viewer->timer_id = g_timeout_add(CURSOR_TIMEOUT, (GSourceFunc) cursor_unvisible, viewer);

	return TRUE;
}

static gboolean
cb_scroll(GtkWidget *wg,
	  GdkEvent *event,
	  gpointer data)
{
	FbImageViewer *viewer = data;

	/*   statusbar_pop(); */

	switch (event->scroll.direction) {
	case 0:
		fb_imageviewer_prev_pages(viewer, 1);
		break;
	case 1:
		fb_imageviewer_next_pages(viewer, 1);
		break;
	}

	return TRUE;
}

static gboolean
cb_leave_notify(GtkWidget *wg,
		GdkEventCrossing *event,
		gpointer data)
{
	FbImageViewer *viewer = data;

	if (!viewer->timer_id)
		viewer->timer_id = g_timeout_add(CURSOR_TIMEOUT, (GSourceFunc) cursor_unvisible, viewer);
	else {
		g_source_remove(viewer->timer_id);
		viewer->timer_id = 0;
	}

	if (viewer->zooming && event->type == GDK_LEAVE_NOTIFY)
		fb_zoom_window_leave(viewer);

	return TRUE;
}

static gboolean
cb_expose(GtkWidget *wg,
	  GdkEventExpose *event,
	  gpointer data)
{
	FbImageViewer *viewer = data;

	fb_imagecanvas_draw(viewer->first, &event->area);

	if (viewer->two_pages)
		fb_imagecanvas_draw(viewer->second, &event->area);

	return TRUE;
}

static void
cb_size_allocate(GtkWidget *wg,
		 GtkAllocation *allocation,
		 gpointer data)
{
	FbImageViewer *viewer = data;

	viewer->width = allocation->width - FRAME;
	viewer->height = allocation->height - FRAME;

	if (!viewer->first || (viewer->two_pages && !viewer->second))
		return;

	/* update position info to display image */
	if (viewer->two_pages) {
		gint width;

		width = viewer->width / 2;

		if (viewer->open_way == RIGHT) {
			viewer->first->canvas->x = width;
			viewer->first->canvas->width = width;
			viewer->first->canvas->height = viewer->height;

			viewer->second->canvas->width = width;
			viewer->second->canvas->height = viewer->height;
		}
		else if (viewer->open_way == LEFT) {
			viewer->first->canvas->width = width;
			viewer->first->canvas->height = viewer->height;

			viewer->second->canvas->x = width;
			viewer->second->canvas->width = width;
			viewer->second->canvas->height = viewer->height;
		}
	}
	else {
		viewer->first->canvas->width = viewer->width;
		viewer->first->canvas->height = viewer->height;
	}

	/* update image or position of image */
	if (viewer->first->fitting) {
		if (GDK_IS_PIXMAP(viewer->first->pixmap)) g_object_unref(viewer->first->pixmap);
		viewer->first->pixmap = NULL;

		if (viewer->zooming)
			fb_imagecanvas_load_zoom(viewer->first);

		viewer->first->scale = fb_imagecanvas_calc_fit_scale(viewer->first);

	}

	viewer->first->x0 = ((viewer->first->raw_width * viewer->first->scale) - viewer->first->canvas->width) / 2;
	viewer->first->y0 = ((viewer->first->raw_height * viewer->first->scale) - viewer->first->canvas->height) / 2;

	if (viewer->two_pages) {
		if (viewer->second->fitting) {
			if (GDK_IS_PIXMAP(viewer->second->pixmap))
				g_object_unref(viewer->second->pixmap);

			viewer->second->pixmap = NULL;

			if (viewer->zooming)
				fb_imagecanvas_load_zoom(viewer->second);

			viewer->second->scale = fb_imagecanvas_calc_fit_scale(viewer->second);
		}

		viewer->second->x0 = ((viewer->second->raw_width * viewer->second->scale) - viewer->second->canvas->width) / 2;
		viewer->second->y0 = ((viewer->second->raw_height * viewer->second->scale) - viewer->second->canvas->height) / 2;
	}

	/* store viewer size */
	if (viewer->size < WINDOW_FULL) {
		viewer_width = allocation->width;
		viewer_height = allocation->height;
	}
}

FbImageViewer *
fb_imageviewer_new(Futaba *futaba)
{
	GtkWidget *top, *frame, *draw;
	FbImageViewer *viewer = g_new0(FbImageViewer, 1);

	top = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_default_size(GTK_WINDOW(top), viewer_width, viewer_height);
	gtk_window_set_title(GTK_WINDOW(top), "Viewer - Futaba");
	gtk_window_add_accel_group(GTK_WINDOW(top), gtk_ui_manager_get_accel_group(futaba->ui_manager));
	g_object_set_data(G_OBJECT(top), "Viewer", viewer);
	g_signal_connect(G_OBJECT(top), "destroy",
			 G_CALLBACK(cb_destroy), futaba);
	g_signal_connect(G_OBJECT(top), "focus-in-event",
			 G_CALLBACK(cb_focus_in), futaba);
	g_signal_connect(G_OBJECT(top), "focus-out-event",
			 G_CALLBACK(cb_focus_out), futaba);
	g_signal_connect(G_OBJECT(top), "key_press_event",
			 G_CALLBACK(cb_key_press), futaba);
	g_signal_connect(G_OBJECT(top), "size_allocate",
			 G_CALLBACK(cb_size_allocate), viewer);
	g_signal_connect(G_OBJECT(top), "leave_notify_event",
			 G_CALLBACK(cb_leave_notify), viewer);
	gtk_widget_show(top);

	frame = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_IN);
	gtk_container_add(GTK_CONTAINER(top), frame);
	gtk_widget_show(frame);

	draw = gtk_drawing_area_new();
	gtk_widget_set_events(draw,
			      GDK_POINTER_MOTION_MASK |
			      GDK_BUTTON_RELEASE_MASK |
			      GDK_BUTTON_PRESS_MASK);
	g_signal_connect_after(G_OBJECT(draw), "expose_event",
			       G_CALLBACK(cb_expose), viewer);
	g_signal_connect(G_OBJECT(draw), "button_press_event",
			 G_CALLBACK(cb_button_press), viewer);
	g_signal_connect(G_OBJECT(draw), "button_release_event",
			 G_CALLBACK(cb_button_release), viewer);
	g_signal_connect(G_OBJECT(draw), "motion_notify_event",
			 G_CALLBACK(cb_motion_notify), viewer);
	g_signal_connect(G_OBJECT(draw), "scroll_event",
			 G_CALLBACK(cb_scroll), viewer);
	gtk_container_add(GTK_CONTAINER(frame), draw);
	gtk_widget_show(draw);

	/* set member */
	viewer->top = top;
	viewer->frame = frame;
	viewer->draw = draw;
	viewer->popup = gtk_ui_manager_get_widget(futaba->ui_manager,
						  "/ViewerPopup");
	viewer->twopopup = gtk_ui_manager_get_widget(futaba->ui_manager,
						     "/ViewerTwoPopup");

	viewer->first = NULL;
	viewer->second = NULL;
	viewer->path_list = fb_dirview_get_path(futaba, IMAGE);
	viewer->position = 0;

	viewer->width = viewer_width - FRAME;
	viewer->height = viewer_height - FRAME;
	viewer->pointer_last_x = 0;
	viewer->pointer_last_y = 0;

	viewer->timer_id = g_timeout_add(CURSOR_TIMEOUT,
					 (GSourceFunc) cursor_unvisible, viewer);

	viewer->size = WINDOW_NORMAL;
	viewer->two_pages = FALSE;
	viewer->open_way = open_way;
	viewer->forward = TRUE;
	viewer->focus = TRUE;
	viewer->zooming = FALSE;

	return viewer;
}

void
fb_imageviewer_start(Futaba *futaba)
{
	gint i;
	gchar *path, *comp;
	FbFileType type;
	FbImageCanvas *fic;
	FbImageViewer *viewer;

	viewer = fb_imageviewer_new(futaba);
	futaba->viewer = viewer;
	futaba->viewer_list = g_list_append(futaba->viewer_list, viewer);

	/* check position of image */
	type = fb_dirview_get_type_at_cursor(futaba);
	if (type == IMAGE) {
		path = fb_dirview_get_path_at_cursor(futaba);
		for (i = 0; comp = g_list_nth_data(viewer->path_list, i); i++) {
			if (!g_ascii_strcasecmp(path, comp)) break;
		}
	}
	else {
		i = 0;
		path = g_list_nth_data(viewer->path_list, i);
	}
	viewer->position = i;

	fic = fb_imagecanvas_new_from_file(viewer->draw, path, 0, 0, viewer->width, viewer->height);
	fb_imagecanvas_draw(fic, fic->canvas);
  
	viewer->first = fic;
}

void
fb_imageviewer_next_pages(FbImageViewer *viewer,
			  gint n)
{
	gint nth;
	gchar *path;

	g_return_if_fail(viewer != NULL);

	nth = viewer->position + n;

	if (viewer->two_pages && !viewer->forward) nth++;

	path = g_list_nth_data(viewer->path_list, nth);
	if (!path) return;

	fb_imagecanvas_set_from_file(viewer->first, path);
	viewer->position = nth;

	if (viewer->two_pages) {
		nth++;

		path = g_list_nth_data(viewer->path_list, nth);
		if (!path) return;

		fb_imagecanvas_set_from_file(viewer->second, path);
		viewer->position = nth;
      
		viewer->forward = TRUE;
	}
}

void
fb_imageviewer_prev_pages(FbImageViewer *viewer,
			  gint n)
{
	gint nth;
	gchar *path;

	g_return_if_fail(viewer != NULL);

	nth = viewer->position - n;

	if (viewer->two_pages) {
		if (viewer->forward) nth--;

		path = g_list_nth_data(viewer->path_list, nth);
		if (!path) return;

		fb_imagecanvas_set_from_file(viewer->second, path);
		viewer->position = nth;

		nth--;
		viewer->forward = FALSE;
	}

	path = g_list_nth_data(viewer->path_list, nth);
	if (!path) return;

	fb_imagecanvas_set_from_file(viewer->first, path);

	viewer->position = nth;
}

void
fb_imageviewer_first_page(FbImageViewer *viewer)
{
	gchar *path;

	g_return_if_fail(viewer != NULL);

	path = g_list_nth_data(viewer->path_list, 0);
	fb_imagecanvas_set_from_file(viewer->first, path);
	viewer->position = 0;

	if (viewer->two_pages) {
		path = g_list_nth_data(viewer->path_list, 1);
		fb_imagecanvas_set_from_file(viewer->second, path);

		viewer->forward = TRUE;
		viewer->position = 1;
	}
}

void
fb_imageviewer_last_page(FbImageViewer *viewer)
{
	gint nth;
	gchar *path;

	g_return_if_fail(viewer != NULL);

	nth = g_list_length(viewer->path_list) - 1;
	if (viewer->two_pages) {
		path = g_list_nth_data(viewer->path_list, nth);
		fb_imagecanvas_set_from_file(viewer->second, path);

		nth--;
		viewer->forward = FALSE;
	}

	path = g_list_nth_data(viewer->path_list, nth);
	fb_imagecanvas_set_from_file(viewer->first, path);

	viewer->position = nth;
}

void
fb_imageviewer_split_window(FbImageViewer *viewer)
{
	gint width;
	gchar *path;

	viewer->two_pages = TRUE;

	/* adjust canvas and create new canvas */
	width = viewer->width / 2;
	if (viewer->open_way == RIGHT) {
		viewer->first->canvas->x = width;
		viewer->first->canvas->width = width;

		viewer->second = fb_imagecanvas_new(viewer->draw, 0, 0, width, viewer->height);
	}
	else if (viewer->open_way == LEFT) {
		viewer->first->canvas->width = width;

		viewer->second = fb_imagecanvas_new(viewer->draw, width, 0, width, viewer->height);
	}
	viewer->second->scale = viewer->first->scale;
	viewer->second->fitting = viewer->first->fitting;
  
	/* first canvas */
	if (GDK_IS_PIXMAP(viewer->first->pixmap)) g_object_unref(viewer->first->pixmap);
	viewer->first->pixmap = NULL;

	if (viewer->zooming) g_object_unref(viewer->first->zoom);
	viewer->first->zoom = NULL;

	viewer->first->x0 = ((viewer->first->raw_width * viewer->first->scale) - viewer->first->canvas->width) / 2;
	viewer->first->y0 = ((viewer->first->raw_height * viewer->first->scale) - viewer->first->canvas->height) / 2;
	if (viewer->first->fitting)
		fb_imagecanvas_zoomfit(viewer->first);
	else
		fb_imagecanvas_draw(viewer->first, viewer->first->canvas);

	/* second canvas */
	if (viewer->position + 1 > g_list_length(viewer->path_list)) {
		gdk_draw_rectangle(viewer->second->widget->window,
				   viewer->second->widget->style->bg_gc[GTK_STATE_NORMAL],
				   TRUE,
				   viewer->second->canvas->x, viewer->second->canvas->y,
				   viewer->second->canvas->width, viewer->second->canvas->height);
		return;
	}

	viewer->position++;
	path = g_list_nth_data(viewer->path_list, viewer->position);
	fb_imagecanvas_set_from_file(viewer->second, path);
}

void
fb_imageviewer_fixup_window(FbImageViewer *viewer)
{

	fb_imagecanvas_free(viewer->second);
	viewer->second = NULL;

	/* adjust first canvas size */
	viewer->first->canvas->x = 0;
	viewer->first->canvas->width = viewer->width;

	if (viewer->first->fitting) {
		if (GDK_IS_PIXMAP(viewer->first->pixmap))
			g_object_unref(viewer->first->pixmap);

		viewer->first->scale = fb_imagecanvas_calc_fit_scale(viewer->first);
	}

	viewer->first->x0 = ((viewer->first->raw_width * viewer->first->scale) - viewer->first->canvas->width) / 2;
	viewer->first->y0 = ((viewer->first->raw_height * viewer->first->scale) - viewer->first->canvas->height) / 2;

	fb_imagecanvas_draw(viewer->first, viewer->first->canvas);

	viewer->position--;
	viewer->two_pages = FALSE;
}

void
fb_imageviewer_destroy(FbImageViewer *viewer)
{

	fb_imagecanvas_free(viewer->first);
	if (viewer->second)
		fb_imagecanvas_free(viewer->second);

	for (; viewer->path_list; viewer->path_list = viewer->path_list->next)
		g_free(viewer->path_list->data);
	g_list_free(viewer->path_list);

	if (GTK_IS_WIDGET(viewer->draw)) gtk_widget_destroy(viewer->draw);

	if (GTK_IS_WIDGET(viewer->frame)) gtk_widget_destroy(viewer->frame);

	if (GTK_IS_WIDGET(viewer->top)) gtk_widget_destroy(viewer->top);

	if (viewer->timer_id) g_source_remove(viewer->timer_id);

	g_free(viewer);

	g_print("Image Viewer destroy\n");
}

void
fb_imageviewer_end(Futaba *futaba)
{
	gint i;

	if (!futaba->viewer) return;

	i = MAX(0, g_list_index(futaba->viewer_list, futaba->viewer) - 1);

	futaba->viewer_list = g_list_remove(futaba->viewer_list, futaba->viewer);
	fb_imageviewer_destroy(futaba->viewer);

	futaba->viewer = g_list_nth_data(futaba->viewer_list, i);
}
