/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2005 Masataka Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <stdio.h>
#include <pwd.h>
#include <grp.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <time.h>
#include <gtk/gtk.h>

#include "futaba.h"
#include "futaba_ui.h"
#include "dirview.h"
#include "fileutil.h"

gboolean cdir_filter;
gboolean pdir_filter;
gboolean show_coltitle;
gboolean show_col1;
gboolean show_col2;
gboolean show_col3;
gboolean show_col4;
gboolean show_col5;
gint sort_mode;
GtkSortType sort_direction;

static const gchar *week[7] = {
	N_("Sun"),
	N_("Mon"),
	N_("Tue"),
	N_("Wed"),
	N_("Thu"),
	N_("Fri"),
	N_("Sat")
};

static const gchar *ani_extensions[] = {
	"ani",
};
static const gchar *bmp_extensions[] = {
	"bmp",
};
static const gchar *gif_extensions[] = {
	"gif",
};
static const gchar *icon_extensions[] = {
	"ico",
	"cur",
};
static const gchar *jpeg_extensions[] = {
	"jpg",
	"jpeg",
	"jpe",
};
static const gchar *png_extensions[] = {
	"png",
};
static const gchar *ppm_extensions[] = {
	"ppm",
};
static const gchar *pgm_extensions[] = {
	"pgm",
};
static const gchar *pbm_extensions[] = {
	"pbm",
};
static const gchar *pnm_extensions[] = {
	"pnm",
	"pbm",
	"pgm",
	"ppm",
};
static const gchar *ras_extensions[] = {
	"ras",
};
static const gchar *tga_extensions[] = {
	"tga",
	"targa",
};
static const gchar *tiff_extensions[] = {
	"tiff",
	"tif",
};
static const gchar *wbmp_extensions[] = {
	"wbmp",
};
static const gchar *xbm_extensions[] = {
	"xbm",
};
static const gchar *xpm_extensions[] = {
	"xpm",
};

static MimeTypeEntry mime_types[] = {
   {
      mime_type:      "application/x-navi-animation",
      description:    N_("The ANI image format"),
      extensions:     ani_extensions,
      extensions_len: sizeof (ani_extensions) / sizeof (gchar *),
      icon:           NULL,
   },
   {
      mime_type:      "image/bmp",
      description:    N_("The BMP image format"),
      extensions:     bmp_extensions,
      extensions_len: sizeof (bmp_extensions) / sizeof (gchar *),
      icon:           NULL,
   },
   {
      mime_type:      "image/x-bmp",
      description:    N_("The BMP image format"),
      extensions:     bmp_extensions,
      extensions_len: sizeof (bmp_extensions) / sizeof (gchar *),
      icon:           NULL,
   },
   {
      mime_type:      "image/x-MS-bmp",
      description:    N_("The BMP image format"),
      extensions:     bmp_extensions,
      extensions_len: sizeof (bmp_extensions) / sizeof (gchar *),
      icon:           NULL,
   },
   {
      mime_type:      "image/gif",
      description:    N_("The GIF image format"),
      extensions:     gif_extensions,
      extensions_len: sizeof (gif_extensions) / sizeof (gchar *),
      icon:           NULL,
   },
   {
      mime_type:      "image/x-icon",
      description:    N_("The ICO image format"),
      extensions:     icon_extensions,
      extensions_len: sizeof (icon_extensions) / sizeof (gchar *),
      icon:           NULL,
   },
   {
      mime_type:      "image/jpeg",
      description:    N_("The JPEG image format"),
      extensions:     jpeg_extensions,
      extensions_len: sizeof (jpeg_extensions) / sizeof (gchar *),
      icon:           NULL,
   },
   {
      mime_type:      "image/png",
      description:    N_("The PNG image format"),
      extensions:     png_extensions,
      extensions_len: sizeof (png_extensions) / sizeof (gchar *),
      icon:           NULL,
   },
   {
      mime_type:      "image/x-portable-anymap",
      description:    N_("Portable Any Map Image"),
      extensions:     pnm_extensions,
      extensions_len: sizeof (pnm_extensions) / sizeof (gchar *),
      icon:           NULL,
   },
   {
      mime_type:      "image/x-portable-bitmap",
      description:    N_("The PBM image format"),
      extensions:     pbm_extensions,
      extensions_len: sizeof (pbm_extensions) / sizeof (gchar *),
      icon:           NULL,
   },
   {
      mime_type:      "image/x-portable-graymap",
      description:    N_("The PGM image format"),
      extensions:     pgm_extensions,
      extensions_len: sizeof (pgm_extensions) / sizeof (gchar *),
      icon:           NULL,
   },
   {
      mime_type:      "image/x-portable-pixmap",
      description:    N_("The PPM image format"),
      extensions:     ppm_extensions,
      extensions_len: sizeof (ppm_extensions) / sizeof (gchar *),
      icon:           NULL,
   },
   {
      mime_type:      "image/x-cmu-raster",
      description:    N_("The Sun raster image format"),
      extensions:     ras_extensions,
      extensions_len: sizeof (ras_extensions) / sizeof (gchar *),
      icon:           NULL,
   },
   {
      mime_type:      "image/x-sun-raster",
      description:    N_("The Sun raster image format"),
      extensions:     ras_extensions,
      extensions_len: sizeof (ras_extensions) / sizeof (gchar *),
      icon:           NULL,
   },
   {
      mime_type:      "image/x-tga",
      description:    N_("The Targa image format"),
      extensions:     tga_extensions,
      extensions_len: sizeof (tga_extensions) / sizeof (gchar *),
      icon:           NULL,
   },
   {
      mime_type:      "image/tiff",
      description:    N_("The TIFF image format"),
      extensions:     tiff_extensions,
      extensions_len: sizeof (tiff_extensions) / sizeof (gchar *),
      icon:           NULL,
   },
   {
      mime_type:      "image/vnd.wap.wbmp",
      description:    N_("The WBMP image format"),
      extensions:     wbmp_extensions,
      extensions_len: sizeof (wbmp_extensions) / sizeof (gchar *),
      icon:           NULL,
   },
   {
      mime_type:      "image/x-xbitmap",
      description:    N_("The XBM image format"),
      extensions:     xbm_extensions,
      extensions_len: sizeof (xbm_extensions) / sizeof (gchar *),
      icon:           NULL,
   },
   {
      mime_type:      "image/x-xpixmap",
      description:    N_("The XPM image format"),
      extensions:     xpm_extensions,
      extensions_len: sizeof (xpm_extensions) / sizeof (gchar *),
      icon:           NULL,
   },
};

/*
 *-------------------------------------
 *     string utility
 *-------------------------------------
 */
static gchar *get_absolute_path(gchar *path);
static gchar *str_mode2str(mode_t mode);
static gchar *str_uid2str(uid_t uid);
static gchar *str_gid2str(gid_t gid);
static gchar *str_size2str(off_t size);
static gchar *str_entries2str(gint count);
static gchar *str_mtime2str(time_t mtime);
static gint count_dir_entries(gchar *path, gboolean dot_filter, gboolean other_filter);

gchar *
str_local_to_utf8(gchar *string)
{
	gchar *utf8;
	GError *error = NULL;

	if (!string) return NULL;

	utf8 = g_locale_to_utf8(string, -1, NULL, NULL, &error);
	if (error) {
		/*       g_print("Unable to convert to UTF-8 from locale:\n%s\n%s\n",string, error->message); */
		g_error_free(error);
	}

	if (!utf8) utf8 = g_strdup(string);

	return utf8;
}

gchar *
str_utf8_to_local(gchar *string)
{
	gchar *local;
	GError *error = NULL;

	if (!string) return NULL;

	local = g_locale_from_utf8(string, -1, NULL, NULL, &error);
	if (error) {
		/*       g_print("Unable to convert to locale from UTF-8:\n%s\n%s\n", string, error->message); */
		g_error_free(error);
	}

	if (!local) local = g_strdup(string);

	return local;
}

static gchar *
get_absolute_path(gchar *path)
{
	gchar *base, *tmp, *abs;

	if (path[0] != '/') return NULL;

	tmp = g_strdup(path);
	if (tmp[0] == '/' && tmp[1] =='/') {
		tmp[0] = ' ';
		tmp = g_strchug(tmp);
	}

	base = g_path_get_basename(tmp);
	if (strcmp(base, ".") == 0)
		abs = g_path_get_dirname(tmp);
	else if (strcmp(base, "..") == 0) {
		gchar *tmp1;

		tmp1 = g_path_get_dirname(tmp);
		abs = g_path_get_dirname(tmp1);
		g_free(tmp1);
	}
	else {
		abs = g_strdup(tmp);
	}

	g_free(base);
	g_free(tmp);

	return abs;
}

static gchar *
str_mode2str(mode_t mode)
{
	gchar permission[11] = {"----------"};

	switch (mode & S_IFMT) {
	case S_IFREG:
		permission[0] = '-';
		break;
	case S_IFLNK:
		permission[0] = 'l';
		break;
	case S_IFDIR:
		permission[0] = 'd';
		break;
	default:
		permission[0] = '?';
		break;
	}

	if (mode & S_IRUSR)
		permission[1] = 'r';
	if (mode & S_IWUSR)
		permission[2] = 'w';
	if (mode & S_IXUSR)
		permission[3] = 'x';

	if (mode & S_IRGRP)
		permission[4] = 'r';
	if (mode & S_IWGRP)
		permission[5] = 'w';
	if (mode & S_IXGRP)
		permission[6] = 'x';

	if (mode & S_IROTH)
		permission[7] = 'r';
	if (mode & S_IWOTH)
		permission[8] = 'w';
	if (mode & S_IXOTH)
		permission[9] = 'x';
   
	if (mode & S_ISUID)
		permission[3] = 'S';
	if (mode & S_ISGID)
		permission[6] = 'S';
	if (mode & S_ISVTX)
		permission[9] = 'T';

	permission[11] = 0;

	return g_strdup(permission);
}

static gchar *
str_uid2str (uid_t uid)
{
	struct passwd *pw = getpwuid(uid);

	if (pw) return g_strdup(pw->pw_name);

	return g_strdup_printf("%d", uid);
}

static gchar *
str_gid2str (gid_t gid)
{
	struct group *gr = getgrgid(gid);

	if (gr) return g_strdup(gr->gr_name);

	return g_strdup_printf("%d", gid);
}

static gchar *
str_size2str(off_t size)
{
	if (size > 1000000)
		return g_strdup_printf(_("%4.1f Mbyte"), (gfloat) size / 1000000);
	else if (size > 1000)
		return g_strdup_printf(_("%4.1f Kbyte"), (gfloat) size / 1000);

	return g_strdup_printf(_("%d byte"), size);
}

static gchar *
str_entries2str(gint count)
{
	if (count <= 0)
		return g_strdup(_("no item"));
	else if (count == 1)
		return g_strdup(_("1 item"));
	else if (count > 1)
		return g_strdup_printf(_("%d items"), count);
}

static gchar *
str_mtime2str(time_t mtime)
{
	/*   struct tm *jst = localtime(&time); */

	/*   return g_strdup_printf("%4d/%02d/%02d %s %02d:%02d", */
	/* 			 jst->tm_year + 1900, jst->tm_mon + 1, jst->tm_mday, */
	/* 			 _(week[jst->tm_wday]), jst->tm_hour, jst->tm_min); */
	return NULL;
}


gboolean
is_archive(gchar *name)
{
	if (!name) return FALSE;

	if (g_str_has_suffix(name, ".tar") ||
	    g_str_has_suffix(name, ".bz2") ||
	    g_str_has_suffix(name, ".gz")  ||
	    g_str_has_suffix(name, ".zip") ||
	    g_str_has_suffix(name, ".ZIP") ||
	    g_str_has_suffix(name, ".lha") ||
	    g_str_has_suffix(name, ".rar"))
		{
			return TRUE;
		}

	return FALSE;
}
 
gboolean
is_imagefile(gchar *name)
{
	if (!name) return FALSE;

	if (g_str_has_suffix(name, ".jpeg") ||
	    g_str_has_suffix(name, ".jpg")  ||
	    g_str_has_suffix(name, ".JPG")  ||
	    g_str_has_suffix(name, ".gif")  ||
	    g_str_has_suffix(name, ".png")  || 
	    g_str_has_suffix(name, ".tiff") ||
	    g_str_has_suffix(name, ".xpm")  ||
	    g_str_has_suffix(name, ".bmp")  ||
	    g_str_has_suffix(name, ".xbm"))
		{
			return TRUE;
		}

	return FALSE;
}

static gint
count_dir_entries(gchar *path,
		  gboolean dot_filter,
		  gboolean other_filter)
{
	gint count = 0;
	GList *entries, *node;

	if (!path) return -1;

	entries = get_dir_entries(path, dot_filter, FALSE);
	for (node = entries; node; node = node->next) {
		if (!node->data) continue;

		if (other_filter &&
		    !g_file_test(node->data, G_FILE_TEST_IS_DIR) &&
		    !(is_imagefile(node->data) || is_archive(node->data)))
			continue;

		count++;
	}

	free_dir_entries(entries);

	return count;
}
/*
 *------------------------------------------------
 *       Directory Data Sort Func
 *------------------------------------------------
 */
static gint check_dot(GtkTreeModel *model, GtkTreeIter *a, GtkTreeIter *b);
static gint sort_path(GtkTreeModel *model, GtkTreeIter *a, GtkTreeIter *b, gpointer data);
static gint sort_type(GtkTreeModel *model, GtkTreeIter *a, GtkTreeIter *b, gpointer data);
static gint sort_size(GtkTreeModel *model, GtkTreeIter *a, GtkTreeIter *b, gpointer data);
static gint sort_mtime(GtkTreeModel *model, GtkTreeIter *a, GtkTreeIter *b, gpointer data);

static gint
check_dot(GtkTreeModel *model,
	  GtkTreeIter *a,
	  GtkTreeIter *b)
{
	gchar *str1, *str2;

	gtk_tree_model_get(model, a, COLUMN_DISPLAY_PATH, &str1, -1);
	gtk_tree_model_get(model, b, COLUMN_DISPLAY_PATH, &str2, -1);

	if (!strcmp(str1, "."))
		return -1;
	else if (!strcmp(str2, "."))
		return 1;
	else if (!strcmp(str1, ".."))
		return -1;
	else if (!strcmp(str2, ".."))
		return 1;
 
	return 0;
}

static gint
sort_path(GtkTreeModel *model,
	  GtkTreeIter *a,
	  GtkTreeIter *b,
	  gpointer data)
{
	gchar *str1, *str2;

	gtk_tree_model_get(model, a, COLUMN_DISPLAY_PATH, &str1, -1);
	gtk_tree_model_get(model, b, COLUMN_DISPLAY_PATH, &str2, -1);


	return strcmp(str1, str2);
}

static gint
sort_type(GtkTreeModel *model,
	  GtkTreeIter *a,
	  GtkTreeIter *b,
	  gpointer data)
{
	gint type1, type2;

	gtk_tree_model_get(model, a, COLUMN_TYPE, &type1, -1);
	gtk_tree_model_get(model, b, COLUMN_TYPE, &type2, -1);

	if ((type1 == DIRECTORY) && (type2 != DIRECTORY))
		return -1;
	else if ((type1 != DIRECTORY) && (type2 == DIRECTORY))
		return 1;
	else if ((type1 == ARCHIVE) && (type2 != ARCHIVE))
		return -1;
	else if ((type1 != ARCHIVE) && (type2 == ARCHIVE))
		return 1;
	else if ((type1 == IMAGE) && (type2 != IMAGE))
		return -1;
	else if ((type1 != IMAGE) && (type2 == IMAGE))
		return 1;
	else if ((type1 == OTHER) && (type2 != OTHER))
		return -1;
	else if ((type1 != OTHER) && (type2 == OTHER))
		return 1;

	return sort_path(model, a, b, data);
}

static gint
sort_size(GtkTreeModel *model,
	  GtkTreeIter *a,
	  GtkTreeIter *b,
	  gpointer data)
{
	gint size1, size2, result;

	gtk_tree_model_get(model, a, COLUMN_SIZE, &size1, -1);
	gtk_tree_model_get(model, b, COLUMN_SIZE, &size2, -1);

	result = check_dot(model, a, b);
	if (result) return result;

	if (size1 > size2)
		return 1;
	else if (size1 < size2)
		return -1;
	else if (size1 == size2)
		return 0;
}

static gint
sort_mtime(GtkTreeModel *model,
	   GtkTreeIter *a,
	   GtkTreeIter *b,
	   gpointer data)
{
	gint result;
	glong time1, time2;

	gtk_tree_model_get(model, a, COLUMN_MTIME, &time1, -1);
	gtk_tree_model_get(model, b, COLUMN_MTIME, &time2, -1);

	result = check_dot(model, a, b);
	if (result) return result;

	if (time1 > time2)
		return 1;
	else if (time1 < time2)
		return -1;
	else if (time1 == time2)
		return 0;
}
/*
 *------------------------------------------------
 *       Directory Data List
 *------------------------------------------------
 */
static gchar *get_icon_id(FbFileType type, gboolean readable);
static GtkListStore *dirlist_new(void);
static void dirlist_add_cdir(GtkListStore *list, gchar *path, gboolean dot_filter, gboolean other_filter);
static void dirlist_add_pdir(GtkListStore *list, gchar *path, gboolean dot_filter, gboolean other_filter);
static GtkTreeIter dirlist_append(GtkListStore *list, gchar *path, gboolean dot_filter, gboolean other_filter);
static void dirlist_remove(GtkListStore *list, GtkTreeIter iter);
static void dirlist_remove_nth(GtkListStore *list, gint nth);
static gboolean dirlist_set_dir(GtkListStore *list, gchar *path, gboolean dot_filter, gboolean other_filter);
static GList *dirlist_get_path(GtkListStore *list, FbFileType type);
static gint dirlist_get_count(GtkListStore *list, FbFileType type);

static GtkListStore *
dirlist_new(void)
{
	GtkListStore *list;
	GtkTreeSortable *sortable;

	list = gtk_list_store_new(NUM_COLUMNS,
				  G_TYPE_INT,
				  G_TYPE_STRING,
				  G_TYPE_INT,
				  G_TYPE_LONG,
				  G_TYPE_BOOLEAN,
				  G_TYPE_STRING,
				  G_TYPE_INT,
				  G_TYPE_STRING,
				  G_TYPE_STRING,
				  G_TYPE_STRING,
				  G_TYPE_STRING,
				  G_TYPE_STRING,
				  G_TYPE_STRING,
				  G_TYPE_BOOLEAN,
				  -1);

	sortable = GTK_TREE_SORTABLE(list);
	gtk_tree_sortable_set_sort_func(sortable,
					SORT_BY_NAME,
					sort_path, NULL, NULL);
	gtk_tree_sortable_set_sort_func(sortable,
					SORT_BY_SIZE,
					sort_size, NULL, NULL);
	gtk_tree_sortable_set_sort_func(sortable,
					SORT_BY_MTIME,
					sort_mtime, NULL, NULL);
	gtk_tree_sortable_set_sort_func(sortable,
					SORT_BY_TYPE,
					sort_type, NULL, NULL);

	gtk_tree_sortable_set_sort_column_id(sortable, sort_mode, sort_direction);

	return list;
}

static void
dirlist_add_cdir(GtkListStore *list,
		 gchar *path,
		 gboolean dot_filter,
		 gboolean other_filter)
{
	struct stat file;
	gchar *permission, *owner, *group, *size, *mtime;
	GtkTreeIter iter;

	if (!GTK_IS_LIST_STORE(list)) return;

	if (stat(path, &file)) return;

	permission = str_mode2str(file.st_mode);
	owner = str_uid2str(file.st_uid);
	group = str_gid2str(file.st_gid);
	size = str_entries2str(count_dir_entries(path, dot_filter, other_filter));
	mtime = str_mtime2str(file.st_mode);

	gtk_list_store_prepend(list, &iter);
	gtk_list_store_set(list, &iter,
			   COLUMN_TYPE, DIRECTORY,
			   COLUMN_PATH, path,
			   COLUMN_SIZE, file.st_size,
			   COLUMN_MTIME, file.st_mtime,
			   COLUMN_READABLE, TRUE,
			   COLUMN_ICON, "fb-directory",
			   COLUMN_ICON_SIZE, GTK_ICON_SIZE_LARGE_TOOLBAR,
			   COLUMN_DISPLAY_PATH, ".",
			   COLUMN_PERMISSION, permission,
			   COLUMN_OWNER, owner,
			   COLUMN_GROUP, group,
			   COLUMN_DISPLAY_SIZE, size,
			   COLUMN_DISPLAY_MTIME, mtime,
			   COLUMN_EDITABLE, FALSE,
			   -1);

	g_free(permission);
	g_free(owner);
	g_free(group);
	g_free(size);
	g_free(mtime);
}

static void
dirlist_add_pdir(GtkListStore *list,
		 gchar *path,
		 gboolean dot_filter,
		 gboolean other_filter)
{
	struct stat file;
	gchar *parent;
	gchar *permission, *owner, *group, *size, *mtime;
	GtkTreeIter iter;

	if (!path) return;
	if (!GTK_IS_LIST_STORE(list)) return;

	parent = g_path_get_dirname(path);
	if (stat(parent, &file)) return;

	permission = str_mode2str(file.st_mode);
	owner = str_uid2str(file.st_uid);
	group = str_gid2str(file.st_gid);
	size = str_entries2str(count_dir_entries(parent, dot_filter, other_filter));
	mtime = str_mtime2str(file.st_mtime);

	gtk_list_store_prepend(list, &iter);
	gtk_list_store_set(list, &iter,
			   COLUMN_TYPE, DIRECTORY,
			   COLUMN_PATH, parent,
			   COLUMN_SIZE, file.st_size,
			   COLUMN_MTIME, file.st_mtime,
			   COLUMN_READABLE, TRUE,
			   COLUMN_ICON, "fb-directory",
			   COLUMN_ICON_SIZE, GTK_ICON_SIZE_LARGE_TOOLBAR,
			   COLUMN_DISPLAY_PATH, "..",
			   COLUMN_PERMISSION, permission,
			   COLUMN_OWNER, owner,
			   COLUMN_GROUP, group,
			   COLUMN_DISPLAY_SIZE, size,
			   COLUMN_DISPLAY_MTIME, mtime,
			   COLUMN_EDITABLE, FALSE,
			   -1);

	g_free(parent);
	g_free(permission);
	g_free(owner);
	g_free(group);
	g_free(size);
	g_free(mtime);
}

static gchar *
get_icon_id(FbFileType type,
	    gboolean readable)
{
	if (type == OTHER) return NULL;

	if (readable) {
		if (type == DIRECTORY)
			return g_strdup("fb-directory");
		else if (type == IMAGE)
			return g_strdup("fb-image");
		else if (type == ARCHIVE)
			return g_strdup("fb-archive");
	}
	else {
		if (type == DIRECTORY)
			return g_strdup("fb-noread-directory");
		else if (type == IMAGE)
			return g_strdup("fb-noread-image");
		else if (type == ARCHIVE)
			return g_strdup("fb-noread-archive");
	}
	
	return NULL;
}

static GtkTreeIter
dirlist_append(GtkListStore *list,
	       gchar *path,
	       gboolean dot_filter,
	       gboolean other_filter)
{
	struct stat file;
	gboolean readable;
	gint type;
	gchar *name, *utf8_name, *icon_id;
	gchar *permission, *owner, *group, *size, *mtime;
	GtkTreeIter iter;

	if (!path) return iter;
	if (!GTK_IS_LIST_STORE(list)) return iter;

	if (stat(path, &file)) return iter;

	name = g_path_get_basename(path);
	if (S_ISDIR(file.st_mode))
		type = DIRECTORY;
	else if (S_ISREG(file.st_mode) && is_imagefile(name))
		type = IMAGE;
	else if (S_ISREG(file.st_mode) && is_archive(name))
		type = ARCHIVE;
	else
		type = OTHER;

	if (other_filter && (type == OTHER)) {
		g_free(name);
		return iter;
	}

	utf8_name = str_local_to_utf8(name);
	g_free(name);

	readable = is_readable(path);

	if (type == DIRECTORY)
		size = str_entries2str(count_dir_entries(path, dot_filter, other_filter));
	else
		size = str_size2str(file.st_size);
	icon_id = get_icon_id(type, readable);
	permission = str_mode2str(file.st_mode);
	owner = str_uid2str(file.st_uid);
	group = str_gid2str(file.st_gid);
	mtime = str_mtime2str(file.st_mtime);

	gtk_list_store_append(list, &iter);
	gtk_list_store_set(list, &iter,
			   COLUMN_TYPE, type,
			   COLUMN_PATH, path,
			   COLUMN_SIZE, file.st_size,
			   COLUMN_MTIME, file.st_mtime,
			   COLUMN_READABLE, readable,
			   COLUMN_ICON, icon_id,
			   COLUMN_ICON_SIZE, GTK_ICON_SIZE_LARGE_TOOLBAR,
			   COLUMN_DISPLAY_PATH, utf8_name,
			   COLUMN_PERMISSION, permission,
			   COLUMN_OWNER, owner,
			   COLUMN_GROUP, group,
			   COLUMN_DISPLAY_SIZE, size,
			   COLUMN_DISPLAY_MTIME, mtime,
			   COLUMN_EDITABLE, FALSE,
			   -1);

	g_free(icon_id);
	g_free(utf8_name);
	g_free(permission);
	g_free(owner);
	g_free(group);
	g_free(size);
	g_free(mtime);

	return iter;
}

static void
dirlist_remove(GtkListStore *list,
	       GtkTreeIter iter)
{
	gtk_list_store_remove(list, &iter);
}

void
dirlist_remove_nth(GtkListStore *list,
		   gint nth)
{
	GtkTreeIter iter;
	GtkTreePath *tpath;

	if (!list) return;

	tpath = gtk_tree_path_new_from_indices(nth, -1);
	if (!tpath) return;

	gtk_tree_model_get_iter(GTK_TREE_MODEL(list), &iter, tpath);
	gtk_list_store_remove(list, &iter);

	gtk_tree_path_free(tpath);
}

static gboolean
dirlist_set_dir(GtkListStore *list,
		gchar *path,
		gboolean dot_filter,
		gboolean other_filter)
{
	gint i;
	gchar *entry;
	GList *entries;

	g_return_val_if_fail(path != NULL, FALSE);
	g_return_val_if_fail(GTK_IS_LIST_STORE(list), FALSE);

	gtk_list_store_clear(list);

	entries = get_dir_entries(path, dot_filter, FALSE);
	for (i = 0; entry = g_list_nth_data(entries, i); i++)
		dirlist_append(list, entry, dot_filter, other_filter);
	free_dir_entries(entries);

	if (!pdir_filter) dirlist_add_pdir(list, path, dot_filter, other_filter);
	if (!cdir_filter) dirlist_add_cdir(list, path, dot_filter, other_filter);

	return TRUE;
}

static GList *
dirlist_get_path(GtkListStore *list,
		 FbFileType type)
{
	gint src_type;
	gchar *path, *src_path;
	GList *path_list = NULL;
	GtkTreeIter iter;

	gtk_tree_model_get_iter_first(GTK_TREE_MODEL(list), &iter);
	while (gtk_tree_model_iter_next(GTK_TREE_MODEL(list), &iter)) {
		gtk_tree_model_get(GTK_TREE_MODEL(list), &iter,
				   COLUMN_TYPE, &src_type,
				   COLUMN_PATH, &src_path,
				   -1);
		if (src_type == type) {
			path = g_strdup(src_path);
			path_list = g_list_append(path_list, path);
		}
	}

	return path_list;
}

static gint
dirlist_get_count(GtkListStore *list,
		  FbFileType type)
{
	gint src_type, count = 0;
	GtkTreeIter iter;

	if (type == DIRECTORY) count = 2;

	gtk_tree_model_get_iter_first(GTK_TREE_MODEL(list), &iter);
	while (gtk_tree_model_iter_next(GTK_TREE_MODEL(list), &iter)) {
		gtk_tree_model_get(GTK_TREE_MODEL(list), &iter,
				   COLUMN_TYPE, &src_type,
				   -1);
		if (src_type == type) count++;
	}

	return count;
}

/*
 *------------------------------------------
 *           PreView
 *------------------------------------------
 */
static void cb_allocate_preview(GtkWidget *vbox, GtkAllocation *allocation, gpointer data);
static FbPreView *preview_new(void);
static void preview_set_image(FbPreView *preview, gchar *path);
static void preview_set_size(FbPreView *preview, gint width, gint height);

static void
cb_allocate_preview(GtkWidget *widget,
		    GtkAllocation *allocation,
		    gpointer data)
{
	FbPreView *preview = data;

	if ((allocation->width <= 0) || (allocation->height <= 0))
		return ;

	if ((preview->width != allocation->width) ||
	    (preview->height != allocation->height))
		preview_set_size(preview, allocation->width, allocation->height);

	preview->width = allocation->width;
	preview->height = allocation->height;
}


static FbPreView *
preview_new(void)
{
	GdkPixmap *pixmap;
	GtkWidget *vbox, *pad;
	GtkWidget *alignment, *frame, *image;
	GtkWidget *label;
	FbPreView *preview = g_new0(FbPreView, 1);

	vbox = gtk_vbox_new(FALSE, 0);
	gtk_widget_set_size_request(vbox, futaba_width / 2, -1);
	g_signal_connect(G_OBJECT(vbox), "size-allocate",
			 G_CALLBACK(cb_allocate_preview), preview);

	/* preview */
	pad = gtk_alignment_new(0.5, 0.5, 1.0, 1.0);
	gtk_alignment_set_padding(GTK_ALIGNMENT(pad), 20, 0, 20, 20);
	gtk_box_pack_start(GTK_BOX(vbox), pad, FALSE, FALSE, 0);
	gtk_widget_show(pad);

	frame = gtk_frame_new(_("PreView"));
	gtk_container_add(GTK_CONTAINER(pad), frame);
	gtk_widget_show(frame);

	alignment = gtk_alignment_new(0.5, 0.5, 1.0, 1.0);
	gtk_alignment_set_padding(GTK_ALIGNMENT(alignment), 1, 1, 1, 1);
	gtk_container_add(GTK_CONTAINER(frame), alignment);
	gtk_widget_show(alignment);

	/*   frame = gtk_aspect_frame_new("Preview", 0.5, 0.5, 1.0, FALSE); */
	/*   gtk_container_set_border_width(GTK_CONTAINER(frame), 1); */
	/*   gtk_container_add(GTK_CONTAINER(pad), frame); */
	/*   gtk_widget_show(frame); */

	image = gtk_image_new();
	gtk_container_add(GTK_CONTAINER(alignment), image);
	gtk_widget_show(image);

	/* label */
	pad = gtk_alignment_new(0.5, 0.5, 1.0, 1.0);
	gtk_alignment_set_padding(GTK_ALIGNMENT(pad), 5, 5, 0, 0);
	gtk_box_pack_start(GTK_BOX(vbox), pad, FALSE, FALSE, 0);
	gtk_widget_show(pad);

	label = gtk_label_new(NULL);
	gtk_container_add(GTK_CONTAINER(pad), label);
	gtk_widget_show(label);

	preview->box = vbox;
	preview->frame = frame;
	preview->image = image;
	preview->label = label;
	preview->path = NULL;
	preview->width = 0;
	preview->height = 0;

	return preview;
}

static void
preview_set_image(FbPreView *preview,
		  gchar *path)
{
	gint width, height, raw_width, raw_height;
	gchar *name, *label;
	GdkPixbuf *pixbuf;
	GError **error = NULL;

	if (!preview) return;

	if (!path) {
		gtk_frame_set_label(GTK_FRAME(preview->frame), NULL);
		gtk_image_set_from_file(GTK_IMAGE(preview->image), NULL);
		gtk_label_set_text(GTK_LABEL(preview->label), NULL);

		return;
	}

	width = preview->frame->allocation.width;
	height = preview->frame->allocation.height;
	
	gdk_pixbuf_get_file_info(path, &raw_width, &raw_height);

	if ((width >= raw_width) && (height >= raw_height)) {
		width = raw_width;
		height = raw_height;
	}

	pixbuf = gdk_pixbuf_new_from_file_at_size(path, width, height, error);
	if (error);

	gtk_image_set_from_pixbuf(GTK_IMAGE(preview->image), pixbuf);
	gdk_pixbuf_unref(pixbuf);
      
	name = g_path_get_basename(path);
	label = str_local_to_utf8(name);
	g_free(name);
	gtk_frame_set_label(GTK_FRAME(preview->frame), label);
	g_free(label);

	label = g_strdup_printf("pixel: %4d x %4d", raw_width, raw_height);
	gtk_label_set_text(GTK_LABEL(preview->label), label);
	g_free(label);

	preview->path = g_strdup(path);
	preview->raw_width = raw_width;
	preview->raw_height = raw_height;
}

static void
preview_set_size(FbPreView *preview,
		 gint width,
		 gint height)
{
	gint frame_width, frame_height;
	GdkPixbuf *pixbuf;

	frame_width = width - 40;
	frame_height = height - 50;
	gtk_widget_set_size_request(preview->frame, frame_width, frame_height);

	if (!preview->path) return;
	if ((frame_width >= preview->raw_width) &&
	    (frame_height >= preview->raw_height)) return;

	pixbuf = gdk_pixbuf_new_from_file_at_size(preview->path, frame_width, frame_height, NULL);
	gtk_image_set_from_pixbuf(GTK_IMAGE(preview->image), pixbuf);
	gdk_pixbuf_unref(pixbuf);
}
/*
 *---------------------------------------------
 *          Dir View
 *---------------------------------------------
 */
static void cb_row_activated(GtkTreeView *treeview, GtkTreePath *treepath, GtkTreeViewColumn *column, gpointer data);
static void cb_change_row(GtkTreeView *treeview, gpointer data);
static gboolean cb_show_popup(GtkWidget *treeview, GdkEventButton *event, gpointer data);
static void cb_cell_edited(GtkCellRendererText *cell, gchar *str1, gchar *str2, gpointer data);
static GtkWidget *dir_view_new(Futaba *futaba);

static void
cb_row_activated(GtkTreeView *treeview,
		 GtkTreePath *tpath,
		 GtkTreeViewColumn *column,
		 gpointer data)
{
	Futaba *futaba = data;

	fb_dirview_row_activated(futaba);
}

static void
cb_change_row(GtkTreeView *treeview,
	      gpointer data)
{
	gint type;
	gchar *path, *utf8, *size, *message;
	GtkTreeIter iter;
	GtkTreeModel *model;
	GtkTreeSelection *select;
	Futaba *futaba = data;

	if (!futaba->liststore) return;
	if (!futaba->preview) return;

	fb_statusbar_pop();

	select = gtk_tree_view_get_selection(treeview);
	if (!gtk_tree_selection_get_selected(select, &model, &iter)) return;

	gtk_tree_model_get(model, &iter,
			   COLUMN_TYPE, &type,
			   COLUMN_PATH, &path,
			   COLUMN_DISPLAY_SIZE, &size,
			   -1);

	utf8 = str_local_to_utf8(path);
	if (type == DIRECTORY)
		message = g_strdup_printf(_("\"%s\" selected (containing %s)"), utf8, size);
	else
		message = g_strdup_printf(_("\"%s\" selected (size %s)"), utf8, size);
	fb_statusbar_push(message);
	g_free(utf8);
	g_free(message);

	if ((type == DIRECTORY) || (type == ARCHIVE) || (type == OTHER)) return;

	if (!futaba->show_preview) return;

	while (gtk_events_pending()) gtk_main_iteration();

	preview_set_image(futaba->preview, path);
}

static gboolean
cb_show_popup(GtkWidget *widget,
	      GdkEventButton *event,
	      gpointer data)
{
	Futaba *futaba = data;

	if (event->button == 3) {
		gboolean flag;
		gchar *path;
		GtkTreeIter iter;
		GtkTreePath *tpath;
		GtkWidget *menu;

		if (!futaba->treeview) return FALSE;
 
		flag = (futaba->image_num > 0);
		fb_ui_set_sensitive(futaba->ui_manager, "/DirViewPopup/OpenViewer", flag);
		fb_ui_set_sensitive(futaba->ui_manager, "/DirViewPopup/OpenThumb", flag);

		/* check write parmission */
		flag = futaba->writable;
		fb_ui_set_sensitive(futaba->ui_manager, "/DirViewPopup/Rename", flag);
		/*       fb_ui_set_sensitive(futaba->ui_manager, "/DirViewPopup/ChangeBulk", flag); */
		fb_ui_set_sensitive(futaba->ui_manager, "/DirViewPopup/MakeDir", flag);
		flag = (flag && (futaba->paste_path != NULL));
		fb_ui_set_sensitive(futaba->ui_manager, "/DirViewPopup/Paste", flag);

		flag = gtk_tree_view_get_path_at_pos(GTK_TREE_VIEW(futaba->treeview),
						     event->x, event->y,
						     &tpath, NULL, NULL, NULL);
  
		if (flag) gtk_tree_view_set_cursor(GTK_TREE_VIEW(futaba->treeview), tpath, NULL, FALSE);

		fb_ui_set_sensitive(futaba->ui_manager, "/DirViewPopup/Open", flag);
		fb_ui_set_sensitive(futaba->ui_manager, "/DirViewPopup/Property", flag);

		if (tpath) {
			gtk_tree_model_get_iter(GTK_TREE_MODEL(futaba->liststore), &iter, tpath);
			gtk_tree_model_get(GTK_TREE_MODEL(futaba->liststore), &iter,
					   COLUMN_DISPLAY_PATH, &path,
					   -1);

			if ((path[0] == '.') &&
			    ((path[1] == '\0') || (path[1] == '.')))
				flag = FALSE;
		}

		fb_ui_set_sensitive(futaba->ui_manager, "/DirViewPopup/Copy", flag);
		fb_ui_set_sensitive(futaba->ui_manager, "/DirViewPopup/Cut", flag);
		fb_ui_set_sensitive(futaba->ui_manager, "/DirViewPopup/Delete", flag);
		fb_ui_set_sensitive(futaba->ui_manager, "/DirViewPopup/Rename", flag);

		gtk_tree_path_free(tpath);

		menu = gtk_ui_manager_get_widget(futaba->ui_manager, "/DirViewPopup");
		gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL,
			       event->button, event->time);

		return TRUE;
	}

	return FALSE;
}

static void
cb_cell_edited(GtkCellRendererText *cell,
	       gchar *string,
	       gchar *name,
	       gpointer data)
{
	GtkTreeIter iter;
	GtkTreePath *tpath;
	Futaba *futaba = data;

	if (!futaba->liststore) return;

	fb_accelerator_unlock(futaba);

	tpath = gtk_tree_path_new_from_string(string);
	gtk_tree_model_get_iter (GTK_TREE_MODEL(futaba->liststore), &iter, tpath);
	gtk_list_store_set(futaba->liststore, &iter,
			   COLUMN_EDITABLE, FALSE,
			   -1);

	fb_dirview_rename(futaba, iter, name);

	gtk_tree_path_free(tpath);
}

static GtkWidget *
dir_view_new(Futaba *futaba)
{
	GtkListStore *liststore;
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *col, *col1, *col2, *col3, *col4, *col5;
	GtkWidget *treeview, *scrolled;

	scrolled = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scrolled),
					    GTK_SHADOW_ETCHED_IN);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
				       GTK_POLICY_AUTOMATIC,
				       GTK_POLICY_AUTOMATIC);

	treeview = gtk_tree_view_new();
	gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(treeview), show_coltitle);
	gtk_tree_view_set_rules_hint(GTK_TREE_VIEW(treeview), TRUE);
	g_signal_connect(G_OBJECT(treeview), "row-activated",
			 G_CALLBACK(cb_row_activated), futaba);
	g_signal_connect(G_OBJECT(treeview), "cursor-changed",
			 G_CALLBACK(cb_change_row), futaba);
	g_signal_connect(G_OBJECT(treeview), "button-press-event",
			 G_CALLBACK(cb_show_popup), futaba);
	gtk_container_add(GTK_CONTAINER(scrolled), treeview);
	gtk_widget_show(treeview);

	liststore = dirlist_new();
	gtk_tree_view_set_model(GTK_TREE_VIEW(treeview), GTK_TREE_MODEL(liststore));

	/* path column */
	col = gtk_tree_view_column_new();
	gtk_tree_view_column_set_resizable(col, TRUE);
	gtk_tree_view_column_set_title(col, _("Name"));
	gtk_tree_view_append_column(GTK_TREE_VIEW(treeview), col);

	renderer = gtk_cell_renderer_pixbuf_new();
	gtk_tree_view_column_pack_start(col, renderer, FALSE);
	gtk_tree_view_column_set_attributes(col, renderer,
					    "stock-id", COLUMN_ICON,
					    "stock-size", COLUMN_ICON_SIZE,
					    NULL);

	renderer = gtk_cell_renderer_text_new();
	g_signal_connect(G_OBJECT(renderer), "edited",
			 G_CALLBACK(cb_cell_edited), futaba);
	gtk_tree_view_column_pack_start(col, renderer, TRUE);
	gtk_tree_view_column_set_attributes(col, renderer,
					    "text", COLUMN_DISPLAY_PATH,
					    "editable", COLUMN_EDITABLE,
					    NULL);

	/* permission column */
	col1 = gtk_tree_view_column_new();
	gtk_tree_view_column_set_resizable(col1, TRUE);
	gtk_tree_view_column_set_visible(col1, show_col1);
	gtk_tree_view_column_set_title(col1, _("Permission"));
	gtk_tree_view_append_column(GTK_TREE_VIEW(treeview), col1);

	renderer = gtk_cell_renderer_text_new();
	gtk_tree_view_column_pack_start(col1, renderer, TRUE);
	gtk_tree_view_column_set_attributes(col1, renderer,
					    "text", COLUMN_PERMISSION,
					    NULL);

	/* owner column */
	col2 = gtk_tree_view_column_new();
	gtk_tree_view_column_set_resizable(col2, TRUE);
	gtk_tree_view_column_set_visible(col2, show_col2);
	gtk_tree_view_column_set_title(col2, _("Owner"));
	gtk_tree_view_append_column(GTK_TREE_VIEW(treeview), col2);

	renderer = gtk_cell_renderer_text_new();
	gtk_tree_view_column_pack_start(col2, renderer, TRUE);
	gtk_tree_view_column_set_attributes(col2, renderer,
					    "text", COLUMN_OWNER,
					    NULL);

	/* group column */
	col3 = gtk_tree_view_column_new();
	gtk_tree_view_column_set_resizable(col3, TRUE);
	gtk_tree_view_column_set_visible(col3, show_col3);
	gtk_tree_view_column_set_title(col3, _("Group"));
	gtk_tree_view_append_column(GTK_TREE_VIEW(treeview), col3);

	renderer = gtk_cell_renderer_text_new();
	gtk_tree_view_column_pack_start(col3, renderer, TRUE);
	gtk_tree_view_column_set_attributes(col3, renderer,
					    "text", COLUMN_GROUP,
					    NULL);

	/* size column */
	col4 = gtk_tree_view_column_new();
	gtk_tree_view_column_set_resizable(col4, TRUE);
	gtk_tree_view_column_set_visible(col4, show_col4);
	gtk_tree_view_column_set_title(col4, _("Size"));
	gtk_tree_view_append_column(GTK_TREE_VIEW(treeview), col4);

	renderer = gtk_cell_renderer_text_new();
	gtk_tree_view_column_pack_start(col4, renderer, TRUE);
	gtk_tree_view_column_set_attributes(col4, renderer,
					    "text", COLUMN_DISPLAY_SIZE,
					    NULL);

	/* mtime column */
	col5 = gtk_tree_view_column_new();
	gtk_tree_view_column_set_resizable(col5, TRUE);
	gtk_tree_view_column_set_visible(col5, show_col5);
	gtk_tree_view_column_set_title(col5, _("Date Modified"));
	gtk_tree_view_append_column(GTK_TREE_VIEW(treeview), col5);

	renderer = gtk_cell_renderer_text_new();
	gtk_tree_view_column_pack_start(col5, renderer, TRUE);
	gtk_tree_view_column_set_attributes(col5, renderer,
					    "text", COLUMN_DISPLAY_MTIME,
					    NULL);

	/* set member */
	futaba->treeview = treeview;
	futaba->liststore = liststore;
	futaba->column1 = col1;
	futaba->column2 = col2;
	futaba->column3 = col3;
	futaba->column4 = col4;
	futaba->column5 = col5;
	futaba->dot_filter = TRUE;
	futaba->other_filter = TRUE;

	return scrolled;
}
/*
 *------------------------------------------
 *           DirView Window
 *------------------------------------------
 */
GtkWidget *
fb_dirview_new(Futaba *futaba)
{
	GtkWidget *hpaned, *scroll_tree;

	hpaned = gtk_hpaned_new();

	/* scrolled treeview create and add */
	scroll_tree = dir_view_new(futaba);
	gtk_paned_pack1(GTK_PANED(hpaned), scroll_tree, TRUE, TRUE);
	gtk_widget_set_size_request(scroll_tree, futaba_width / 2, -1);
	gtk_widget_show(scroll_tree);

	/* preview create and add */
	futaba->preview = preview_new();
	futaba->show_preview = TRUE;
	gtk_paned_pack2(GTK_PANED(hpaned), futaba->preview->box, TRUE, TRUE);
	gtk_widget_show(futaba->preview->box);

	return hpaned;
}


gboolean
fb_dirview_set_history(Futaba *futaba,
		       gchar *path)
{
	gint num;
	gchar *abs_path, *utf8, *msg_num, *msg_path, *message;
	GtkTreeModel *filter;
	GtkAdjustment *adjust;

	if (!path) return FALSE;
	if (!futaba->treeview) return FALSE;
	if (!futaba->liststore) return FALSE;

	if (!g_file_test(path, G_FILE_TEST_EXISTS)) {
		fb_statusbar_push(_("Directory don't exist."));

		return FALSE;
	}

	abs_path = get_absolute_path(path);
	/* set dirview */
	if (!dirlist_set_dir(futaba->liststore, abs_path, futaba->dot_filter, futaba->other_filter)) {
		g_free(abs_path);
		return FALSE;
	}

	futaba->workdir = g_strdup(abs_path);
	futaba->writable = is_writable(abs_path);
	futaba->image_num = dirlist_get_count(futaba->liststore, IMAGE);

	/* set row position */
	adjust = gtk_tree_view_get_vadjustment(GTK_TREE_VIEW(futaba->treeview));
	gtk_adjustment_set_value(adjust, 0.0);

	fb_locationbar_set_text(futaba, abs_path);

	/* set statusbar */
	num = gtk_tree_model_iter_n_children(GTK_TREE_MODEL(futaba->liststore), NULL);
	if (!pdir_filter) num--;
	if (!cdir_filter) num--;

	if (num == 0)
		msg_num = g_strdup(_(" No item"));
	else if (num == 1)
		msg_num = g_strdup_printf(_(" 1 item exist"));
	else if (num > 1)
		msg_num = g_strdup_printf(_(" %d items exist"), num);

	utf8 = str_local_to_utf8(abs_path);
	msg_path = g_strdup_printf(_("\"%s\" open,"), utf8);
	g_free(utf8);

	message = g_strconcat(msg_path, msg_num, NULL);
	fb_statusbar_swap(message);

	g_free(abs_path);
	g_free(msg_num);
	g_free(msg_path);
	g_free(message);

	return TRUE;
}

gboolean
fb_dirview_set_dir(Futaba *futaba,
		   gchar *path)
{
	if (!fb_dirview_set_history(futaba, path))
		return FALSE;

	fb_dir_history_add(futaba, path);

	return TRUE;
}

void
fb_dirview_set_preview(Futaba *futaba,
		       gboolean show_preview)
{
	if (!futaba->preview) return;

	if (show_preview)
		gtk_widget_show(futaba->preview->box);
	else
		gtk_widget_hide(futaba->preview->box);

	futaba->show_preview = show_preview;
}

void
fb_dirview_edit_row(Futaba *futaba)
{
	GtkTreeIter iter;
	GtkTreePath *tpath;
	GtkTreeModel *model;
	GtkTreeViewColumn *column;
	GtkTreeSelection *select;

	if (!futaba->treeview) return;
	if (!futaba->liststore) return;

	select = gtk_tree_view_get_selection(GTK_TREE_VIEW(futaba->treeview));
	if (!gtk_tree_selection_get_selected(select, &model, &iter)) return;

	gtk_list_store_set(GTK_LIST_STORE(model), &iter,
			   COLUMN_EDITABLE, TRUE,
			   -1);

	gtk_tree_view_get_cursor(GTK_TREE_VIEW(futaba->treeview), &tpath, &column);
	gtk_tree_view_set_cursor(GTK_TREE_VIEW(futaba->treeview), tpath, column, TRUE);

	gtk_tree_path_free(tpath);

	fb_accelerator_lock(futaba);
}

void
fb_dirview_add_file(Futaba *futaba,
		    gchar *path)
{
	GtkTreeIter iter;
	GtkTreePath *tpath;

	if (!path) return;
	if (!futaba->liststore) return;

	iter = dirlist_append(futaba->liststore, path, futaba->dot_filter, futaba->other_filter);

	tpath = gtk_tree_model_get_path(GTK_TREE_MODEL(futaba->liststore), &iter);
	gtk_tree_view_set_cursor(GTK_TREE_VIEW(futaba->treeview), tpath, NULL, FALSE);

	gtk_tree_path_free(tpath);

	return;
}

void
fb_dirview_add_new_dir(Futaba *futaba)
{
	gint i;
	gchar *path, *name;

	if (!futaba->workdir) return;

	path = g_strconcat(futaba->workdir, _("/New1"), NULL);
	for (i = 2; g_file_test(path, G_FILE_TEST_EXISTS); i++) {
		g_free(path);
		name = g_strdup_printf(_("/New%d"), i);
		path = g_strconcat(futaba->workdir, name, NULL);
		g_free(name);
	}

	if (!make_dir(path)) {
		g_free(path);
		return;
	}

	/* FIX ME (treeviewƥ֤ˤʤäƤʤ硢
	   ǥ쥯ȥ꤬ǥåȾ֤ˤʤʤ)*/
	fb_dirview_add_file(futaba, path);
	fb_dirview_edit_row(futaba);
	/* END */

	g_free(path);

	return;
}

void
fb_dirview_remove(Futaba *futaba,
		  GtkTreeIter iter)
{
	if (!futaba->liststore) return;

	dirlist_remove(futaba->liststore, iter);
}

void
fb_dirview_remove_row(Futaba *futaba,
		      gint nth)
{
	if (!futaba->liststore) return;

	dirlist_remove_nth(futaba->liststore, nth);
}

void
fb_dirview_remove_at_cursor(Futaba *futaba)
{
	FbFileType type;
	GtkTreeIter iter;
	GtkTreeModel *model;
	GtkTreeSelection *select;

	if (!futaba->treeview) return;
	if (!futaba->liststore) return;

	select = gtk_tree_view_get_selection(GTK_TREE_VIEW(futaba->treeview));
	if (!gtk_tree_selection_get_selected(select, &model, &iter)) return;

	gtk_tree_model_get(model, &iter,
			   COLUMN_TYPE, &type,
			   -1);

	if (type == IMAGE) preview_set_image(futaba->preview, NULL);

	dirlist_remove(futaba->liststore, iter);
}

gboolean
fb_dirview_rename(Futaba *futaba,
		  GtkTreeIter iter,
		  gchar *name)
{
	gchar *tmp, *src_path, *new_path;

	if (!futaba->liststore) return FALSE;
	if (!name) return FALSE;

	gtk_tree_model_get(GTK_TREE_MODEL(futaba->liststore), &iter,
			   COLUMN_PATH, &src_path,
			   -1);
	/* FIX ME (message dialog) */
	tmp = g_path_get_basename(src_path);
	if (!strcmp(name, tmp)) {
		g_free(tmp);
		return FALSE;
	}
	g_free(tmp);

	tmp = g_path_get_dirname(src_path);
	new_path = g_strconcat(tmp, "/", name, NULL);
	g_free(tmp);

	if (rename(src_path, new_path)) {
		g_free(new_path);
		return FALSE;
	}
	/* END */

	dirlist_remove(futaba->liststore, iter);
	dirlist_append(futaba->liststore, new_path, futaba->dot_filter, futaba->other_filter);

	g_free(new_path);

	return TRUE;
}

gint
fb_dirview_get_pos(Futaba *futaba)
{
	gint *pos;
	GtkTreePath *tpath;
	GtkTreeSelection *select;

	if (!futaba->treeview) return -1;

	select = gtk_tree_view_get_selection(GTK_TREE_VIEW(futaba->treeview));
	gtk_tree_selection_select_path(select, tpath);
	gtk_tree_view_get_cursor(GTK_TREE_VIEW(futaba->treeview), &tpath, NULL);
	pos = gtk_tree_path_get_indices(tpath);

	gtk_tree_path_free(tpath);

	return *pos;
}

GList *
fb_dirview_get_path(Futaba *futaba,
		    FbFileType type)
{
	GList *path_list;

	if (!futaba->liststore) return NULL;

	path_list = dirlist_get_path(futaba->liststore, type);
	path_list = g_list_sort(path_list, (GCompareFunc) strcmp);

	return path_list;
}

gboolean
fb_dirview_get_readable_at_cursor(Futaba *futaba)
{
	gboolean readable;
	GtkTreeIter iter;
	GtkTreeModel *model;
	GtkTreeSelection *select;

	if (!futaba->treeview) return FALSE;

	select = gtk_tree_view_get_selection(GTK_TREE_VIEW(futaba->treeview));
	if (!gtk_tree_selection_get_selected(select, &model, &iter)) return FALSE;

	gtk_tree_model_get(model, &iter,
			   COLUMN_READABLE, &readable,
			   -1);

	return readable;
}

gchar *
fb_dirview_get_path_at_cursor(Futaba *futaba)
{
	gchar *path;
	GtkTreeIter iter;
	GtkTreeModel *model;
	GtkTreeSelection *select;

	if (!futaba->treeview) return FALSE;

	select = gtk_tree_view_get_selection(GTK_TREE_VIEW(futaba->treeview));
	if (!gtk_tree_selection_get_selected(select, &model, &iter)) return NULL;

	gtk_tree_model_get(model, &iter,
			   COLUMN_PATH, &path,
			   -1);

	return g_strdup(path);
}

FbFileType
fb_dirview_get_type_at_cursor(Futaba *futaba)
{
	FbFileType type;
	GtkTreeIter iter;
	GtkTreeModel *model;
	GtkTreeSelection *select;

	if (!futaba->treeview) return -1;

	select = gtk_tree_view_get_selection(GTK_TREE_VIEW(futaba->treeview));
	if (!gtk_tree_selection_get_selected(select, &model, &iter)) return -1;

	gtk_tree_model_get(model, &iter,
			   COLUMN_TYPE, &type,
			   -1);

	return type;
}

void
fb_dirview_set_sort_mode(Futaba *futaba,
			 FbDirViewSortMode mode)
{
	GtkSortType order;
	GtkTreeSortable *sortable;

	if (!futaba->liststore) return;

	sortable = GTK_TREE_SORTABLE(futaba->liststore);

	gtk_tree_sortable_get_sort_column_id(sortable, NULL, &order);
	gtk_tree_sortable_set_sort_column_id(sortable, mode, order);
}

void
fb_dirview_row_activated(Futaba *futaba)
{
	gboolean readable;
	gint type;
	GtkTreeIter iter;
	GtkTreeModel *model;
	GtkTreeSelection *select;

	if (!futaba->treeview) return;

	select = gtk_tree_view_get_selection(GTK_TREE_VIEW(futaba->treeview));
	if (!gtk_tree_selection_get_selected(select, &model, &iter)) return;

	gtk_tree_model_get(model, &iter,
			   COLUMN_TYPE, &type,
			   COLUMN_READABLE, &readable,
			   -1);

	if (!readable) {
		fb_statusbar_push(_("Parmission denied."));
		return;
	}

	switch (type) {
	case DIRECTORY: {
		gchar *path;

		gtk_tree_model_get(model, &iter,
				   COLUMN_PATH, &path,
				   -1);
		fb_dirview_set_dir(futaba, path);
	}
		break;
	case IMAGE:
		fb_imageviewer_start(futaba);
		break;
	case ARCHIVE:
		break;
	case OTHER:
	default:
		break;
	}
}
