/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2005 Masataka Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <gtk/gtk.h>

#include "futaba.h"
#include "dirview.h"
#include "imageviewer.h"
#include "canvas.h"
#include "pixbuf_utils.h"

extern gfloat zoom_scale;

/*
 *----------------------------------------------------------
 *                Utility
 *----------------------------------------------------------
 */
gboolean
fb_rectangle_diff(GdkRectangle *src,
		  GdkRectangle *dest,
		  GdkRectangle *harea,
		  GdkRectangle *varea)
{
	GdkRectangle *common;

	common = g_new0(GdkRectangle, 1);

	if (!gdk_rectangle_intersect(src, dest, common)) return FALSE;

	if ((dest->x == common->x) && (dest->y == common->y)) {
		harea->y = dest->y + common->height;
		varea->x = dest->x + common->width;
	}
	else if ((dest->x == common->x) && (dest->y != common->y)) {
		harea->y = dest->y;
		varea->x = dest->x + common->width;
	}
	else if ((dest->x != common->x) && (dest->y == common->y)) {
		harea->y = dest->y + common->height;
		varea->x = dest->x;
	}
	else if ((dest->x != common->x) && (dest->y != common->y)) {
		harea->y = dest->y;
		varea->x = dest->x;
	}

	harea->x = dest->x;
	harea->width = dest->width;
	harea->height = dest->height - common->height;

	varea->y = dest->y;
	varea->width = dest->width - common->width;
	varea->height = dest->height;

	g_free(common);

	return TRUE;
}

/*
 *----------------------------------------------------------
 *              Image Canvas
 *----------------------------------------------------------
 */
FbImageCanvas *
fb_imagecanvas_new(GtkWidget *drawable,
		   gint x,
		   gint y,
		   gint width,
		   gint height)
{
	GdkRectangle *canvas;
	FbImageCanvas *fic;

	/* position info to display image */
	canvas = g_new0(GdkRectangle, 1);
	canvas->x = x;
	canvas->y = y;
	canvas->width = width;
	canvas->height = height;

	/* set member */
	fic = g_new0(FbImageCanvas, 1);
	fic->widget = drawable;
	fic->pixmap = NULL;
	fic->zoom = NULL;
	fic->pixbuf = NULL;
	fic->canvas = canvas;
	fic->path = NULL;
	fic->fitting = TRUE;

	return fic;
}

FbImageCanvas *
fb_imagecanvas_new_from_file(GtkWidget *drawable,
			     gchar *path,
			     gint x,
			     gint y,
			     gint width,
			     gint height)
{
	gint raw_w, raw_h;
	gfloat scale;
	GdkRectangle *canvas;
	GdkPixbuf *pixbuf;
	FbImageCanvas *fic;

	pixbuf = gdk_pixbuf_new_from_file(path, NULL);
	g_return_val_if_fail(pixbuf != NULL, NULL);

	raw_w = gdk_pixbuf_get_width(pixbuf);
	raw_h = gdk_pixbuf_get_height(pixbuf);

	scale = (gfloat) height / (gfloat) raw_h;
	if (width < (raw_w * scale))
		scale = (gfloat) width / (gfloat) raw_w;

	/* position info to display image */
	canvas = g_new0(GdkRectangle, 1);
	canvas->x = 0;
	canvas->y = 0;
	canvas->width = width;
	canvas->height = height;

	/* set member */
	fic = g_new0(FbImageCanvas, 1);
	fic->widget = drawable;
	fic->pixmap = NULL;
	fic->zoom = NULL;
	fic->pixbuf = pixbuf;
	fic->canvas = canvas;
	fic->path = g_strdup(path);
	fic->raw_width = raw_w;
	fic->raw_height = raw_h;
	fic->scale = scale;
	fic->x0 = ((raw_w * scale) - width) / 2;
	fic->y0 = ((raw_h * scale) - height) / 2;
	fic->fitting = TRUE;

	return fic;
}

gboolean
fb_imagecanvas_load_pixmap(FbImageCanvas *fic)
{
	gint width, height;
	GdkPixbuf *pixbuf;
	GdkPixmap *pixmap;

	width = fic->scale * fic->raw_width;
	height = fic->scale * fic->raw_height;

	pixbuf = gdk_pixbuf_scale_simple(fic->pixbuf, width, height, GDK_INTERP_BILINEAR);

	g_return_val_if_fail(pixbuf != NULL, FALSE);

	if (GDK_IS_PIXMAP(fic->pixmap)) g_object_unref(fic->pixmap);

	pixmap = gdk_pixmap_new(fic->widget->window, width, height, -1);
	gdk_draw_rectangle(pixmap,
			   fic->widget->style->bg_gc[GTK_STATE_NORMAL],
			   TRUE,
			   0, 0, width, height);
	gdk_draw_pixbuf(pixmap,
			fic->widget->style->fg_gc[GTK_WIDGET_STATE(fic->widget)],
			pixbuf, 0, 0,
			0, 0, width, height,
			GDK_RGB_DITHER_NONE, 0, 0);

	gdk_pixbuf_unref(pixbuf);

	fic->pixmap = pixmap;

	return TRUE;
}

gboolean
fb_imagecanvas_load_zoom(FbImageCanvas *fic)
{
	gint width, height;
	GdkPixbuf *pixbuf;
	GdkPixmap *zoom;

	width = zoom_scale * fic->scale * fic->raw_width;
	height = zoom_scale * fic->scale * fic->raw_height;

	pixbuf = gdk_pixbuf_scale_simple(fic->pixbuf, width, height, GDK_INTERP_BILINEAR);

	g_return_val_if_fail(pixbuf != NULL, FALSE);

	if (GDK_IS_PIXMAP(fic->zoom)) g_object_unref(fic->zoom);

	zoom = gdk_pixmap_new(fic->widget->window, width, height, -1);
	gdk_draw_pixbuf(zoom,
			fic->widget->style->fg_gc[GTK_WIDGET_STATE(fic->widget)],
			pixbuf, 0, 0,
			0, 0, width, height,
			GDK_RGB_DITHER_NONE, 0, 0);

	gdk_pixbuf_unref(pixbuf);

	fic->zoom = zoom;

	return TRUE;
}

void
fb_imagecanvas_free(FbImageCanvas *fic)
{

	g_return_if_fail(fic != NULL);

	if (GDK_IS_PIXBUF(fic->pixbuf)) gdk_pixbuf_unref(fic->pixbuf);
	fic->pixbuf = NULL;

	if (GDK_IS_PIXMAP(fic->pixmap)) g_object_unref(fic->pixmap);
	fic->pixmap = NULL;

	if (GDK_IS_PIXMAP(fic->zoom)) g_object_unref(fic->zoom);
	fic->zoom = NULL;

	g_free(fic->canvas);
	g_free(fic->path);
	g_free(fic);
}

gfloat
fb_imagecanvas_calc_fit_scale(FbImageCanvas *fic)
{
	gfloat scale;

	scale = (gfloat) fic->canvas->height / (gfloat) fic->raw_height;

	if (fic->canvas->width < (fic->raw_width * scale))
		scale = (gfloat) fic->canvas->width / (gfloat) fic->raw_width;

	return scale;
}

void
fb_imagecanvas_draw(FbImageCanvas *fic,
		    GdkRectangle *draw_area)
{
	gint src_x, src_y;
	GdkRectangle *dest = g_new0(GdkRectangle, 1);

	/* FIX ME (use pixbuf to draw "draw_area" before loading pixmap) */

	g_return_if_fail(fic != NULL);

	if (!GDK_IS_PIXMAP(fic->pixmap))
		fb_imagecanvas_load_pixmap(fic);

	gdk_rectangle_intersect(draw_area, fic->canvas, dest);

	src_x = fic->x0 - fic->canvas->x + dest->x;
	src_y = fic->y0 - fic->canvas->y + dest->y;
      
	gdk_draw_drawable(fic->widget->window, 
			  fic->widget->style->fg_gc[GTK_WIDGET_STATE(fic->widget)],
			  fic->pixmap, src_x, src_y,
			  dest->x, dest->y,
			  dest->width, dest->height);

	g_free(dest);
	/*   g_print("draw x: %4d y: %4d width: %4d height: %4d\n", */
	/* 	  draw_area->x, draw_area->y, draw_area->width, draw_area->height); */
	/* END */
}

void
fb_imagecanvas_zooming(FbImageCanvas *fic,
		       gfloat scale)
{
	gint src_x, src_y, width, height;
	GdkRectangle *dest;
	GdkPixbuf *pixbuf;

	if (!fic->pixbuf) return;

	/* background clean up */
	gdk_draw_rectangle(fic->widget->window,
			   fic->widget->style->bg_gc[GTK_STATE_NORMAL],
			   TRUE,
			   fic->canvas->x, fic->canvas->y,
			   fic->canvas->width, fic->canvas->height);

	if (GDK_IS_PIXMAP(fic->pixmap)) g_object_unref(fic->pixmap);
	fic->pixmap = NULL;

	if (GDK_IS_PIXMAP(fic->zoom)) g_object_unref(fic->zoom);
	fic->zoom = NULL;

	/* scaled image create and draw */
	fic->scale = CLAMP(scale, 0.1, 5.0);
	width = fic->raw_width * fic->scale;
	height = fic->raw_height * fic->scale;
	fic->x0 = (width - fic->canvas->width) / 2;
	fic->y0 = (height - fic->canvas->height) / 2;

	pixbuf = gdk_pixbuf_scale_simple(fic->pixbuf, width, height, GDK_INTERP_BILINEAR);

	src_x = MAX(0, (width - fic->canvas->width) / 2);
	src_y = MAX(0, (height - fic->canvas->height) / 2);

	dest = g_new0(GdkRectangle, 1);
	dest->x = MAX(0, (fic->canvas->width - width) / 2) + fic->canvas->x;
	dest->y = MAX(0, (fic->canvas->height - height) / 2) + fic->canvas->y;
	dest->width = MIN(width, fic->canvas->width);
	dest->height = MIN(height, fic->canvas->height);

	gdk_draw_pixbuf(fic->widget->window,
			fic->widget->style->fg_gc[GTK_WIDGET_STATE(fic->widget)],
			pixbuf, src_x, src_y,
			dest->x, dest->y, dest->width, dest->height,
			GDK_RGB_DITHER_NONE, 0, 0);

	gdk_pixbuf_unref(pixbuf);
	g_free(dest);

	fic->fitting = FALSE;
	/* FIX ME (correspond to zoom) */
}

void
fb_imagecanvas_zoomfit(FbImageCanvas *fic)
{
	gfloat scale;

	scale = fb_imagecanvas_calc_fit_scale(fic);

	fb_imagecanvas_zooming(fic, scale);
	fic->fitting = TRUE;
}

void
fb_imagecanvas_rotate(FbImageCanvas *fic,
		      gboolean counter_clockwise)
{
	gint raw_w, raw_h;
	GdkPixbuf *src, *new;

	g_return_if_fail(fic != NULL);

	if (fic->pixmap) g_object_unref(fic->pixmap);
	if (fic->zoom) g_object_unref(fic->zoom);

	if (fic->pixbuf)
		src = fic->pixbuf;
	else
		src = gdk_pixbuf_new_from_file(fic->path, NULL);

	new = pixbuf_copy_rotate_90(src, counter_clockwise);
	raw_w = gdk_pixbuf_get_width(new);
	raw_h = gdk_pixbuf_get_height(new);
	gdk_pixbuf_unref(src);

	fic->pixbuf = new;
	fic->raw_width = raw_w;
	fic->raw_height = raw_h;
	fic->x0 = ((raw_w * fic->scale) - fic->canvas->width) / 2;
	fic->y0 = ((raw_h * fic->scale) - fic->canvas->height) / 2;

	fb_imagecanvas_draw(fic, fic->canvas);
}

void
fb_imagecanvas_mirror(FbImageCanvas *fic,
		      gboolean mirror,
		      gboolean flip)
{
	gint raw_w, raw_h;
	GdkPixbuf *src, *new;

	g_return_if_fail(fic != NULL);

	if (fic->pixmap) g_object_unref(fic->pixmap);
	if (fic->zoom) g_object_unref(fic->zoom);

	if (fic->pixbuf)
		src = fic->pixbuf;
	else
		src = gdk_pixbuf_new_from_file(fic->path, NULL);

	new = pixbuf_copy_mirror(src, mirror, flip);
	raw_w = gdk_pixbuf_get_width(new);
	raw_h = gdk_pixbuf_get_height(new);
	gdk_pixbuf_unref(src);

	fic->pixbuf = new;
	fic->raw_width = raw_w;
	fic->raw_height = raw_h;
	fic->x0 = ((raw_w * fic->scale) - fic->canvas->width) / 2;
	fic->y0 = ((raw_h * fic->scale) - fic->canvas->height) / 2;

	fb_imagecanvas_draw(fic, fic->canvas);
}

void
fb_imagecanvas_move_image(FbImageCanvas *fic,
			  gint dx,
			  gint dy)
{
	gint width, height;

	g_return_if_fail(fic != NULL);

	if (fic->fitting) return;
	if ((fic->x0 < 0) && (fic->y0 < 0)) return;

	/*   g_return_if_fail(GDK_PIXMAP(fic->pixmap)); */

	width = fic->scale * fic->raw_width;
	height = fic->scale * fic->raw_height;

	if (fic->x0 >= 0)
		fic->x0 = CLAMP((fic->x0 - dx), 0, (width - fic->canvas->width));

	if (fic->y0 >= 0)
		fic->y0 = CLAMP((fic->y0 - dy), 0, (height - fic->canvas->height));

	fb_imagecanvas_draw(fic, fic->canvas);
}

void
fb_imagecanvas_set_from_file(FbImageCanvas *fic,
			     gchar *path)
{
	GdkPixbuf *pixbuf;

	if (!path) return;
	if (fic->path && !g_ascii_strcasecmp(path, fic->path)) return;

	g_return_if_fail(g_file_test(path, G_FILE_TEST_EXISTS) == TRUE);

	pixbuf = gdk_pixbuf_new_from_file(path, NULL);
	g_return_if_fail(pixbuf != NULL);

	if (fic->pixbuf) g_object_unref(fic->pixbuf);
	if (fic->pixmap) g_object_unref(fic->pixmap);
	if (fic->zoom) g_object_unref(fic->zoom);
	g_free(fic->path);

	fic->pixmap = NULL;
	fic->zoom = NULL;
	fic->pixbuf = pixbuf;
	fic->path = g_strdup(path);
	fic->raw_width = gdk_pixbuf_get_width(pixbuf);
	fic->raw_height = gdk_pixbuf_get_height(pixbuf);
	fic->x0 = ((fic->raw_width * fic->scale) - fic->canvas->width) / 2;
	fic->y0 = ((fic->raw_height * fic->scale) - fic->canvas->height) / 2;

	if (fic->fitting)
		fb_imagecanvas_zoomfit(fic);
	else
		fb_imagecanvas_draw(fic, fic->canvas);
}
