/* -*- Mode: C; tab-width: 2; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2005 Masataka Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <stdio.h>
#include <gtk/gtk.h>

#include "futaba.h"
#include "futaba_ui.h"
#include "dirview.h"
#include "bookmark.h"
#include "dialog.h"

#define MENU_BM_POSITION 2

struct _FbBookmark
{
				GtkWidget *item;
				gchar *path;
				gchar *label;
				gchar *message;
				FbFileType type;
};

static void cb_change_dir(GtkMenuItem *item, gpointer data);
static FbBookmark *bookmark_new(Futaba *futaba, gchar *path);
static void bookmark_free(FbBookmark *fbm);

static void
cb_change_dir(GtkMenuItem *item,
							gpointer data)
{
				gchar *path = g_object_get_data(G_OBJECT(item), "path");
				Futaba *futaba = data;

				fb_dirview_set_dir(futaba, path);
}

static FbBookmark *
bookmark_new(Futaba *futaba, 
						 gchar *path)
{
				gchar *base, *stock_path, *label, *message;
				GtkWidget *item, *icon;
				FbFileType type;
				FbBookmark *fbm = g_new0(FbBookmark, 1);

				if (!path) return NULL;
				if (!g_file_test(path, G_FILE_TEST_EXISTS)) return NULL;

				base = g_path_get_basename(path);
				label = str_local_to_utf8(base);
				g_free(base);

				stock_path = g_strdup(path);
				message = g_strdup_printf(_("Go to \"%s\""), stock_path);

				item = gtk_image_menu_item_new_with_label(label);
				g_signal_connect(G_OBJECT(item), "activate",
												 G_CALLBACK(cb_change_dir), futaba);
				g_object_set_data(G_OBJECT(item), "path", stock_path);
				fb_widget_set_message(item, message);

				if (g_file_test(path, G_FILE_TEST_IS_DIR)) {
								type = DIRECTORY;
								icon = gtk_image_new_from_stock("fb-directory", GTK_ICON_SIZE_MENU);
								gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(item), icon);
								gtk_widget_show(icon);
				}
				else if (g_file_test(path, G_FILE_TEST_IS_REGULAR) &&
								 is_imagefile(path))
								type = IMAGE;

				fbm->item = item;
				fbm->path = stock_path;
				fbm->label = label;
				fbm->message = message;
				fbm->type = type;

				return fbm;
}

static void
bookmark_free(FbBookmark *fbm)
{
				if (GTK_IS_WIDGET(fbm->item)) gtk_widget_destroy(fbm->item);
				if (fbm->path) g_free(fbm->path);
				if (fbm->label) g_free(fbm->label);
				if (fbm->message) g_free(fbm->message);

				g_free(fbm);
}

GList *
fb_load_bookmark(void)
{
				FILE *bm_file;
				gchar *path;
				gchar buf[1024];
				GList *paths = NULL;

				path = g_strconcat(g_get_home_dir(), "/", FUTABA_RC_DIR, "/", BOOKMARK_FILE, NULL);

				bm_file = fopen(path, "r");
				g_free(path);

				if (!bm_file) return NULL;

				while (fgets(buf, 1024, bm_file)) {
								if (buf[0] == '#') continue;
								if (buf[0] == '\n') continue;

								path = g_strdup(g_strstrip(buf));
								paths = g_list_append(paths, path);
				}
				fclose(bm_file);

				return paths;
}

void
fb_save_bookmark(Futaba *futaba)
{
				FILE *save_file;
				gint i;
				gchar *path;
				FbBookmark *fbm;

				path = g_strconcat(g_get_home_dir(), FUTABA_RC_DIR, BOOKMARK_FILE, NULL);

				save_file = fopen(path, "w");
				g_free(path);

				if (!save_file) return;

				g_fprintf(save_file, 
									"####################################################\n"
									"#                   bookmarks                      #\n"
									"####################################################\n"
									"\n");

				for (i = 0;fbm = g_list_nth_data(futaba->bookmarks, i); i++)
								if (fbm) g_fprintf(save_file, "%s\n", fbm->path);

				fclose(save_file);
}

void
fb_menu_add_bookmark(Futaba *futaba,
										 gchar *path)
{
				gchar *message;
				GtkWidget *menu, *item;
				FbBookmark *fbm;

				g_return_if_fail(path != NULL);

				item = gtk_ui_manager_get_widget(futaba->ui_manager, "/MenuBar/BookmarkMenu");
				menu = gtk_menu_item_get_submenu(GTK_MENU_ITEM(item));

				fbm = bookmark_new(futaba, path);
				if (!fbm) return;

				message = g_strdup_printf(_("Add \"%s\" to bookmark"), path);
				fb_statusbar_push(message);
				g_free(message);

				futaba->bookmarks = g_list_append(futaba->bookmarks, fbm);

				gtk_menu_shell_append(GTK_MENU_SHELL(menu), fbm->item);
				gtk_widget_show(fbm->item);
}

void
fb_menu_remove_bookmark(Futaba *futaba,
												gint pos)
{
				FbBookmark *fbm;

				fbm = g_list_nth_data(futaba->bookmarks, pos);
				if (!fbm) return;

				futaba->bookmarks = g_list_remove(futaba->bookmarks, fbm);

				bookmark_free(fbm);
}

void
fb_menu_init_bookmark(Futaba *futaba)
{
				gint i;
				gchar *path;
				GList *path_list;
				GtkWidget *menu, *item;
				FbBookmark *fbm;

				g_return_if_fail(futaba->bookmarks == NULL);

				item = gtk_ui_manager_get_widget(futaba->ui_manager, "/MenuBar/BookmarkMenu");
				menu = gtk_menu_item_get_submenu(GTK_MENU_ITEM(item));

				item = gtk_separator_menu_item_new();
				gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
				gtk_widget_show(item);

				path_list = fb_load_bookmark();

				for (i = 0;path = g_list_nth_data(path_list, i); i++) {
								fbm = bookmark_new(futaba, path);
								if (!fbm) continue;

								futaba->bookmarks = g_list_append(futaba->bookmarks, fbm);
								gtk_menu_shell_append(GTK_MENU_SHELL(menu), fbm->item);
								gtk_widget_show(fbm->item);
				}
}
/*
 *--------------------------------------
 *       BookMark Edit Dialog
 *--------------------------------------
 */
#define BM_DIALOG_W 400
#define BM_DIALOG_H 300

enum{
				PATH,
				ALIAS,
				EDITABLE,
				BOOKMARKS
};

typedef struct _BookmarkManager BookmarkManager;

struct _BookmarkManager
{
				GtkWidget *dialog;
				GtkWidget *treeview;
				GtkWidget *entry;
				GtkListStore *liststore;
				GList *bookmarks;
};

static void cb_cell_edited(GtkCellRendererText *cell, gchar *arg1, gchar *arg2, gpointer data);
static void cb_add_bookmark(GtkEntry *entry, gpointer data);
static GtkListStore *bookmark_list_new(Futaba *gmagna);
static GtkWidget *tree_view_new(GtkListStore *liststore);

static void
cb_cell_edited(GtkCellRendererText *cell,
							 gchar *arg1,
							 gchar *arg2,
							 gpointer data)
{
				GtkTreePath *path = gtk_tree_path_new_from_string (arg1);
				GtkTreeIter iter;

				gtk_tree_model_get_iter (GTK_TREE_MODEL(data), &iter, path);
				gtk_list_store_set(GTK_LIST_STORE(data), &iter, 1,
													 arg2, -1);

				gtk_tree_path_free(path);
}

static GtkListStore *
bookmark_list_new(Futaba *futaba)
{
				gint i;
				gchar *path;
				GtkListStore *liststore;
				GtkTreeIter iter;
				FbBookmark *fbm;

				liststore = gtk_list_store_new(BOOKMARKS,
																			 G_TYPE_STRING,
																			 G_TYPE_STRING,
																			 G_TYPE_BOOLEAN);

				for (i = 0; fbm = g_list_nth_data(futaba->bookmarks, i); i++) {
								path = str_local_to_utf8(fbm->path);
								gtk_list_store_append(liststore, &iter);
								gtk_list_store_set(liststore, &iter,
																	 PATH, path,
																	 -1);
				}

				return liststore;
}

static GtkWidget *
tree_view_new(GtkListStore *liststore)
{
				GtkCellRenderer *render;
				GtkTreeViewColumn *column;
				GtkWidget *treeview;

				treeview = gtk_tree_view_new();
				gtk_tree_view_set_rules_hint(GTK_TREE_VIEW(treeview), TRUE);
				gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(treeview), FALSE);
				gtk_tree_view_set_model(GTK_TREE_VIEW(treeview), GTK_TREE_MODEL(liststore));

				column = gtk_tree_view_column_new();
				gtk_tree_view_column_set_min_width(column, (gint)(BM_DIALOG_W * 5 / 7));
				gtk_tree_view_append_column(GTK_TREE_VIEW(treeview), column);

				render = gtk_cell_renderer_text_new();
				gtk_tree_view_column_pack_start(column, render, FALSE);
				gtk_tree_view_column_set_attributes(column, render,
																						"text", PATH,
																						NULL);


				return treeview;
}

static void
cb_bm_activate(GtkEntry *entry,
							 gpointer data)
{
				GtkWidget *dialog = data;

				gtk_dialog_response(GTK_DIALOG(dialog), GTK_RESPONSE_ACCEPT);
}

static void
cb_bm_response(GtkDialog *dialog,
							 gint arg,
							 gpointer data)
{
				Futaba *futaba = data;

				fb_statusbar_pop();

				switch (arg) {
				case GTK_RESPONSE_REJECT: {
								gint *pos;
								GtkTreeIter iter;
								GtkTreePath *treepath;
								GtkTreeModel *treemodel;
								GtkWidget *treeview = g_object_get_data(G_OBJECT(dialog), "bm-treeview");
								gtk_tree_view_get_cursor(GTK_TREE_VIEW(treeview), &treepath, NULL);
								if (!treepath) return;

								pos = gtk_tree_path_get_indices(treepath);

								fb_menu_remove_bookmark(futaba, *pos);

								treemodel = gtk_tree_view_get_model(GTK_TREE_VIEW(treeview));
								gtk_tree_model_get_iter(treemodel, &iter, treepath);
								gtk_list_store_remove(GTK_LIST_STORE(treemodel), &iter);

								gtk_tree_path_free(treepath);
				}
								break;
				case GTK_RESPONSE_ACCEPT: {
								gchar *input, *path;
								GtkTreeIter iter;
								GtkListStore *liststore;
								GtkWidget *entry, *treeview;

								entry = g_object_get_data(G_OBJECT(dialog), "bm-entry");
								input = gtk_editable_get_chars(GTK_EDITABLE(entry), 0, -1);
								path = str_local_to_utf8(input);

								if (!g_file_test(path, G_FILE_TEST_EXISTS)) {
												fb_statusbar_swap(_("Don't exist"));
												return;
								}

								treeview = g_object_get_data(G_OBJECT(dialog), "bm-treeview");
								liststore = GTK_LIST_STORE(gtk_tree_view_get_model(GTK_TREE_VIEW(treeview)));

								gtk_list_store_append(liststore, &iter);
								gtk_list_store_set(liststore, &iter,
																	 PATH, path,
																	 -1);

								fb_menu_add_bookmark(futaba, path);
								gtk_editable_delete_text(GTK_EDITABLE(entry), 0, -1);

								g_free(path);
				}
								break;
				case GTK_RESPONSE_OK: {
								fb_save_bookmark(futaba);
								gtk_widget_destroy(GTK_WIDGET(dialog));
				}
								break;
				}
}

GtkWidget *
fb_bookmark_dialog_new(Futaba *futaba)
{
				GtkListStore *liststore;
				GtkWidget *dialog, *scroll, *treeview;
				GtkWidget *hbox, *label, *entry;

				dialog = gtk_dialog_new_with_buttons(_("Bookmark Editor - Futaba"),
																						 GTK_WINDOW(futaba->top),
																						 GTK_DIALOG_MODAL |
																						 GTK_DIALOG_NO_SEPARATOR,
																						 GTK_STOCK_REMOVE, GTK_RESPONSE_REJECT,
																						 GTK_STOCK_OK, GTK_RESPONSE_OK,
																						 NULL);
				gtk_window_set_default_size(GTK_WINDOW(dialog), BM_DIALOG_W, BM_DIALOG_H);
				g_signal_connect(G_OBJECT(dialog), "response",
												 G_CALLBACK(cb_bm_response), futaba);

				/* dialog->vbox area */
				/*   label = gtk_label_new(_("Bookmarks.")); */
				/*   gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), label, FALSE, FALSE, 0); */
				/*   gtk_widget_show(label); */

				scroll = gtk_scrolled_window_new(NULL, NULL);
				gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scroll),
																						GTK_SHADOW_ETCHED_IN);
				gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scroll),
																			 GTK_POLICY_NEVER,
																			 GTK_POLICY_AUTOMATIC);
				gtk_widget_set_size_request(scroll, -1, BM_DIALOG_H - 30);
				gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), scroll, TRUE, TRUE, 0);
				gtk_widget_show(scroll);

				liststore = bookmark_list_new(futaba);

				treeview = tree_view_new(liststore);
				g_object_set_data(G_OBJECT(dialog), "bm-treeview", treeview);
				gtk_container_add(GTK_CONTAINER(scroll), treeview);
				gtk_widget_show(treeview);

				hbox = gtk_hbox_new(FALSE, 0);
				gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), hbox, FALSE, FALSE, 0);
				gtk_widget_show(hbox);
	
				label = gtk_label_new(_("add : "));
				gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
				gtk_widget_show(label);

				entry = gtk_entry_new_with_max_length(128);
				g_object_set_data(G_OBJECT(dialog), "bm-entry", entry);
				g_signal_connect(G_OBJECT(entry), "activate",
												 G_CALLBACK(cb_bm_activate), dialog);
				gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);
				gtk_widget_show(entry);

				return dialog;
}
