/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2005 Masataka Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <gtk/gtk.h>

#include "futaba.h"
#include "futaba_ui.h"
#include "action.h"
#include "dirview.h"
#include "imageviewer.h"
#include "canvas.h"
#include "dialog.h"
#include "prefs.h"
#include "bookmark.h"
#include "history.h"
#include "thumbnail.h"

#define ZOOMIN 0.77
#define ZOOMOUT 1.3
/*
 *------------------------------------------------------
 *       Actions
 *------------------------------------------------------
 */
static void cb_show_file_menu(GtkAction *action, gpointer data);
static void cb_show_edit_menu(GtkAction *action, gpointer data);
static void cb_show_view_menu(GtkAction *action, gpointer data);
static void cb_show_move_menu(GtkAction *action, gpointer data);
static void cb_open(GtkAction *action, gpointer data);
static void cb_open_viewer(GtkAction *action, gpointer data);
static void cb_open_thumb(GtkAction *action, gpointer data);
static void cb_close_viewer(GtkAction *action, gpointer data);
static void cb_close_thumb(GtkAction *action, gpointer data);
static void cb_quit(GtkAction *action, gpointer data);
static void cb_copy(GtkAction *action, gpointer data);
static void cb_cut(GtkAction *action, gpointer data);
static void cb_paste(GtkAction *action, gpointer data);
static void cb_delete(GtkAction *action, gpointer data);
static void cb_make_dir(GtkAction *action, gpointer data);
static void cb_rename(GtkAction *action, gpointer data);
static void cb_change_bulk(GtkAction *action, gpointer data);
static void cb_property(GtkAction *action, gpointer data);
static void cb_preferences(GtkAction *action, gpointer data);
static void cb_add_bookmark(GtkAction *action, gpointer data);
static void cb_edit_bookmark(GtkAction *action, gpointer data);
static void cb_about(GtkAction *action, gpointer data);
static void cb_back_history(GtkAction *action, gpointer data);
static void cb_next_history(GtkAction *action, gpointer data);
static void cb_show_back_history(GtkAction *action, gpointer data);
static void cb_show_next_history(GtkAction *action, gpointer data);
static void cb_clear_history(GtkAction *action, gpointer data);
static void cb_up_dir(GtkAction *action, gpointer data);
static void cb_go_home(GtkAction *action, gpointer data);
static void cb_refresh_dir(GtkAction *action, gpointer data);
static void cb_change_column_type(GtkAction *action, gpointer data);
static void cb_zoom_in(GtkAction *action, gpointer data);
static void cb_zoom_out(GtkAction *action, gpointer data);
static void cb_zoom_100(GtkAction *action, gpointer data);
static void cb_fit_image(GtkAction *action, gpointer data);
static void cb_right_rotation(GtkAction *action, gpointer data);
static void cb_left_rotation(GtkAction *action, gpointer data);
static void cb_mirror_rotation(GtkAction *action, gpointer data);
static void cb_r_zoom_in(GtkAction *action, gpointer data);
static void cb_r_zoom_out(GtkAction *action, gpointer data);
static void cb_r_zoom_100(GtkAction *action, gpointer data);
static void cb_r_fit_image(GtkAction *action, gpointer data);
static void cb_r_right_rotation(GtkAction *action, gpointer data);
static void cb_r_left_rotation(GtkAction *action, gpointer data);
static void cb_r_mirror_rotation(GtkAction *action, gpointer data);
static void cb_l_zoom_in(GtkAction *action, gpointer data);
static void cb_l_zoom_out(GtkAction *action, gpointer data);
static void cb_l_zoom_100(GtkAction *action, gpointer data);
static void cb_l_fit_image(GtkAction *action, gpointer data);
static void cb_l_right_rotation(GtkAction *action, gpointer data);
static void cb_l_left_rotation(GtkAction *action, gpointer data);
static void cb_l_mirror_rotation(GtkAction *action, gpointer data);
static void cb_switch_file_display(GtkAction *action, gpointer data);
static void cb_switch_dot_file(GtkAction *action, gpointer data);
static void cb_switch_preview(GtkAction *action, gpointer data);
static void cb_show_tbar(GtkAction *action, gpointer data);
static void cb_show_lbar(GtkAction *action, gpointer data);
static void cb_show_sbar(GtkAction *action, gpointer data);
static void cb_set_reverse(GtkAction *action, gpointer data);
static void cb_show_col1(GtkAction *action, gpointer data);
static void cb_show_col2(GtkAction *action, gpointer data);
static void cb_show_col3(GtkAction *action, gpointer data);
static void cb_show_col4(GtkAction *action, gpointer data);
static void cb_show_col5(GtkAction *action, gpointer data);
static void cb_show_coltitle(GtkAction *action, gpointer data);
static void cb_set_dot_filter(GtkAction *action, gpointer data);
static void cb_set_other_filter(GtkAction *action, gpointer data);
static void cb_two_pages(GtkAction *action, gpointer data);
static void cb_max_window(GtkAction *action, gpointer data);
static void cb_fullscreen(GtkAction *action, gpointer data);
static void cb_sort_mode(GtkRadioAction *action, GtkRadioAction *current, gpointer data);

/* GtkActionEntry {
   const gchar *name,
   const gchar *stock_id,
   const gchar *label,
   const gchar *accelerator,
   const gchar *tooltip,
   GCallback func} */
static GtkActionEntry actions[] = {
  {"FileMenu", NULL, N_("File"), NULL, NULL, G_CALLBACK(cb_show_file_menu)},
  {"EditMenu", NULL, N_("Edit"), NULL, NULL, G_CALLBACK(cb_show_edit_menu)},
  {"ViewMenu", NULL, N_("View"), NULL, NULL, G_CALLBACK(cb_show_view_menu)},
  {"MoveMenu", NULL, N_("Move"), NULL, NULL, G_CALLBACK(cb_show_move_menu)},
  {"BookmarkMenu", NULL, N_("Bookmark")},
  {"HelpMenu", NULL, N_("Help")},

  {"Sort", NULL, N_("Sort")},
  {"Column", NULL, N_("Column")},
  {"Filter", NULL, N_("Filter")},
  {"LeftImage", NULL, N_("left image")},
  {"RightImage", NULL, N_("right image")},

  {"Open", "gtk-open", N_("Open"), NULL, N_("Open"), G_CALLBACK(cb_open)},
  {"OpenViewer", NULL, N_("Open Viewer"), "<control>V", N_("Open Image Viewer"), G_CALLBACK(cb_open_viewer)},
  {"OpenThumb", NULL, N_("Open Thumbnail"), "<control>T", N_("Open Thumbnail Window"), G_CALLBACK(cb_open_thumb)},
  {"CloseViewer", "gtk-close", N_("Close Viewer"), "<control>W", N_("Close Image Viewer"), G_CALLBACK(cb_close_viewer)},
  {"CloseThumb", "gtk-close", N_("Close Thumbnail"), NULL, N_("Close Thumbnail Window"), G_CALLBACK(cb_close_thumb)},
  {"Quit", "gtk-quit", N_("Quit"), "<control>Q", N_("Quit"), G_CALLBACK(cb_quit)},

  {"MakeDir", NULL, N_("Create new dir"), NULL, N_("Create new directory"), G_CALLBACK(cb_make_dir)},
  {"ChangeBulk", NULL, N_("Change bulk"), NULL, N_("Change file in bulk"), G_CALLBACK(cb_change_bulk)},
  {"Rename", NULL, N_("Rename"), NULL, N_("Rename the selected item"), G_CALLBACK(cb_rename)},
  {"Property", NULL, N_("Property"), NULL, N_("Display the properties of selected item"), G_CALLBACK(cb_property)},
  {"Copy", "gtk-copy", N_("Copy"), "<control>C", N_("Copy"), G_CALLBACK(cb_copy)},
  {"Cut", "gtk-cut", N_("Cut"), "<control>U", N_("Cut"), G_CALLBACK(cb_cut)},
  {"Paste", "gtk-paste", N_("Paste"), "<control>P", N_("Paste"), G_CALLBACK(cb_paste)},
  {"Delete", NULL, N_("Delete"), "<control>D", N_("Delete"), G_CALLBACK(cb_delete)},

  {"Preferences", "gtk-preferences", N_("Preferences..."), "<cotrol>P", NULL, G_CALLBACK(cb_preferences)},

  {"AddBookmark", "gtk-add", N_("Add bookmark"), "<control>B", NULL, G_CALLBACK(cb_add_bookmark)},
  {"EditBookmark", NULL, N_("Edit bookmark"), NULL, NULL, G_CALLBACK(cb_edit_bookmark)},
   
  {"About", NULL, N_("About"), NULL, N_("Show credits"), G_CALLBACK(cb_about)},

  {"BackHistory", "gtk-go-back", N_("Back"), "<alt>Left", N_("Go to the previous visited directory"), G_CALLBACK(cb_back_history)},
  {"NextHistory", "gtk-go-forward", N_("Forward"), "<alt>Right", N_("Go to the next visited directory"), G_CALLBACK(cb_next_history)},
  {"ShowBackHistory", NULL, N_("Back"), NULL, N_("Go back a few pages"), G_CALLBACK(cb_show_back_history)},
  {"ShowNextHistory", NULL, N_("Next"), NULL, N_("Go forward a number of pages"), G_CALLBACK(cb_show_next_history)},
  {"ClearHistory", NULL, N_("Clear History"), NULL, N_("Clear Back/Forward lists"), G_CALLBACK(cb_clear_history)},

  {"UpDir", "gtk-go-up", N_("Up"), "<alt>Up", N_("Go up one level"), G_CALLBACK(cb_up_dir)},
  {"GoToHome", "gtk-home", N_("Home"), "<alt>Home", N_("Go to the home"), G_CALLBACK(cb_go_home)},
  {"RefreshDir", "gtk-refresh", N_("Refresh"), "<control>R", N_("Refresh directory"), G_CALLBACK(cb_refresh_dir)},

  {"SwitchFileDisplay", "fb-another-file", N_("File Display"), NULL, N_("Show/Hide another file"), G_CALLBACK(cb_switch_file_display)},
  {"SwitchDotFile", "fb-dot", N_("Dot file"), NULL, N_("Show/Hide dot file"), G_CALLBACK(cb_switch_dot_file)},
  {"ChangeColumnType", "fb-list", N_("fileinfo"), NULL, NULL, G_CALLBACK(cb_change_column_type)},

  {"SwitchPreview", "fb-preview", N_("preview"), "<alt>P", N_("Show/Hide Preview"), G_CALLBACK(cb_switch_preview)},

  {"ZoomIn", "gtk-zoom-in", N_("zoom in"), "<release>J", NULL, G_CALLBACK(cb_zoom_in)},
  {"ZoomOut", "gtk-zoom-out", N_("zoom out"), "<release>L", NULL, G_CALLBACK(cb_zoom_out)},
  {"Zoom100", "gtk-zoom-100", N_("zoom 1:1"), "< release>O", NULL, G_CALLBACK(cb_zoom_100)},
  {"ZoomFit", "gtk-zoom-fit", N_("fit image to window"),"<release>K", NULL, G_CALLBACK(cb_fit_image)},
  {"RightRotation", NULL, N_("rigth rotation"), NULL, NULL, G_CALLBACK(cb_right_rotation)},
  {"LeftRotation", NULL, N_("left rotation"), NULL, NULL, G_CALLBACK(cb_left_rotation)},
  {"MirrorRotation", NULL, N_("mirror rotation"), NULL, NULL, G_CALLBACK(cb_mirror_rotation)},

  {"R_ZoomIn", "gtk-zoom-in", N_("zoom in"), "<alt>J", NULL, G_CALLBACK(cb_r_zoom_in)},
  {"R_ZoomOut", "gtk-zoom-out", N_("zoom out"), "<alt>L",  NULL, G_CALLBACK(cb_r_zoom_out)},
  {"R_Zoom100", "gtk-zoom-100", N_("zoom 1:1"), "<alt>O", NULL, G_CALLBACK(cb_r_zoom_100)},
  {"R_ZoomFit", "gtk-zoom-fit", N_("fit image to window"),"<alt>K", NULL, G_CALLBACK(cb_r_fit_image)},
  {"R_RightRotation", NULL, N_("rigth rotation"), NULL, NULL, G_CALLBACK(cb_r_right_rotation)},
  {"R_LeftRotation", NULL, N_("left rotation"), NULL, NULL, G_CALLBACK(cb_r_left_rotation)},
  {"R_MirrorRotation", NULL, N_("mirror rotation"), NULL, NULL, G_CALLBACK(cb_r_mirror_rotation)},

  {"L_ZoomIn", "gtk-zoom-in", N_("zoom in"), "<control>J", NULL, G_CALLBACK(cb_l_zoom_in)},
  {"L_ZoomOut", "gtk-zoom-out", N_("zoom out"), "<control>L",  NULL, G_CALLBACK(cb_l_zoom_out)},
  {"L_Zoom100", "gtk-zoom-100", N_("zoom 1:1"), "<control>O", NULL, G_CALLBACK(cb_l_zoom_100)},
  {"L_ZoomFit", "gtk-zoom-fit", N_("fit image to window"),"<control>K", NULL, G_CALLBACK(cb_l_fit_image)},
  {"L_RightRotation", NULL, N_("rigth rotation"), NULL, NULL, G_CALLBACK(cb_l_right_rotation)},
  {"L_LeftRotation", NULL, N_("left rotation"), NULL, NULL, G_CALLBACK(cb_l_left_rotation)},
  {"L_MirrorRotation", NULL, N_("mirror rotation"), NULL, NULL, G_CALLBACK(cb_l_mirror_rotation)},
};
static guint actions_len = G_N_ELEMENTS(actions);

static GtkToggleActionEntry toggle_actions[] = {
  {"TBarShow", NULL, N_("Toolbar"), NULL, NULL, G_CALLBACK(cb_show_tbar), TRUE},
  {"LBarShow", NULL, N_("Locationbar"), NULL, NULL, G_CALLBACK(cb_show_lbar), TRUE},
  {"SBarShow", NULL, N_("Statusbar"), NULL, NULL, G_CALLBACK(cb_show_sbar), TRUE},

  {"Reverse", NULL, N_("Reversed oeder"), NULL, NULL, G_CALLBACK(cb_set_reverse), FALSE},

  {"Col1", NULL, N_("Permission"), NULL, NULL, G_CALLBACK(cb_show_col1), FALSE},
  {"Col2", NULL, N_("Owner"), NULL, NULL, G_CALLBACK(cb_show_col2), FALSE},
  {"Col3", NULL, N_("Group"), NULL, NULL, G_CALLBACK(cb_show_col3), FALSE},
  {"Col4", NULL, N_("Size"), NULL, NULL, G_CALLBACK(cb_show_col4), FALSE},
  {"Col5", NULL, N_("Mtime"), NULL, NULL, G_CALLBACK(cb_show_col5), FALSE},
  {"ColTitle", NULL, N_("Title"), NULL, NULL, G_CALLBACK(cb_show_coltitle), FALSE},

  {"FilterDot", NULL, N_("Dot file"), NULL, NULL, G_CALLBACK(cb_set_dot_filter), TRUE},
  {"FilterOther", NULL, N_("Other file"), NULL, NULL, G_CALLBACK(cb_set_other_filter), TRUE},

  {"TwoWindow", NULL, N_("two pages display"), "<release>space", NULL, G_CALLBACK(cb_two_pages), FALSE},
  {"MaxWindow", NULL, N_("Max Window"), "<release>M", NULL, G_CALLBACK(cb_max_window), FALSE},
  {"FullScreen", NULL, N_("full screen"), "<release>F", NULL, G_CALLBACK(cb_fullscreen), FALSE},
};
static guint toggle_actions_len = G_N_ELEMENTS(toggle_actions);

static GtkRadioActionEntry radio_actions[] = {
  {"SortName", NULL, N_("Sort by name"), NULL, NULL, SORT_BY_NAME},
  {"SortSize", NULL, N_("Sort by size"), NULL, NULL, SORT_BY_SIZE},
  {"SortMtime", NULL, N_("Sort by mtime"), NULL, NULL, SORT_BY_MTIME},
  {"SortType", NULL, N_("Sort by type"), NULL, NULL, SORT_BY_TYPE}
};
static guint radio_actions_len = G_N_ELEMENTS(radio_actions);

static void
cb_show_file_menu(GtkAction *action,
		  gpointer data)
{
	gboolean flag;
	Futaba *futaba = data;

	flag = (futaba->image_num > 0);
	fb_ui_set_sensitive(futaba->ui_manager, "/MenuBar/FileMenu/OpenViewer", flag);
	fb_ui_set_sensitive(futaba->ui_manager, "/MenuBar/FileMenu/OpenThumb", flag);

	flag = (futaba->viewer != NULL);
	fb_ui_set_sensitive(futaba->ui_manager, "/MenuBar/FileMenu/CloseViewer", flag);

	flag = (futaba->thumbwin != NULL);
	fb_ui_set_sensitive(futaba->ui_manager, "/MenuBar/FileMenu/CloseThumb", flag);
}

static void
cb_show_edit_menu(GtkAction *action,
		  gpointer data)
{
	gboolean flag;
	gchar *path;
	GtkTreeIter iter;
	GtkTreePath *tpath;
	Futaba *futaba = data;

	flag = (futaba->writable && (futaba->paste_path != NULL));
	fb_ui_set_sensitive(futaba->ui_manager, "/MenuBar/EditMenu/Paste", flag);

	gtk_tree_view_get_cursor(GTK_TREE_VIEW(futaba->treeview), &tpath, NULL);

	flag = (tpath != NULL);
	if (flag) {
		gtk_tree_model_get_iter(GTK_TREE_MODEL(futaba->liststore), &iter, tpath);
		gtk_tree_model_get(GTK_TREE_MODEL(futaba->liststore), &iter,
				   COLUMN_DISPLAY_PATH, &path,
				   -1);
		
		if ((path[0] == '.') &&
		    ((path[1] == '\0') || (path[1] == '.')))
			flag = FALSE;
	}

	fb_ui_set_sensitive(futaba->ui_manager, "/MenuBar/EditMenu/Copy", flag);
	fb_ui_set_sensitive(futaba->ui_manager, "/MenuBar/EditMenu/Cut", flag);
	fb_ui_set_sensitive(futaba->ui_manager, "/MenuBar/EditMenu/Delete", flag);

	gtk_tree_path_free(tpath);
}

static void
cb_show_view_menu(GtkAction *action,
		  gpointer data)
{
	Futaba *futaba = data;


}

static void
cb_show_move_menu(GtkAction *action,
		  gpointer data)
{
	gint length;
	gboolean flag;
	Futaba *futaba = data;

	flag = g_strcasecmp(futaba->workdir, "/");
	fb_ui_set_sensitive(futaba->ui_manager, "/MenuBar/MoveMenu/UpDir", flag);

	length = g_list_length(futaba->history->record) - 1;

	flag = (futaba->history->position != length);
	fb_ui_set_sensitive(futaba->ui_manager, "/MenuBar/MoveMenu/BackHistory", flag);

	flag = (futaba->history->position != 0);
	fb_ui_set_sensitive(futaba->ui_manager, "/MenuBar/MoveMenu/NextHistory", flag);

}

static void
cb_open(GtkAction *action,
	gpointer data)
{
	Futaba *futaba = data;

	/*   if (!futaba->treeview) return; */

	/*   g_signal_emit_by_name(G_OBJECT(futaba->treeview), "row-activated", futaba); */
	fb_dirview_row_activated(futaba);
}

static void
cb_open_viewer(GtkAction *action,
	       gpointer data)
{
	Futaba *futaba = data;

	if (!futaba->image_num) {
		fb_statusbar_push(_("No image !!"));

		return;
	}

	fb_imageviewer_start(futaba);
}

static void
cb_open_thumb(GtkAction *action,
	      gpointer data)
{
	Futaba *futaba = data;

	if (!futaba->image_num) {
		fb_statusbar_push(_("No image !!"));

		return;
	}
	
	fb_thumbwin_start(futaba);
}

static void
cb_close_viewer(GtkAction *action,
		gpointer data)
{
	Futaba *futaba = data;

	if (!futaba->viewer) return;

	g_signal_emit_by_name(G_OBJECT(futaba->viewer->top), "destroy", futaba);
}

static void
cb_close_thumb(GtkAction *action,
	       gpointer data)
{
	Futaba *futaba = data;

	if (!futaba->thumbwin) return;

	g_signal_emit_by_name(G_OBJECT(futaba->thumbwin->top), "destroy", futaba);
}

static void
cb_quit(GtkAction *action,
	gpointer data)
{
	Futaba *futaba = data;

	futaba_quit(futaba);
}

static void
cb_copy(GtkAction *action,
	gpointer data)
{
	Futaba *futaba = data;

	if (futaba->paste_path) g_free(futaba->paste_path);
 
	futaba->paste_path = fb_dirview_get_path_at_cursor(futaba);
	futaba->is_copy = TRUE;

	fb_ui_set_message(futaba->ui_manager, "/MenuBar/EditMenu/Paste", futaba->paste_path);
	fb_ui_set_message(futaba->ui_manager, "/DirViewPopup/Paste", futaba->paste_path);
}

static void
cb_cut(GtkAction *action,
       gpointer data)
{
	Futaba *futaba = data;

	if (futaba->paste_path) g_free(futaba->paste_path);

	futaba->paste_path = fb_dirview_get_path_at_cursor(futaba);
	futaba->is_copy = FALSE;

	fb_ui_set_message(futaba->ui_manager, "/MenuBar/EditMenu/Paste", futaba->paste_path);
	fb_ui_set_message(futaba->ui_manager, "/DirViewPopup/Paste", futaba->paste_path);
}

static void
cb_paste(GtkAction *action,
	 gpointer data)
{
	gchar *base, *dest_path;
	Futaba *futaba = data;

	if (!futaba->paste_path) return;

	base = g_path_get_basename(futaba->paste_path);
	if (futaba->workdir == "/")
		dest_path = g_strconcat("/", base, NULL);
	else
		dest_path = g_strconcat(futaba->workdir, "/", base, NULL);
	g_free(base);

	/* FIX ME */
	if (fb_paste_start(futaba, dest_path) && !futaba->is_copy) {
		g_free(futaba->paste_path);
		futaba->paste_path = NULL;
	}
	/* END */

	g_free(dest_path);
}

static void
cb_delete(GtkAction *action,
	  gpointer data)
{
	gchar *path;
	Futaba *futaba = data;

	path = fb_dirview_get_path_at_cursor(futaba);
	if (fb_rmdir_start(path))
		fb_dirview_remove_at_cursor(futaba);

	g_free(path);
}

static void
cb_make_dir(GtkAction *action,
	    gpointer data)
{
	GtkWidget *dialog;
	Futaba *futaba = data;

	/*   dialog = fb_mkdir_dialog_new(futaba); */
	/*   gtk_widget_show_all(dialog); */
	fb_dirview_add_new_dir(futaba);
}

static void
cb_rename(GtkAction *action,
	  gpointer data)
{
	Futaba *futaba = data;

	fb_dirview_edit_row(futaba);
}

static void
cb_change_bulk(GtkAction *action,
	       gpointer data)
{
	GtkWidget *dialog;
	Futaba *futaba = data;

	dialog = fb_bulk_change_dialog_new(futaba);
	gtk_widget_show(dialog);
}

static void
cb_property(GtkAction *action,
	    gpointer data)
{
	GtkWidget *dialog;
	Futaba *futaba = data;

	dialog = fb_property_dialog_new(futaba);
	gtk_widget_show(dialog);
}

static void
cb_preferences(GtkAction *action,
	       gpointer data)
{
	GtkWidget *win;
	Futaba *futaba = data;

	win = fb_prefs_win_new();
	gtk_widget_show(win);
}

static void
cb_add_bookmark(GtkAction *action,
		gpointer data)
{
	Futaba *futaba = data;

	fb_menu_add_bookmark(futaba, futaba->workdir);
}

static void
cb_edit_bookmark(GtkAction *action,
		 gpointer data)
{
	GtkWidget *dialog;
	Futaba *futaba = data;

	dialog = fb_bookmark_dialog_new(futaba);
	gtk_widget_show_all(dialog);
}

static void
cb_about(GtkAction *action,
	 gpointer data)
{
	GtkWidget *dialog;
	Futaba *futaba = data;

	dialog = fb_about_dialog_new(futaba->top);
	gtk_widget_show_all(dialog);
}

static void
cb_back_history(GtkAction *action,
		gpointer data)
{
	gchar *path;
	Futaba *futaba = data;

	fb_statusbar_pop();

	path = fb_dir_history_back(futaba->history);

	if (!path) return;

	fb_dirview_set_history(futaba, path);
	fb_toolbar_set_sensitive(futaba);

	g_free(path);
}

static void
cb_next_history(GtkAction *action,
		gpointer data)
{
	gchar *path;
	Futaba *futaba = data;

	fb_statusbar_pop();

	path = fb_dir_history_next(futaba->history);

	if (!path) return;

	fb_dirview_set_history(futaba, path);
	fb_toolbar_set_sensitive(futaba);

	g_free(path);
}

static void
cb_show_back_history(GtkAction *item,
		     gpointer data)
{
	gint i, limit, length;
	Futaba *futaba = data;

	fb_statusbar_pop();

	length = g_list_length(futaba->history->record);
	if (length == 0) return;

	i = futaba->history->position + 1;
	if (i == length) return;

	gtk_menu_popup(GTK_MENU(futaba->history->prev), NULL, NULL, NULL, NULL,
		       0, gtk_get_current_event_time());
}

static void
cb_show_next_history(GtkAction *action,
		     gpointer data)
{
	Futaba *futaba = data;

	fb_statusbar_pop();

	if (futaba->history->position <= 0) return;

	gtk_menu_popup(GTK_MENU(futaba->history->next), NULL, NULL, NULL, NULL,
		       0, gtk_get_current_event_time());
}

static void
cb_clear_history(GtkAction *action,
		 gpointer data)
{
	Futaba *futaba = data;

	fb_dir_history_clear(futaba);
}

static void
cb_up_dir(GtkAction *action,
	  gpointer data)
{
	gchar *path;
	Futaba *futaba = data;

	fb_statusbar_pop();

	if (!g_ascii_strcasecmp(futaba->workdir, "/")) {
		fb_statusbar_push(_("Top directory !!"));

		return;
	}

	path = g_path_get_dirname(futaba->workdir);

	fb_dirview_set_dir(futaba, path);

	g_free(path);
}

static void
cb_go_home(GtkAction *action,
	   gpointer data)
{
	Futaba *futaba = data;

	fb_statusbar_pop();

	if (!g_ascii_strcasecmp(futaba->workdir, home_page)) return;

	fb_dirview_set_dir(futaba, home_page);
}

static void
cb_refresh_dir(GtkAction *action,
	       gpointer data)
{
	Futaba *futaba = data;

	fb_statusbar_pop();

	fb_dirview_set_history(futaba, futaba->workdir);
}

static void
cb_change_column_type(GtkAction *action,
		      gpointer data)
{
	Futaba *futaba = data;

}

static void
cb_switch_file_display(GtkAction *action,
		       gpointer data)
{
	Futaba *futaba = data;

	fb_statusbar_pop();

	futaba->other_filter = !futaba->other_filter;
	fb_dirview_set_history(futaba, futaba->workdir);
}

static void
cb_switch_dot_file(GtkAction *action,
		   gpointer data)
{
	Futaba *futaba = data;

	fb_statusbar_pop();

	futaba->dot_filter = !futaba->dot_filter;
	fb_dirview_set_history(futaba, futaba->workdir);
}

static void
cb_switch_preview(GtkAction *action,
		  gpointer data)
{
	Futaba *futaba = data;

	fb_statusbar_pop();

	fb_dirview_set_preview(futaba, !futaba->show_preview);
}

static void
cb_zoom_in(GtkAction *action,
	   gpointer data)
{
	Futaba *futaba = data;

	if (!futaba->viewer) return;

	fb_imagecanvas_zooming(futaba->viewer->first, futaba->viewer->first->scale * ZOOMIN);

	if (futaba->viewer->two_pages)
		fb_imagecanvas_zooming(futaba->viewer->second, futaba->viewer->second->scale * ZOOMIN);
}

static void
cb_zoom_out(GtkAction *action,
	    gpointer data)
{
	Futaba *futaba = data;

	if (!futaba->viewer) return;

	fb_imagecanvas_zooming(futaba->viewer->first, futaba->viewer->first->scale * ZOOMOUT);

	if (futaba->viewer->two_pages)
		fb_imagecanvas_zooming(futaba->viewer->second, futaba->viewer->second->scale * ZOOMOUT);
}

static void
cb_zoom_100(GtkAction *action,
	    gpointer data)
{
	Futaba *futaba = data;

	if (!futaba->viewer) return;

	fb_imagecanvas_zooming(futaba->viewer->first, 1.0);

	if (futaba->viewer->two_pages)
		fb_imagecanvas_zooming(futaba->viewer->second, 1.0);
}

static void
cb_fit_image(GtkAction *action,
	     gpointer data)
{
	Futaba *futaba = data;

	if (!futaba->viewer) return;

	fb_imagecanvas_zoomfit(futaba->viewer->first);

	if (futaba->viewer->two_pages)
		fb_imagecanvas_zoomfit(futaba->viewer->second);
}

static void
cb_right_rotation(GtkAction *action,
		  gpointer data)
{
	Futaba *futaba = data;

	if (!futaba->viewer) return;

	fb_imagecanvas_rotate(futaba->viewer->first, FALSE);

	if (futaba->viewer->two_pages)
		fb_imagecanvas_rotate(futaba->viewer->second, FALSE);
}

static void
cb_left_rotation(GtkAction *action,
		 gpointer data)
{
	Futaba *futaba = data;

	if (!futaba->viewer) return;

	fb_imagecanvas_rotate(futaba->viewer->first, TRUE);

	if (futaba->viewer->two_pages)
		fb_imagecanvas_rotate(futaba->viewer->second, TRUE);
}

static void
cb_mirror_rotation(GtkAction *action,
		   gpointer data)
{
	Futaba *futaba = data;

	if (!futaba->viewer) return;

	fb_imagecanvas_mirror(futaba->viewer->first, TRUE, FALSE);

	if (futaba->viewer->two_pages)
		fb_imagecanvas_mirror(futaba->viewer->second, TRUE, FALSE);
}


static void
cb_r_zoom_in(GtkAction *action,
	     gpointer data)
{
	Futaba *futaba = data;

	if (!futaba->viewer) return;

	if (futaba->viewer->open_way == RIGHT)
		fb_imagecanvas_zooming(futaba->viewer->first, futaba->viewer->first->scale * ZOOMIN);
	else if (futaba->viewer->open_way == LEFT)
		fb_imagecanvas_zooming(futaba->viewer->second,  futaba->viewer->second->scale * ZOOMIN);
}

static void
cb_r_zoom_out(GtkAction *action,
	      gpointer data)
{
	Futaba *futaba = data;

	if (!futaba->viewer) return;

	if (futaba->viewer->open_way == RIGHT)
		fb_imagecanvas_zooming(futaba->viewer->first, futaba->viewer->first->scale * ZOOMOUT);
	else if (futaba->viewer->open_way == LEFT)
		fb_imagecanvas_zooming(futaba->viewer->second, futaba->viewer->second->scale * ZOOMOUT);
}

static void
cb_r_zoom_100(GtkAction *action,
	      gpointer data)
{
	Futaba *futaba = data;

	if (!futaba->viewer) return;

	if (futaba->viewer->open_way == RIGHT)
		fb_imagecanvas_zooming(futaba->viewer->first, 1.0);
	else if (futaba->viewer->open_way == LEFT)
		fb_imagecanvas_zooming(futaba->viewer->second, 1.0);
}

static void
cb_r_fit_image(GtkAction *action,
	       gpointer data)
{
	Futaba *futaba = data;

	if (!futaba->viewer) return;

	if (futaba->viewer->open_way == RIGHT)
		fb_imagecanvas_zoomfit(futaba->viewer->first);
	else if (futaba->viewer->open_way == LEFT)
		fb_imagecanvas_zoomfit(futaba->viewer->second);
}

static void
cb_r_right_rotation(GtkAction *action,
		    gpointer data)
{
	Futaba *futaba = data;

	if (!futaba->viewer) return;

	if (futaba->viewer->open_way == RIGHT)
		fb_imagecanvas_rotate(futaba->viewer->first, FALSE);
	else if (futaba->viewer->open_way == LEFT)
		fb_imagecanvas_rotate(futaba->viewer->second, FALSE);
}

static void
cb_r_left_rotation(GtkAction *action,
		   gpointer data)
{
	Futaba *futaba = data;

	if (!futaba->viewer) return;

	if (futaba->viewer->open_way == RIGHT)
		fb_imagecanvas_rotate(futaba->viewer->first, TRUE);
	else if (futaba->viewer->open_way == LEFT)
		fb_imagecanvas_rotate(futaba->viewer->second, TRUE);
}

static void
cb_r_mirror_rotation(GtkAction *action,
		     gpointer data)
{
	Futaba *futaba = data;

	if (!futaba->viewer) return;

	if (futaba->viewer->open_way == RIGHT)
		fb_imagecanvas_mirror(futaba->viewer->first, TRUE, FALSE);
	else if (futaba->viewer->open_way == LEFT)
		fb_imagecanvas_mirror(futaba->viewer->second, TRUE, FALSE);
}


static void
cb_l_zoom_in(GtkAction *action,
	     gpointer data)
{
	Futaba *futaba = data;

	if (!futaba->viewer) return;

	if (futaba->viewer->open_way == RIGHT)
		fb_imagecanvas_zooming(futaba->viewer->second, futaba->viewer->second->scale * ZOOMIN);
	else if (futaba->viewer->open_way == LEFT)
		fb_imagecanvas_zooming(futaba->viewer->first, futaba->viewer->first->scale * ZOOMIN);
}

static void
cb_l_zoom_out(GtkAction *action,
	      gpointer data)
{
	Futaba *futaba = data;

	if (!futaba->viewer) return;

	if (futaba->viewer->open_way == RIGHT)
		fb_imagecanvas_zooming(futaba->viewer->second, futaba->viewer->second->scale * ZOOMOUT);
	else if (futaba->viewer->open_way == LEFT)
		fb_imagecanvas_zooming(futaba->viewer->first, futaba->viewer->first->scale * ZOOMOUT);
}

static void
cb_l_zoom_100(GtkAction *action,
	      gpointer data)
{
	Futaba *futaba = data;

	if (!futaba->viewer) return;

	if (futaba->viewer->open_way == RIGHT)
		fb_imagecanvas_zooming(futaba->viewer->second, 1.0);
	else if (futaba->viewer->open_way == LEFT)
		fb_imagecanvas_zooming(futaba->viewer->first, 1.0);
}

static void
cb_l_fit_image(GtkAction *action,
	       gpointer data)
{
	Futaba *futaba = data;

	if (!futaba->viewer) return;

	if (futaba->viewer->open_way == RIGHT)
		fb_imagecanvas_zoomfit(futaba->viewer->second);
	else if (futaba->viewer->open_way == LEFT)
		fb_imagecanvas_zoomfit(futaba->viewer->first);
}

static void
cb_l_right_rotation(GtkAction *action,
		    gpointer data)
{
	Futaba *futaba = data;

	if (!futaba->viewer) return;

	if (futaba->viewer->open_way == RIGHT)
		fb_imagecanvas_rotate(futaba->viewer->second, FALSE);
	else if (futaba->viewer->open_way == LEFT)
		fb_imagecanvas_rotate(futaba->viewer->first, FALSE);
}

static void
cb_l_left_rotation(GtkAction *action,
		   gpointer data)
{
	Futaba *futaba = data;

	if (!futaba->viewer) return;

	if (futaba->viewer->open_way == RIGHT)
		fb_imagecanvas_rotate(futaba->viewer->second, TRUE);
	else if (futaba->viewer->open_way == LEFT)
		fb_imagecanvas_rotate(futaba->viewer->first, TRUE);
}

static void
cb_l_mirror_rotation(GtkAction *action,
		     gpointer data)
{
	Futaba *futaba = data;

	if (!futaba->viewer) return;

	if (futaba->viewer->open_way == RIGHT)
		fb_imagecanvas_mirror(futaba->viewer->second, TRUE, FALSE);
	else if (futaba->viewer->open_way == LEFT)
		fb_imagecanvas_mirror(futaba->viewer->first, TRUE, FALSE);
}

static void
cb_two_pages(GtkAction *action,
	     gpointer data)
{
	Futaba *futaba = data;

	if (!futaba->viewer) return;

	if (futaba->viewer->two_pages)
		fb_imageviewer_fixup_window(futaba->viewer);
	else
		fb_imageviewer_split_window(futaba->viewer);
}

static void
cb_max_window(GtkAction *action,
	      gpointer data)
{
	Futaba *futaba = data;

	if (!futaba->viewer) return;
	if (futaba->viewer->size > WINDOW_FULL) return;

	if (futaba->viewer->size > WINDOW_MAX) {
		gtk_window_unmaximize(GTK_WINDOW(futaba->viewer->top));
		futaba->viewer->size ^= WINDOW_MAX;
	}
	else {
		gtk_window_maximize(GTK_WINDOW(futaba->viewer->top));
		gtk_window_move(GTK_WINDOW(futaba->viewer->top), 0, 0);
		futaba->viewer->size ^= WINDOW_MAX;
	}
}

static void
cb_fullscreen(GtkAction *action,
	      gpointer data)
{
	Futaba *futaba = data;

	if (!futaba->viewer) return;
	/*   if (!futaba->viewer->focus) return; */

	if (futaba->viewer->size > WINDOW_FULL) {
		gtk_window_unfullscreen(GTK_WINDOW(futaba->viewer->top));
		futaba->viewer->size ^= WINDOW_FULL;
	}
	else {
		gtk_window_fullscreen(GTK_WINDOW(futaba->viewer->top));
		futaba->viewer->size ^= WINDOW_FULL;
	}
}

static void
cb_show_tbar(GtkAction *action,
	     gpointer data)
{
	gboolean is_active;
	Futaba *futaba = data;

	is_active = gtk_toggle_action_get_active(GTK_TOGGLE_ACTION(action));

	if (is_active)
		gtk_widget_show(futaba->tool_bar);
	else
		gtk_widget_hide(futaba->tool_bar);

}
      
static void
cb_show_lbar(GtkAction *action,
	     gpointer data)
{
	gboolean is_active;
	Futaba *futaba = data;

	is_active = gtk_toggle_action_get_active(GTK_TOGGLE_ACTION(action));

	if (is_active)
		gtk_widget_show(futaba->location_bar);
	else
		gtk_widget_hide(futaba->location_bar);

}
      
static void
cb_show_sbar(GtkAction *action,
	     gpointer data)
{
	gboolean is_active;
	Futaba *futaba = data;

	is_active = gtk_toggle_action_get_active(GTK_TOGGLE_ACTION(action));

	if (is_active)
		gtk_widget_show(futaba->status_bar);
	else
		gtk_widget_hide(futaba->status_bar);

}

static void
cb_set_reverse(GtkAction *action,
	       gpointer data)
{
	gint sort_id;
	GtkSortType order;
	GtkTreeSortable *sortable;
	Futaba *futaba = data;

	if (!futaba->liststore) return;

	sortable = GTK_TREE_SORTABLE(futaba->liststore);
	gtk_tree_sortable_get_sort_column_id(sortable, &sort_id, NULL);

	if (gtk_toggle_action_get_active(GTK_TOGGLE_ACTION(action)))
		order = GTK_SORT_DESCENDING;
	else
		order = GTK_SORT_ASCENDING;

	gtk_tree_sortable_set_sort_column_id(sortable, sort_id, order);
}      
      
static void
cb_show_col1(GtkAction *action,
	     gpointer data)
{
	gboolean is_active;
	Futaba *futaba = data;

	if (!futaba->column1) return;

	is_active = gtk_toggle_action_get_active(GTK_TOGGLE_ACTION(action));

	gtk_tree_view_column_set_visible(futaba->column1, is_active);

	show_col1 = is_active;
}

static void
cb_show_col2(GtkAction *action,
	     gpointer data)
{
	gboolean is_active;
	Futaba *futaba = data;

	if (!futaba->column2) return;

	is_active = gtk_toggle_action_get_active(GTK_TOGGLE_ACTION(action));

	gtk_tree_view_column_set_visible(futaba->column2, is_active);

	show_col2 = is_active;
}

static void
cb_show_col3(GtkAction *action,
	     gpointer data)
{
	gboolean is_active;
	Futaba *futaba = data;

	if (!futaba->column3) return;

	is_active = gtk_toggle_action_get_active(GTK_TOGGLE_ACTION(action));

	gtk_tree_view_column_set_visible(futaba->column3, is_active);

	show_col3 = is_active;
}

static void
cb_show_col4(GtkAction *action,
	     gpointer data)
{
	gboolean is_active;
	Futaba *futaba = data;

	if (!futaba->column4) return;

	is_active = gtk_toggle_action_get_active(GTK_TOGGLE_ACTION(action));

	gtk_tree_view_column_set_visible(futaba->column4, is_active);

	show_col4 = is_active;
}

static void
cb_show_col5(GtkAction *action,
	     gpointer data)
{
	gboolean is_active;
	Futaba *futaba = data;

	if (!futaba->column5) return;

	is_active = gtk_toggle_action_get_active(GTK_TOGGLE_ACTION(action));

	gtk_tree_view_column_set_visible(futaba->column5, is_active);

	show_col5 = is_active;
}

static void
cb_show_coltitle(GtkAction *action,
		 gpointer data)
{
	gboolean is_active;
	Futaba *futaba = data;

	if (!futaba->treeview) return;

	is_active = gtk_toggle_action_get_active(GTK_TOGGLE_ACTION(action));

	gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(futaba->treeview), is_active);

	show_coltitle = is_active;
}

static void
cb_set_dot_filter(GtkAction *action,
		  gpointer data)
{
	Futaba *futaba = data;

	fb_statusbar_pop();

	futaba->dot_filter = gtk_toggle_action_get_active(GTK_TOGGLE_ACTION(action));
	fb_dirview_set_history(futaba, futaba->workdir);
}

static void
cb_set_other_filter(GtkAction *action,
		    gpointer data)
{
	Futaba *futaba = data;

	fb_statusbar_pop();

	futaba->other_filter = gtk_toggle_action_get_active(GTK_TOGGLE_ACTION(action));
	fb_dirview_set_history(futaba, futaba->workdir);
}

static void
cb_sort_mode(GtkRadioAction *action,
	     GtkRadioAction *current,
	     gpointer data)
{
	FbDirViewSortMode mode;
	Futaba *futaba = data;

	mode = gtk_radio_action_get_current_value(GTK_RADIO_ACTION(current));
	fb_dirview_set_sort_mode(futaba, mode);
}
/*
 *------------------------------------------------------------
 *       Public 
 *------------------------------------------------------------
 */
void
action_connect_accelerator(GtkActionGroup *action_group,
			   const gchar *path)
{
	GtkAction *action;

	action = gtk_action_group_get_action(action_group, path);
	gtk_action_connect_accelerator(action);
}

GtkActionGroup *
fb_actions_new(Futaba *futaba)
{
	GtkActionGroup *action_group;

	action_group = gtk_action_group_new("Futaba");

	gtk_action_group_set_translation_domain(action_group, NULL);

	gtk_action_group_add_actions(action_group,
				     actions, actions_len,
				     futaba);

	gtk_action_group_add_toggle_actions(action_group,
					    toggle_actions, toggle_actions_len,
					    futaba);

	gtk_action_group_add_radio_actions(action_group,
					   radio_actions, radio_actions_len,
					   sort_mode,
					   G_CALLBACK(cb_sort_mode), futaba);

	return action_group;
}
