<?php
/**
 * クライアントからの要求を提供するHTTPリクエスト、
 * クライアントへの応答を送信する際の処理を行うHTTPレスポンス
 * それぞれで使用するデータを保持するクラスです。
 * 
 * @package spider spiderのコアパッケージ
 * @version 1.0.0
 * @copyright Copyright &copy; 2008, Multimedia Digital Contents Systems.Co.,Ltd.<info@md-systems.net> http://www.md-systems.net/
 * @author Multimedia Digital Contents Systems.Co.,Ltd. m.nakashima <m_nakashima@md-systems.net>
 * @since PHP 4.3
 */
define( 'SPIDER_SESSION_SCOPE_GLOBAL', 'GLOBAL' );
define( 'SPIDER_SESSION_SCOPE_AUTO', 'AUTO' );
require_once(dirname(dirname(__FILE__)).DIRECTORY_SEPARATOR.'util'.DIRECTORY_SEPARATOR.'CharUtility.class.php');
class spider_HttpRequest {
	/** Attributes			*/
	var $attribute_array	= array();
	/** Errors				*/
	var $errors				= array();
	/** Global Errors		*/
	var $global_errors		= array();
	/** Response Headers	*/
	var $headers			= array();
	/** Response Body		*/
	var $response_body		= null;
	/** Response File Path	*/
	var $response_file_path	= null;
	/** Redirect To URL		*/
	var $redirect_url		= null;
	/** Attribute Prefix	*/
	var $attribute_prefix	= null;

	/**
	 * コンストラクタ
	 */
	function spider_HttpRequest() {
		$this->attribute_array	= array();
		$this->errors			= array();
	}
	/**
	 * 属性を設定します。
	 * @param $key 属性名
	 * @param $value 属性値
	 */
	function setAttribute( $key, & $value ) {
		$reg_key		= $key;
		$callstack		= debug_backtrace();
		$caller_file	= $callstack[0]['file'];
		unset($callstack);
		if( strpos($caller_file,DIR_PATH_LIB) !== false
			&& strpos($caller_file,DIR_PATH_LIB.DIRECTORY_SEPARATOR.'spider') === false ) {
			// 呼び出し元がlib内のファイルの場合
			$lib_uri		= str_replace(DIR_PATH_LIB,'',$caller_file);
			$caller_class	= preg_replace('/^_/','',str_replace('.class.php','',str_replace(DIRECTORY_SEPARATOR,'_',$lib_uri)));
			$caller_prefix	= str_replace('_','.',$caller_class);
			$package_prefix	= preg_replace('/\\.[^\\.]+$/','',$caller_prefix);
			if( preg_match('/^'.CharUtility::escape_regx_str($caller_prefix).'/',$key) == 0
				&& preg_match('/^'.CharUtility::escape_regx_str($package_prefix).'/',$key) == 0
			) {
				// パッケージ名もクラス名も属性名に含まれないならクラスプレフィックスつけて登録
				$reg_key	= $caller_prefix.'.'.$reg_key;
			}
		} else {
			// lib内でない場合、下位互換の為、$GLOBALSにも登録する
			$reg_key	= 'page.'.$key;
			$GLOBALS[$reg_key]	= $value;
			$GLOBALS[$key]		= $value;
		}
		if( !is_null( $this->attribute_prefix ) && strlen($this->attribute_prefix) > 0 ) {
			$reg_key	= $this->attribute_prefix.'.'.$reg_key;
		}
		$this->attribute_array[$reg_key]	= $value;
		$this->attribute_array[$key]		= $value;
	}
	/**
	 * 指定キーの属性を取得します。
	 * @param $key 属性名
	 */
	function getAttribute( $key ) {
		$ref	= & $this->attribute_array[$key];
		return $ref;
	}
	/**
	 * 指定キーの性がセットされたか確認します
	 */
	function existAttribute( $key ) {
		return array_key_exists( $key, $this->attribute_array );
	}
	/**
	 * エラーメッセージを追加します。
	 * @param $message エラーメッセージ
	 * @param $log_level ログ出力を同時に行う場合に指定します。falseを指定すると記録しません。デフォルトはfalseです。
	 */
	function addError( $message, $log_level=false ) {
		array_push( $this->errors, $message );
		if( false !== $log_level && is_numeric($log_level) ) {
			$this->writeLog( $message, $log_level );
		}
	}
	/**
	 * エラーがあるか確認します。
	 * @return true:エラーあり、false:それ以外
	 */
	function isError() {
		if( count( $this->errors ) > 0 ) {
			return true;
		} else {
			return false;
		}
	}
	/**
	 * グローバルエラーを追加します
	 */
	function addGlobalError( $message ) {
		array_push( $this->global_errors, $message );
	}
	/**
	 * レスポンスヘッダを設定します
	 */
	function setResponseHeader( $key, $value ) {
		$this->headers[$key]	= $value;
	}
	/**
	 * レスポンスボディを設定します
	 */
	function setResponseBody( $body ) {
		$this->response_body	= $body;
	}
	/**
	 * レスポンスボディをファイルパスで設定します。
	 */
	function setResponseFile( $file_path ) {
		$this->response_file_path	= $file_path;
	}
	/**
	 * リダイレクト先を設定します
	 */
	function redirectTo($url) {
		$this->redirect_url	= $url;
	}
	/**
	 * セッション変数を設定します
	 * @param $key
	 * @param $value
	 * @param $scopse GLOBALまたは有効フォルダURI
	 */
	function setSession( $key, $value, $scope=SPIDER_SESSION_SCOPE_AUTO ) {
		$this->optimizeSession();
		if( $scope == SPIDER_SESSION_SCOPE_AUTO ) {
			$scope	= dirname($_SERVER['REQUEST_URI']);
		}
		if( SPIDER_SESSION_SCOPE_GLOBAL == $scope ) {
			$key	= $this->_getGlobalSessionKey( $key );
		} else {
			$key	= $scope.'/'.$key;
		}
		$_SESSION[$key]	= serialize( $value );
	}
	/**
	 * セッション変数を取得します
	 * @param $key
	 */
	function getSession( $key ) {
		$this->optimizeSession();
		$value			= null;
		$current_scope	= dirname( $_SERVER['REQUEST_URI'] );
		while( strlen($current_scope) > 0 ) {
			$target_key		= $current_scope.'/'.$key;
			if( isset( $_SESSION ) && isset($_SESSION[$target_key]) ) {
				return unserialize($_SESSION[$target_key]);
			}
			$current_scope	= dirname( $current_scope );
		}
		$target_key	= $this->_getGlobalSessionKey( $key );
		if( isset( $_SESSION ) && isset($_SESSION[$target_key]) ) {
			return unserialize($_SESSION[$target_key]);
		}
		
		return null;
	}
	/**
	 * セッション変数が登録されているか確認します
	 * @param $key
	 */
	function existsSession( $key ) {
		$this->optimizeSession();
		$current_scope	= dirname( $_SERVER['REQUEST_URI'] );
		while( strlen($current_scope) > 0 ) {
			$target_key		= $current_scope.'/'.$key;
			if( isset( $_SESSION ) && isset($_SESSION[$target_key]) ) {
				return true;
			}
			$current_scope	= dirname( $current_scope );
		}
		$target_key	= $this->_getGlobalSessionKey( $key );
		if( isset( $_SESSION ) && isset($_SESSION[$target_key]) ) {
			return true;
		}
		return false;
	}
	/**
	 * セッション登録済み情報を最適化します
	 */
	function optimizeSession() {
		if( isset( $_SESSION ) && is_array( $_SESSION ) ) {
			$current_scope	= dirname( $_SERVER['REQUEST_URI'] );
			foreach( $_SESSION as $key => $value ) {
				if( preg_match('/^spider\\_GLOBAL\\./',$key) == 0
					&& preg_match('/^'.CharUtility::escape_regx_str($current_scope).'/',$key) == 0 ) {
					unset( $_SESSION[$key] );	
				}
			}
		}
	}
	/**
	 * グローバルセッションキー
	 */
	function _getGlobalSessionKey( $key ) {
		return 'spider_GLOBAL.'.$key;
	}
	/**
	 * ログを出力します
	 * @param $message ログメッセージ
	 */
	function writeLog( $message, $log_level=SPIDER_LOG_LEVEL_INFO ) {
		$system_log_level	= SPIDER_LOG_LEVEL_INFO;
		if( defined('SYSTEM_LOG_LEVEL') && is_numeric(SYSTEM_LOG_LEVEL)
			&& preg_match('/^[0-4]$/',SYSTEM_LOG_LEVEL) > 0 ) {
			$system_log_level	= SYSTEM_LOG_LEVEL;
		}
		// システム設定のログレベルより大きいレベルのログは出力しない
		if( $log_level > $system_log_level ) {
			return true;
		}

		$message	= str_replace( "\r\n", "\n", $message );
		$message	= str_replace( "\r", "\n", $message );
		$message	= str_replace( "\n", " ", $message );

		$callstack		= debug_backtrace();
		$caller_file	= $callstack[0]['file'];
		$caller_line	= $callstack[0]['line'];
		unset($callstack);

		$log_message	= date('Y-m-d H:i:s')
			. "\t".$message
			. "\t"."[".$caller_file."] "
			. "\t"."[".$caller_line."] "
			. "\t" . $_SERVER['REMOTE_ADDR']
			. "\t" . $_SERVER['REMOTE_HOST']
			. "\t" . $_SERVER['HTTP_USER_AGENT'];
		
		$log_type	= SPIDER_LOG_TYPE_FILE;
		if( defined('SYSTEM_LOG_TYPE') && is_numeric(SYSTEM_LOG_TYPE) ) {
			if( SPIDER_LOG_TYPE_SYSLOG == SYSTEM_LOG_TYPE ) {
				$log_type	= SPIDER_LOG_TYPE_SYSLOG;
			}
		}
		if( SPIDER_LOG_TYPE_SYSLOG == $log_type ) {
			// syslogdに出力する場合
			openlog('spiderLog', LOG_PID | LOG_NDELAY, LOG_SYSLOG);
			$log_level	= LOG_INFO;
			switch ($log_level) {
				case SPIDER_LOG_LEVEL_DEBUG:
				$log_level	= LOG_DEBUG;
				break;
				case SPIDER_LOG_LEVEL_INFO:
				$log_level	= LOG_INFO;
				break;
				case SPIDER_LOG_LEVEL_NOTICE:
				$log_level	= LOG_NOTICE;
				break;
				case SPIDER_LOG_LEVEL_WARNING:
				$log_level	= LOG_WARNING;
				break;
				case SPIDER_LOG_LEVEL_ERROR:
				$log_level	= LOG_ERR;
				break;
				case SPIDER_LOG_LEVEL_FATAL:
				$log_level	= LOG_CRIT;
				break;
				default:
				break;
			}
			syslog($log_level, $log_message);
			closelog();
		} else {
			// ファイルに出力する
			// ログファイル名の決定
			$log_file_path	= DIR_PATH_LOG.DIRECTORY_SEPARATOR.'spider';
			if( !file_exists( $log_file_path ) ) {
				if( @mkdir( $log_file_path, 0777 ) ) {
					@chmod( $log_file_path, 0777 );
				}
			}
			$log_file_path	= $log_file_path.DIRECTORY_SEPARATOR.date('Y');
			if( !file_exists( $log_file_path ) ) {
				if( @mkdir( $log_file_path, 0777 ) ) {
					@chmod( $log_file_path, 0777 );
				}
			}
			$log_file_path	= $log_file_path.DIRECTORY_SEPARATOR.date('m');
			if( !file_exists( $log_file_path ) ) {
				if( @mkdir( $log_file_path, 0777 ) ) {
					@chmod( $log_file_path, 0777 );
				}
			}
			$log_file_path	= $log_file_path.DIRECTORY_SEPARATOR.'system_'.date('d').'.log';
			if( !file_exists( $log_file_path ) ) {
				if(@touch( $log_file_path, 0666 )){
					@chmod( $log_file_path, 0666 );
				}
			}
			error_log($log_message."\n" , 3, $log_file_path );
		}
		return true;
	}
	/**
	 * Fatalレベルのログを出力します。
	 * @param $message ログメッセージ
	 */
	function fatal($message){
		$this->writeLog( $message, SPIDER_LOG_LEVEL_FATAL );
	}
	/**
	 * Errorレベルのログを出力します。
	 * @param $message ログメッセージ
	 */
	function error($message){
		$this->writeLog( $message, SPIDER_LOG_LEVEL_ERROR );
	}
	/**
	 * Warningレベルのログを出力します。
	 * @param $message ログメッセージ
	 */
	function warn($message){
		$this->writeLog( $message, SPIDER_LOG_LEVEL_WARNING );
	}
	/**
	 * Noticeレベルのログを出力します。
	 * @param $message ログメッセージ
	 */
	function notice($message){
		$this->writeLog( $message, SPIDER_LOG_LEVEL_NOTICE );
	}
	/**
	 * Infoレベルのログを出力します。
	 * @param $message ログメッセージ
	 */
	function info($message){
		$this->writeLog( $message, SPIDER_LOG_LEVEL_INFO );
	}
	/**
	 * Debugレベルのログを出力します。
	 * @param $message ログメッセージ
	 */
	function debug($message){
		$this->writeLog( $message, SPIDER_LOG_LEVEL_DEBUG );
	}
	/**
	 * システム設定された値でメール送信オブジェクトを作成して取得します。
	 * @return mixed util_Mailクラスの実装オブジェクト・インスタンスを作成できなかった場合はfalse
	 */
	function getSystemMailer() {
		if( defined('SYSTEM_MAIL_SEND_METHOD') ) {
			require_once( dirname(dirname(__FILE__))
				. DIRECTORY_SEPARATOR . "util"
				. DIRECTORY_SEPARATOR . "Mail.class.php" );
			// 送信オプション
			$params		= array();
			// 送信方法の決定
			$class_name	= 'PHP';
			if( preg_match('/^[sS][mM][tT][pP]$/',SYSTEM_MAIL_SEND_METHOD) > 0 ) {
				$class_name	= 'SMTP';
			} else if( preg_match('/^[sS][eE][nN][dD][mM][aA][iI][lL]$/',SYSTEM_MAIL_SEND_METHOD) > 0 ) {
				$class_name	= 'SendMail';
			}
			$params		= $GLOBALS['SYSTEM_MAIL_SEND_METHOD_OPTIONS'];
			// 送信オブジェクト作成
			$send_object	= util_Mail::get_instance( $class_name, $params );
			if( $send_object === false ) {
				return false;
			} else {
				return $send_object;
			}
		} else {
			return false;
		}
	}
	/**
	 * メールを送信します。
	 * 宛先、配信元、返信先、戻り先のメールアドレスを指定しなかった場合、下記に定義された定数もしはグローバル配列を探して
	 * 定義済みの変数を元に送信を実行します。定義がなかった場合はfalseを返します。
	 * 
	 * 宛て先メールアドレス $GLOBALS['SYSTEM_MAIL_REPORT_TO_ADDRESSES'] 配列で定義してください。
	 * 配信元メールアドレス SYSTEM_MAIL_FROM_ADDRESS 文字列で定義してください。
	 * 配信先メールアドレス SYSTEM_MAIL_REPLY_ADDRESS 指定がない場合はSYSTEM_MAIL_FROM_ADDRESSを利用します。
	 * 戻り先メールアドレス SYSTEM_MAIL_RETURN_ADDRESS 指定がない場合はSYSTEM_MAIL_FROM_ADDRESSを利用します。
	 * 
	 * @param $subject メール件名
	 * @param $message メール本文
	 * @param $mailto 宛て先メールアドレス
	 * @param $from 配信元メールアドレス指定
	 * @param $reply 返信先メールアドレス指定
	 * @param $return エラー戻り先メールアドレス指定
	 * @return boolean
	 */
	function mailTo($subject,$message,$mailto=null,$from=null,$reply=null,$return=null) {
		$mailto_addresses	= array();
		if( isset($GLOBALS['SYSTEM_MAIL_REPORT_TO_ADDRESSES'])
		&& is_array($GLOBALS['SYSTEM_MAIL_REPORT_TO_ADDRESSES']) ) {
			$mailto_addresses	= $GLOBALS['SYSTEM_MAIL_REPORT_TO_ADDRESSES'];
		}
		$mailto	= trim($mailto);
		if(!is_null($mailto) && strlen($mailto) > 0 ) {
			array_push($mailto_addresses, $mailto);
		}
		if(count($mailto_addresses)==0){
			$this->warn('HttpRequest->MailTo: derivery mail address is required!: '.$subject.':'.$message);
			return false;
		}
		if(defined('SYSTEM_MAIL_SUBJECT_PREFIX')){
			$subject	= SYSTEM_MAIL_SUBJECT_PREFIX.$subject;
		}
		if( is_null($from) || strlen($from) == 0 ) {
			if(defined('SYSTEM_MAIL_FROM_ADDRESS') && strlen(SYSTEM_MAIL_FROM_ADDRESS)>0 ) {
				if(defined('SYSTEM_MAIL_FROM_NAME') && strlen(SYSTEM_MAIL_FROM_NAME)>0 ) {
					$from	= SYSTEM_MAIL_FROM_NAME.' <'.SYSTEM_MAIL_FROM_ADDRESS.'>';
				} else {
					$from	= SYSTEM_MAIL_FROM_ADDRESS;
				}
			} else {
				$this->warn('HttpRequest->MailTo: from address is required!: '.$subject.':'.$message);
				return false;
			}
		}
		if( is_null($reply) || strlen($reply) == 0 ) {
			if(defined('SYSTEM_MAIL_REPLY_ADDRESS') && strlen(SYSTEM_MAIL_REPLY_ADDRESS)>0 ) {
				$reply	= SYSTEM_MAIL_REPLY_ADDRESS;
			} else {
				$reply	= $from;
			}
		}
		if( is_null($return) || strlen($return) == 0 ) {
			if(defined('SYSTEM_MAIL_RETURN_ADDRESS') && strlen(SYSTEM_MAIL_RETURN_ADDRESS)>0 ) {
				$return	= SYSTEM_MAIL_RETURN_ADDRESS;
			} else {
				$return	= $from;
			}
		}
		
		if( $send_object = $this->getSystemMailer() ) {
			foreach( $mailto_addresses as $address ) {
				$result	= $send_object->send( $address, $subject, $message, $from, $reply, $return );
				if( $result ) {
					$this->info( 'HttpRequest->MailTo: '.$address.':'.$subject.':'.$message );
					return true;
				} else {
					$this->warn('HttpRequest->MailTo: delivey error: '.$subject.':'.$message);
					return false;
				}
			}
		} else {
			$this->warning( 'HttpRequest->MailTo: can\'t create mailer instance!: '.$mailto.':'.$subject.':'.$message );
			return false;
		}
	}
}
?>
