package jp.sourceforge.foolishmerge.text;

import java.util.Arrays;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;

import jp.sourceforge.foolishmerge.merge.ConflictDeltas;

import org.apache.commons.jrcs.diff.Chunk;
import org.apache.commons.jrcs.diff.Delta;
import org.apache.commons.jrcs.diff.PatchFailedException;
import org.apache.commons.jrcs.util.ToString;

/**
 * 編集前のオリジナルのテキストを表すクラス。
 */
public final class Original extends AbstractText {

  /**
   * マージされたテキスト
   */
  private List merged = null;

  /**
   * オフセット
   */
  private int mergeOffset = 0;

  /**
   * オブジェクトを生成する。
   * @param text テキスト
   */
  public Original(String text) {
    // スーパークラスのコンストラクタを呼び出す。
    super(text);
    // マージされたテキストを生成。
    merged = new LinkedList(Arrays.asList(content));
  }

  /**
   * マージされたテキストを返す。
   * @return マージされたテキスト
   */
  public String getMergerd() {
    // リストを配列に変換。
    String[] mergedArray = new String[merged.size()];
    merged.toArray(mergedArray);

    // 配列を文字列に変換。
    String mergedText = ToString.arrayToString(mergedArray);

    // 文字列を返す。
    return mergedText;
  }

  /**
   * パッチを当てる。
   * @param delta デルタ
   * @throws PatchFailedException パッチに失敗した場合
   */
  public void patch(Delta delta) throws PatchFailedException {
    // オリジナルのチャンクを取得。
    Chunk org = delta.getOriginal();
    // 変更されたチャンクを取得。
    Chunk rev = delta.getRevised();

    // 新しいポジションでチャンクを生成。
    Chunk newOrg = newChunk(org, org.first() + mergeOffset);
    Chunk newRev = newChunk(rev, rev.first() + mergeOffset);

    // 新しいデルタを生成。
    Delta newDlt = Delta.newDelta(newOrg, newRev);

    // テキストにパッチを当てる。
    newDlt.patch(merged);

    // チャンクのサイズの差分を取得。
    int size = rev.size() - org.size();

    // オフセットをシフト。
    mergeOffset += size;
  }

  /**
   * 新しいポジションのチャンクを生成する。
   * @param chunk 元のチャンク
   * @param pos 新しいポジション
   * @return 新しいポジションのチャンク
   */
  private static Chunk newChunk(Chunk chunk, int pos) {
    // 新しいポジションでチャンクを生成。
    Chunk newChunk = new Chunk(chunk.chunk(), 0, chunk.chunk().size(), pos);

    // 新しいチャンクを返す。
    return newChunk;
  }

  /**
   * 競合するデルタをマージする。
   * @param conflictDeltas 競合するデルタ
   * @param filename コンフリクト時に表示するファイル名
   * @param revision コンフリクト時に表示するリビジョン
   * @throws PatchFailedException パッチに失敗した場合
   */
  public void merge(
    ConflictDeltas conflictDeltas,
    String filename,
    String revision)
    throws PatchFailedException {
    // 開始/終了ポジションを取得。
    int first = conflictDeltas.first();
    int last = conflictDeltas.last();

    // オリジナルのテキストの変更部分を切り出す。
    List part = cutPert(first, last);

    // 変更部分にパッチを当てる。
    List patched1 = pactch(part, conflictDeltas.getDeltas1(), first);
    List patched2 = pactch(part, conflictDeltas.getDeltas2(), first);

    // コンフリクト時のテキストを生成。
    if (!patched1.equals(patched2)) {
      patched1.add(0, "<<<<<<< " + filename);
      patched1.add("=======");
      patched1.addAll(patched2);
      patched1.add(">>>>>>> " + revision);
    }

    // コンフリクト時のテキストを、オリジナルのテキストに挿入。
    merged.addAll(first + mergeOffset, patched1);

    // 修正サイズを取得。
    int size = patched1.size() - part.size();

    // オフセットをシフト。
    mergeOffset += size;
  }

  /**
   * パッチを当てる。
   * @param part パッチを当てる部分
   * @param deltas デルタのセット
   * @param first 開始ポジション
   * @return パッチを当てたテキスト
   * @throws PatchFailedException パッチに失敗した場合
   */
  private static List pactch(List part, Set deltas, int first)
    throws PatchFailedException {
    // パッチを当てる部分をコピー
    List patched = copy(part);

    // デルタのセットからイテレータを取得。
    Iterator ite = deltas.iterator();

    // すべてのデルタのパッチを当てる。
    for (int offset = 0; ite.hasNext();) {
      // デルタを取得。
      Delta delta = (Delta) ite.next();
      // テキストにパッチを当てる。
      offset += patch(patched, delta, first, offset);
    }

    // パッチを当てたテキストを返す。
    return patched;
  }

  /**
   * 指定されたテキストにパッチを当てる。
   * @param text テキスト
   * @param delta デルタ
   * @param first 開始ポジション
   * @param offset オフセット
   * @return 修正のサイズ
   * @throws PatchFailedException パッチに失敗した場合
   */
  private static int patch(List text, Delta delta, int first, int offset)
    throws PatchFailedException {
    // オリジナルのチャンクを取得。
    Chunk org = delta.getOriginal();
    // 変更されたチャンクを取得。
    Chunk rev = delta.getRevised();

    // 新しいポジションでチャンクを生成。
    Chunk newOrg = newChunk(org, org.first() - first);
    Chunk newRev = newChunk(rev, rev.first() - first);

    // 新しいデルタを生成。
    Delta newDlt = Delta.newDelta(newOrg, newRev);

    // テキストにパッチを当てる。
    newDlt.patch(text);

    // チャンクのサイズの差分を取得。
    int size = rev.size() - org.size();

    // 修正のサイズを返す。
    return size;
  }

  /**
   * テキストのリストをコピーする。
   * @param src コピー元
   * @return コピーされたテキストのリスト
   */
  private static List copy(List src) {
    // コピー元を文字列配列に変換
    String[] array = new String[src.size()];
    src.toArray(array);

    // 文字列配列をリストに変換。
    List dst = new LinkedList(Arrays.asList(array));

    // コピーされたリストを返す。
    return dst;
  }

  /**
   * オリジナルのテキストから一部分を切り出す。
   * @param first 開始行(これを含む)
   * @param last 終了行(これを含む)
   * @return オリジナルのテキストの一部分
   */
  private List cutPert(int first, int last) {
    // 部分を格納するリストを生成。
    List part = new LinkedList();

    // 取得した部分を削除し、リストに格納。
    for (int i = first; i <= last; i++) {
      Object line = merged.remove(first + mergeOffset);
      part.add(line);
    }

    // オリジナルのテキストの一部分を返す。
    return part;
  }

}
