package jp.sourceforge.foolishmerge.merge;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;

import org.apache.commons.jrcs.diff.Delta;
import org.apache.commons.jrcs.diff.DifferentiationFailedException;
import org.apache.commons.jrcs.diff.PatchFailedException;
import org.apache.commons.jrcs.diff.Revision;

import jp.sourceforge.foolishmerge.text.Modified;
import jp.sourceforge.foolishmerge.text.Original;

public final class Merge {

  /**
   * テキストをマージする。
   * @param original オリジナルのテキスト
   * @param modified1 編集後のテキスト1
   * @param modified2 編集後のテキスト2
   * @param filename コンフリクト時に表示するファイル名
   * @param revision コンフリクト時に表示するリビジョン名
   * @return コンフリクトしなかった場合はtrue
   * @throws DifferentiationFailedException 差分の取得に失敗した場合
   * @throws PatchFailedException パッチに失敗した場合
   */
  public static boolean merge(
    Original original,
    Modified modified1,
    Modified modified2,
    String filename,
    String revision)
    throws DifferentiationFailedException, PatchFailedException {

    // コンフリクトフラグを宣言。
    boolean isNotConflict = true;

    // 差分を取得。
    Revision rev1 = original.diff(modified1);
    Revision rev2 = original.diff(modified2);

    // デルタのイテレータを取得。
    ListIterator dlts1 = iterator(rev1);
    ListIterator dlts2 = iterator(rev2);

    // デルタを取得。
    Delta delta1 = getDelta(dlts1);
    Delta delta2 = getDelta(dlts2);

    // 差分をオリジナルのテキストにマージ。
    while (delta1 != null || delta2 != null) {
      // コンフリクトしているかチェック。
      int conflict = Conflict.conflict(delta1, delta2);

      // コンフリクトの内容を判定。
      switch (conflict) {
        case Conflict.D1_LT_D2 :
          // コンフリクトしていない場合。(デルタ1 < デルタ2)
          // オリジナルのテキストにパッチ。(デルタ1)
          original.patch(delta1);
          // デルタ1を取得。
          delta1 = getDelta(dlts1);
          break;

        case Conflict.D1_GT_D2 :
          // コンフリクトしていない場合。(デルタ1 > デルタ2)
          // オリジナルのテキストにパッチ。(デルタ2)
          original.patch(delta2);
          // デルタ2を取得。
          delta2 = getDelta(dlts2);
          break;

        default :
          // コンフリクトしていた場合。
          // コンフリクトフラグをfalseにする。
          isNotConflict = false;
          // 競合するデルタをロード。
          ConflictDeltas conflictDeltas = new ConflictDeltas(dlts1, dlts2);
          conflictDeltas.loadDeltas(delta1, delta2);
          // オリジナルのテキストにデルタをマージ。
          original.merge(conflictDeltas, filename, revision);
          // デルタ1、デルタ2を取得。
          delta1 = getDelta(dlts1);
          delta2 = getDelta(dlts2);
          break;
      }
    }

    // コンフリクトフラグを返す。
    return isNotConflict;
  }

  /**
   * デルタのイテレータを取得する。
   * @param rev 差分
   * @return デルタのイテレータ
   */
  private static ListIterator iterator(Revision rev) {
    // リストを生成。
    List list = new ArrayList();

    // 差分のデルタをリストに格納。
    for (int i = 0; i < rev.size(); i++) {
      Delta dlt = rev.getDelta(i);
      list.add(dlt);
    }

    // リストからイテレータを取得。
    ListIterator ite = list.listIterator();

    // イテレータを返す。
    return ite;
  }

  /**
   * イテレータからデルタを取得する。
   * @param dlts デルタのイテレータ
   * @return デルタ
   */
  private static Delta getDelta(Iterator dlts) {
    // 戻り値の変数を宣言。
    Delta dlt = null;

    // イテレータに要素がある場合、デルタを取得。
    if (dlts.hasNext()) {
      dlt = (Delta) dlts.next();
    }

    // デルタを返す。
    return dlt;
  }

}