package jp.sourceforge.foolishmerge.merge;

import java.util.Iterator;
import java.util.ListIterator;
import java.util.Set;

import org.apache.commons.jrcs.diff.Delta;

/**
 * 競合する差分情報を保持するクラス。
 */
public final class ConflictDeltas {

  /**
   * デルタのイテレータ1
   */
  private ListIterator dltsIte1 = null;

  /**
   * デルタのイテレータ2
   */
  private ListIterator dltsIte2 = null;

  /**
   * デルタ1のセット
   */
  private Set dlts1 = new DeltaSet();

  /**
   * デルタ2のセット
   */
  private Set dlts2 = new DeltaSet();

  /**
   * 開始ポジション
   */
  private int first = 0;

  /**
   * 開始ポジションのセットフラグ
   */
  private boolean setFirst = false;

  /**
   * 終了ポジション
   */
  private int last = 0;

  /**
   * オブジェクトを生成する。
   * @param dlts1 デルタのイテレータ1
   * @param dlts2 デルタのイテレータ2
   */
  ConflictDeltas(ListIterator dlts1, ListIterator dlts2) {
    // デルタのイテレータをフィールドにセット。
    dltsIte1 = dlts1;
    dltsIte2 = dlts2;
  }

  /**
   * 競合しているデルタをロードする。
   * @param dltsIte
   * @return
   */
  void loadDeltas(Delta delta1, Delta delta2) {
    // いずれかのデルタがnullの場合は処理を抜ける。
    if (delta1 == null || delta2 == null) {
      return;
    }

    // コンフリクトしているかチェック。
    int conflict = Conflict.conflict(delta1, delta2);

    // コンフリクトの内容を判定。
    switch (conflict) {
      case Conflict.CONFLICT_D1_LT_D2 :
        // デルタ1 <= デルタ2の場合
        // デルタをセットに追加。
        dlts1.add(delta1);
        dlts2.add(delta2);
        // 開始/終了ポジションをセット。
        setFirst(delta1);
        setLast(delta2);
        // デルタ1を取得。
        delta1 = getDelta(dltsIte1);
        // 再帰的にメソッドを呼び出す。
        loadDeltas(delta1, delta2);
        break;

      case Conflict.CONFLICT_D1_GT_D2 :
        // デルタ1 >= デルタ2の場合
        // デルタをセットに追加。
        dlts1.add(delta1);
        dlts2.add(delta2);
        // 開始/終了ポジションをセット。
        setFirst(delta2);
        setLast(delta1);
        // デルタ2を取得。
        delta2 = getDelta(dltsIte2);
        // 再帰的にメソッドを呼び出す。
        loadDeltas(delta1, delta2);
        break;

      case Conflict.CONFLICT_D1_EQ_D2 :
        // デルタ1 == デルタ2の場合
        // デルタをセットに追加。
        dlts1.add(delta1);
        dlts2.add(delta2);
        // 開始/終了ポジションをセット。
        setFirst(delta1, delta2);
        setLast(delta1);
        break;

      case Conflict.D1_GT_D2 :
        // デルタ1 > デルタ2の場合
        // イテレータ1をデクリメント。
        dltsIte1.previous();
        break;

      case Conflict.D1_LT_D2 :
        // デルタ1 < デルタ2の場合
        // イテレータ2をデクリメント。
        dltsIte2.previous();
        break;
    }
  }

  /**
   * イテレータからデルタを取得する。
   * @param dlts デルタのイテレータ
   * @return デルタ
   */
  private static Delta getDelta(Iterator dlts) {
    // 戻り値の変数を宣言。
    Delta dlt = null;

    // イテレータに要素がある場合、デルタを取得。
    if (dlts.hasNext()) {
      dlt = (Delta) dlts.next();
    }

    // デルタを返す。
    return dlt;
  }

  /**
   * 開始ポジションをセットする。
   * @param delta デルタ
   */
  private void setFirst(Delta delta) {
    // すでに開始ポジションがセットされている場合は処理を抜ける。
    if (setFirst) {
      return;
    }

    // 開始ポジションをセット。
    first = delta.getOriginal().first();

    // フラグをtrueにする。
    setFirst = true;
  }

  /**
   * 開始ポジションをセットする。
   * @param delta1 デルタ
   * @param delta2 デルタ
   */
  private void setFirst(Delta delta1, Delta delta2) {
    // すでに開始ポジションがセットされている場合は処理を抜ける。
    if (setFirst) {
      return;
    }

    // 開始ポジションをセット。
    int first1 = delta1.getOriginal().first();
    int first2 = delta2.getOriginal().first();
    first = first1 <= first2 ? first1 : first2;

    // フラグをtrueにする。
    setFirst = true;
  }

  /**
   * 終了ポジションをセットする。
   * @param delta デルタ
   */
  private void setLast(Delta delta) {
    // 終了ポジションをセット。
    last = delta.getOriginal().last();
  }

  /**
   * デルタ1のセットを取得する。
   * @return デルタ1のセット
   */
  public Set getDeltas1() {
    // デルタ1のセットを返す。
    return dlts1;
  }

  /**
   * デルタ2のセットを取得する。
   * @return デルタ2のセット
   */
  public Set getDeltas2() {
    // デルタ2のセットを返す。
    return dlts2;
  }

  /**
   * 開始ポジションを取得する。
   * @return 開始ポジション
   */
  public int first() {
    // 開始ポジションを返す。
    return first;
  }

  /**
   * 終了ポジションを取得する。
   * @return 終了ポジション
   */
  public int last() {
    // 終了ポジションを返す。
    return last;
  }

}
