package jp.sourceforge.foolishmerge.merge;

import org.apache.commons.jrcs.diff.Chunk;
import org.apache.commons.jrcs.diff.Delta;

/**
 * コンフリクトをチェックするクラス。
 */
final class Conflict {

  /**
   * コンフリクトなし(デルタ1 &lt; デルタ2)<BR>
   * ┌<BR>
   * │<BR>
   * └<BR>
   * 　┐<BR>
   * 　│<BR>
   * 　┘
   */
  static final int D1_LT_D2 = -2;

  /**
   * コンフリクト(デルタ1 &lt;= デルタ2)<BR>
   * ┌　┌┐　┐<BR>
   * │┐││┌│<BR>
   * └│└│└│<BR>
   * 　┘　┘　┘
   */
  static final int CONFLICT_D1_LT_D2 = -1;

  /**
   * コンフリクト(デルタ1 == デルタ2)<BR>
   * ┌┐┌　　┐<BR>
   * │││┐┌│<BR>
   * ││││││<BR>
   * └┘└┘└┘
   */
  static final int CONFLICT_D1_EQ_D2 = 0;

  /**
   * コンフリクト(デルタ1 &gt;= デルタ2)<BR> 
   * 　┐┌┐┌<BR>
   * ┌││││┐<BR>
   * │┘│┘│┘<BR>
   * └　└　└
   */
  static final int CONFLICT_D1_GT_D2 = 1;

  /**
   * コンフリクトなし(デルタ1 &gt; デルタ2)<BR>
   * 　┐<BR>
   * 　│<BR>
   * 　┘<BR>
   * ┌<BR>
   * │<BR>
   * └
   */
  static final int D1_GT_D2 = 2;

  /**
   * コンフリクトしているか判定する。
   * @param delta1 デルタ1
   * @param delta2 デルタ2
   * @return コンフリクト定数
   */
  static int conflict(Delta delta1, Delta delta2) {
    // いずれかのデルタがnullの場合、コンフリクトなしとする。
    if (delta2 == null) {
      return D1_LT_D2;
    } else if (delta1 == null) {
      return D1_GT_D2;
    }
    // オリジナルのチャンクを取得。
    Chunk chunk1 = delta1.getOriginal();
    Chunk chunk2 = delta2.getOriginal();

    // チャンクを比較。
    return compare(chunk1, chunk2);
  }

  /**
   * コンフリクトしているか判定する。
   * @param chunk1 チャンク1
   * @param chunk2 チャンク2
   * @return コンフリクト定数
   */
  private static int compare(Chunk chunk1, Chunk chunk2) {
    // 戻り値に「コンフリクト(デルタ1 == デルタ2)」をセット。
    // ┌┐ ┌     ┐
    // ││ │┐ ┌│
    // ││ ││ ││
    // └┘ └┘ └┘
    int ret = CONFLICT_D1_EQ_D2;

    if (chunk1.last() < chunk2.first()) {
      // チャンク1の末尾がチャンク2の先頭より前にある場合、
      // 戻り値に「コンフリクトなし(デルタ1 < デルタ2)」をセット。
      // ┌
      // │
      // └
      //   ┐
      //   │
      //   ┘	
      ret = D1_LT_D2;
    } else if (chunk1.first() > chunk2.last()) {
      // チャンク2の末尾がチャンク1の先頭より前にある場合、
      // 戻り値に「コンフリクトなし(デルタ1 > デルタ2)」をセット。
      //   ┐
      //   │
      //   ┘	
      // ┌
      // │
      // └
      ret = D1_GT_D2;
    } else if (chunk1.last() < chunk2.last()) {
      // チャンク1の末尾がチャンク2の末尾より前にある場合、
      // 戻り値に「コンフリクト(デルタ1 <= デルタ2)」をセット。
      // ┌   ┌┐   ┐
      // │┐ ││ ┌│
      // └│ └│ └│
      //   ┘   ┘   ┘
      ret = CONFLICT_D1_LT_D2;
    } else if (chunk1.last() > chunk2.last()) {
      // チャンク2の末尾がチャンク1の末尾より前にある場合、
      // 戻り値に「コンフリクト(デルタ1 >= デルタ2)」をセット。
      //   ┐ ┌┐ ┌
      // ┌│ ││ │┐
      // │┘ │┘ │┘
      // └   └   └
      ret = CONFLICT_D1_GT_D2;
    }

    // コンフリクト定数を返す。
    return ret;
  }

}
