<?php
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2007 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.net
 */
/**
 * @file URI.php
 * @brief URI for php
 * @author <a href="mailto:kent@guarana.cc">ISHITOYA Kentaro</a>
 * @author <a href="mailto:sumi@wakhok.ac.jp">SUMI Masafumi</a>
 * @version $Id: URI.php 2 2007-07-11 10:37:48Z ishitoya $
 * 
 * URI for php
 */
require_once("ficus/lang/Object.php");
require_once("ficus/lang/Types.php");
require_once("ficus/net/IRISyntax.php");
require_once("ficus/net/URISyntax.php");
require_once("ficus/lang/Unicode.php");
require_once("ficus/exception/IllegalArgumentException.php");
require_once("ficus/exception/URISyntaxException.php");
/**
 * @class Ficus_URI
 */
class Ficus_URI extends Ficus_Object {

    /** 
     * @var $uri string Passed plain string.
     */
    protected $uri;

    /**
     * @var $scheme string scheme.
     */
    protected $scheme;

    /**
     * @var $specific string authority.
     */
    protected $authority;

    /**
     * @var string userInfo.
     */
    protected $userInfo;

    /**
     * @var string host.
     */
    protected $host;

    /**
     * @var string port.
     */
    protected $port;

    /**
     * @var $specific string path.
     */
    protected $path;

    /**
     * @var $fragment string query.
     */
    protected $query;

    /**
     * @var $fragment string fragment.
     */
    protected $fragment;

    /**
     * @var $syntax URI syntax.
     */
    protected $syntax; 

    const IRI = 'IRI';
    const URI = 'URI';

    /**
     * Constructor.
     *
     * @param $uri mixed URI to be parsed.
     */
    public function __construct($uri,
                                Ficus_URISyntax $syntax = null,
                                $encoding = Ficus_Unicode::UTF8) {
        $this->uri = rawurldecode($uri);
        if (is_null($syntax)) {
            $this->syntax = Ficus_URISyntax::createInstance();
        } else {
            $this->syntax = $syntax;
        }
        $this->parse();
    }

    /**
     * Validate scheme.
     */
    public function validateScheme() {
        if (!is_null($this->scheme)) {
            $this->syntax->validateScheme($this->scheme);
        }
        if (strcasecmp($this->getDefaultScheme(), $this->getScheme()) !== 0) {
            throw new Ficus_URISyntaxException("Illegal URI scheme : '{$this->getScheme()}' is not '{$this->getDefaultScheme()}'.");
        }
    }

    /**
     * Split specific.
     *
     * @throw Ficus_URISyntaxException invalid URI.
     */
    final protected function splitComponent() {
        list($this->scheme, $this->authority,
             $this->path, $this->query, $this->fragment)
            = $this->syntax->splitComponent($this->uri);
    }

    /**
     * Constructor.
     *
     * @param $uri mixed URI to be parsed.
     */
    static public function createIRI($uri) {
        return new Ficus_URI($uri, Ficus_IRISyntax::createInstance());
    }

    /**
     * Constructor.
     *
     * @param $uri mixed URI to be parsed.
     */
    static public function createURI($uri) {
        return new Ficus_URI($uri, Ficus_URISyntax::createInstance());
    }

    /**
     * Parse URL.
     */
    protected function parse() {
        $this->splitComponent();
        $this->parseAuthority();
        $this->validateScheme();
    }

    /**
     * Parse authority.
     */
    protected function parseAuthority() {
        if (is_null($this->authority) || strlen($this->authority) == 0) {
            return;
        }
        if (preg_match('/^(?:([^@]*)(@))?(.*)(?:(:)([0-9]*))?$/', $this->authority, $matches) == 0) {
            throw new Ficus_URISyntaxException('Bad authority');
        }
        $this->userInfo = isset($matches[2]) && $matches[2] == '@' ? $matches[1] : null;
        $this->host = isset($matches[3]) ? $matches[3] : null;
        $this->port = isset($matches[4]) && $matches[4] == ':' ? $matches[5] : null;
    }

    /**
     * Get default scheme.
     *
     * @return string scheme name.
     */
    public function getDefaultScheme() {
        return strtolower($this->getScheme());
    }

    /**
     * Get scheme.
     *
     * @return string scheme.
     */
    public function getScheme() {
        return $this->scheme;
    }

    /**
     * Get raw scheme.
     *
     * @return string raw scheme.
     */
    public function getRawScheme() {
        return $this->getScheme();
    }

    /**
     * Get scheme specific part.
     *
     * @return string scheme specific part.
     */
    public function getSpecific() {
        $specific = '';
        $specific .= is_null($this->authority) ? '' : '//' . $this->getAuthority();
        $specific .= is_null($this->path) ? '' : $this->getPath();
        $specific .= is_null($this->query) ? '' : '?' . $this->getQuery();
        return $specific;
    }

    /**
     * Get raw scheme specific part.
     *
     * @return string raw scheme specific part.
     */
    public function getRawSpecific() {
        $specific = '';
        $specific .= is_null($this->authority) ? '' : '//' . $this->authority;
        $specific .= is_null($this->path) ? '' : $this->path;
        $specific .= is_null($this->query) ? '' : '?' . $this->query;
        return $specific;
    }

    /**
     * Get scheme authority part.
     *
     * @return string scheme authority part.
     */
    public function getAuthority() {
        if (is_null($this->authority)) {
            return null;
        }
        $ret = '';
        $ret .= is_null($this->userInfo) ? '' : $this->getUserInfo() . '@';
        $ret .= $this->getHost();
        $ret .= is_null($this->port) ? '' : $this->port;
        return $ret;
    }

    /**
     * Get raw authority.
     *
     * @return string raw authority.
     */
    public function getRawAuthority() {
        return $this->authority;
    }

    /**
     * Get user.
     *
     * @return string user.
     */
    final public function getUserInfo() {
        return $this->syntax->encodeUserInfo($this->userInfo);
    }

    /**
     * Get raw user.
     *
     * @return string raw user.
     */
    final public function getRawUserInfo() {
        return $this->userInfo;
    }

    /**
     * Get host.
     *
     * @return string host.
     */
    final public function getHost() {
        return $this->syntax->encodeHost($this->host);
    }

    /**
     * Get host.
     *
     * @return string host.
     */
    final public function getRawHost() {
        return $this->host;
    }

    /**
     * Get port.
     *
     * @return string port.
     */
    final public function getPort() {
        return $this->port;
    }

    /**
     * Get scheme path part.
     *
     * @return string scheme path part.
     */
    public function getPath() {
        return $this->syntax->encodePath($this->path, is_null($this->scheme));
    }

    /**
     * Get raw path.
     *
     * @return string raw path.
     */
    public function getRawPath() {
        return $this->path;
    }

    /**
     * Get query.
     *
     * @return string query.
     */
    public function getQuery() {
        return $this->syntax->encodeQuery($this->query);
    }

    /**
     * Get raw query.
     *
     * @return string raw query.
     */
    public function getRawQuery() {
        return $this->query;
    }

    /**
     * Get fragment.
     *
     * @return string fragment.
     */
    public function getFragment() {
        return $this->syntax->encodeFragment($this->fragment);
    }

    /**
     * Get raw fragment.
     *
     * @return string raw fragment.
     */
    public function getRawFragment() {
        return $this->fragment;
    }

    /**
     * Get raw URI.
     *
     * @return string raw URI.
     */
    public function getRawURI() {
        return $this->uri;
    }

    /**
     * To string.
     *
     * @return String URI.
     */
    public function toString() {
        return $this->uri;
    }

    /**
     * To string.
     *
     * @return String URI.
     */
    public function __toString() {
        return $this->uri;
    }

    /**
     * Get URI.
     *
     * @return string URI.
     */
    public function getURI() {
        return $this->uri;
    }

    /**
     *
     */
    public function toAsciiString() {
        $uri = '';
        $uri .= is_null($this->getScheme()) ? '' : $this->getScheme() . ":";
        $uri .= is_null($this->getAuthority()) ? '' : '//' . $this->getAuthority();
        $uri .= is_null($this->getPath()) ? '' : $this->getPath();
        $uri .= is_null($this->getQuery()) ? '' : '?' . $this->getQuery();
        $uri .= is_null($this->getFragment()) ? '' : '#' . $this->getFragment();
        return new Ficus_URI($uri);
    }

    /**
     * Get URI.
     *
     * @return string URI.
     */
    public function toURI() {
        if ($this->syntax instanceof Ficus_URISyntax) {
            return $this;
        }
        return new Ficus_URI($this->uri);
    }

    /**
     * Convert to IRI.
     *
     * @return Ficus_URI IRI.
     */
    public function toIRI() {
        if ($this->syntax instanceof Ficus_IRISyntax) {
            return $this;
        }
        return new Ficus_URI(rawurldecode($this->uri),
                            Ficus_IRISyntax::createInstance());
    }

    /**
     * Resovle relative URI.
     *
     * @param $relative mixed string or Ficus_URI relative URI.
     */
    public function resolve($uri) {
        Ficus_Assert::typeHinting('Ficus_URI', $uri);

        if ($uri->isAbsolute()) {
            if ($uri instanceof Ficus_HierarchicalURI
             && $this->getScheme() == $uri->getScheme()) {
                 // 3)
                 $uriScheme = null;
            } else {
                return $uri;
            }
        } else {
            $uriScheme = $uri->getRawScheme();
        }


        // add query and fragment.
        if (strlen($uriScheme) == 0
          && is_null($uri->getAuthority())
          && strlen($uri->getPath()) == 0
          && strlen($uri->getQuery()) == 0
          && !is_null($uri->getFragment())) {

              if ($this->getFragment() === $uri->getFragment()) {
                  return $this;
              }

              $uri = Ficus_URIBuilder::createURIStringByParts($this->getRawScheme(),
                                                             $this->getRawAuthority(),
                                                             $this->getRawPath(),
                                                             $this->getRawQuery(),
                                                             $uri->getRawFragment());
              return new self($uri, $this->syntax);
        }

        // 4), 5)
        if ($uri->getAuthority()) {
            // 7)
            $uri = Ficus_URIBuilder::createURIStringByParts($this->getRawScheme(),
                                                           $uri->getRawAuthority(),
                                                           $uri->getRawPath(),
                                                           $uri->getRawQuery(),
                                                           $uri->getRawFragment());
            return new self($uri, $this->syntax);
        } else {

            if (substr($uri->getRawPath(), 0, 1) == '/') {
                $path = $uri->getRawPath();
            } else {
                // 4), 6)
                $path = substr($this->getRawPath(), 0, strlen($this->getRawPath()) - strlen(strrchr($this->getRawPath(), '/')));
                if ($path) {
                    $path = Ficus_Dir::resolve($path, $uri->getRawPath(), '/');
                } else {
                    $path = $uri->getRawPath();
                }
                if ($path{0} != '/') {
                    $path = '/' . $path;
                }
            }


            $uri = Ficus_URIBuilder::createURIStringByParts($this->getRawScheme(),
                                                           $this->getRawAuthority(),
                                                           $path,
                                                           $uri->getRawQuery(),
                                                           $uri->getRawFragment());
            return new self($uri, $this->syntax);
        }

    }

    /**
     * Is absolute URI.
     *
     * @return boolean true if absolute URL.
     */
    public function isAbsolute() {
        return !is_null($this->getScheme());
    }

    /**
     * Normalize URI.
     *
     * @return Ficus_URI URI.
     */
    public function normalize() {
    }

}
?>
