<?php
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2006 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.lang
 */
/**
 * @file Bytes.php
 * @brief byte array
 * @author <a href="mailto:kent@guarana.cc">ISHITOYA Kentaro</a>
 * @version $Id: Bytes.php 2 2007-07-11 10:37:48Z ishitoya $
 * 
 * Byte util
 */

require_once("ficus/lang/Types.php");

/**
 * @class Ficus_Bytes
 */
class Ficus_Bytes
{
    /**
     * @var Array of bytes
     */
    protected $bytes = array();

    /**
     * constructor
     * @param $string data to format bytes
     */
    public function __construct($string = null){
        if(is_null($string)){
            return;
        }
        $data = Ficus_Types::toStringOf($string);
        $bytes = array();
        for($i = 0; $i < strlen($data); $i++){
            $bytes[] = ord($data[$i]);
        }
        $this->bytes = $bytes;
    }

    /**
     * length
     * @return integer length of bytes
     */
    public function length(){
        return count($this->bytes);
    }
    
    /**
     * fill bytes with $pad
     * @param $length integer length of fill
     * @param $byte integer to fill
     * @param Ficus_Bytes filled Byte
     */
    public function fill($length, $byte){
        $bytes = array();
        for($i = 0; $i < $length; $i++){
            $bytes[] = $byte;
        }
        $byte = new Ficus_Bytes();
        $byte->bytes = $bytes;
        return $byte;
    }
    
    /**
     * xor
     * @param $right Ficus_Bytes to xor
     * @return Ficus_Bytes byte
     */
    public function xorWith($right){
        if(count($right->bytes) != count($this->bytes)){
            throw new Ficus_IllegalArgumentException("count of bytes is mismatch");
        }

        $bytes = array();
        foreach($this->bytes as $index => $byte){
            $bytes[] = $byte ^ $right->bytes[$index];
        }
        $byte = new Ficus_Bytes();
        $byte->bytes = $bytes;
        return $byte;
    }

    /**
     * padding
     * @param $length integer padding length with this length
     * @param $pad integer padding integer
     * @return Ficus_Bytes byte object
     */
    public function padding($length, $pad){
        $count = count($this->bytes);
        $bytes = $this->bytes;
        for($i = $count; $i < $length; $i++){
            $bytes[] = $pad;
        }
        $byte = new Ficus_Bytes();
        $byte->bytes = $bytes;
        return $byte;
    }
        
    /**
     * get bytes
     * @return array of bytes
     */
    public function getBytes(){
        return $this->bytes;
    }
    
    /**
     * toString
     * @return string bytes
     */
    public function __toString(){
        $string = "";
        foreach($this->bytes as $byte){
            $string .= chr($byte);
        }
        return $string;
    }
}
?>
