<?php
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2006 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.io
 */
/**
 * @file PathPattern.php
 * @brief path pattern
 * @author <a href="mailto:kent@guarana.cc">ISHTIOYA Kentaro</a>
 * @version $Id: PathPattern.php 2 2007-07-11 10:37:48Z ishitoya $
 * 
 * path pattern to describe pattern
 */

/**
 * @class Ficus_PathPattern
 */
class Ficus_PathPattern
{
    /**
     * @var $includes array include patterns
     */
    protected $includes = array();

    /**
     * @var $excludes array exclude patterns
     */
    protected $excludes = array();

    /**
     * constructor
     * @pattern $defaultExclude boolean true to enable default Exclude pattern
     */
    public function __construct($defaultExclude=true){
        if($defaultExclude){
            $this->addExclude("**/*~");
            $this->addExclude("**/#*#");
            $this->addExclude("**/.#*");
            $this->addExclude("**/%*%");
            $this->addExclude("**/._*");
            $this->addExclude("**/CVS");
            $this->addExclude("**/CVS/**");
            $this->addExclude("**/.cvsignore");
            $this->addExclude("**/SCCS");
            $this->addExclude("**/SCCS/**");
            $this->addExclude("**/vssver.scc");
            $this->addExclude("**/.svn");
            $this->addExclude("**/.svn/**");
        }
    }
    
    /**
     * add include
     * @param $include string include pattern
     * @return Ficus_PathPattern fluent this object
     */
    public function addInclude($include){
        $this->includes[] = $this->convertPatternToRE($include);
        return $this;
    }

    /**
     * add exclude
     * @param $exclude string exclude pattern
     * @return Ficus_PathPattern fluent this object
     */
    public function addExclude($exclude){
        $this->excludes[] = $this->convertPatternToRE($exclude);
        return $this;
    }

    /**
     * match
     * @param $entry string subject
     * @return boolean true if match
     */
    public function match($entry){
        if(empty($this->includes) &&
           empty($this->excludes)){
            return true;
        }

        $matched = false;
        foreach($this->includes as $include){
            if(preg_match($include, $entry)){
                $matched = true;
                break;
            }
        }
        if($matched === false && empty($this->includes) == false){
            return false;
        }
        foreach($this->excludes as $exclude){
            if(preg_match($exclude, $entry)){
                return false;
            }
        }
        return true;
    }

    /**
     * pattern to Reguler expression
     * @param $patten string pattern
     * @return reguler expression
     */
    protected function convertPatternToRE($pattern){
        $pattern = preg_replace('/^\/\*\*/', '**', $pattern);
        $pattern = preg_replace('/^\*\*\//', '**', $pattern);
        $pattern = str_replace(array('.', '/', '~', '^', '$', '(', ')',
                                     '{', '}', '|'),
                               array('\.','\/', '\~', '\^', '\$', '\(', '\)',
                                     '\{', '\}', '\|'),
                               $pattern);
        $pattern = str_replace('**', ".*?", $pattern);
        $pattern = preg_replace('/(?<!(\*|\.))\*(?!\*)/', '[^\/]*', $pattern);
        $pattern = '/^' . $pattern . '$/';
        return $pattern;
    }        
}
?>
