<?php 
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2007 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.db.s2dao.models
 */
/**
 * @file DataAccessObject.php
 * @brief DataAccessObject.
 * @author <a href="mailto:kent@guarana.cc">ISHITOYA Kentaro</a>
 * @version $Id: S2DaoDataAccessObject.php 2 2007-07-11 10:37:48Z ishitoya $
 *
 */

require_once("ficus/db/s2dao/models/S2DaoModelConstants.php");

/**
 * @class Ficus_S2DaoDataAccessObject
 */
class Ficus_S2DaoDataAccessObject implements Ficus_S2DaoModelConstants{
    /**
     * dao object
     */
    protected $dao = null;
    
    /**
     * subject
     */
    protected $subject = "";

    /**
     * constructor
     */
    public function __construct($target){
        if(is_string($target)){
            $this->dao = Ficus_S2DaoComponentFactory::getComponent($target);
            $classname = get_class($this);
        }else{
            $this->dao = $target;
            $classname = get_class($this->dao);
        }
        if(preg_match("/.*?_(Auto)?([^_]+)Dao/", $classname, $regs)){
            $this->subject = $regs[2];
        }else{
            $this->subject = $classname;
        }
    }

    /**
     * get dao instance from argument
     * @param $subject string subject name
     * @param $auto boolean true if auto dto
     * @return DAO class
     */
    protected function getDataAccessObject($subject = null){
        if(is_null($subject)){
            $subject = $this->subject;
        }
        return Ficus_S2DaoComponentFactory::getModelFactory()->getDataAccessObject($subject);
    }

    /**
     * get dto instance from argument
     * @param $subject string subject name
     * @param $auto boolean true if auto dto
     * @return DTO class
     */
    protected function getDto($subject = null){
        if(is_null($subject)){
            $subject = $this->subject;
        }
        return Ficus_S2DaoComponentFactory::getModelFactory()->getDto($subject);
    }

    /**
     * get entity instance from argument
     * @param $subject string subject name
     * @param $auto boolean true if auto dto
     * @return Entity class
     */
    protected function getEntity($subject = null, $auto = false){
        if(is_null($subject)){
            $subject = $this->subject;
        }
        return Ficus_S2DaoComponentFactory::getModelFactory()->getEntity($subject);
    }

    /**
     * get entity class name
     */
    public static function getEntityWithSubject($subject, $auto = false){
        $entity = self::getClassName($subject, $auto,
                                     self::ENTITY_PACKAGE,
                                     self::ENTITY_SUFFIX);
        return Ficus_ClassLoader::load($entity);
    }
    
    /**
     * access dao with dto
     * @param $dto DTO class of condition
     * @return S2Dao_ArrayList result
     */
    public function getWithCondition($dto){
        $entities = $this->dao->getWithConditionList($dto);
        if($entities instanceof S2Dao_ArrayList &&
           $entities->isEmpty() == false){
            foreach($entities as $key => $entity){
                $entities[$key] =
                    $this->castEntity($entity, $this->getEntity());
            }
            return $this->processResult($entities);
        }
        if($entities === 0){
            return new S2Dao_ArrayList();
        }
        return $entities;
    }

    /**
     * cast entity
     * @param $entity Ficus_S2DaoEntity entity
     * @param $class string class
     * @return Ficus_S2DaoEntity casted entity
     */
    protected function castEntity($entity, $class){
        return Ficus_Class::cast($entity, $class);
    }
    
    /**
     * process result
     * @param $results S2Dao_ArrayList 
     * @return S2Dao_ArrayList result
     */
    protected function processResult($results){
        foreach($results as $entity){
            $reader = new Ficus_S2DaoEntityAnnotationReader($entity);
            $properties = $reader->properties();
            foreach($properties as $property){
                if($reader->hasExpand($property)){
                    $expand = $reader->expand($property);
                    $expand = explode(self::EXPAND_SEPARATOR, $expand);
                    foreach($expand as $target){
                        $this->getExpandedEntity($entity, $property, $target);
                    }
                }
            }
        }
        return $results;
    }

    /**
     * get expanded entity
     * @param $entity Ficus_S2DaoEntity TargetEntity
     * @param $property string property
     * @param $target string target 
     */
    protected function getExpandedEntity($entity, $property, $target){
        $dao = $this->getDataAccessObject($target);
        $dto = $this->getDto($target);
        $target = $entity->get($property);
        if(($target instanceof Ficus_S2DaoEntity) == false ||
           $target->hasId() == false){
            return;
        }
        
        $dto->set($property, $target->id());
        $entities = $dao->getWithCondition($dto);
        if($entities->size() == 0){
            return;
        }
        $entity->set($property, $entities);
    }

    /**
     * get new Entity
     */
    public function entity(){
        return $this->getEntity();
    }

    /**
     * get new Dto
     */
    public function dto(){
        return $this->getDto();
    }

    /**
     * call interface function
     */
    public function __call($name, $args){
        $entities = $this->dao->{$name}($args[0]);
        if($entities instanceof S2Dao_ArrayList){
            foreach($entities as $key => $entity){
                $entities[$key] = $this->castEntity($entity, $this->getEntity());
            }
            return $this->processResult($entities);
        }else if($entities instanceof Ficus_S2DaoEntity){
            $list = new S2Dao_ArrayList();
            $list->append($entities);
            return $this->processResult($list);
        }else{
            return new S2Dao_ArrayList();
        }
        throw new Ficus_MethodNotFoundException($e->getMessage());
    }
}
?>
