<?php
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2007 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.parameters.datatypes
 */
/**
 * @file DateTimeParameter.php
 * @brief DateTimeParameter for php
 * @author <a href="mailto:kent@guarana.cc">ISHITOYA Kentaro</a>
 * @author <a href="mailto:sumi@wakhok.ac.jp">SUMI Masafumi</a>
 * @version $Id: DateTimeParameter.php 2 2007-07-11 10:37:48Z ishitoya $
 * 
 * ValidatableSimpleParameter with normailizer.
 * treat param as datetime
 */

require_once("ficus/exception/ValidationException.php");
require_once("ficus/parameters/ValidatableSimpleParameter.php");

/**
 * @class Ficus_DateTimeParameter
 */
class Ficus_DateTimeParameter extends Ficus_ValidatableSimpleParameter
{
	const TYPENAME = "dateTime";
	const PATTERN = '^(-?[0-9]{4})-?(0[1-9]|1[0-2])-?(0[1-9]|[1-2][0-9]|3[0-1])T([0-1][0-9]|2[0-4]):?([0-5][0-9]):?([0-6][0-9])(\.[0-9]*)?(Z|[+\-][[0-5][0-9]:?[0-5][0-9])?$';
    const DATE_W3C = 'Y-m-d\TH:i:sO';
	
	/**
 	 * construct parameter with name, value, and specific validator.
 	 * @param $name string name of this parameter.
 	 * @param $value string value of this parameter.
 	 * @param $validator array array of Ficus_Validator.
 	 */
 	public function __construct($name, $value = null, $validator = array()){
		parent::__construct($name, self::TYPENAME, $value);
		$this->setValidator($validator);
	}

	/**
	 * check param that associated with simple parameter
	 * YYYY-MM-DDThh:mm:ss(Z|mm:ss) or YYYYMMDD
	 * but 1981-11-31 will throw exception, because 11-31 is not exist.
	 * @param $parameter array data
	 * @return array validated param
     * @throw Ficus_ValidationException invalidate parameter.
	 */
	public function validate($parameter){
		$value = trim($parameter["value"]);
		
        if (self::match($value)) {
			parent::validate($parameter);
		}else{
			throw new Ficus_ValidationException("parameter {$this->name()} is not legal dateTime format");
		}
	}

    public function match($str) {
        return empty($str)
            || (ereg(self::PATTERN, $str, $regs) && (integer)$regs[1] !== 0);
    }

    /**
     * Date format of W3C.
     *
     * @return string Date format of W3C.
     */
    public static function dateFormatW3C() {
        return defined('DATE_W3C') ? DATE_W3C: self::DATE_W3C;
    }

    /**
     * Parse date by W3C format.
     *
     * The date time formar must be
     *   YYYY-MM-DDThh:mm:ss.sTZD (eg 1997-07-16T19:20:30.45+01:00)
     * @param $str string string of date.
     * @return int gmt date.
     * @see http://www.w3.org/TR/NOTE-datetime
     */
    public static function parse($str) {
        if (!self::match($str)) {
            throw new Ficus_ValidationException("'$str' is not legal dateTime format.");
        }
        if (function_exists('strptime')) {
            return strtotime($str);
        } else {
            $ret = preg_match('/(-?)([0-9]{4})-([0-1][0-9])-([0-3][0-9])T([0-2][0-9]):([0-6][0-9]):([0-6][0-9])(?:([zZ])|([\\-\\+])([0-2][0-9]):?([0-6][0-9]))/', $str, $matches);
            if ($ret === false || !isset($matches[1])) {
                throw new Ficus_ValidationException("parameter $str is not legal dateTime format");
            }
            //preg_match('/' . self::PATTERN . '/', $str, $matches);
            $minus = $matches[1];
            $year = intval($matches[2]);
            $month = intval($matches[3]);
            $day = intval($matches[4]);
            $hour = intval($matches[5]);
            $min = intval($matches[6]);
            $sec = intval($matches[7]);
            if (sizeof($matches) >= 9 && strtoupper($matches[8]) == 'Z') {
                $timezone = 'Z';
                $plusminus = 1;
                $timezoneHour = 0;
                $timezoneMin = 0;
            } else {
                $timezone = null;
                $plusminus = $matches[9] == '+' ? 1 : -1;
                $timezoneHour = intval($matches[10]);
                $timezoneMin = intval($matches[11]);
            }
            $tz = date_default_timezone_get();
            date_default_timezone_set('utc');
            $local = mktime($hour, $min, $sec, $month, $day, $year);
            date_default_timezone_set($tz);
            $gmt = $local + -1 * $plusminus * (($timezoneHour * 60 + $timezoneMin) * 60);
            return $gmt;
        }
    }

    /**
     * Date format.
     *
     * @param $date int Date.
     * @return string formated date.
     */
    public static function format($date = null) {
        $formated =  is_null($date) ? date(self::dateFormatW3C()) : date(self::dateFormatW3C(), $date);
        return $formated;
    }

}
?>
