<?php 
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2007 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.pages
 */
/**
 * @file PageController.php
 * @brief controll page
 * @author <a href="mailto:kent@guarana.cc">ISHITOYA Kentaro</a>
 * @version $Id: PageController.php 2 2007-07-11 10:37:48Z ishitoya $
 */

require_once("smarty/Smarty.class.php");
require_once("ficus/pages/PageModeExtractor.php");
require_once("ficus/lang/ClassLoader.php");
require_once("ficus/exception/ClassNotFoundException.php");
require_once("ficus/exception/PageNotFoundException.php");
require_once("ficus/lang/Class.php");
require_once("ficus/lang/String.php");
require_once("ficus/pages/PageComponentFactory.php");
require_once("ficus/pages/PageConstants.php");

/**
 * @class Ficus_PageController
 */
class Ficus_PageController implements Ficus_PageConstants{
    /**
     * cache dir
     */
    protected $cacheDir = null;
    /**
     * base dir
     */
    protected $baseDir = null;
    /**
     * Smarty
     */
    protected $smarty = null;
    /**
     * pages
     */
    protected $pages = array();
    /**
     * mode extractor
     */
    protected $extractor = null;

    /**
     * set base dir
     */
    public function setBaseDir($base){
        $this->baseDir = $base;
    }

    /**
     * call method
     * @param $name string name of page
     * @param $mode string name of mode
     * @param $buffering boolean buffering
     * @return mixed return value of method
     */
    public function execute($name = null, $mode = null, $buffering = false){
        Ficus_PageComponentFactory::getTemplateLoader();
        $request = new Ficus_PageRequest(func_get_args());
        if(is_null($name)){
            $name =
                Ficus_PageComponentFactory::getPageLoader()->dispatch($request);
        }
        if($this->isPageExists($name) == false){
            $this->addPage($name);
        }

        if(is_null($mode)){
            $mode = $this->getMode($name);
        }

        $page = $this->pages[$name]["page"];

        if($page->authorization() == self::FORWARDED){
            return;
        }
        if($buffering == true){
            try{
                Ficus_Assert::isInstanceOf($page, "Ficus_BufferedPage");
            }catch(Ficus_TypeMismatchException $e){
                throw new Ficus_IllegalArgumentException("out buffering is on, but page has not ability to buffer.");
            }
            $page->enableOutputBuffering();
        }else{
            if($page instanceof Ficus_BufferedPage){
                $page->disableOutputBuffering();
            }
        }
        return call_user_func(array($page, $mode), $request);
    }

    /**
     * get mode from extractor
     * @param $name String name of page
     * @return String of mode name
     */
    protected function getMode($name){
        $mode = $this->pages[$name]["extractor"]->getMode();
        return $mode;
    }
    
    /**
     * add a page to pages array
     * @param $name String name of page
     */
    protected function addPage($name){
        $page = Ficus_PageComponentFactory::getPageLoader()->load($name);
        $extractor = new Ficus_PageModeExtractor($page);
        $extractor->setDefaultMode($page->getDefaultMode());
        $this->pages[$name]["page"] = $page;
        $this->pages[$name]["extractor"] = $extractor;
    }

    /**
     * checks page exist in pages array
     * @param $name String name of page
     * @return boolean return true if page exists
     */
    protected function isPageExists($name){
        return array_key_exists($name, $this->pages);
    }


    /**
     * set Cache dir specify this in full path.
     * @param $cacheDir string cache dir
     */
    public function setCacheDir($cacheDir){
        $this->cacheDir = $cacheDir;
        if(is_dir($this->cacheDir) == false){
            Ficus_Dir::mkdir($this->cacheDir);
        }
        $smarty = Ficus_PageComponentFactory::getSmarty();
        $smarty->compile_dir = $this->cacheDir;
        $smarty->compile_id = basename($this->baseDir);
    }
}
?>