<?php
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2007 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.io
 */
/**
 * @file File.php
 * @brief File path utility.
 * @author <a href="mailto:sumi@wakhok.ac.jp">SUMI Masafumi</a>
 * @version $Id: File.php 2 2007-07-11 10:37:48Z ishitoya $
 * 
 * File utility.
 */
require_once('ficus/io/Dir.php');
require_once('ficus/exception/MethodNotFoundException.php');
require_once('ficus/lang/ClassPath.php');
require_once('mime_magic/class.mime_magic.inc.php');
/**
 * @class Ficus_File
 */
class Ficus_File {

    /**
     * @var string path.
     */
    private $path;

    /**
     * Constructor.
     *
     * @param $path string path.
     */
    public function __construct($path = '') {
        $this->path = $path;
    }

    /**
     * Get path.
     *
     * @return string path.
     */
    public function getPath() {
        return $this->path;
    }

    /**
     * Get file name
     *
     * @return string file name
     */
    public function getFilename(){
        return basename($this->path);
    }

    /**
     * Get extension
     */
    public function getExtension(){
        if(preg_match('/\.([^\.]+)$/', $this->path, $regs)){
            return $regs[1];
        }
        return null;
    }          

    /**
     * Get Dir
     *
     * @return string dir name
     */
    public function getDirname(){
        return dirname($this->path);
    }

    /**
     * is Dir contains
     *
     * @param $pattern string pattern
     * @return boolean dir name contains patten
     */
    public function dirnameContains($pattern){
        $result = strstr($this->getDirname(), $pattern);
        if($result != false){
            return true;
        }
        return false;
    }

    /**
     * To string.
     *
     * @return string path.
     */
    public function toString() {
        return $this->getPath();
    }

    /**
     * To string.
     *
     * @return string path.
     */
    public function __toString() {
        return $this->getPath();
    }

    /**
     * Call PHP filesystem function.
     *
     * @param $name string function name.
     * @param $arguments function arguments.
     * @return mixed function return value.
     * @throw Ficus_MethodNotFoundException method not found.
     */
    public function __call($name, $arguments) {
        switch ($name) {

            case 'basename':
            case 'chgrp':
            case 'chmod':
            case 'chown':
            case 'copy':
            //case 'delete':
            case 'dirname':
            case 'file_exists':
            case 'file_get_contents':
            case 'file_put_contents':
            case 'file':
            case 'fileatime':
            case 'filectime':
            case 'filegroup':
            case 'fileinode':
            case 'filemtime':
            case 'fileperms':
            case 'filesize':
            case 'filetype':
            case 'is_dir':
            case 'is_executable':
            case 'is_file':
            case 'is_link':
            case 'is_readable':
            case 'is_uploaded_file':
            case 'is_writable':
            case 'is_writeable':
            case 'link':
            case 'linkinfo':
            case 'lstat':
            case 'mkdir':
            case 'move_uploaded_file':
            case 'parse_ini_file':
            case 'pathinfo':
            case 'readfile':
            case 'readlink':
            case 'realpath':
            case 'rename':
            case 'rmdir':
            case 'stat':
            case 'symlink':
            case 'tempnam':
            case 'touch':
            case 'unlink':
            if (is_array($arguments)) {
                $argumentsWithPath = array_merge(array($this->path), $arguments);
            } else {
                $argumentsWithPath = array($this->path);
            }
            return call_user_func_array($name, $argumentsWithPath);
            break;

            // path is not first argument.
            case 'fnmatch':
            $argumentsWithPath = array_merge(array($arguments[0], $this->path), array_slice($arguments, 1));
            return call_user_func_array($name, $argumentsWithPath);
            break;

            // don't use path.
            case 'glob':
            case 'tempfile':
            return call_user_func_array($name, $arguments);
            break;

        }
        throw new Ficus_MethodNotFoundException($name);
    }

    /**
     * Normailze path separator.
     *
     * @param $separator string directory separator.
     * @return Ficus_File this.
     */
    public function normalizePathSeparator($separator = null) {
        return new Ficus_File(Ficus_Dir::normalizePathSeparator($this->path, $separator));
    }

    /**
     * Normailze path.
     *
     * @param $separator string directory separator.
     * @return Ficus_File this.
     */
    public function normalize($separator = null) {
        return new Ficus_File(Ficus_Dir::normalize($this->path, $separator));
    }

    /**
     * Resolve path.
     *
     * @param $relative string relative path.
     * @param $separator string directory separator.
     * @return Ficus_File this.
     */
    public function resolve($relative, $separator = null) {
        return new Ficus_File(Ficus_Dir::resolve($this->path, $relative, $separator));
    }

    /**
     * Current dir.
     *
     * @return Ficus_File current dir.
     */
    public static function currentDir() {
        $trace = debug_backtrace();
        $class = new ReflectionClass($trace[1]["class"]);
        return new Ficus_File(dirname($class->getFileName()));
    }

    /**
     * Current dir.
     *
     * @return string current dir.
     */
    public static function currentDirStr() {
        $trace = debug_backtrace();
        $class = new ReflectionClass($trace[1]["class"]);
        return dirname($class->getFileName());
    }

    /**
     * Current dir.
     *
     * alias currentDir().
     *
     * @return Ficus_File current dir.
     */
    public static function pwd() {
        $trace = debug_backtrace();
        $class = new ReflectionClass($trace[1]["class"]);
        return new Ficus_File(dirname($class->getFileName()));
    }

    /**
     * Current dir.
     *
     * alias currentDir().
     *
     * @return string current dir.
     */
    public static function pwdStr() {
        $trace = debug_backtrace();
        $class = new ReflectionClass($trace[1]["class"]);
        return dirname($class->getFileName());
    }

    /**
     * Current PHP file.
     *
     * @return Ficus_File current PHP file.
     */
    public static function currentFile() {
        $trace = debug_backtrace();
        $class = new ReflectionClass($trace[1]["class"]);
        return new Ficus_File($class->getFileName());
    }

    /**
     * Current PHP file.
     *
     * @return string current PHP file.
     */
    public static function currentFileStr() {
        $trace = debug_backtrace();
        $class = new ReflectionClass($trace[1]["class"]);
        return $class->getFileName();
    }

    /**
     * Create File.
     *
     * @param $path string path.
     * @return Ficus_File file.
     */
    public static function create($path) {
        return new Ficus_File($path);
    }

    /** 
     * Check if a file exists in the include path 
     * 
     * @version     1.2.1 
     * @author      Aidan Lister <aidan@php.net> 
     * @link        http://aidanlister.com/repos/v/function.file_exists_incpath.php 
     * @param       string     $file       Name of the file to look for 
     * @return      mixed      The full path if file exists, FALSE if it does not 
     */ 
    public static function fileExistsInIncludePath ($file) { 
          $paths = explode(PATH_SEPARATOR, get_include_path()); 
          foreach ($paths as $path) { 
              // Formulate the absolute path 
              $fullpath = $path . DIRECTORY_SEPARATOR . $file; 
              // Check it 
              if (file_exists($fullpath)) { 
                  return $fullpath; 
              } 
          } 
          
          return false; 
      } 

    /** 
     * Check if a file exists in the classpath
     * 
     * @param       string     $file       Name of the file to look for 
     * @return      mixed      The full path if file exists, FALSE if it does not 
     */ 
    public static function fileExistsInClassPath ($file) { 
        $paths = Ficus_ClassPath::search($file);
        if(empty($paths)){
            return false;
        }
        return $paths[0];
    }

    /**
     * get mime type from filename
     */
    public function mimetype(){
        $mime = new mime_magic();
        return $mime->filename2mime($this->path);
    }

    /**
     * add
     */
    public function add($path){
        $this->path .= $path;
    }
}
?>
